/* This file performs the MOUNT and UMOUNT system calls.
 *
 * The entry points into this file are
 *   do_mount:	perform the MOUNT system call
 *   do_umount:	perform the UMOUNT system call
 */

#include "fs.h"
#include <minix/com.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include "buf.h"
#include "assert.h"
INIT_ASSERT
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "name.h"
#include "param.h"
#include "super.h"

FORWARD _PROTOTYPE( int super_type, (struct super_block *sp)		);
FORWARD _PROTOTYPE( dev_t ip_to_dev, (struct inode *ip, int *err_p)	);


/*===========================================================================*
 *				do_mount				     *
 *===========================================================================*/
PUBLIC int do_mount()
{
/* Perform the mount(name, mfile, flags) system call. */

  register struct inode *rip, *rip2, *root_ip;
  struct super_block *sp, *dsp;
  struct mount *mp;
  struct buf *bp;
  dev_t dev;
  int r, loaded, bits;
  int flags, type, devtype;
  int fs_err= EGENERIC;

  assert(fs_call == MOUNT);

  flags = mount_flags & M_FLAGS;
  type = mount_flags & M_TYPE;
  if (type == 0) type = M_DEV;		/* Old style mount. */

  /* Only the super-user may do MOUNT. */
  if (!super_user) return(EPERM);

  if ((fs_err= fetch_name(mount_name, mount_namelen, M1)) != OK)
  	return(fs_err);

  /* Everything starts out OK, with no resources allocated. */
  r = OK;
  loaded = FALSE;
  root_ip = rip = NIL_INODE;
  dev = NO_DEV;
  sp = NIL_SUPER;

  if (type != M_LO) {
	/* If 'name' is not for a block special file, return error. */
	if ((rip2 = eat_path(user_path, &fs_err)) == NIL_INODE)
		return(fs_err);
	if ((dev = ip_to_dev(rip2, &fs_err)) == NO_DEV)
		return(fs_err);

	/* Scan super block table to see if dev already mounted. */
	sp = search_super(dev);
  }

  if (sp != NIL_SUPER) {
	/* Already mounted, then this must be a remount. */
	if (!(flags & M_REMOUNT)) return(EBUSY);

	/* Change the flags, but not to read only. */
	if (!sp->s_rd_only && (flags & M_RDONLY)) return(EINVAL);

	if (sp->s_rd_only && !(flags & M_RDONLY)) {
		/* Changing from ro to rw.  Try to reopen the device in
		 * rw mode, and close the ro mode.  See if the file system
		 * is now clean.
		 */
		r = dev_opcl(1, dev, FS_PROC_NR, R_BIT|W_BIT);
		if (r != OK) return(r);
		(void) dev_opcl(0, dev, FS_PROC_NR, 0);

		bp = get_block(sp->s_dev, SUPER_BLOCK, BF_NORMAL);
		dsp = (struct super_block *) &bp->b_data;
		sp->s_flags = dsp->s_flags;
		sp->s_fsck_magic[0] = dsp->s_fsck_magic[0];
		sp->s_fsck_magic[1] = dsp->s_fsck_magic[1];

		if (!(sp->s_flags & S_CLEAN)) {
			/* If its not clean now, then don't mark it clean
			 * later.
			 */
			sp->s_fsck_magic[0] = !FSCK_MAGIC0;
			put_block(bp);
		} else {
			/* If the filesystem is both read/write and clean,
			 * then we have to keep the superblock around to be
			 * able to mark it dirty.
			 */
			assert(sp->s_superblock == NULL);
			sp->s_superblock = bp;
		}
	}

	sp->s_rd_only = (flags & M_RDONLY) != 0;
	sp->s_nosuid = (flags & M_NOSUID) != 0;
	sp->s_grpid = (flags & M_GRPID) != 0;
	sp->s_nf = (flags & M_NF) != 0;

	return(super_type(sp) | flags);
  }

  /* It can't be a remount if not already mounted. */
  if (flags & M_REMOUNT) return(EINVAL);

  /* Find a free slot in the mount table. */
  for (mp = mount_table; mp < &mount_table[NR_MOUNTS]; mp++) {
	if (mp->m_imount == NIL_INODE) break;
  }
  if (mp == &mount_table[NR_MOUNTS]) return(ENFILE); /* no entry available */
  if (strlen(user_path) > sizeof(mp->m_special)-1) return(ENAMETOOLONG);
  strcpy(mp->m_special, user_path);

  if (type != M_LO) {
	/* Find a free superblock. */
	sp = search_super(NO_DEV);
	if (sp == NIL_SUPER) return(ENFILE);	/* no super block available */

	bits = (flags & M_RDONLY) ? R_BIT : R_BIT|W_BIT;
	if ((r = dev_opcl(1, dev, FS_PROC_NR, bits)) != OK) return(r);

	/* Fill in the super block. */
	sp->s_dev = dev;	/* read_super() needs to know which dev */
	read_super(sp);

	/* Find out the device type. */
	devtype = super_type(sp);
	if (type == M_DEV) type = devtype;	/* "open" mount */

	/* Check the type and a few other key values to see if super block
	 * looks reasonable.
	 */
	if (devtype == 0 || devtype != type
			|| sp->s_imap_blocks < 1 || sp->s_zmap_blocks < 1
				|| sp->s_ninodes < 1 || sp->s_zones < 1 )
		r = EINVAL;

	/* Get the root inode of the mounted file system. */
	if (r == OK && (root_ip = get_inode(dev, ROOT_INODE, &fs_err)) == NIL_INODE)
	{
		r = fs_err;
		assert(fs_err != EGENERIC);
		fs_err= EGENERIC;
	}

	if (r == OK && root_ip->i_mode == 0) r = EINVAL;
  } else {
	/* Get the "root" inode to be the loopback mounted. */
	if ((root_ip = eat_path(user_path, &fs_err)) == NIL_INODE)
		return(fs_err);
  }

  /* Now get the inode of the file to be mounted on. */
  if (r == OK && (fs_err= fetch_name(mount_name2, mount_namelen2, M1)) != OK)
  {
  	r = fs_err;
	assert(fs_err != EGENERIC);
	fs_err= EGENERIC;
  }

  if (r == OK && strlen(user_path) > sizeof(mp->m_mountp)-1) r = ENAMETOOLONG;
  if (r == OK) strcpy(mp->m_mountp, user_path);

  if (r == OK && (rip = eat_path(user_path, &fs_err)) == NIL_INODE)
  {
  	r = fs_err;
	assert(fs_err != EGENERIC);
	fs_err= EGENERIC;
  }
  
  /* Load the i-node and zone bit maps from the new device. */
  if (r == OK && type != M_LO) {
	if (load_bit_maps(dev) != OK) r = ENFILE;	/* load bit maps */
	if (r == OK) loaded = TRUE;
  }
  
  /* You can only mount a directory on a directory. */
  if (r == OK && ((root_ip->i_mode & I_TYPE) == I_DIRECTORY) &&
	((rip->i_mode & I_TYPE) != I_DIRECTORY)) r = ENOTDIR;

  /* If error, return the super block and both inodes; release the maps. */
  if (r != OK) {
	put_inode(rip);
	put_inode(root_ip);

	if (type != M_LO) {
		if (loaded) (void) unload_bit_maps(dev);
		flushall_inodes(dev);
		flushall(dev);
		invalidate_inodes(dev);
		invalidate(dev);

		(void) dev_opcl(0, dev, FS_PROC_NR, 0);
		sp->s_dev = NO_DEV;
	}
	return(r);
  }

  /* Nothing else can go wrong.  Perform the mount. */
  rip->i_mounted_by = mp->m_iroot = root_ip;
  root_ip->i_mounted_on = mp->m_imount = rip;    /* Inodes connected. */

  if (type != M_LO) {
	sp->s_rd_only = (flags & M_RDONLY) != 0;
	sp->s_nosuid = (flags & M_NOSUID) != 0;
	sp->s_grpid = (flags & M_GRPID) != 0;
	sp->s_nf = (flags & M_NF) != 0;

	/* If its not clean now, then don't mark it clean later. */
	if (!(sp->s_flags & S_CLEAN)) sp->s_fsck_magic[0] = !FSCK_MAGIC0;

	/* If the filesystem is clean but not readonly, then we have to keep
	 * the superblock around to be able to mark it dirty.
	 */
	if (!sp->s_rd_only && (sp->s_flags & S_CLEAN)) {
		sp->s_superblock = get_block(sp->s_dev, SUPER_BLOCK, BF_NORMAL);
	} else {
		sp->s_superblock = NULL;
	}
  }

  if (mount_flags <= M_RDONLY) return(OK);	/* old mount */
  return(type | flags);
}


/*===========================================================================*
 *				do_umount				     *
 *===========================================================================*/
PUBLIC int do_umount()
{
/* Perform the umount(name) system call. */

  register struct inode *rip;
  struct mount *mp;
  dev_t dev;
  int fs_err= EGENERIC;

  assert(fs_call == UMOUNT);

  /* Only the super-user may do UMOUNT. */
  if (!super_user) return(EPERM);

  if ((fs_err= fetch_name(umount_name, umount_namelen, M3)) != OK)
  	return(fs_err);
  if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) return(fs_err);

  if (rip->i_mounted_on == NIL_INODE) {
	/* If rip is not mounted on something then it must be a device. */

	/* If 'name' is not for a block special file, return error. */
	if ( (dev = ip_to_dev(rip, &fs_err)) == NO_DEV) return(fs_err);
	for (mp = mount_table; mp < &mount_table[NR_MOUNTS]; mp++) {
		rip = mp->m_iroot;
		if (rip != NIL_INODE && rip->i_dev == dev
					&& rip->i_num == ROOT_INODE) break;
	}
	if (mp == &mount_table[NR_MOUNTS]) return(EINVAL);  /* not mounted */
  } else {
  	/* The unmount is done from the mount point. */
	for (mp = mount_table; mp < &mount_table[NR_MOUNTS]; mp++) {
		if (mp->m_iroot == rip) break;
	}
	assert(mp != &mount_table[NR_MOUNTS]);
	put_inode(rip);
  }

  return(unmount(mp));
}


/*===========================================================================*
 *				unmount					     *
 *===========================================================================*/
PUBLIC int unmount(mp)
struct mount *mp;
{
/* Do the actual unmount.  This routine is also called during a reboot. */

  register struct inode *rip;
  struct super_block *sp;
  int count;

  if (mp->m_iroot->i_num == ROOT_INODE) {
	sp = mp->m_iroot->i_sp;

	/* See if the mounted device is busy.  Only 1 inode using it should be
	 * open -- the root inode -- and that inode only 1 time (usually).
	 */
	count = 0;
	for (rip = &inode_table[0]; rip < &inode_table[NR_INODES]; rip++) {
		if (rip->i_count > 0 && rip->i_dev == sp->s_dev) {
			count += rip->i_count;
		}
	}
	if (mp->m_iroot->i_mounted_by != NIL_INODE) count--; /* layered mount */

	if (count > 1) return(EBUSY);	/* can't unmount a busy file system */

	/* Pipes can no longer be allocated on this file system. */
	if (pipe_fs == sp) {
		pipe_fs = &super_block[0];
		if (pipe_fs == sp) pipe_fs = NULL;
	}
  } else {
	sp = NIL_SUPER;		/* loopback mount */
  }

  if (mp->m_iroot->i_mounted_by != NIL_INODE) {	/* layered mount */
	mp->m_iroot->i_mounted_by->i_mounted_on = mp->m_imount;
	put_inode(mp->m_iroot);
	dup_inode(mp->m_imount);
  }

  mp->m_imount->i_mounted_by = mp->m_iroot->i_mounted_by;

  /* mp->m_iroot is disconnected. */
  mp->m_iroot->i_mounted_on = mp->m_iroot->i_mounted_by = NIL_INODE;

  put_inode(mp->m_imount);		/* release the inode mounted on */
  put_inode(mp->m_iroot);		/* release the inode mounted on it */
  mp->m_imount = mp->m_iroot = NIL_INODE;

  if (sp != NIL_SUPER) {
	/* Release the bit maps, sync the disk, mark clean, and invalidate
	 * caches.
	 */
	if (unload_bit_maps(sp->s_dev) != OK) panic("unmount", sp->s_dev);
	flushall_inodes(sp->s_dev);
	flushall(sp->s_dev);		/* flush cached blocks */

	super_clean(sp, TRUE);
	if (sp->s_superblock)
	{
		put_block(sp->s_superblock);
		sp->s_superblock= NULL;
	}
	invalidate_inodes(sp->s_dev);	/* invalidate cached inodes */
	invalidate(sp->s_dev);		/* invalidate cached blocks */
	flush_names(sp->s_dev, 0);	/* flush cached names */

	(void) dev_opcl(0, sp->s_dev, FS_PROC_NR, 0);

	sp->s_dev = NO_DEV;
  }
  return(OK);
}


/*===========================================================================*
 *				super_type				     *
 *===========================================================================*/
PRIVATE int super_type(sp)
struct super_block *sp;
{
/* Translate a magic number to a device type, 0 on failure. */

  switch (sp->s_version) {
  case 1:	return(sp->s_flags & S_FLEX ? M_V1F : M_V1);
  case 2:	return(sp->s_flags & S_FLEX ? M_V2F : M_V2);
  }
  return(0);
}

/*===========================================================================*
 *				ip_to_dev				     *
 *===========================================================================*/
PRIVATE dev_t ip_to_dev(rip, err_p)
register struct inode *rip;
int *err_p;				/* pointer to error variable */
{
/* Convert the block special file 'rip' to a device number.  If 'rip'
 * is not a block special file, return error code in '*err_p'.
 */

  register dev_t dev;

  assert(rip != NIL_INODE);

  /* If 'rip' is not a block special file, return error. */
  if ( (rip->i_mode & I_TYPE) != I_BLOCK_SPECIAL) {
	put_inode(rip);
	*err_p = ENOTBLK;
	return(NO_DEV);
  }

  /* Extract the device number. */
  dev = (dev_t) rip->i_zone[0];
  put_inode(rip);
  return(dev);
}

/*
 * $PchId: mount.c,v 1.6 1995/11/28 07:56:58 philip Exp $
 */
