/* This file manages the inode table.  There are procedures to allocate and
 * deallocate inodes, acquire, erase, and release them, and read and write
 * them from the disk.
 *
 * The entry points into this file are
 *   get_inode:	   search inode table for a given inode; if not there, read it
 *   put_inode:	   indicate that an inode is no longer needed in memory
 *   alloc_inode:  allocate a new, unused inode
 *   free_inode:   mark an inode as available for a new file
 *   update_times: update atime, ctime, and mtime
 *   rw_inode:	   read a disk block and extract an inode, or corresp. write
 *   old_icopy:	   copy to/from in-core inode struct and disk inode (V1.x)
 *   new_icopy:	   copy to/from in-core inode struct and disk inode (V2.x)
 *   dup_inode:	   indicate that someone else is using an inode table entry
 */

#include "fs.h"
#include <sys/stat.h>
#include "assert.h"
INIT_ASSERT
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "super.h"

FORWARD _PROTOTYPE( void old_icopy, (struct inode *rip, d1_inode *dip,
						int direction, int norm));
FORWARD _PROTOTYPE( void new_icopy, (struct inode *rip, d2_inode *dip,
						int direction, int norm));


#define hash_inode(var, dev, num) \
	(*(var)= (dev) ^ (num), \
	*(var)= (*(var) ^ (*(var) >> 8)) & (NR_INODE_HASH-1))

/*===========================================================================*
 *				get_inode				     *
 *===========================================================================*/
PUBLIC struct inode *get_inode(dev, numb, err_p)
dev_t dev;			/* device on which inode resides */
int numb;			/* inode number (ANSI: may not be unshort) */
int *err_p;			/* pointer to error variable */
{
/* Find a slot in the inode table, load the specified inode into it, and
 * return a pointer to the slot.
 */
  static struct inode *next_inode= &inode_table[0];

  register struct inode *rip, *prev, *curr;
  int h, h1, i;

  assert(dev != NO_DEV);

  hash_inode(&h, dev, numb);

  /* Search the inode table for (dev, numb). */
  for (rip = inode_hash[h]; rip != NULL; rip= rip->i_next) {
	if (rip->i_num == numb && rip->i_dev == dev)
	{
		/* This is the inode that we are looking for. */
		rip->i_count++;
		return(rip);	/* (dev, numb) found */
	}
  }

  /* Search the inode table for a free slot. */
  for (i= 0, rip = next_inode; i < 2 * NR_INODES; i++, rip++) {
  	if (rip == &inode_table[NR_INODES])
  		rip= &inode_table[0];
  	if (rip->i_used)
  	{
  		rip->i_used= 0;
  		continue;
  	}
  	if (rip->i_count)
  		continue;
  	if (rip->i_dev != NO_DEV)
  	{
		if (rip->i_dirt != CLEAN) rw_inode(rip, WRITING);
		hash_inode(&h1, rip->i_dev, rip->i_num);
		for (curr= inode_hash[h1], prev= NULL;
			curr != rip && curr != NULL;
			prev= curr, curr= curr->i_next)
		{
			; /* Do nothing. */
		}
		assert(curr == rip);
		if (prev == NULL)
			inode_hash[h1]= curr->i_next;
		else
			prev->i_next= curr->i_next;

		rip->i_dev= NO_DEV;
		rip->i_sp= NULL;
		rip->i_next= NULL;
  	}
	break;
  }

  /* Did we find a free slot? */
  if (i == 2 * NR_INODES) {	/* inode table completely full */
	*err_p = ENFILE;
	return(NIL_INODE);
  }

  /* A free inode slot has been located.  Load the inode into it. */
  rip->i_dev = dev;
  rip->i_num = numb;
  rip->i_count = 1;
  rip->i_used = 0;
  rip->i_sp= get_super(dev);
  rip->i_mounted_on = rip->i_mounted_by = NIL_INODE;
  rw_inode(rip, READING);	/* get inode from disk */
  rip->i_update = 0;		/* all the times are initially up-to-date */
  rip->i_next= inode_hash[h];
  inode_hash[h]= rip;

  /* Remember where next to search. */
  next_inode= rip+1;

  return(rip);
}


/*===========================================================================*
 *				put_inode				     *
 *===========================================================================*/
PUBLIC void put_inode(rip)
register struct inode *rip;	/* pointer to inode to be released */
{
/* The caller is no longer using this inode.  If no one else is using it either
 * and it has no links then truncate it.
 */

  if (rip == NIL_INODE) return;	/* checking here is easier than in caller */
  rip->i_used++;
  assert(rip->i_count > 0);
  if (--rip->i_count > 0)
  	return;
  /* i_count == 0 means no one is using it now */
  assert(rip->i_lock == NULL);
  if (rip->i_nlinks == 0) {
	/* Not present in the file system, so free the inode. */
	freespace(rip, 0, MAX_FILE_POS);/* return all the disk blocks */
	rip->i_size = 0;
	rip->i_mode = I_NOT_ALLOC;	/* clear I_TYPE field */
	free_inode(rip);
  } else {
	/* Free pipe blocks unless the underlying file system is read-only. */
	if (S_ISFIFO(rip->i_mode) && !rip->i_sp->s_rd_only) {
		freespace(rip, 0, MAX_FILE_POS);
		rip->i_size = 0;
	}
	if (rip->i_update) update_times(rip);
  }
}

/*===========================================================================*
 *				inode_vrfy_ptr				     *
 *===========================================================================*/
PUBLIC struct inode *inode_vrfy_ptr(rip, dev, num)
register struct inode *rip;	/* pointer to inode to be checked */
dev_t dev;
ino_t num;
{
	int fs_err;

	/* Follow an old inode pointer to see if the same inode as before is
	 * still there.  Otherwise get the proper inode.
	 */

	if (rip != NULL && rip->i_dev == dev && rip->i_num == num)
	{
		rip->i_count++;
		return rip;
	}
#if DEBUG & 0
 { where(); printf("inode_vrfy_ptr: 0x%x, %d\n", dev, num); }
#endif
	return get_inode(dev, num, &fs_err);
}


/*===========================================================================*
 *				alloc_inode				     *
 *===========================================================================*/
PUBLIC struct inode *alloc_inode(sp, bits, err_p)
struct super_block *sp;		/* filesystem on which to allocate the inode */
mode_t bits;			/* mode of the inode */
int *err_p;			/* pointer to error variable */
{
/* Allocate a free inode on 'dev', and return a pointer to it. */

  register struct inode *rip;
  int major, minor, inumb;
  bit_t b, bit;
  int i, fs_err= EGENERIC;

  if (sp == NULL)
  {
	*err_p= EINVAL;
	return NIL_INODE;
  }
  if (sp->s_rd_only) {	/* can't allocate an inode on a read only device. */
	*err_p = EROFS;
	return(NIL_INODE);
  }

  /* Acquire an inode from the bit map. */
  bit = sp->s_isearch;		/* start at first unused inode */
  b = alloc_bit(sp, MP_INODE, bit);
  if (b == NO_BIT) {
	major = (int) (sp->s_dev >> MAJOR) & BYTE;
	minor = (int) (sp->s_dev >> MINOR) & BYTE;
	printf("Out of i-nodes on %sdevice %d/%d\n",
		sp->s_dev == root_dev ? "root " : "", major, minor);
	*err_p = ENFILE;
	return(NIL_INODE);
  }
  sp->s_isearch = b;		/* next time start here */
  inumb = (int) b;		/* be careful not to pass unshort as param */

  /* Try to acquire a slot in the inode table. */
  if ( (rip = get_inode(sp->s_dev, inumb, &fs_err)) == NIL_INODE) {
	/* No inode table slots available.  Free the inode just allocated. */
	free_bit(sp, MP_INODE, b);
	*err_p= fs_err;
  } else {
	/* An inode slot is available. Put the inode just allocated into it. */
	rip->i_mode = bits;		/* set up RWX bits */
	rip->i_nlinks = (nlink_t) 0;	/* initially no links */
	rip->i_uid = fp->fp_effuid;	/* file's uid is owner's */
	rip->i_gid = fp->fp_effgid;	/* ditto group id */
	rip->i_sp = sp;			/* pointer to super block */
	rip->i_size = 0;
	rip->i_update = ATIME | CTIME | MTIME;	/* update all times later */
	rip->i_dirt = GRIMY;
	for (i = 0; i < V2_NR_TZONES; i++) rip->i_zone[i] = NO_ZONE;
  }

  return(rip);
}


/*===========================================================================*
 *				free_inode				     *
 *===========================================================================*/
PUBLIC void free_inode(rip)
register struct inode *rip;
{
/* Return an inode to the pool of unallocated inodes. */

  register struct super_block *sp;
  bit_t b;

  /* Locate the appropriate super_block. */
  sp = rip->i_sp;
  if (rip->i_num <= 0 || rip->i_num > sp->s_ninodes)
  	panic("free_inode:", rip->i_num);
  b = rip->i_num;
  free_bit(sp, MP_INODE, b);
  if (b < sp->s_isearch) sp->s_isearch = b;
}

/*===========================================================================*
 *				update_times				     *
 *===========================================================================*/
PUBLIC void update_times(rip)
register struct inode *rip;	/* pointer to inode to be read/written */
{
/* Various system calls are required by the standard to update atime, ctime,
 * or mtime.  Since updating a time requires sending a message to the clock
 * task--an expensive business--the times are marked for update by setting
 * bits in i_update.  When a stat, fstat, or sync is done, or an inode is 
 * released, update_times() may be called to actually fill in the times.
 */

  time_t cur_time;
  struct super_block *sp;

  sp = rip->i_sp;		/* get pointer to super block. */
  if (sp->s_rd_only) return;	/* no updates for read-only file systems */

  cur_time = clock_time();
  if (rip->i_update & ATIME) rip->i_atime = cur_time;
  if (rip->i_update & CTIME) rip->i_ctime = cur_time;
  if (rip->i_update & MTIME) rip->i_mtime = cur_time;
  rip->i_update = 0;		/* they are all up-to-date now */
}


/*===========================================================================*
 *				rw_inode				     *
 *===========================================================================*/
PUBLIC void rw_inode(rip, rw_flag)
register struct inode *rip;	/* pointer to inode to be read/written */
int rw_flag;			/* READING or WRITING */
{
/* An entry in the inode table is to be copied to or from the disk. */

  register struct buf *bp;
  register struct super_block *sp;
  d1_inode *dip;
  d2_inode *dip2;
  block_t b, offset;

#if DEBUG & 0
 { where(); printf("rw_inode: from "); stacktrace(); }
#endif

  /* Get the block where the inode resides. */
  sp = rip->i_sp;		/* get pointer to super block */
  offset = sp->s_imap_blocks + sp->s_zmap_blocks + 2;
  b = (block_t) (rip->i_num - 1)/sp->s_inodes_per_block + offset;
  assert(rip->i_dev == sp->s_dev);
  bp = get_block(rip->i_dev, b, BF_NORMAL);

  assert(rip->i_dev == bp->b_dev ||
	(printf("i_dev= 0x%x, i_num= %d, b_dev= 0x%x, b_blocknr= %d\n",
		rip->i_dev, rip->i_num, bp->b_dev, bp->b_blocknr), 0));

  dip  = bp->b_v1_ino + (rip->i_num - 1) % V1_INODES_PER_BLOCK;
  dip2 = bp->b_v2_ino + (rip->i_num - 1) % V2_INODES_PER_BLOCK;

  /* Do the read or write. */
  if (rw_flag == WRITING) {
	if (rip->i_update) update_times(rip);	/* times need updating */
	if (rip->i_nlinks == 0) rip->i_dirt = GRIMY;
	if (!sp->s_rd_only) {
		bp->b_dirt |= rip->i_dirt;
	}
  }

  /* Copy the inode from the disk block to the in-core table or vice versa.
   * If the fourth parameter below is FALSE, the bytes are swapped.
   */
  if (sp->s_version == V1)
	old_icopy(rip, dip,  rw_flag, sp->s_native);
  else
	new_icopy(rip, dip2, rw_flag, sp->s_native);
  
  put_block(bp);
  rip->i_dirt = CLEAN;
}


/*===========================================================================*
 *				old_icopy				     *
 *===========================================================================*/
PRIVATE void old_icopy(rip, dip, direction, norm)
register struct inode *rip;	/* pointer to the in-core inode struct */
register d1_inode *dip;		/* pointer to the d1_inode inode struct */
int direction;			/* READING (from disk) or WRITING (to disk) */
int norm;			/* TRUE = do not swap bytes; FALSE = swap */

{
/* The V1.x IBM disk, the V1.x 68000 disk, and the V2 disk (same for IBM and
 * 68000) all have different inode layouts.  When an inode is read or written
 * this routine handles the conversions so that the information in the inode
 * table is independent of the disk structure from which the inode came.
 * The old_icopy routine copies to and from V1 disks.
 */

  int i;

  if (direction == READING) {
	/* Copy V1.x inode to the in-core table, swapping bytes if need be. */
	rip->i_mode    = conv2(norm, (int) dip->d1_mode);
	rip->i_uid     = conv2(norm, (int) dip->d1_uid );
	rip->i_size    = conv4(norm,       dip->d1_size);
	rip->i_mtime   = conv4(norm,       dip->d1_mtime);
	rip->i_atime   = rip->i_mtime;
	rip->i_ctime   = rip->i_mtime;
	rip->i_nlinks  = (nlink_t) dip->d1_nlinks;	/* 1 char */
	if (dip->d1_gid == 0xff)
		rip->i_gid= -1;				/* nobody */
	else
		rip->i_gid     = dip->d1_gid;	/* 1 char */
	for (i = 0; i < V1_NR_TZONES; i++)
		rip->i_zone[i] = conv2(norm, (int) dip->d1_zone[i]);
  } else {
	/* Copying V1.x inode to disk from the in-core table. */
	dip->d1_mode   = conv2(norm, (int) rip->i_mode);
	dip->d1_uid    = conv2(norm, (int) rip->i_uid );
	dip->d1_size   = conv4(norm,       rip->i_size);
	dip->d1_mtime  = conv4(norm,       rip->i_mtime);
	dip->d1_nlinks = (nlink_t) rip->i_nlinks;	/* 1 char */
	dip->d1_gid    = (u8_t) rip->i_gid;		/* 1 char */
	if (dip->d1_gid != rip->i_gid)
		dip->d1_gid= 0xff;
	for (i = 0; i < V1_NR_TZONES; i++)
		dip->d1_zone[i] = conv2(norm, (int) rip->i_zone[i]);
  }
}


/*===========================================================================*
 *				new_icopy				     *
 *===========================================================================*/
PRIVATE void new_icopy(rip, dip, direction, norm)
register struct inode *rip;	/* pointer to the in-core inode struct */
register d2_inode *dip;	/* pointer to the d2_inode struct */
int direction;			/* READING (from disk) or WRITING (to disk) */
int norm;			/* TRUE = do not swap bytes; FALSE = swap */

{
/* Same as old_icopy, but to/from V2 disk layout. */

  int i;

  if (direction == READING) {
	/* Copy V2.x inode to the in-core table, swapping bytes if need be. */
	rip->i_mode    = conv2(norm,dip->d2_mode);
	rip->i_uid     = conv2(norm,dip->d2_uid );
	rip->i_nlinks  = conv2(norm,(int) dip->d2_nlinks);
	rip->i_gid     = conv2(norm,(int) dip->d2_gid );
	rip->i_size    = conv4(norm,dip->d2_size);
	rip->i_atime   = conv4(norm,dip->d2_atime);
	rip->i_ctime   = conv4(norm,dip->d2_ctime);
	rip->i_mtime   = conv4(norm,dip->d2_mtime);
	for (i = 0; i < V2_NR_TZONES; i++)
		rip->i_zone[i] = conv4(norm, (long) dip->d2_zone[i]);
  } else {
	/* Copying V2.x inode to disk from the in-core table. */
	dip->d2_mode   = conv2(norm,rip->i_mode);
	dip->d2_uid    = conv2(norm,rip->i_uid );
	dip->d2_nlinks = conv2(norm,rip->i_nlinks);
	dip->d2_gid    = conv2(norm,rip->i_gid );
	dip->d2_size   = conv4(norm,rip->i_size);
	dip->d2_atime  = conv4(norm,rip->i_atime);
	dip->d2_ctime  = conv4(norm,rip->i_ctime);
	dip->d2_mtime  = conv4(norm,rip->i_mtime);
	for (i = 0; i < V2_NR_TZONES; i++)
		dip->d2_zone[i] = conv4(norm, (long) rip->i_zone[i]);
  }
}


/*===========================================================================*
 *				dup_inode				     *
 *===========================================================================*/
PUBLIC void dup_inode(ip)
struct inode *ip;		/* The inode to be duplicated. */
{
/* This routine is a simplified form of get_inode() for the case where
 * the inode pointer is already known.
 */

  ip->i_count++;
}

/*===========================================================================*
 *				flushall_inodes				     *
 *===========================================================================*/
PUBLIC void flushall_inodes(dev)
dev_t dev;
{
	register struct inode *rip;

	for (rip = &inode_table[0]; rip < &inode_table[NR_INODES]; rip++)
	{
		if (rip->i_dev == NO_DEV)
			continue; /* only check used slots. */
		if (rip->i_dev != dev)
			continue;
		if (rip->i_dirt != CLEAN) rw_inode(rip, WRITING);
	}
}

/*===========================================================================*
 *				invalidate_inodes			     *
 *===========================================================================*/
PUBLIC void invalidate_inodes(dev)
dev_t dev;
{
	register struct inode *rip, *prev, *curr;
	int h;

	/* Nothing to do, but check if all inodes for this device are really
	 * gone.
	 */
	for (rip = &inode_table[0]; rip < &inode_table[NR_INODES]; rip++)
	{
		if (rip->i_dev == NO_DEV)
			continue; /* only check used slots. */
		if (rip->i_dev != dev)
			continue;
		if (rip->i_count != 0 || rip->i_dirt != CLEAN)
		{
			panic("invalidate_inodes: found dirty/busy inode:",
				rip->i_num);
		}

		hash_inode(&h, rip->i_dev, rip->i_num);
		for (curr= inode_hash[h], prev= NULL;
			curr != rip && curr != NULL; prev= curr, curr= curr->i_next)
		{
			; /* Do nothing. */
		}
		assert(curr == rip);
		if (prev == NULL)
			inode_hash[h]= curr->i_next;
		else
			prev->i_next= curr->i_next;

		rip->i_dev= NO_DEV;
	}
}


/*===========================================================================*
 *				check_ihash				     *
 *===========================================================================*/
PUBLIC int check_ihash()
{
	int hash, i, i_count, i_hashcount;
	struct inode *ip;

	i_hashcount= 0;
	for(i= 0; i<NR_INODE_HASH; i++)
	{
		for(ip= inode_hash[i]; ip; ip= ip->i_next)
		{
			assert(ip->i_dev != NO_DEV && ip->i_num != 0);
			hash_inode(&hash, ip->i_dev, ip->i_num);
			assert(hash == i);
			i_hashcount++;
		}
	}
	i_count= i_hashcount;
	for(i= 0; i<NR_INODES; i++)
	{
		if (inode_table[i].i_dev == NO_DEV)
			i_count++;
	}
	if (i_count != NR_INODES)
	{
		printf(
	"check_ihash: wrong number of inodes in hash chains: %d, expected %d\n",
			i_hashcount, NR_INODES-(i_count-i_hashcount));
		return 0;
	}
	assert(i_count == NR_INODES);
	return 1;
}

/*
 * $PchId: inode.c,v 1.9 1996/01/19 23:12:35 philip Exp $
 */
