/* When a needed block is not in the cache, it must be fetched from the disk.
 * Special character files also require I/O.  The routines for these are here.
 *
 * The entry points in this file are:
 *   dev_io:	 perform a read or write on a block or character device
 *   dev_opcl:   perform generic device-specific processing for open & close
 *   tty_open:   perform tty-specific processing for open
 *   tty_close:  perform tty-specific processing for close
 *   do_ioctl:	 perform the IOCTL system call
 *   call_task:	 procedure that actually calls the kernel tasks
 *   rw_dev2:	 procedure that actually calls task for /dev/tty
 *   no_call:	 dummy procedure (e.g., used when device need not be opened)
 */

#include "fs.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <fcntl.h>
#include "assert.h"
INIT_ASSERT
#include "dev.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"

PUBLIC struct dmap *dmap[256];

#define VALID_MAJOR(m) ((m) > 0 && (m) < 255 && dmap[(m)])

/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PUBLIC int do_ioctl()
{
/* Perform the ioctl(ls_fd, request, argx) system call (uses m2 fmt). */

  struct filp *f;
  register struct inode *rip;
  dev_t dev;
  message dev_mess;
  int status;
  int fs_err= EGENERIC;

  assert(fs_call == OLD_IOCTL);

  if ((m.TTY_REQUEST >> 8) != 't') {
	/* It is not one of those weird old sgtty ioctls, so the new ioctl
	 * can handle it.
	 */
	int fd = ioctl_fd;
	ioreq_t req = m.TTY_REQUEST;
	void *address = m.ADDRESS;

	fs_call = IOCTL;
	ioctl3_fd = fd;
	ioctl3_req = req;
	ioctl3_buffer = address;
	return(do_ioctl3());
  }

  if ( (f = get_filp(ioctl_fd, &fs_err)) == NIL_FILP) return(fs_err);
  if (f->filp_int_flags & FIF_CLOSED)
	return(EIO);
  rip = f->filp_ino;		/* get inode pointer */
  if ( (rip->i_mode & I_TYPE) != I_CHAR_SPECIAL) return(ENOTTY);
  dev = (dev_t) rip->i_zone[0];

  dev_mess= m;

  dev_mess.m_type  = DEV_IOCTL;
  dev_mess.PROC_NR = who;

  /* Call the task. */
  status= dev_msg(dev, who, &dev_mess);
  m1.TTY_SPEK = dev_mess.TTY_SPEK;	/* erase and kill */
  m1.TTY_FLAGS = dev_mess.TTY_FLAGS;	/* flags */
  return status;
}


/*===========================================================================*
 *				do_ioctl3				     *
 *===========================================================================*/
PUBLIC int do_ioctl3()
{
/* Perform the ioctl(fd, request, arg) system call (uses m2 fmt). */

  struct filp *f;
  register struct inode *rip;
  dev_t dev;
  int status;
  int oflags;
  int fs_err= EGENERIC;

  assert(fs_call == IOCTL);

  if ( (f = get_filp(ioctl3_fd, &fs_err)) == NIL_FILP) return(fs_err);
  if (f->filp_int_flags & FIF_CLOSED)
	return(EIO);
  oflags = f->filp_flags;
  rip = f->filp_ino;		/* get inode pointer */
  if ( (rip->i_mode & I_TYPE) != I_CHAR_SPECIAL
		&& (rip->i_mode & I_TYPE) != I_BLOCK_SPECIAL) return(ENOTTY);
  dev = (dev_t) rip->i_zone[0];

  /* Stubborn developers had trouble agreeing on the request field type,
   * finally a compromise was reached (ioreq_t), but all dumb things are
   * carved in stone till the end of time by the demands of backwards
   * compatibility:
   */
  if (ioctl3_req == 0) ioctl3_req= m.m2_i3;

  status= dev_ioctl(dev, who, ioctl3_req, ioctl3_buffer, ioctl3_fd,
  	oflags & O_NONBLOCK);
  return status;
}


/*===========================================================================*
 *				dev_init				     *
 *===========================================================================*/
PUBLIC void dev_init()
{
	int i;
	int major;
	
	for (i= 0; i<256; i++)
		dmap[i]= NULL;

	for (i= 0; i < NR_DMAPS; i++)
	{
		major= dmap_table[i].dmap_major;
		if (major == 0) continue;
		if (major < 1 || major > 255 || dmap[major] != NULL)
			panic("inconsistent dmap, major= ", major);
		dmap[major]= &dmap_table[i];
	}
}

/*===========================================================================*
 *				dev_opcl				     *
 *===========================================================================*/
PUBLIC int dev_opcl(open_close, device, user_task, flags)
int open_close;			/* open=1, close=0 */
dev_t device;			/* which device. */
int user_task;			/* which user. */
int flags;
{
	int major, dev_task, status;
	dmap_opcl_t func;
	struct dmap *dmapp;
	int r, tasknr;

	major = (device >> MAJOR) & BYTE;	/* major device nr */
	if (!VALID_MAJOR(major))
		return ENODEV;
	dmapp= dmap[major];
	dev_task = dmapp->dmap_task;			/* device task nr */
	if (dev_task == ANY && dmapp->dmap_taskname != NULL)
	{
		/* Task has not been found yet, ask the kernel. */
		r= sys_findproc(dmapp->dmap_taskname, &tasknr, 0);
		if (r == ESRCH)
		{
			printf("unable to find task '%s'\n",
				dmapp->dmap_taskname);
			return ENODEV;
		}
		assert(r == OK);
		dev_task= dmapp->dmap_task= tasknr;
	}
	func= open_close ? dmapp->dmap_open : dmapp->dmap_close;
	status= (*func)(open_close, device, dev_task, user_task, flags);
	return status;
}


/*===========================================================================*
 *				dev_rw					     *
 *===========================================================================*/
PUBLIC int dev_rw(op, device, user_task, pos, buf, bytes, ref, flags)
int op;				/* DEV_READ, DEV_WRITE, DEV_SCATTER, ... */
dev_t device;			/* which device. */
int user_task;			/* which user. */
u64_t pos;			/* offset, for block special devices. */
char *buf;			/* buffer is user space. */
unsigned bytes;			/* number of bytes. */
int ref;			/* extra reference for matching replies. */
int flags;			/* flags like O_NONBLOCK */
{
	int major, dev_task, status;
	dmap_rw_t func;

	major = (device >> MAJOR) & BYTE;	/* major device nr */
	if (!VALID_MAJOR(major))
		panic("read on non existing device.", NO_NUM);
	dev_task = dmap[major]->dmap_task;		/* device task nr */
	func= dmap[major]->dmap_rw;
	status= (*func)(op, device, dev_task, user_task, pos,
						buf, bytes, ref, flags);

	/* Task has completed.  See if call completed. */
	if (status == SUSPEND) {
		assert(fproc[user_task].fp_event != EV_NONE);
	}
	return status;
}


/*===========================================================================*
 *				dev_ioctl				     *
 *===========================================================================*/
PUBLIC int dev_ioctl(device, user_task, function, buf, ref, flags)
dev_t device;
int user_task;
ioreq_t function;
char *buf;
int ref;
int flags;
{
	int major, dev_task, status;
	dmap_ioctl_t func;

	major = (device >> MAJOR) & BYTE;	/* major device nr */
	if (!VALID_MAJOR(major))
		panic("ioctl on non existing device.", NO_NUM);
	dev_task = dmap[major]->dmap_task;		/* device task nr */
	func= dmap[major]->dmap_ioctl;
	status= (*func)(device, dev_task, user_task, function, buf, ref, flags);

	/* Task has completed.  See if call completed. */
	if (status == SUSPEND) {
		assert(fproc[user_task].fp_event != EV_NONE);
	}
	return status;
}


/*===========================================================================*
 *				dev_msg					     *
 *===========================================================================*/
PUBLIC int dev_msg(device, user_task, m_ptr)
dev_t device;
int user_task;
message *m_ptr;
{
	int major, dev_task, status;
	dmap_mess_t func;

	major = (device >> MAJOR) & BYTE;	/* major device nr */
	if (!VALID_MAJOR(major))
		panic("operation on non existing device.", NO_NUM);
	dev_task = dmap[major]->dmap_task;		/* device task nr */
	func= dmap[major]->dmap_mess;
	status= (*func)(device, dev_task, user_task, m_ptr);

	/* Task has completed.  See if call completed. */
	if (status == SUSPEND) {
		assert(fproc[user_task].fp_event != EV_NONE);
	}
	return status;
}
	

/*===========================================================================*
 *				dev_cancel				     *
 *===========================================================================*/
PUBLIC int dev_cancel(what, device, user_task, ref)
int what;
dev_t device;
int user_task;
int ref;
{
	int major, dev_task, status;
	dmap_cancel_t func;

	major = (device >> MAJOR) & BYTE;	/* major device nr */
	if (!VALID_MAJOR(major))
		panic("cancel on non existing device.", NO_NUM);
	dev_task = dmap[major]->dmap_task;		/* device task nr */
	func= dmap[major]->dmap_cancel;
	status= (*func)(what, device, dev_task, user_task, ref);

	return status;
}


/*===========================================================================*
 *				gen_opcl				     *
 *===========================================================================*/
PUBLIC int gen_opcl(open_close, device, dev_task, user_task, flags)
int open_close;			/* open=1, close=0 */
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
int flags;			/* special flags */
{
  message m;

  m.DEVICE= (device >> MINOR) & BYTE;
  m.PROC_NR= user_task;
  m.COUNT= flags;
  m.m_type= open_close ? DEV_OPEN : DEV_CLOSE;

  call_task(dev_task, &m);

  if (m.REP_STATUS == SUSPEND)
	panic("don't know how to suspend an open or close.", NO_NUM);

  return m.REP_STATUS;
}


/*===========================================================================*
 *				gen_rw					     *
 *===========================================================================*/
PUBLIC int gen_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags)
int op;				/* DEV_READ, DEV_WRITE, ... */
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
u64_t pos;			/* position for block special files. */
char *buf;			/* buffer in user space. */
unsigned bytes;			/* number of bytes. */
int ref;			/* extra reference for matching replies. */
int flags;			/* special flags */
{
/* Read or write from a device. */

  message m;
  int r, r1;

  /* Set up the message passed to task. */
  m.m_type	= op;
  m.DEVICE	= (device >> MINOR) & BYTE;
  m.POSITION	= ex64lo(pos);
  m.HIGHPOS	= ex64hi(pos);
  m.PROC_NR	= user_task;
  m.ADDRESS	= buf;
  m.COUNT	= bytes;

  call_task(dev_task, &m);
  assert (m.REP_STATUS != SUSPEND);	/* use ogen_rw instead */

  /* Task has completed.  See if call completed. */

  r= m.REP_STATUS;
  if (r != E_NOPAGE)
  	return r;
  r= sys_vm_lock(user_task, (vir_bytes)buf, bytes);
  if (r != OK)
  	return r;
  r1= gen_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags);
  r= sys_vm_unlock(user_task, (vir_bytes)buf, bytes);
  assert(r == OK);
  return r1;
}


/*===========================================================================*
 *				gen_ioctl				     *
 *===========================================================================*/
PUBLIC int gen_ioctl(device, dev_task, user_task, function, buf, ref, flags)
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
ioreq_t function;		/* which function to perform. */
char *buf;			/* buffer in user space. */
int ref;			/* extra reference for matching replies. */
int flags;			/* special flags */
{
/* Perform an ioctl on a device. */

  message m;
  int ip_flag;
  struct od *odp;
  int fs_err= EGENERIC;

  odp= get_fd(ref, &fs_err);	/* Assume ref=fd */
  if (odp == NIL_ODP && user_task != FS_PROC_NR)
  	panic("invalid fd", NO_NUM);
  ip_flag= ODF_RD_IP|ODF_WR_IP; 

  /* Set up the message passed to task. */
  m.m_type	= DEV_IOCTL;
  m.DEVICE	= (device >> MINOR) & BYTE;
  m.PROC_NR	= user_task;
  m.REQUEST	= function;
  m.ADDRESS	= buf;

  call_task(dev_task, &m);

  /* Task has completed.  See if call completed. */
  if (m.REP_STATUS == SUSPEND) {
	odp->od_flags |= ip_flag;
	fd_suspend(EV_IOCTL, ref);		/* suspend user */
  }

  return(m.REP_STATUS);
}


/*===========================================================================*
 *				gen_cancel				     *
 *===========================================================================*/
PUBLIC int gen_cancel(operation, device, dev_task, user_task, ref)
int operation;
dev_t device;
int dev_task;
int user_task;
int ref;
{
/* Cancel a read, write or ioctl. */

  message m;

  /* Set up the message passed to task. */
  m.m_type	= DEV_CANCEL;
  m.DEVICE	= (device >> MINOR) & BYTE;
  m.PROC_NR	= user_task;
  m.COUNT	= operation;

  call_task(dev_task, &m);

  return(m.REP_STATUS);
}


/*===========================================================================*
 *				gen_msg					     *
 *===========================================================================*/
PUBLIC int gen_msg(device, dev_task, user_task, m_ptr)
dev_t device;
int dev_task;
int user_task;
message *m_ptr;
{
/* This function is very tricky. It is used to implement the old ioctls
 * that passed many fields in a message instead of passing a pointer
 * to the structure in user space.
 */

  m_ptr->DEVICE = (device >> MINOR) & BYTE;

  call_task(dev_task, m_ptr);

  return(m_ptr->REP_STATUS);
}


/*===========================================================================*
 *				ngen_opcl				     *
 *===========================================================================*/
PUBLIC int ngen_opcl(open_close, device, dev_task, user_task, flags)
int open_close;
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* This procedure is called from the dmap struct in table.c on opens or closes.
 */

  int r;
  message m;

  m.m_type	= open_close ? DEV_OPEN : DEV_CLOSE;
  m.NDEV_MINOR	= (device >> MINOR) & BYTE;
  m.NDEV_PROC	= user_task;
  m.NDEV_REF	= 0;

  /* Call the task. */
  call_ntask(dev_task, &m);
  r= m.REP_STATUS;

  if (r == SUSPEND)
	panic("don't know how to suspend an open or close.", NO_NUM);

  return r;
}


/*===========================================================================*
 *				ngen_rw					     *
 *===========================================================================*/
PUBLIC int ngen_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags)
int op;				/* DEV_READ, DEV_WRITE, ... */
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
u64_t pos;			/* position for block special files. */
char *buf;			/* buffer in user space. */
unsigned bytes;			/* number of bytes. */
int ref;			/* extra reference for matching replies. */
int flags;			/* special flags */
{
/* Read or write from a device. */

  message m;
  struct od *odp;
  int ip_flag, asio_op, event;
  int r;
  int fs_err= EGENERIC;

  /* Check if the operation is already in progress. */
  odp= get_fd(ref, &fs_err);	/* Assume ref=fd */
  if (odp == NIL_ODP)	panic("invalid fd", NO_NUM);
  event= EV_NONE;
  switch (op)
  {
  case DEV_WRITE:
  case DEV_SCATTER:
  	ip_flag= ODF_WR_IP; asio_op= ASIO_WRITE; event= EV_WRITE;
  	break;
  case DEV_READ:
  case DEV_GATHER:
  	ip_flag= ODF_RD_IP; asio_op= ASIO_READ; event= EV_READ;
  	break;
  default:
  	panic("invalid operation", NO_NUM);
  }
  if (odp->od_flags & ip_flag)	/* Operation already in progress */
	return EALREADY;

  /* Set up the message passed to task. */
  m.m_type		= op;
  m.NDEV_MINOR		= (device >> MINOR) & BYTE;
  m.NDEV_PROC		= user_task;
  m.NDEV_REF		= ref;
  m.NDEV_POSITION	= ex64lo(pos);
  m.NDEV_HIGHPOS	= ex64hi(pos);
  m.NDEV_COUNT		= bytes;
  m.NDEV_BUFFER		= buf;

  call_ntask(dev_task, &m);

  /* Task has completed.  See if call completed. */
  if (m.REP_STATUS == SUSPEND) {
  	if (flags & O_NONBLOCK)
  	{
		r= ngen_cancel(asio_cancel[asio_op], device, dev_task,
			user_task, ref);
		if (r == EINTR)
			r= EAGAIN;
		return r;
  	}
	odp->od_flags |= ip_flag;
	if (odp->od_flags & ODF_ASYNCH)
		return EINPROGRESS;
	fd_suspend(event, ref);
  }

  return(m.REP_STATUS);
}


/*===========================================================================*
 *				ngen_ioctl					     *
 *===========================================================================*/
PUBLIC int ngen_ioctl(device, dev_task, user_task, function, buf, ref, flags)
dev_t device;
int dev_task;
int user_task;
ioreq_t function;
char *buf;
int ref;
int flags;
{
/* Ioctl on a device. */

  message m;
  struct od *odp;
  int r;
  int fs_err= EGENERIC;

  /* Check is the operation is already in progress. */
  odp= get_fd(ref, &fs_err);	/* Assume ref=fd */
  if (odp == NIL_ODP)	panic("invalid fd", NO_NUM);
  if (odp->od_flags & ODF_IOC_IP)	/* Operation already in progress */
	return EALREADY;

  /* Set up the message passed to task. */
  m.m_type		= DEV_IOCTL3;
  m.NDEV_MINOR		= (device >> MINOR) & BYTE;
  m.NDEV_PROC		= user_task;
  m.NDEV_REF		= ref;
  m.NDEV_MODE		= flags & O_NONBLOCK;
  m.NDEV_IOCTL		= function;
  m.NDEV_BUFFER		= buf;

  call_ntask(dev_task, &m);

  /* Task has completed.  See if call completed. */
  if (m.REP_STATUS == SUSPEND) {
  	if (flags & O_NONBLOCK)
  	{
		r= ngen_cancel(DEV_IOCTL3, device, dev_task, user_task, ref);
		if (r == EINTR)
			r= EAGAIN;
		return r;
  	}
	odp->od_flags |= ODF_IOC_IP;
	if (odp->od_flags & ODF_ASYNCH)
	{
		return EINPROGRESS;
	}
	fd_suspend(EV_IOCTL, ref);
  }

  return(m.REP_STATUS);
}


/*===========================================================================*
 *				ngen_cancel				     *
 *===========================================================================*/
PUBLIC int ngen_cancel(operation, device, dev_task, user_task, ref)
int operation;
dev_t device;
int dev_task;
int user_task;
int ref;
{
/* Cancel a read, write or ioctl. */

  message m;

  /* Set up the message passed to task. */
  m.m_type		= DEV_CANCEL;
  m.NDEV_MINOR		= (device >> MINOR) & BYTE;
  m.NDEV_PROC		= user_task;
  m.NDEV_REF		= ref;
  m.NDEV_OPERATION	= operation;

  call_ntask(dev_task, &m);

  return(m.REP_STATUS);
}


/*===========================================================================*
 *				ogen_rw					     *
 *===========================================================================*/
PUBLIC int ogen_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags)
int op;				/* DEV_READ, DEV_WRITE, ... */
dev_t device;			/* major/minor */
int dev_task;			/* which task to call */
int user_task;			/* for whom is the call made */
u64_t pos;			/* position for block special files. */
char *buf;			/* buffer in user space. */
unsigned bytes;			/* number of bytes. */
int ref;			/* extra reference for matching replies. */
int flags;			/* special flags */
{
/* Read or write from a device. */

  message m;
  int r, r1;
  int ip_flag;
  int fs_err= EGENERIC;
  struct od *odp;

  odp= get_fd(ref, &fs_err);	/* Assume ref=fd */
  if (odp == NIL_ODP)	panic("invalid fd", NO_NUM);
  switch (op)
  {
  case DEV_WRITE:
  case DEV_SCATTER:	ip_flag= ODF_WR_IP; break;
  case DEV_READ:
  case DEV_GATHER:	ip_flag= ODF_RD_IP; break;
  default:		panic("invalid operation", NO_NUM);
  }

  /* Set up the message passed to task. */
  m.m_type	= op;
  m.DEVICE	= (device >> MINOR) & BYTE;
  m.POSITION	= ex64lo(pos);
  m.HIGHPOS	= ex64hi(pos);
  m.PROC_NR	= user_task;
  m.ADDRESS	= buf;
  m.COUNT	= bytes;

  call_task(dev_task, &m);

  /* Task has completed.  See if call completed. */
  if (m.REP_STATUS == SUSPEND) {
	odp->od_flags |= ip_flag;
	fd_suspend(ip_flag == ODF_WR_IP ? EV_WRITE : EV_READ, ref);	
							/* suspend user */
  }

  r= m.REP_STATUS;
  if (r != E_NOPAGE)
  	return r;
  r= sys_vm_lock(user_task, (vir_bytes)buf, bytes);
  if (r != OK)
  	return r;
  r1= ogen_rw(op, device, dev_task, user_task, pos, buf, bytes, ref, flags);
  r= sys_vm_unlock(user_task, (vir_bytes)buf, bytes);
  assert(r == OK);
  return r1;
}


/*===========================================================================*
 *				no_open					     *
 *===========================================================================*/
PUBLIC int no_opcl(open_close, device, dev_task, user_task, flags)
int open_close;			/* open=1, close=0 */
dev_t device;
int dev_task;
int user_task;
int flags;
{
/* Null operation always succeeds. */

  return (OK);
}

/*===========================================================================*
 *				no_ioctl				     *
 *===========================================================================*/
PUBLIC int no_ioctl(device, dev_task, user_task, function, buf, ref, flags)
dev_t device;
int dev_task;
int user_task;
ioreq_t function;
char *buf;
int ref;
int flags;
{
	return (ENOTTY);
}


/*===========================================================================*
 *				no_cancel				     *
 *===========================================================================*/
PUBLIC int no_cancel(operation, device, dev_task, user_task, ref)
int operation;
dev_t device;
int dev_task;
int user_task;
int ref;
{
	panic("no_cancel called\n", NO_NUM);
	/*NOTREACHED*/
}


/*===========================================================================*
 *				no_msg					     *
 *===========================================================================*/
PUBLIC int no_msg(device, dev_task, user_task, m_ptr)
dev_t device;
int dev_task;
int user_task;
message *m_ptr;
{
  return(EIO);
}


/*===========================================================================*
 *				call_ntask				     *
 *===========================================================================*/
PUBLIC void call_ntask(task_nr, mess_ptr)
int task_nr;			/* which task to call */
message *mess_ptr;		/* pointer to message for task */
{
/* All file system I/O ultimately comes down to I/O on major/minor device
 * pairs.  These lead to calls on the following routines via the dmap table.
 */

  int r, ref, operation;
  message local_m;
  int user_proc;
  int op;

  user_proc= mess_ptr->NDEV_PROC;
  ref= mess_ptr->NDEV_REF;
  operation= mess_ptr->m_type;
  if (operation == DEV_CANCEL)
  	operation= mess_ptr->NDEV_OPERATION;
  
  while ((r = sendrec(task_nr, mess_ptr)) == ELOCKED) {
	/* sendrec() failed to avoid deadlock. The task 'task_nr' is
	 * trying to send a REVIVE message for an earlier request.
	 * Handle it and go try again.
	 */
	if (receive(task_nr, &local_m) != OK)
		panic("call_task: can't receive", NO_NUM);

	/* If we're trying to send a cancel message to a task which has just
	 * sent a completion reply, ignore the reply and abort the cancel
	 * request. The caller will do the revive for the process. 
	 */
	if (mess_ptr->m_type == DEV_CANCEL &&
		local_m.REP_PROC_NR == mess_ptr->NDEV_PROC &&
		local_m.REP_REF == mess_ptr->NDEV_REF &&
		local_m.REP_OPERATION == mess_ptr->NDEV_OPERATION)
	{
		*mess_ptr= local_m;
		return;
	}

	if (local_m.m_type != DEVICE_REPLY)
	{
		printf(
	"%s, %d: got a strange message from %d for %d (m_type == %d)\n",
			__FILE__, __LINE__, local_m.m_source,
			local_m.REP_PROC_NR,
			local_m.m_type);
		printf("REP_PROC_NR= %d, expecting %d\n", local_m.REP_PROC_NR,
			user_proc);
		assert(0);
		continue;
	}
	switch(local_m.REP_OPERATION)
	{
	case DEV_READ: op= ASIO_READ; break;
	case DEV_WRITE: op= ASIO_WRITE; break;
	case DEV_IOCTL3: op= ASIO_IOCTL; break;
	default: panic("invalid command", NO_NUM);
	}
	fd_revive(local_m.REP_PROC_NR, local_m.REP_REF, op, local_m.REP_STATUS);
  }
  if (r != OK) panic("call_task: can't send", NO_NUM);
  while (TRUE) {
  
  	if (mess_ptr->REP_PROC_NR == user_proc && mess_ptr->REP_REF == ref &&
		mess_ptr->REP_OPERATION == operation)
		return;

	if (mess_ptr->m_type != DEVICE_REPLY)
	{
		printf("%s, %d: got a strange message for %d (m_type == %d)\n",
			__FILE__, __LINE__, mess_ptr->REP_PROC_NR,
			mess_ptr->m_type);
		printf("REP_PROC_NR= %d, expecting %d\n", mess_ptr->REP_PROC_NR,
			user_proc);
		r = receive (task_nr, mess_ptr);
		if (r != OK) panic("call_ntask: can't receive", NO_NUM);
		continue;
	}
	switch(mess_ptr->REP_OPERATION)
	{
	case DEV_READ: op= ASIO_READ; break;
	case DEV_WRITE: op= ASIO_WRITE; break;
	case DEV_IOCTL3: op= ASIO_IOCTL; break;
	default: 
		where(); printf("operation= %d, REP_OPERATION= %d\n", 
			operation, mess_ptr->REP_OPERATION);
		panic("invalid command", NO_NUM);
	}
	fd_revive(mess_ptr->REP_PROC_NR, mess_ptr->REP_REF, op, 
							mess_ptr->REP_STATUS);

	r = receive (task_nr, mess_ptr);
	if (r != OK) panic("call_ntask: can't receive", NO_NUM);
  }
}


/*===========================================================================*
 *				clone_open				     *
 *===========================================================================*/
PUBLIC int clone_open(open_close, device, dev_task, user_task, flags)
int open_close;			/* open=1 */
dev_t device;			/* major/minor */
int dev_task;			/* task to send message to */
int user_task;
int flags;
{
/* Clone devices need special processing upon open. */

  dev_t dev;
  struct inode *rip, *nrip;
  int result;
  int fs_err= EGENERIC;

  rip= fp->fp_filp[curr_fd]->od_filp->filp_ino; 

  nrip= alloc_inode(pipe_fs, ALL_MODES | I_CHAR_SPECIAL, &fs_err);
  if (nrip == NIL_INODE) {
	return (fs_err);
  }

  result= ngen_opcl(1, device, dev_task, user_task, flags);

  if (result < 0)
  {
	put_inode(nrip);
	return result;
  }

  dev= device; 
  dev= (dev & ~(BYTE << MINOR)) | ((result & BYTE) << MINOR); 

  nrip->i_zone[0]= dev;
  put_inode (rip);
  fp->fp_filp[curr_fd]->od_filp->filp_ino= nrip;
  return OK;
}


/*===========================================================================*
 *				call_task				     *
 *===========================================================================*/
PUBLIC void call_task(task_nr, mess_ptr)
int task_nr;			/* which task to call */
message *mess_ptr;		/* pointer to message for task */
{
/* All file system I/O ultimately comes down to I/O on major/minor device
 * pairs.  These lead to calls on the following routines via the dmap table.
 */

  int r;
  message local_m;
  int user_proc;
  int event, op, proc_nr;
  struct fproc *rfp;

  user_proc= mess_ptr->PROC_NR;

  while ((r = sendrec(task_nr, mess_ptr)) == ELOCKED) {
	/* sendrec() failed to avoid deadlock. The task 'task_nr' is
	 * trying to send a REVIVE message for an earlier request.
	 * Handle it and go try again.
	 */
	if (receive(task_nr, &local_m) != OK)
		panic("call_task: can't receive", NO_NUM);

	/* If we're trying to send a cancel message to a task which has just
	 * sent a completion reply, ignore the reply and abort the cancel
	 * request. The caller will do the revive for the process. 
	 */
	if (mess_ptr->m_type == DEV_CANCEL
	    && local_m.REP_PROC_NR == mess_ptr->PROC_NR)
		return;
	proc_nr= mess_ptr->REP_PROC_NR;
	if (proc_nr < 0 || proc_nr >= NR_PROCS) panic("bad REVIVE", proc_nr);
	rfp = &fproc[proc_nr];
	event= rfp->fp_event;
	op= ASIO_NOOP;
	if (event == EV_READ) op= ASIO_READ;
	else if (event == EV_WRITE) op= ASIO_WRITE;
	else if (event == EV_IOCTL) op= ASIO_IOCTL;
	assert(op != ASIO_NOOP);

	fd_revive(mess_ptr->REP_PROC_NR, rfp->fp_state.fp_rwio.fp_fd,
		op, mess_ptr->REP_STATUS);
  }
  if (r != OK) panic("call_task: can't send", NO_NUM);
  while (TRUE) {
  	if (mess_ptr->REP_PROC_NR == user_proc)
		return;

	if (mess_ptr->m_type != REVIVE)
	{
		printf("%s, %d: got a strange message for %d (m_type == %d)\n",
			__FILE__, __LINE__, mess_ptr->REP_PROC_NR,
			mess_ptr->m_type);
		printf("REP_PROC_NR= %d, expecting %d\n", mess_ptr->REP_PROC_NR,
			user_proc);
		assert(0);
		continue;
	}
	proc_nr= mess_ptr->REP_PROC_NR;
	if (proc_nr < 0 || proc_nr >= NR_PROCS) panic("bad REVIVE", proc_nr);
	rfp = &fproc[proc_nr];
	event= rfp->fp_event;
	op= ASIO_NOOP;
	if (event == EV_READ) op= ASIO_READ;
	else if (event == EV_WRITE) op= ASIO_WRITE;
	else if (event == EV_IOCTL) op= ASIO_IOCTL;
	assert(op != ASIO_NOOP);

	fd_revive(mess_ptr->REP_PROC_NR, rfp->fp_state.fp_rwio.fp_fd,
		op, mess_ptr->REP_STATUS);

	r = receive (task_nr, mess_ptr);
	if (r != OK) panic("call_task: can't receive", NO_NUM);
  }
}

/*
 * $PchId: device.c,v 1.5 1995/11/28 08:21:56 philip Exp $
 */
