/* ps - print status 			Author: Peter Valkenburg */

/* Ps.c, Peter Valkenburg (valke@psy.vu.nl), january 1990.
 *
 * This is a V7 ps(1) look-alike for MINIX >= 1.5.0.  It can use a database
 * with information on system addresses and terminal names as an extra, and
 * has some additional fields.
 * It does not support the 'k' option (i.e. cannot read memory from core file).
 * If you want to compile this for non-IBM PC architectures, the header files
 * require that you have your CHIP, MACHINE etc. defined.
 * Full syntax:
 *	ps [-][alx]
 * Option `a' gives all processes, `l' for detailed info, `x' includes even
 * processes without a terminal.
 *
 * VERY IMPORTANT NOTE:
 *	To compile ps, the kernel/, fs/ and mm/ source directories must be in
 *	../ relative to the directory where ps is compiled (normally the
 *	command source directory).
 *
 *	If you want your ps to be useable by anyone, yet disallow modification
 *	of the database by anyone other than root and bin, you can arrange the
 *	following access permissions (note the protected memory files and set
 *	*group* id on ps):
 *	-rwxr-sr-x  1 bin         11916 Jul  4 15:31 /bin/ps
 *	crw-r-----  1 bin        1,   1 Jan  1  1970 /dev/mem
 */

/* Some technical comments on this implementation:
 *
 * Most fields are similar to V7 ps(1), except for CPU, NICE, PRI which are
 * absent, RECV which replaces WCHAN, and PGRP that is an extra.
 * The info is obtained from the following fields of proc, mproc and fproc:
 * F	- kernel status field, p_flags
 * S	- kernel status field, p_flags; mm status field, mp_flags (R if p_flags
 * 	  is 0; Z if mp_flags == HANGING; T if mp_flags == STOPPED; else W).
 * UID	- mm eff uid field, mp_effuid
 * PID	- mm pid field, mp_pid
 * PPID	- mm parent process index field, mp_parent (used as index in proc).
 * PGRP - mm process group field, mp_procgrp
 * SZ	- kernel text size + data size + stack size
 * 	  p_map[SEG_T].mem_len + p_map[SEG_D].mem_len + p_map[SEG_S].mem_len
 * RECV	- kernel process index field for message receiving, p_getfrom
 *	  If sleeping, mm's mp_flags, or fs's fp_task are used for more info.
 * TTY	- fs controlling tty device field, fp_tty.
 * TIME	- kernel user + system times fields, user_time + sys_time
 * CMD	- system process index (converted to mnemonic name obtained by reading
 *	  tasktab array from kmem), or user process argument list (obtained by
 *	  reading the stack frame; the resulting address is used to get
 *	  the argument vector from user space and converted into a concatenated
 *	  argument list).
 */

#include <minix/cfg_public.h>
#include <limits.h>
#include <sys/types.h>

#include <minix/const.h>
#include <minix/type.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ttyent.h>
#include <sys/svrctl.h>
#include <minix/queryparam.h>

#include <kernel/const.h>
#include <kernel/type.h>
#include <kernel/proc.h>
#undef printf			/* kernel's const.h defined this */

#include <mm/mproc.h>
#include <fs/fproc.h>
#include <fs/const.h>
#undef printf			/* fs's const.h defined this */

/*----- ps's local stuff below this line ------*/

#include <minix/com.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#include <signal.h>
#include <stdio.h>
#include <limits.h>
#include <alloca.h>

/* A scalar of the same size as a pointer. */
#if PTR_SIZE == WORD_SIZE
typedef unsigned	ptr_t;
#else
typedef unsigned long	ptr_t;
#endif

int nr_procs;			/* Number of processes. */
int nr_tasks;			/* Number of tasks. */
ptr_t a_mproc;			/* &mproc */
size_t s_mproc;			/* sizeof(mproc) */
size_t s_mproc0;		/* sizeof(mproc[0]) */
size_t s_p_name;		/* sizeof(proc[0].p_name) */

#define mindev(dev)	(((dev)>>MINOR) & 0377)	/* yield minor device */
#define majdev(dev)	(((dev)>>MAJOR) & 0377)	/* yield major device */

#define	TTY_MAJ		4	/* major device of console */

/* Structure for tty name info. */
typedef struct {
  char tty_name[NAME_MAX + 1];	/* file name in /dev */
  dev_t tty_dev;		/* major/minor pair */
} ttyinfo_t;

ttyinfo_t *ttyinfo;		/* ttyinfo holds actual tty info */
size_t n_ttyinfo;		/* Number of tty info slots */
int tty_major[256];		/* Device that are ttys. */

#define	MEM_PATH	"/dev/mem"	/* opened for argument vectors */

int memfd;		/* file descriptor of /dev/mem */

int uid;			/* real uid of caller */
int opt_all = FALSE;		/* -a */
int opt_long = FALSE;		/* -l */
int opt_notty = FALSE;		/* -x */
char *opt_pidlist = NULL;	/* -p */

/* Short and long listing formats:
 *
 *   PID TTY  TIME CMD
 * ppppp tttmmm:ss cccccccccccccccccccccccccccccccccccccccccccccccccccccccc...
 *
 *   F S UID   PID  PPID  PGRP   SZ       RECV TTY  TIME CMD
 * fff s uuu ppppp ppppp ppppp ssss rrrrrrrrrr ttt mm:ss cccccccccccccccccc...
 */
#define S_HEADER "  PID TTY  TIME CMD\n"
#define S_FORMAT "%5s %3s %s %s%s\n"
#define L_HEADER "  F S UID   PID  PPID  PGRP   SZ       RECV TTY  TIME CMD\n"
#define L_FORMAT "%3o %c %3d %5s %5d %5d %4s %10s %3s %s %s%s\n"

struct pstat {			/* process information. */
  struct pstat *ps_next;
  struct pstat *ps_sort;
  int ps_nr;			/* process table slot number, MM = 0. */
  int ps_wanted;		/* do we want to print this one? */
  dev_t ps_dev;			/* major/minor of controlling tty */
  uid_t ps_ruid;		/* real uid */
  uid_t ps_euid;		/* effective uid */
  pid_t ps_pid;			/* process id */
  int ps_parent;		/* parent proc table index */
  pid_t ps_ppid;		/* parent process id */
  int ps_pgrp;			/* process group id */
  int ps_flags;			/* kernel flags */
  int ps_mflags;		/* mm flags */
  int ps_fevent;		/* fs suspend event */
  int ps_state;			/* process state */
  size_t ps_tsize;		/* text size (in bytes) */
  size_t ps_dsize;		/* data size (in bytes) */
  size_t ps_ssize;		/* stack size (in bytes) */
  off_t ps_text;		/* physical text offset */
  off_t ps_data;		/* physical data offset */
  off_t ps_stack;		/* physical stack offset */
  int ps_recv;			/* process number to receive from */
  time_t ps_utime;		/* accumulated user time */
  time_t ps_stime;		/* accumulated system time */
  char *ps_name;		/* basename on exec() */
} *proctab, *procsort;

/* Ps_state field values in pstat struct above */
#define NO_STATE	 0	/* Not filled in yet */
#define	Z_STATE		'Z'	/* Zombie */
#define	W_STATE		'W'	/* Waiting */
#define	S_STATE		'S'	/* Sleeping */
#define	R_STATE		'R'	/* Runnable */
#define	T_STATE		'T'	/* stopped (Trace) */

char *tname(Dev_t dev_nr);
char *prrecv(struct pstat *bufp);
int main(int argc, char *argv[]);
void mmfs_gather(ptr_t list);
void kernel_gather1(int p_nr);
struct pstat *mm_getpid(int p_nr);
void mergesort(struct pstat **app);
struct pstat *find(int p_nr);
struct pstat *search(int p_nr);
char *off_to_k(off_t off);
int member(pid_t pid, char *pidlist);
char *get_args(struct pstat *bufp);
int addrread(int fd, phys_clicks base, vir_bytes addr, char *buf, int nbytes);
void usage(char *pname);
void err(char *s);
int gettynames(void);


/*
 * Tname returns mnemonic string for dev_nr. This is "?" for maj/min pairs that
 * are not found.  It uses the ttyinfo array (prepared by gettynames).
 * Tname assumes that the first three letters of the tty's name can be omitted
 * and returns the rest (except for the console, which yields "co").
 */
char *tname(Dev_t dev_nr)
{
  int i;

  if (majdev(dev_nr) == TTY_MAJ && mindev(dev_nr) == 0) return "co";

  for (i = 0; i < n_ttyinfo && ttyinfo[i].tty_name[0] != '\0'; i++)
	if (ttyinfo[i].tty_dev == dev_nr)
		return ttyinfo[i].tty_name + 3;

  return "?";
}

/* Prrecv prints the RECV field for process with pstat buffer pointer bufp.
 * This is either "ANY", "taskname", or "(blockreason) taskname".
 */
char *prrecv(struct pstat *bufp)
{
  char *blkstr, *task;		/* reason for blocking and task */
  static char recvstr[20];
  struct pstat *prec;

  if (bufp->ps_recv == ANY) return "ANY";

  prec= find(bufp->ps_recv);

  task = prec->ps_name;
  if (bufp->ps_state != S_STATE) return task;

  blkstr = "?";
  if (bufp->ps_recv == MM_PROC_NR) {
	if (bufp->ps_mflags & PAUSED)
		blkstr = "pause";
	else if (bufp->ps_mflags & WAITING)
		blkstr = "wait";
  } else if (bufp->ps_recv == FS_PROC_NR) {
  	switch(bufp->ps_fevent)
  	{
  	case EV_LOCK:		blkstr= "lock"; break;
  	case EV_POPEN:		blkstr= "popen"; break;
  	case EV_FWAIT:		blkstr= "fwait"; break;
  	case EV_READ:		blkstr= "read"; break;
  	case EV_WRITE:		blkstr= "write"; break;
  	case EV_IOCTL:		blkstr= "ioctl"; break;
  	default:		blkstr= "???"; break;
  	}
  }
  (void) sprintf(recvstr, "(%s) %s", blkstr, task);
  return recvstr;
}

/* Main interprets arguments, gets system addresses, opens [k]mem, reads in
 * process tables from kernel/mm/fs and calls pstat() for relevant entries.
 */
int main(int argc, char *argv[])
{
  int i, n;
  struct svrqueryparam qpar;
  char values[6 * (2 * sizeof(long) + 1)];
  char *pval;
  char pid[2 + sizeof(pid_t) * 3];
  char cpu[sizeof(clock_t) * 3 + 1 + 2];
  unsigned long ustime;
  char *args, *ap, *args0;
  ptr_t mproc_inuse, mproc_syslist, mproc_deadlist;
  size_t s_proc, s_proc0;
  struct pstat *pp, **app, *pref;

  uid = getuid();

  /* Parse arguments; a '-' need not be present (V7/BSD compatability) */
  i = 1;
  while (i < argc && (i == 1 || argv[i][0] == '-')) {
	char *opt = argv[i++];

	if (*opt == '-') {
		opt++;
		if (opt[0] == '-' && opt[1] == 0) break;
	}
	while (*opt != 0) {
		switch (*opt++) {
		case 'a':
			opt_all = TRUE;
			break;
		case 'e':
			opt_all = opt_notty = TRUE;
			break;
		case 'f':
		case 'l':
			opt_long = TRUE;
			break;
		case 'p':
			if (*opt == 0) {
				if (i == argc) usage(argv[0]);
				opt = argv[i++];
			}
			opt_pidlist = opt;
			if (opt_pidlist[0] == 0 || member(-1, opt_pidlist) < 0)
				usage(argv[0]);
			opt = "";
			break;
		case 'x':
			opt_notty = TRUE;
			break;
		default:
			usage(argv[0]);
		}
	}
  }
  if (i != argc) usage(argv[0]);

  if ((memfd = open(MEM_PATH, O_RDONLY)) == -1) err(MEM_PATH);

  if (gettynames() == -1) err("Can't get tty names");

  /* Get address, size, element size, and pointers of the MM process table. */
  qpar.param =
	"&mproc,$mproc,$mproc[0],mproc_inuse,mproc_syslist,mproc_deadlist";
  qpar.psize = strlen(qpar.param);
  qpar.value = values;
  qpar.vsize = sizeof(values);
  if (svrctl(MMQUERYPARAM, &qpar) < 0) err("svrctl(MMQUERYPARAM) failed");
  pval = values;
  if (paramvalue(&pval, &a_mproc, sizeof(a_mproc)) != sizeof(a_mproc)
  	|| paramvalue(&pval, &s_mproc, sizeof(s_mproc)) != sizeof(s_mproc)
  	|| paramvalue(&pval, &s_mproc0, sizeof(s_mproc0)) != sizeof(s_mproc0)
  	|| paramvalue(&pval, &mproc_inuse, sizeof(mproc_inuse))
						!= sizeof(mproc_inuse)
  	|| paramvalue(&pval, &mproc_syslist, sizeof(mproc_syslist))
						!= sizeof(mproc_syslist)
  	|| paramvalue(&pval, &mproc_deadlist, sizeof(mproc_deadlist))
						!= sizeof(mproc_deadlist)
  )
  	err("Can't get MM process data");

  /* Get size, element size, and proc name size of the kernel process table. */
  qpar.param = "$proc,$proc[0],$proc[0].p_name";
  qpar.psize = strlen(qpar.param);
  qpar.value = values;
  qpar.vsize = sizeof(values);
  if (svrctl(SYSQUERYPARAM, &qpar) < 0) err("svrctl(SYSQUERYPARAM) failed");
  pval = values;
  if (paramvalue(&pval, &s_proc, sizeof(s_proc)) != sizeof(s_proc)
  	|| paramvalue(&pval, &s_proc0, sizeof(s_proc0)) != sizeof(s_proc0)
  	|| paramvalue(&pval, &s_p_name, sizeof(s_p_name)) != sizeof(s_p_name)
  )
  	err("Can't get kernel process data");

  /* NR_PROCS == sizeof(mproc)/sizeof(mproc[0]). */
  nr_procs = s_mproc / s_mproc0;

  /* NR_TASKS + NR_PROCS == sizeof(proc)/sizeof(proc[0]). */
  nr_tasks = s_proc / s_proc0 - nr_procs;

  /* Gather MM & FS process table data by following MM's lists. */
  mmfs_gather(mproc_inuse);
  mmfs_gather(mproc_syslist);
  mmfs_gather(mproc_deadlist);

  /* Gather kernel process table data. */
  for (pp = proctab; pp != NULL; pp= pp->ps_next) kernel_gather1(pp->ps_nr);

  if (opt_all && opt_notty) {
	for (i = -nr_tasks; i < 0; i++) kernel_gather1(i);
  }

  /* Put the processes we are interested in on the sort list. */
  app= &procsort;
  for (pp = proctab; pp != NULL; pp= pp->ps_next) {
	if (!pp->ps_wanted) continue;
	*app= pp;
	app= &pp->ps_sort;
  }
  *app= NULL;

  /* Follow the recv and parent references. */
  for (pp = procsort; pp != NULL; pp= pp->ps_sort) {
	if (opt_long && (pp->ps_flags & RECEIVING) && pp->ps_recv != ANY) {
		pref= find(pp->ps_recv);
		if (pref->ps_state == NO_STATE)
			kernel_gather1(pp->ps_recv);
	}
	if (pp->ps_parent != 0) {
		pref= search(pp->ps_parent);
		if (pref == NULL) pref= mm_getpid(pp->ps_parent);
		pp->ps_ppid= pref->ps_pid;
	}
  }

  /* Sort the result. */
  if (procsort != NULL && procsort->ps_sort != NULL) mergesort(&procsort);

  args0= alloca(s_p_name + 4);

  /* Now loop through process table and handle each entry */
  printf("%s", opt_long ? L_HEADER : S_HEADER);
  for (pp = procsort; pp != NULL; pp= pp->ps_sort) {
	args0[0]= 0;
	if (pp->ps_nr <= INIT_PROC_NR)
		args = pp->ps_name;
	else
	if (pp->ps_state == Z_STATE)
		args = "<defunct>";
	else {
		if ((args= get_args(pp)) == NULL) args= "";

		/* First argument matches ps_name somehow? */
		n= strlen(pp->ps_name);
		for (ap= args; *ap != 0 && *ap != ' '; ap++) {
			if (strncmp(ap, pp->ps_name, n) == 0) {
				n= 0;
				break;
			}
		}

		/* Insert "(ps_name)" before the argument strings? */
		if (n > 0) {
			args0[0]= '(';
			strcpy(args0+1, pp->ps_name);
			strcat(args0, ")");
			if (args[0] != 0) strcat(args0, " ");
		}
	}

	if (pp->ps_pid == 0) {
		sprintf(pid, "(%d)", pp->ps_nr);
	} else {
		sprintf(pid, "%d", pp->ps_pid);
	}

	ustime = (pp->ps_utime + pp->ps_stime) / HZ;
	if (ustime < 60 * 60) {
		sprintf(cpu, "%2lu:%02lu", ustime / 60, ustime % 60);
	} else
	if (ustime < 100 * 60 * 60) {
		sprintf(cpu, "%2luh%02lu", ustime / 3600, (ustime / 60) % 60);
	} else {
		sprintf(cpu, "%4luh", ustime / 3600);
	}

	if (opt_long) {
		printf(L_FORMAT,
			pp->ps_flags, pp->ps_state,
			pp->ps_euid, pid, pp->ps_ppid,
			pp->ps_pgrp,
			off_to_k((pp->ps_tsize
				 + pp->ps_dsize
				 + pp->ps_ssize)),
			(pp->ps_flags & RECEIVING ? prrecv(pp) : ""),
			tname((Dev_t) pp->ps_dev),
			cpu, args0, args);
	} else {
		printf(S_FORMAT,
			pid, tname((Dev_t) pp->ps_dev),
			cpu, args0, args);
	}
  }
  return(0);
}

void mmfs_gather(ptr_t list)
{
/* Follow one of MM's lists of processes and gather data. */
  struct svrqueryparam qpar;
  char param[200];
  char values[7 * (2 * sizeof(long) + 1)];
  char *pval;
  unsigned mp_flags;
  uid_t mp_realuid, mp_effuid;
  pid_t mp_pid, mp_procgrp;
  pid_t mp_parent;
  ptr_t mp_next;
  dev_t fp_tty;
  int fp_event;
  int p_nr;
  struct pstat *pp;
  int wanted;

  while (list != 0) {
	p_nr = (list - a_mproc) / s_mproc0;

	/* Get parameters from MM. */
	sprintf(param,
		"mproc[%d].mp_flags,"
		"mproc[%d].mp_next,"
		"mproc[%d].mp_pid,"
		"mproc[%d].mp_procgrp,"
		"mproc[%d].mp_realuid,"
		"mproc[%d].mp_effuid,"
		"mproc[%d].mp_parent",
		p_nr, p_nr, p_nr, p_nr, p_nr, p_nr, p_nr);

	qpar.param = param;
	qpar.psize = sizeof(param);
	qpar.value = values;
	qpar.vsize = sizeof(values);
	if (svrctl(MMQUERYPARAM, &qpar) < 0)
		err("svrctl(MMQUERYPARAM) failed");
	pval = values;
	if (paramvalue(&pval, &mp_flags, sizeof(mp_flags))
							!= sizeof(mp_flags)
		|| paramvalue(&pval, &mp_next, sizeof(mp_next))
							!= sizeof(mp_next)
		|| paramvalue(&pval, &mp_pid, sizeof(mp_pid))
							!= sizeof(mp_pid)
		|| paramvalue(&pval, &mp_procgrp, sizeof(mp_procgrp))
							!= sizeof(mp_procgrp)
		|| paramvalue(&pval, &mp_realuid, sizeof(mp_realuid))
							!= sizeof(mp_realuid)
		|| paramvalue(&pval, &mp_effuid, sizeof(mp_effuid))
							!= sizeof(mp_effuid)
		|| paramvalue(&pval, &mp_parent, sizeof(mp_parent))
							!= sizeof(mp_parent)
	) {
		err("Can't get MM process data");
	}

	/* Next process in MM's list. */
	list= mp_next;

	/* Do we want this process? */
	wanted= 1;
	if (opt_pidlist != NULL) {
		if (!member(mp_pid, opt_pidlist)) wanted= 0;
	} else {
		if (!opt_all && mp_realuid != uid && mp_effuid != uid)
			wanted= 0;
	}

	if (wanted) {
		/* Get tty data from FS. */
		sprintf(param,
			"fproc[%d].fp_tty,"
			"fproc[%d].fp_event",
			p_nr, p_nr);

		qpar.param = param;
		qpar.psize = sizeof(param);
		qpar.value = values;
		qpar.vsize = sizeof(values);
		if (svrctl(FSQUERYPARAM, &qpar) < 0)
			err("svrctl(FSQUERYPARAM) failed");
		pval = values;
		if (paramvalue(&pval, &fp_tty, sizeof(fp_tty))
							!= sizeof(fp_tty)
			|| paramvalue(&pval, &fp_event, sizeof(fp_event))
							!= sizeof(fp_event)
		) {
			err("Can't get FS process data");
		}
	}

	/* Do we still want it? */
	if (opt_pidlist == NULL && !opt_notty && !tty_major[majdev(fp_tty)])
		wanted= 0;

	/* Put the collected data into a process status structure. */
	pp= find(p_nr);
	pp->ps_wanted= wanted;
	pp->ps_mflags= mp_flags;
	pp->ps_ruid= mp_realuid;
	pp->ps_euid= mp_effuid;
	pp->ps_pid= mp_pid;
	pp->ps_pgrp= mp_procgrp;
	pp->ps_parent= mp_parent;
	pp->ps_dev= fp_tty;
	pp->ps_fevent= fp_event;
  }
}

void kernel_gather1(int p_nr)
{
  struct svrqueryparam qpar;
  char param[200];
  size_t s_values;
  char *values;
  char *pval;
  struct pstat *pp;
  int p_flags;
  struct mem_map p_map[NR_SEGS];
  int p_tofrom;
  clock_t user_time;
  clock_t sys_time;
  char *p_name;
  int k_nr;

  /* Arrays */
  s_values= 4 * (2 * sizeof(long) + 1) + (2 * sizeof(p_map) + 1)
						+ (2 * s_p_name + 1);
  values= alloca(s_values);
  p_name= alloca(s_p_name);

  /* Index in the kernel table. */
  k_nr= p_nr + nr_tasks;

  /* Get parameters from the kernel. */
  sprintf(param,
	"proc[%d].p_flags,"
	"proc[%d].user_time,"
	"proc[%d].sys_time,"
	"proc[%d].p_name,"
	"proc[%d].p_map,"
	"proc[%d].p_tofrom",
	k_nr, k_nr, k_nr, k_nr, k_nr, k_nr);

  qpar.param = param;
  qpar.psize = sizeof(param);
  qpar.value = values;
  qpar.vsize = s_values;
  if (svrctl(SYSQUERYPARAM, &qpar) < 0)
	err("svrctl(SYSQUERYPARAM) failed");
  pval = values;
  if (paramvalue(&pval, &p_flags, sizeof(p_flags))
						!= sizeof(p_flags)
	|| paramvalue(&pval, &user_time, sizeof(user_time))
						!= sizeof(user_time)
	|| paramvalue(&pval, &sys_time, sizeof(sys_time))
						!= sizeof(sys_time)
	|| paramvalue(&pval, p_name, s_p_name)
						!= s_p_name
	|| paramvalue(&pval, &p_map, sizeof(p_map))
						!= sizeof(p_map)
	|| paramvalue(&pval, &p_tofrom, sizeof(p_tofrom))
						!= sizeof(p_tofrom)
  ) {
	err("Can't get kernel process data");
  }

  /* Something there? */
  if ((pp= search(p_nr)) == NULL) {
  	if ((p_flags & P_SLOT_FREE)) return;
  	pp= find(p_nr);
  }

  /* Add the kernel data to the process structure. */
  pp->ps_flags = p_flags;

  /* State is interpretation of combined kernel/mm flags for non-tasks */
  if (pp->ps_nr >= 0) {		/* non-tasks */
	if (pp->ps_mflags & HANGING)
		pp->ps_state = Z_STATE;		/* zombie */
	else
	if (pp->ps_mflags & STOPPED)
		pp->ps_state = T_STATE;		/* stopped (traced) */
	else
	if (p_flags == 0)
		pp->ps_state = R_STATE;		/* in run-queue */
	else
	if (pp->ps_mflags & (WAITING | PAUSED) || pp->ps_fevent != EV_NONE)
		pp->ps_state = S_STATE;		/* sleeping */
	else
		pp->ps_state = W_STATE;		/* a short wait */
  } else {			/* tasks are simple */
	if (p_flags == 0)
		pp->ps_state = R_STATE;		/* in run-queue */
	else
		pp->ps_state = W_STATE;		/* other i.e. waiting */
  }

  pp->ps_tsize = (size_t) p_map[SEG_T].mem_len << CLICK_SHIFT;
  pp->ps_dsize = (size_t) p_map[SEG_D].mem_len << CLICK_SHIFT;
  pp->ps_ssize = (size_t) p_map[SEG_S].mem_len << CLICK_SHIFT;
  pp->ps_text = (off_t) p_map[SEG_T].mem_phys << CLICK_SHIFT;
  pp->ps_data = (off_t) p_map[SEG_D].mem_phys << CLICK_SHIFT;
  pp->ps_stack = (off_t) p_map[SEG_S].mem_phys << CLICK_SHIFT;
#if (CHIP == INTEL)
  pp->ps_stack += (off_t) p_map[SEG_S].mem_vir << CLICK_SHIFT;
#endif

  pp->ps_recv = p_tofrom;

  pp->ps_utime = user_time;
  pp->ps_stime = sys_time;
  memcpy(pp->ps_name, p_name, s_p_name);
}

struct pstat *mm_getpid(int p_nr)
/* Look for the process id of a given process.  Only for finding the as yet
 * unknown parent of a process we are interested in.
 */
{
	struct pstat *pp;
	struct svrqueryparam qpar;
	char param[14 + 3*sizeof(int)];
	char values[2 * sizeof(pid_t) + 1];
	char *pval;
	pid_t mp_pid;

	/* Get parameters from MM. */
	sprintf(param, "mproc[%d].mp_pid", p_nr);

	qpar.param = param;
	qpar.psize = sizeof(param);
	qpar.value = values;
	qpar.vsize = sizeof(values);
	if (svrctl(MMQUERYPARAM, &qpar) < 0)
		err("svrctl(MMQUERYPARAM) failed");
	pval = values;
	if (paramvalue(&pval, &mp_pid, sizeof(mp_pid))
					!= sizeof(mp_pid)) {
		err("Can't get MM process data");
	}
	pp= find(p_nr);
	pp->ps_pid= mp_pid;
	return pp;
}

void mergesort(struct pstat **app)
/* This is either a stable mergesort, or thermal noise, I'm no longer sure.
 * It must be called like this: if (L != NULL && L->next != NULL) mergesort(&L);
 */
{
	/* static */ struct pstat *pp1, **mid;  /* Need not be local */
	struct pstat *pp2;

	pp1= *(mid= &(*app)->ps_sort);
	do {
		if ((pp1= pp1->ps_sort) == NULL) break;
		mid= &(*mid)->ps_sort;
	} while ((pp1= pp1->ps_sort) != NULL);

	pp2= *mid;
	*mid= NULL;

	if ((*app)->ps_sort != NULL) mergesort(app);
	if (pp2->ps_sort != NULL) mergesort(&pp2);

	pp1= *app;
	for (;;) {
		/* Group on parent processes, then sort on process id. */
		if (pp1->ps_ppid < pp2->ps_ppid
				|| (pp1->ps_ppid == pp2->ps_ppid
					&& pp1->ps_pid <= pp2->ps_pid)) {
			if ((pp1= *(app= &pp1->ps_sort)) == NULL) {
				*app= pp2;
				break;
			}
		} else {
			*app= pp2;
			pp2= *(app= &pp2->ps_sort);
			*app= pp1;
			if (pp2 == NULL) break;
		}
	}
}

struct pstat *find(int p_nr)
{
/* Obtain a slot for a process. */
  struct pstat *pp, **app;

  app= &proctab;
  for (;;) {
	if ((pp= *app) == NULL || p_nr < pp->ps_nr) {
		/* Make a new slot. */
		if ((pp= malloc(sizeof(*pp) + s_p_name + 1)) == NULL)
			err("Out of memory");
		memset(pp, 0, sizeof(*pp));
		pp->ps_name= (char *) pp + sizeof(*pp);
		pp->ps_name[0]= 0;
		pp->ps_nr= p_nr;
		pp->ps_wanted= 1;
		pp->ps_next= *app;
		*app= pp;
		break;
	}
	if (p_nr == pp->ps_nr) break;
	app= &pp->ps_next;
  }
  return pp;
}

struct pstat *search(int p_nr)
{
/* See if a process is in our tables. */
  struct pstat *pp;

  for (pp= proctab; pp != NULL && p_nr >= pp->ps_nr; pp= pp->ps_next) {
	if (p_nr == pp->ps_nr) return pp;
  }
  return NULL;
}

char *off_to_k(off_t off)
{
/* Convert memory offsets to rounded kilo-units or mega-units */
  unsigned kb;
  static char mem[1 + sizeof(kb) * 3];

  kb = (off + 512) / 1024;
  if (kb < 10000) {
	sprintf(mem, "%u", kb);
  } else {
	sprintf(mem, "%uM", (kb + 512) / 1024);
  }
  return mem;
}

int member(pid_t pid, char *pidlist)
{
  unsigned long p;
  char *next;

  while (*pidlist != 0) {
	p = strtoul(pidlist, &next, 10);
	if (next == pidlist || (*next != ',' && *next != 0)) return -1;
	if (pid != -1 && pid == p) return 1;
	pidlist = next;
	if (*pidlist == ',') pidlist++;
  }
  return 0;
}

/* Get_args inspects /dev/mem, using bufp, and tries to locate the initial
 * stack frame pointer, i.e. the place where the stack started at exec time.
 * It is assumed that the end of the stack frame looks as follows:
 *	argc	<-- initial stack frame starts here
 *	argv[0]
 *	...
 *	NULL	(*)
 *	envp[0]
 *	...
 *	NULL	(**)
 *	argv[0][0] ... '\0'
 *	...
 *	argv[argc - 1][0] ... '\0'
 *	envp[0][0] ... '\0'
 *	...
 *	[trailing '\0']
 * Where the total space occupied by this original stack frame <= ARG_MAX.
 * Get_args reads in the last ARG_MAX bytes of the process' data, and
 * searches back for two NULL ptrs (hopefully the (*) & (**) above).
 * If it finds such a portion, it continues backwards, counting ptrs until:
 * a) either a word is found that has as its value the count (supposedly argc),
 * b) another NULL word is found, in which case the algorithm is reiterated, or
 * c) we wind up before the start of the buffer and fail.
 * Upon success, get_args returns a pointer to the conactenated arg list.
 * Warning: this routine is inherently unreliable and probably doesn't work if
 * ptrs and ints have different sizes.
 */
char *get_args(struct pstat *bufp)
{
  static union {
	ptr_t stk_i;
	char *stk_cp;
	char stk_c;
  } stk[64*1024 / sizeof(char *)], *sp;
  enum {
	INITIAL, DONE, FAIL, NULL1, NULL2
  } state;
  int nargv;			/* guessed # of (non-NULL) argv pointers seen */
  int cnt;			/* # of bytes read from stack frame */
  int neos;			/* # of '\0's seen in argv string space */
  off_t l;
  char *cp, *args;


  if (bufp->ps_ssize < 64*1024)
	cnt = bufp->ps_ssize;
  else
	cnt = 64*1024;

  /* Get last cnt bytes from user stack */
  if (addrread(memfd, (phys_clicks) (bufp->ps_stack >> CLICK_SHIFT),
	     (vir_bytes) (bufp->ps_ssize - cnt),
	     (char *) stk, cnt) != cnt)
	return NULL;

  state = INITIAL;
  sp = &stk[cnt / sizeof(char *)];
  while (state != DONE && state != FAIL) {
	if (sp-- == &stk[0])
		state = FAIL;	/* wound up before start of buffer */
	switch (state) {
	    case INITIAL:	/* no NULL seen yet */
		if (sp[0].stk_cp == NULL) state = NULL1;
		break;
	    case NULL1:		/* one NULL seen */
		if (sp[0].stk_cp == NULL) {
			nargv = 0;	/* start counting argv ptrs */
			state = NULL2;
		}

		/* What follows is a dirty patch to recognize sh's
		 * stack frame when it has assigned argv[0] to
		 * argv[1], and has thus blown away its NULL pointer
		 * there. */
		else if (sp > &stk[0] && sp[0].stk_cp == sp[-1].stk_cp) {
			nargv = 0;
			state = NULL2;
		}
		break;
	    case NULL2:		/* two NULLs seen */
		if (sp[0].stk_cp == NULL)
			nargv = 0;	/* restart counting */
		else if (sp[0].stk_i == nargv)
			state = DONE;	/* think i got it */
		/* Next is same ugly patch as above */
		else if (sp > &stk[0] && sp[0].stk_cp == sp[-1].stk_cp)
			nargv = 0;
		else
			nargv++;/* ? some argv pointer ? */
		break;
	    default:		/* FAIL or DONE */
		break;
	}
  }

  if (state != DONE) return NULL;

  /* Get a local version of argv[0]; l is offset back from end of stack */
  l = bufp->ps_stack + bufp->ps_ssize -
#if (CHIP == INTEL)
	bufp->ps_data -
#endif
	(vir_bytes) sp[1].stk_cp;
  if (l < 0 || l > cnt) return NULL;
  args = &((char *) stk)[cnt - (int) l];
  neos = 0;
  for (cp = args; cp < &((char *) stk)[cnt]; cp++)
	if (*cp == '\0')
		if (++neos >= sp[0].stk_i)
			break;
		else
			*cp = ' ';
  if (neos != sp[0].stk_i) return NULL;

  return args;
}

/* Addrread reads nbytes from offset addr to click base of fd into buf. */
int addrread(int fd, phys_clicks base, vir_bytes addr, char *buf, int nbytes)
{
  if (lseek(fd, ((long) base << CLICK_SHIFT) + (long) addr, 0) < 0)
	return -1;

  return read(fd, buf, nbytes);
}

void usage(char *pname)
{
  fprintf(stderr, "Usage: %s [-][alx] [-p pidlist]\n", pname);
  exit(1);
}

void err(char *s)
{
  extern int errno;

  if (errno == 0)
	fprintf(stderr, "ps: %s\n", s);
  else
	perror(s);

  exit(2);
}

/* Fill ttyinfo by fstatting character specials in /dev. */
int gettynames(void)
{
  static char dev_path[] = "/dev/";
  struct stat statbuf;
  static char path[sizeof(dev_path) + NAME_MAX];
  int index;
  struct ttyent *ttyp;

  index = 0;
  while ((ttyp = getttyent()) != NULL) {
	strcpy(path, dev_path);
	strcat(path, ttyp->ty_name);
	if (stat(path, &statbuf) == -1 || !S_ISCHR(statbuf.st_mode))
		continue;
	if (index >= n_ttyinfo) {
		n_ttyinfo= (index+16) * 2;
		ttyinfo = realloc(ttyinfo, n_ttyinfo * sizeof(ttyinfo[0]));
		if (ttyinfo == NULL) err("Out of memory");
	}
	ttyinfo[index].tty_dev = statbuf.st_rdev;
	tty_major[majdev(statbuf.st_rdev)]= 1;
	strcpy(ttyinfo[index].tty_name, ttyp->ty_name);
	index++;
  }
  endttyent();
  while (index < n_ttyinfo) ttyinfo[index++].tty_dev= 0;

  return 0;
}

/*
 * $PchId: ps.c,v 1.6 1996/02/29 23:19:37 philip Exp $
 */
