#define INODE_MAX ((unsigned) 65535)

/* mkfs  -  make the MINIX filesystem	Authors: A. Tanenbaum & P. Ogilvie */


/*		 Andy Tanenbaum & Paul Ogilvie, June 1986
 *
 *	This program was initially designed to build a filesystem
 *	with blocksize = zonesize. During the course of time the
 *	program is being converted to handle zone_size > blocksize
 *	but this isn't complete yet. Where routines can handle the
 *	situation this is mentioned in the comment.
 *
 *	To compile this program for MS-DOS, use: cc -DDOS mkfs.c diskio.asm
 *	To compile this program for UNIX,   use: cc -DUNIX mkfs.c
 *	To compile this program for MINIX,  use: cc mkfs.c
 */

#define _MINIX_SOURCE

#include <sys/types.h>
#include <dirent.h>
#include <fcntl.h>
#include <limits.h>
#include <minix/config.h>
#include <minix/const.h>
#include <minix/type.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fs/const.h>

#undef EXTERN
#define EXTERN			/* get rid of EXTERN by making it null */
#include <fs/type.h>
#include <fs/super.h>

/* Temporary macros to handle the two V2 versions. */
#if V2_NR_DZONES == 10
#define OLD_V2		1	/* old version of V2 file system */
#define Z_STRIDE	3	/* stride for zone nrs in inodes (3 bytes) */
#define chkizones	chk3zones
#define dtoz(cptr)	c3_i4(cptr)
#define ztod(cptr, z)	i4_c3((cptr), (z))
#else				/* new V2 */
#define NEW_V2		1	/* new version of V2 file system */
#define Z_STRIDE	1	/* stride for zone nrs in inodes (1 long) */
#define chkizones	chkzones
#define dtoz(lptr)	(*(lptr))
#define ztod(lptr, z)	(*(lptr) = (z))
#endif


#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <minix/minlib.h>


#define STICKY_BIT	01000	/* not defined anywhere else */
#ifndef DOS
#undef printf			/* printf is a macro for printk */
#ifndef UNIX
#define UNIX
#endif
#endif

#define INODE_MAP            2
#define MAX_TOKENS          10
#define LINE_LEN           200
#define BIT_MAP_SHIFT       13
#define N_BLOCKS       (Z_MAP_SLOTS * BLOCK_SIZE * 8L)

#ifdef DOS
   maybedefine O_RDONLY	     4	/* O_RDONLY | BINARY_BIT */
   maybedefine BWRITE	     5	/* O_WRONLY | BINARY_BIT */
#endif

#if (MACHINE == ATARI)
int	isdev;
#endif

int next_zone, next_inode, zone_size, zone_shift = 0, zoff;
block_t nrblocks;
int inode_offset, nrinodes, lct = 1, disk, fd, print = 0, file = 0;
int override = 0, simple = 0, dflag;
int donttest;			/* skip test if it fits on medium */

long current_time, bin_time;
char zero[BLOCK_SIZE], *lastp;
char umap[(N_BLOCKS + 8) / 8];	/* bit map tells if block read yet */
block_t zone_map;		/* where is zone map? (depends on # inodes) */

FILE *proto;
char *protoarg;
char gwarning[] = {65, 46, 83, 46, 84, 97, 110, 101, 110, 98, 97, 117, 109, 10};
char *mtab = "/etc/mtab";	/* file where mount table is kept */

_PROTOTYPE(int main, (int argc, char **argv));
_PROTOTYPE(int super, (zone_t zones, Ino_t inodes));
_PROTOTYPE(PRIVATE int bitmapsize, (bit_t nr_bits));
_PROTOTYPE(int rootdir, (Ino_t inode));
_PROTOTYPE(int eat_dir, (Ino_t parent));
_PROTOTYPE(int eat_file, (Ino_t inode, int f));
_PROTOTYPE(int enter_dir, (Ino_t parent, char *name, Ino_t child));
_PROTOTYPE(int add_zone, (Ino_t n, zone_t z, long bytes, long cur_time));
_PROTOTYPE(int incr_link, (Ino_t n));
_PROTOTYPE(int incr_size, (Ino_t n, long count));
_PROTOTYPE(PRIVATE ino_t alloc_inode, (int mode, int usrid, int grpid));
_PROTOTYPE(PRIVATE zone_t alloc_zone, (void));
_PROTOTYPE(int insert_bit, (block_t block, int bit));
_PROTOTYPE(int mode_con, (char *p));
_PROTOTYPE(int getline, (char line [LINE_LEN ], char *parse [MAX_TOKENS ]));
_PROTOTYPE(long file_time, (int f));
_PROTOTYPE(int pexit, (char *s));
_PROTOTYPE(int copy, (char *from, char *to, int count));
_PROTOTYPE(int print_fs, (void));
_PROTOTYPE(int read_and_set, (block_t n));
_PROTOTYPE(int special, (char *string));
_PROTOTYPE(int get_block, (block_t n, char buf [BLOCK_SIZE ]));
_PROTOTYPE(int put_block, (block_t n, char buf [BLOCK_SIZE ]));
_PROTOTYPE(int cache_init, (void));
_PROTOTYPE(int flush, (void));
_PROTOTYPE(int mx_read, (int blocknr, char buf [BLOCK_SIZE ]));
_PROTOTYPE(int mx_write, (int blocknr, char buf [BLOCK_SIZE ]));
_PROTOTYPE(int dexit, (char *s, int sectnum, int err));
_PROTOTYPE(int special, (char *string));
_PROTOTYPE(int get_block, (block_t n, char *buf));
_PROTOTYPE(int put_block, (block_t n, char buf [BLOCK_SIZE ]));
_PROTOTYPE(int flush, (void));
_PROTOTYPE(int cache_init, (void));
_PROTOTYPE(PRIVATE long c3_i4, (char *c3));
_PROTOTYPE(PRIVATE void i4_c3, (char *c3, long i4));

/*================================================================
 *                    mkfs  -  make filesystem
 *===============================================================*/

main(argc, argv)
int argc;
char *argv[];
{
  int nread, mode, usrid, grpid, badusage = 0;
  block_t blocks;
  block_t i;
  ino_t inodes, root_inum;
  zone_t zones;
  char *token[MAX_TOKENS], line[LINE_LEN];
  struct stat statbuf;


  /* Get two times, the current time and the mod time of the binary of
   * mkfs itself.  When the -d flag is used, the later time is put into
   * the i_mtimes of all the files.  This feature is useful when
   * producing a set of file systems, and one wants all the times to be
   * identical. First you set the time of the mkfs binary to what you
   * want, then go. */
  current_time = time(0L);	/* time mkfs is being run */
  stat(argv[0], &statbuf);
  bin_time = statbuf.st_mtime;	/* time when mkfs binary was last modified */

  /* Process parameters and switches */
  if (argc != 3 && argc != 4) badusage = 1;
  if (stat(argv[argc - 1], &statbuf) == 0) {
	if ((statbuf.st_mode & S_IFMT) != S_IFREG) badusage = 1;
  }
  if (badusage) {
	write(2, "Usage: mkfs [-ldt] special proto\n", 33);
	exit(1);
  }

  while (--argc) {
	switch (argv[argc][0]) {
	    case '-':
		while (*++argv[argc]) switch (*argv[argc]) {
			    case 'l':
			    case 'L':	print = 1;	break;
			    case 'o':
			    case 'O':
				override = 1;
				break;
			    case 'd':
			    case 'D':
				current_time = bin_time;
				dflag = 1;
				break;
			    case 't':
			    case 'T':
				donttest = 1;
				break;
			    default:
				printf("Bad switch %c, ignored.\n", *argv[argc]);
			}
		break;

	    default:

		/* Process proto & special */
		protoarg = argv[argc];
		proto = fopen(protoarg, "r");
		if (proto != NULL || strncmp(protoarg, "boot\n", 5) == 0) {
			/* Prototype file is readable. */
			getline(line, token);	/* skip boot block info. */

			/* Read the line with the block and inode counts. */
			getline(line, token);
			blocks = atol(token[0]);
			if (blocks > N_BLOCKS)
				pexit("Block count too large");
			inodes = atoi(token[1]);

			/* Process mode line for root directory. */
			getline(line, token);
			mode = mode_con(token[0]);
			usrid = atoi(token[1]);
			grpid = atoi(token[2]);

		} else {

			/* Maybe the prototype file is just a size.
			 * Check for that. */
			blocks = atol(argv[argc]);
			if (blocks == 0) pexit("Can't open prototype file");
			if (blocks < 4) pexit("Block count too small");
			if (blocks > N_BLOCKS) pexit("Block count too large");

			/* Ok, make simple file system of given size, using
			 * defaults.  The default for inodes is 3 blocks per
			 * inode, rounded up to fill an inode block .
			 */
			i = (blocks/3 + V2_INODES_PER_BLOCK - 1);
			i = i / V2_INODES_PER_BLOCK * V2_INODES_PER_BLOCK;
			if (i > INODE_MAX) i = INODE_MAX;
			inodes = (ino_t) i;
			mode = 041777;
			usrid = 0;
			grpid = 0;
			simple = 1;
		}

		/* Open special */
		argc--;
		special(argv[argc]);

		nrblocks = blocks;
		nrinodes = inodes;
	}			/* end switch */
  }				/* end while */



#ifdef UNIX
  if (!donttest) {
	static short testb[BLOCK_SIZE / sizeof(short)];

	/* Try writing the last block of partition or diskette. */
	lseek(fd, (off_t) (blocks - 1) * BLOCK_SIZE, SEEK_SET);
	testb[0] = 0x3245;
	testb[1] = 0x11FF;
	if (write(fd, (char *) testb, BLOCK_SIZE) != BLOCK_SIZE)
		pexit("File system is too big for minor device");
	sync();			/* flush write, so if error next read fails */
	lseek(fd, (off_t) (blocks - 1) * BLOCK_SIZE, SEEK_SET);
	testb[0] = 0;
	testb[1] = 0;
	nread = read(fd, (char *) testb, BLOCK_SIZE);
	if (nread != BLOCK_SIZE || testb[0] != 0x3245 || testb[1] != 0x11FF)
		pexit("File system is too big for minor device");
	lseek(fd, (off_t) (blocks - 1) * BLOCK_SIZE, SEEK_SET);
	testb[0] = 0;
	testb[1] = 0;
	if (write(fd, (char *) testb, BLOCK_SIZE) != BLOCK_SIZE)
		pexit("File system is too big for minor device");
	lseek(fd, 0L, SEEK_SET);
  }
#endif

  /* Make the file-system */

  cache_init();
#if (MACHINE == ATARI)
  if (isdev) {
    char block0[BLOCK_SIZE];
    get_block((block_t) 0, block0);	
    /* need to read twice; first time gets an empty block */
    get_block((block_t) 0, block0);
    /* zero parts of the boot block so the disk won't be recognized as
       a tos disk any more. */
    block0[0] = block0[1] = 0;	/* branch code to boot code    */
    strncpy(&block0[2], "MINIX ", 6);
    block0[16] = 0;		/* number of FATS              */
    block0[17] = block0[18] = 0;/* number of dir entries       */
    block0[22] = block0[23] = 0;/* sectors/FAT                 */
    bzero(&block0[30], 480);	/* boot code                   */
    put_block((block_t) 0, block0);	
  }
  else
#endif
  put_block((block_t) 0, zero);	/* Write a null boot block. */

  zone_shift = 0;		/* for future use */
  zones = nrblocks >> zone_shift;

  super(zones, inodes);

  root_inum = alloc_inode(mode, usrid, grpid);
  rootdir(root_inum);
  if (simple == 0) eat_dir(root_inum);

  if (print) print_fs();
  flush();
  exit(0);

  /* NOTREACHED */
}				/* end main */




/*================================================================
 *                 super  -  construct a superblock
 *===============================================================*/

super(zones, inodes)
zone_t zones;
ino_t inodes;
{

  unsigned int i;
  int inodeblks;
  int initblks;
  zone_t initzones;
  zone_t nrzones;
  zone_t zo;
  struct super_block *sup;
  char buf[BLOCK_SIZE], *cp;

  for (cp = buf; cp < &buf[BLOCK_SIZE]; cp++) *cp = 0;
  sup = (struct super_block *) buf;	/* lint - might use a union */

  sup->s_ninodes = inodes;
  sup->s_nzones = 0;		/* not used in V2 - 0 forces errors early */
  sup->s_zones = zones;
  sup->s_flags = S_FLEX|S_CLEAN;  /* making an fs with flex dir entries */
  sup->s_imap_blocks = bitmapsize((bit_t) (1 + inodes));
  sup->s_zmap_blocks = bitmapsize((bit_t) zones);
  inode_offset = sup->s_imap_blocks + sup->s_zmap_blocks + 2;
  inodeblks = (inodes + V2_INODES_PER_BLOCK - 1) / V2_INODES_PER_BLOCK;
  initblks = inode_offset + inodeblks;
  initzones = (initblks + (1 << zone_shift) - 1) >> zone_shift;
  nrzones = nrblocks >> zone_shift;
  sup->s_firstdatazone = (initblks + (1 << zone_shift) - 1) >> zone_shift;
  zoff = sup->s_firstdatazone - 1;
  sup->s_log_zone_size = zone_shift;
  sup->s_magic = SUPER_V2;	/* identify super blocks */
  zo = V2_NR_DZONES + (zone_t) V2_INDIRECTS + (zone_t) V2_INDIRECTS * V2_INDIRECTS;
  sup->s_max_size = zo * BLOCK_SIZE;
  sup->s_fsck_magic[0] = FSCK_MAGIC0;	/* Enable clean flag. */
  sup->s_fsck_magic[1] = FSCK_MAGIC1;
  zone_size = 1 << zone_shift;	/* nr of blocks per zone */

  put_block((block_t) 1, buf);

  /* Clear maps and inodes. */
  for (i = 2; i < initblks; i++) put_block((block_t) i, zero);

  next_zone = sup->s_firstdatazone;
  next_inode = 1;

  zone_map = INODE_MAP + sup->s_imap_blocks;

  insert_bit(zone_map, 0);	/* bit zero must always be allocated */
  insert_bit((block_t) INODE_MAP, 0);	/* inode zero not used but must be
				 * allocated */
}




/* The next routine is copied from fsck.c.  Modify some names for consistency.
 * This sharing should be done better.
 */
#define BITMAPSHIFT BIT_MAP_SHIFT

/* Convert from bit count to a block count. The usual expression
 *
 *	(nr_bits + (1 << BITMAPSHIFT) - 1) >> BITMAPSHIFT
 *
 * doesn't work because of overflow.
 *
 * Other overflow bugs, such as the expression for N_ILIST overflowing when
 * s_inodes is just over V2_INODES_PER_BLOCK less than the maximum+1, are not
 * fixed yet, because that number of inodes is silly.
 */
 
/* The above comment doesn't all apply now bit_t is long.  Overflow is now
 * unlikely, but negative bit counts are now possible (though unlikely)
 * and give silly results.
 */ 
PRIVATE int bitmapsize(nr_bits)
bit_t nr_bits;
{
  int nr_blocks;

  nr_blocks = (int) (nr_bits >> BITMAPSHIFT);  /* lint but OK */
  if (((bit_t) nr_blocks << BITMAPSHIFT) < nr_bits) ++nr_blocks;
  return(nr_blocks);
}





/*================================================================
 *              rootdir  -  install the root directory
 *===============================================================*/

rootdir(inode)
ino_t inode;
{
  zone_t z;

  z = alloc_zone();
  add_zone(inode, z, 16L, current_time);
  enter_dir(inode, ".", inode);
  enter_dir(inode, "..", inode);
  incr_link(inode);
  incr_link(inode);
}





/*================================================================
 *	    eat_dir  -  recursively install directory
 *===============================================================*/

eat_dir(parent)
ino_t parent;
{
  /* Read prototype lines and set up directory. Recurse if need be. */
  char *token[MAX_TOKENS], *p;
  char line[LINE_LEN];
  int mode, usrid, grpid, maj, min, f;
  ino_t n;
  zone_t z;
  long size;

  while (1) {
	getline(line, token);
	p = token[0];
	if (*p == '$') return;
	p = token[1];
	mode = mode_con(p);
	usrid = atoi(token[2]);
	grpid = atoi(token[3]);
	if (grpid & 0200) write(2, gwarning, 14);
	n = alloc_inode(mode, usrid, grpid);

	/* Enter name in directory and update directory's size. */
	enter_dir(parent, token[0], n);
	incr_size(parent, (1 + _EXTENT(strlen(token[0]))) * 8L);

	/* Check to see if file is directory or special. */
	incr_link(n);
	if (*p == 'd') {
		/* This is a directory. */
		z = alloc_zone();	/* zone for new directory */
		add_zone(n, z, 16L, current_time);
		enter_dir(n, ".", n);
		enter_dir(n, "..", parent);
		incr_link(parent);
		incr_link(n);
		eat_dir(n);
	} else if (*p == 'b' || *p == 'c') {
		/* Special file. */
		maj = atoi(token[4]);
		min = atoi(token[5]);
		size = 0;
		if (token[6]) size = atoi(token[6]);
		size = BLOCK_SIZE * size;
		add_zone(n, (zone_t) ((maj << 8) | min), size, current_time);
	} else {
		/* Regular file. Go read it. */
		if ((f = open(token[4], O_RDONLY)) < 0) {
			write(2, "Can't open file ", 16);
			write(2, token[4], strlen(token[4]));
			write(2, "\n", 1);
		} else
			eat_file(n, f);
	}
  }

}



/*================================================================
 * 		eat_file  -  copy file to MINIX
 *===============================================================*/

/* Zonesize >= blocksize */
eat_file(inode, f)
ino_t inode;
int f;
{
  int ct, i, j, k;
  zone_t z;
  char buf[BLOCK_SIZE];
  long timeval;

  do {
	for (i = 0, j = 0; i < zone_size; i++, j += ct) {
		for (k = 0; k < BLOCK_SIZE; k++) buf[k] = 0;
		if ((ct = read(f, buf, BLOCK_SIZE)) > 0) {
			if (i == 0) z = alloc_zone();
			put_block((z << zone_shift) + i, buf);
		}
	}
	timeval = (dflag ? current_time : file_time(f));
	if (ct) add_zone(inode, z, (long) j, timeval);
  } while (ct == BLOCK_SIZE);
  close(f);
}





/*================================================================
 *	    directory & inode management assist group
 *===============================================================*/

enter_dir(parent, name, child)
ino_t parent, child;
char *name;
{
  /* Enter child in parent directory */
  /* Works for dir > 1 block and zone > block */
  int i, j, k, l, off, need;
  block_t b;
  zone_t z;
  char *p1, *p2;
  struct _fl_direct dir_entry[FL_NR_DIR_ENTRIES];
  d2_inode ino[V2_INODES_PER_BLOCK];

  need = 1 + _EXTENT(strlen(name));

  b = ((parent - 1) / V2_INODES_PER_BLOCK) + inode_offset;
  off = (parent - 1) % V2_INODES_PER_BLOCK;
  get_block(b, (char *) ino);

  for (k = 0; k < V2_NR_DZONES; k++) {
	z = dtoz(&ino[off].d2_zone[Z_STRIDE*k]);
	if (z == 0) {
		z = alloc_zone();
		ztod(&ino[off].d2_zone[Z_STRIDE*k], z);
	}
	for (l = 0; l < zone_size; l++) {
		get_block((z << zone_shift) + l, (char *) dir_entry);
		for (i = 0; i < FL_NR_DIR_ENTRIES; i+= 1+dir_entry[i].d_extent)
		{
			j = i;
			while (j-i < need && j < FL_NR_DIR_ENTRIES
					&& dir_entry[j].d_ino == 0) j++;

			if (j-i == need) {
				dir_entry[i].d_ino = child;
				dir_entry[i].d_extent = need - 1;
				p1 = name;
				p2 = dir_entry[i].d_name;
				j = 8 * need - 3;
				while (j--) {
					*p2++ = *p1;
					if (*p1 != 0) p1++;
				}
				put_block((z << zone_shift) + l, (char *) dir_entry);
				put_block(b, (char *) ino);
				return;
			}
		}
	}
  }

  printf("Directory-inode %d beyond direct blocks.  Could not enter %s\n",
         parent, name);
  pexit("Halt");
}




add_zone(n, z, bytes, cur_time)
ino_t n;
zone_t z;
long bytes, cur_time;
{
  /* Add zone z to inode n. The file has grown by 'bytes' bytes. */

  int off, i;
  block_t b;
  zone_t indir;
  zone_t blk[V2_INDIRECTS];
  d2_inode *p;
  d2_inode inode[V2_INODES_PER_BLOCK];

  b = ((n - 1) / V2_INODES_PER_BLOCK) + inode_offset;
  off = (n - 1) % V2_INODES_PER_BLOCK;
  get_block(b, (char *) inode);
  p = &inode[off];
  p->d2_size += bytes;
  p->d2_mtime = cur_time;
  for (i = 0; i < V2_NR_DZONES; i++)
	if (dtoz(&p->d2_zone[Z_STRIDE*i]) == 0) {
		ztod(&p->d2_zone[Z_STRIDE*i], z);
		put_block(b, (char *) inode);
		return;
	}
  put_block(b, (char *) inode);

  /* File has grown beyond a small file. */
  if (dtoz(&p->d2_zone[Z_STRIDE*V2_NR_DZONES]) == 0)
	ztod(&p->d2_zone[Z_STRIDE*V2_NR_DZONES], alloc_zone());
  indir = dtoz(&p->d2_zone[Z_STRIDE*V2_NR_DZONES]);
  put_block(b, (char *) inode);
  b = indir << zone_shift;
  get_block(b, (char *) blk);
  for (i = 0; i < V2_INDIRECTS; i++)
	if (blk[i] == 0) {
		blk[i] = z;
		put_block(b, (char *) blk);
		return;
	}
  pexit("File has grown beyond single indirect");
}




incr_link(n)
ino_t n;
{
  /* Increment the link count to inode n */
  block_t b;
  int off;
  d2_inode inode[V2_INODES_PER_BLOCK];

  b = ((n - 1) / V2_INODES_PER_BLOCK) + inode_offset;
  off = (n - 1) % V2_INODES_PER_BLOCK;
  get_block(b, (char *) inode);
  inode[off].d2_nlinks++;
  put_block(b, (char *) inode);
}




incr_size(n, count)
ino_t n;
long count;
{
  /* Increment the file-size in inode n */
  block_t b;
  int off;
  d2_inode inode[V2_INODES_PER_BLOCK];

  b = ((n - 1) / V2_INODES_PER_BLOCK) + inode_offset;
  off = (n - 1) % V2_INODES_PER_BLOCK;
  get_block(b, (char *) inode);
  inode[off].d2_size += count;
  put_block(b, (char *) inode);
}




/*================================================================
 * 	 	     allocation assist group
 *===============================================================*/

PRIVATE ino_t alloc_inode(mode, usrid, grpid)
int mode, usrid, grpid;
{
  block_t b;
  ino_t num;
  int off;
  d2_inode inode[V2_INODES_PER_BLOCK];

  num = next_inode++;
  if (num >= nrinodes) pexit("File system does not have enough inodes");
  b = ((num - 1) / V2_INODES_PER_BLOCK) + inode_offset;
  off = (num - 1) % V2_INODES_PER_BLOCK;
  get_block(b, (char *) inode);
  inode[off].d2_mode = mode;
  inode[off].d2_uid = usrid;
  inode[off].d2_gid = grpid;
  put_block(b, (char *) inode);

  /* Set the bit in the bit map. */
  /* DEBUG FIXME.  This assumes the bit is in the first inode map block. */
  insert_bit((block_t) INODE_MAP, (int) num);
  return(num);
}




PRIVATE zone_t alloc_zone()
{
  /* Allocate a new zone */
  /* Works for zone > block */
  block_t b;
  int i;
  zone_t z;

  z = next_zone++;
  b = z << zone_shift;
  if ((b + zone_size) > nrblocks)
	pexit("File system not big enough for all the files");
  for (i = 0; i < zone_size; i++)
	put_block(b + i, zero);	/* give an empty zone */
  /* DEBUG FIXME.  This assumes the bit is in the first zone map block. */
  insert_bit(zone_map, (int) (z - zoff));	/* lint, NOT OK because z
  						 * hasn't been broken up
  						 * into block + offset yet.
  						 */
  return(z);
}




insert_bit(block, bit)
block_t block;
int bit;
{
  /* Insert 'count' bits in the bitmap */
  int w, s;
  bitchunk_t buf[BLOCK_SIZE / sizeof(bitchunk_t)];

  get_block(block, (char *) buf);
  w = bit / (8 * sizeof(bitchunk_t));
  s = bit % (8 * sizeof(bitchunk_t));
  buf[w] |= (1 << s);
  put_block(block, (char *) buf);
}




/*================================================================
 * 		proto-file processing assist group
 *===============================================================*/

int mode_con(p)
char *p;
{
  /* Convert string to mode */
  int type = I_REGULAR, mode = 0;

  switch (*p) {
  case 'd':	type = I_DIRECTORY; p++; break;
  case 'b':	type = I_BLOCK_SPECIAL; p++; break;
  case 'c':	type = I_CHAR_SPECIAL; p++; break;
  }

  for (;;) {
  	if (*p == 'u') type |= I_SET_UID_BIT;
  	else
  	if (*p == 'g') type |= I_SET_GID_BIT;
  	else
  	if (*p == 't') type |= STICKY_BIT;
  	else
  	if (*p != '-') break;
  	p++;
  }
  while ('0' <= *p && *p <= '7') mode = (mode << 3) | (*p++ - '0');
  if (*p != 0) pexit("Bad mode");
  return(type | mode);
}



getline(line, parse)
char *parse[MAX_TOKENS];
char line[LINE_LEN];
{
  /* Read a line and break it up in tokens */
  int k;
  char c, *p;
  int d;

  for (k = 0; k < MAX_TOKENS; k++) parse[k] = 0;
  for (k = 0; k < LINE_LEN; k++) line[k] = 0;
  k = 0;
  parse[0] = 0;
  p = line;
  while (1) {
	if (++k > LINE_LEN) pexit("Line too long");
	if (proto == NULL) {
		d = *protoarg == 0 ? EOF : *protoarg++ & 0xFF;
	} else {
		d = fgetc(proto);
	}
	if (d == EOF) pexit("Unexpected end-of-file");
	*p = d;
	if (*p == '\n') lct++;
	if (*p == ' ' || *p == '\t') *p = 0;
	if (*p == '\n') {
		*p++ = 0;
		*p = '\n';
		break;
	}
	p++;
  }

  k = 0;
  p = line;
  lastp = line;
  while (1) {
	c = *p++;
	if (c == '\n') return;
	if (c == 0) continue;
	parse[k++] = p - 1;
	do {
		c = *p++;
	} while (c != 0 && c != '\n');
  }
}




/*================================================================
 *			other stuff
 *===============================================================*/


long file_time(f)
int f;
{
#ifdef UNIX
  struct stat statbuf;
  fstat(f, &statbuf);
  return(statbuf.st_mtime);
#else				/* fstat not supported by DOS */
  return(0L);
#endif
}


pexit(s)
char *s;
{
  char *s0;

  s0 = s;
  while (*s0 != 0) s0++;
  write(2, "Error: ", 7);
  write(2, s, (int) (s0 - s));
  write(2, "\n", 1);
  printf("Line %d being processed when error detected.\n", lct);
  flush();
  exit(2);
}


copy(from, to, count)
char *from, *to;
int count;
{
  while (count--) *to++ = *from++;
}


print_fs()
{
  int i, j;
  ino_t k;
  d2_inode inode[V2_INODES_PER_BLOCK];
  unsigned short usbuf[BLOCK_SIZE/sizeof(unsigned short)];
  block_t b;
  struct _fl_direct dir[FL_NR_DIR_ENTRIES];

  get_block((block_t) 1, (char *) usbuf);
  printf("\nSuperblock: ");
  for (i = 0; i < 8; i++) printf("%06o ", usbuf[i]);
  get_block((block_t) 2, (char *) usbuf);
  printf("...\nInode map:  ");
  for (i = 0; i < 9; i++) printf("%06o ", usbuf[i]);
  get_block((block_t) 3, (char *) usbuf);
  printf("...\nZone  map:  ");
  for (i = 0; i < 9; i++) printf("%06o ", usbuf[i]);
  printf("...\n");
  
  k = 0;
  for (b = inode_offset; k < nrinodes; b++) {
	get_block(b, (char *) inode);
	for (i = 0; i < V2_INODES_PER_BLOCK; i++) {
		k = V2_INODES_PER_BLOCK * (int) (b - inode_offset) + i + 1;
				/* lint but OK */
		if (k > nrinodes) break;
		if (inode[i].d2_mode != 0) {
			printf("Inode %2d:  mode=", k, inode[i].d2_mode);
			printf("%06o", inode[i].d2_mode);
			printf("  uid=%2d  gid=%2d  size=",
			       inode[i].d2_uid, inode[i].d2_gid);
			printf("%6ld", inode[i].d2_size);
			printf("  zone[0]=%ld\n",
			       dtoz(&inode[i].d2_zone[Z_STRIDE*0]));
		}
		if ((inode[i].d2_mode & I_TYPE) == I_DIRECTORY) {
			/* This is a directory */
			get_block(dtoz(&inode[i].d2_zone[Z_STRIDE*0]),
				  (char *) dir);
			for (j = 0; j < FL_NR_DIR_ENTRIES; j+=1+dir[j].d_extent)
				if (dir[j].d_ino)
					printf("\tInode %2d: %s\n", dir[j].d_ino, dir[j].d_name);
		}
	}
  }

  printf("%d inodes used.     %d zones used.\n", next_inode - 1, next_zone);
}


int read_and_set(n)
block_t n;
{
/*  The first time a block is read, it returns alls 0s, unless there has
 *  been a write.  This routine checks to see if a block has been accessed.
 */

  int w, s, mask, r;

  w = n / 8;
  s = n % 8;
  mask = 1 << s;
  r = (umap[w] & mask ? 1 : 0);
  umap[w] |= mask;
  return(r);
}



/*================================================================
 *		      get_block & put_block for MS-DOS
 *===============================================================*/

#ifdef DOS

/*
 *	These are the get_block and put_block routines
 *	when compiling & running mkfs.c under MS-DOS.
 *
 *	It requires the (asembler) routines absread & abswrite
 *	from the file diskio.asm. Since these routines just do
 *	as they are told (read & write the sector specified),
 *	a local cache is used to minimize the i/o-overhead for
 *	frequently used blocks.
 *
 *	The global variable "file" determines whether the output
 *	is to a disk-device or to a binary file.
 */


#define PH_SECTSIZE	   512	/* size of a physical disk-sector */


char *derrtab[14] = {
	     "no error",
	     "disk is read-only",
	     "unknown unit",
	     "device not ready",
	     "bad command",
	     "data error",
	     "internal error: bad request structure length",
	     "seek error",
	     "unknown media type",
	     "sector not found",
	     "printer out of paper (??)",
	     "write fault",
	     "read error",
	     "general error"
};

#define	CACHE_SIZE	20	/* 20 block-buffers */


struct cache {
  char blockbuf[BLOCK_SIZE];
  block_t blocknum;
  int dirty;
  int usecnt;
} cache[CACHE_SIZE];




special(string)
char *string;
{

  if (string[1] == ':' && string[2] == 0) {
	/* Format: d: or d:fname */
	disk = (string[0] & ~32) - 'A';
	if (disk > 1 && !override)	/* safety precaution */
		pexit("Bad drive specifier for special");
  } else {
	file = 1;
	if ((fd = creat(string, BWRITE)) == 0)
		pexit("Can't open special file");
  }
}



get_block(n, buf)
block_t n;
char buf[BLOCK_SIZE];
{
  /* Get a block to the user */
  struct cache *bp, *fp;

  /* First access returns a zero block */
  if (read_and_set(n) == 0) {
	copy(zero, buf, BLOCK_SIZE);
	return;
  }

  /* Look for block in cache */
  fp = 0;
  for (bp = cache; bp < &cache[CACHE_SIZE]; bp++) {
	if (bp->blocknum == n) {
		copy(bp, buf, BLOCK_SIZE);
		bp->usecnt++;
		return;
	}

	/* Remember clean block */
	if (bp->dirty == 0)
		if (fp) {
			if (fp->usecnt > bp->usecnt) fp = bp;
		} else
			fp = bp;
  }

  /* Block not in cache, get it */
  if (!fp) {
	/* No clean buf, flush one */
	for (bp = cache, fp = cache; bp < &cache[CACHE_SIZE]; bp++)
		if (fp->usecnt > bp->usecnt) fp = bp;
	mx_write(fp->blocknum, fp);
  }
  mx_read(n, fp);
  fp->dirty = 0;
  fp->usecnt = 0;
  fp->blocknum = n;
  copy(fp, buf, BLOCK_SIZE);
}



put_block(n, buf)
block_t n;
char buf[BLOCK_SIZE];
{
  /* Accept block from user */
  struct cache *fp, *bp;

  (void) read_and_set(n);

  /* Look for block in cache */
  fp = 0;
  for (bp = cache; bp < &cache[CACHE_SIZE]; bp++) {
	if (bp->blocknum == n) {
		copy(buf, bp, BLOCK_SIZE);
		bp->dirty = 1;
		return;
	}

	/* Remember clean block */
	if (bp->dirty == 0)
		if (fp) {
			if (fp->usecnt > bp->usecnt) fp = bp;
		} else
			fp = bp;
  }

  /* Block not in cache */
  if (!fp) {
	/* No clean buf, flush one */
	for (bp = cache, fp = cache; bp < &cache[CACHE_SIZE]; bp++)
		if (fp->usecnt > bp->usecnt) fp = bp;
	mx_write(fp->blocknum, fp);
  }
  fp->dirty = 1;
  fp->usecnt = 1;
  fp->blocknum = n;
  copy(buf, fp, BLOCK_SIZE);
}



cache_init()
{
  struct cache *bp;
  for (bp = cache; bp < &cache[CACHE_SIZE]; bp++) bp->blocknum = -1;
}



flush()
{
  /* Flush all dirty blocks to disk */
  struct cache *bp;

  for (bp = cache; bp < &cache[CACHE_SIZE]; bp++)
	if (bp->dirty) {
		mx_write(bp->blocknum, bp);
		bp->dirty = 0;
	}
}



/*==================================================================
 *			hard read & write etc.
 *=================================================================*/

#define MAX_RETRIES	5


mx_read(blocknr, buf)
int blocknr;
char buf[BLOCK_SIZE];
{

  /* Read the requested MINIX-block in core */
  char (*bp)[PH_SECTSIZE];
  int sectnum, retries, err;

  if (file) {
	lseek(fd, (off_t) blocknr * BLOCK_SIZE, 0);
	if (read(fd, buf, BLOCK_SIZE) != BLOCK_SIZE)
		pexit("mx_read: error reading file");
  } else {
	sectnum = blocknr * (BLOCK_SIZE / PH_SECTSIZE);
	for (bp = buf; bp < &buf[BLOCK_SIZE]; bp++) {
		retries = MAX_RETRIES;
		do
			err = absread(disk, sectnum, bp);
		while (err && --retries);

		if (retries) {
			sectnum++;
		} else {
			dexit("mx_read", sectnum, err);
		}
	}
  }
}



mx_write(blocknr, buf)
int blocknr;
char buf[BLOCK_SIZE];
{
  /* Write the MINIX-block to disk */
  char (*bp)[PH_SECTSIZE];
  int retries, sectnum, err;

  if (file) {
	lseek(fd, blocknr * BLOCK_SIZE, 0);
	if (write(fd, buf, BLOCK_SIZE) != BLOCK_SIZE) {
		pexit("mx_write: error writing file");
	}
  } else {
	sectnum = blocknr * (BLOCK_SIZE / PH_SECTSIZE);
	for (bp = buf; bp < &buf[BLOCK_SIZE]; bp++) {
		retries = MAX_RETRIES;
		do {
			err = abswrite(disk, sectnum, bp);
		} while (err && --retries);

		if (retries) {
			sectnum++;
		} else {
			dexit("mx_write", sectnum, err);
		}
	}
  }
}


dexit(s, sectnum, err)
int sectnum, err;
char *s;
{
  printf("Error: %s, sector: %d, code: %d, meaning: %s\n",
         s, sectnum, err, derrtab[err]);
  exit(2);
}

#endif

/*================================================================
 *		      get_block & put_block for UNIX
 *===============================================================*/

#ifdef UNIX

special(string)
char *string;
{
  fd = open(string, O_RDWR|O_CREAT, 0666);
  if (fd < 0) pexit("Can't open special file");
#if (MACHINE == ATARI)
  {
	struct stat statbuf;

	if (fstat(fd, &statbuf) < 0)
		return;
	isdev = (statbuf.st_mode & S_IFMT) == S_IFCHR
		||
		(statbuf.st_mode & S_IFMT) == S_IFBLK
		;
  }
#endif
}





get_block(n, buf)
block_t n;
char *buf;
{
/* Read a block. */

  int k;

  /* First access returns a zero block */
  if (read_and_set(n) == 0) {
	copy(zero, buf, BLOCK_SIZE);
	return;
  }
  lseek(fd, (off_t) n * BLOCK_SIZE, SEEK_SET);
  k = read(fd, buf, BLOCK_SIZE);
  if (k != BLOCK_SIZE) {
	pexit("get_block couldn't read");
  }
}



put_block(n, buf)
block_t n;
char buf[BLOCK_SIZE];
{
/* Write a block. */

  (void) read_and_set(n);

  if (lseek(fd, (off_t) n * BLOCK_SIZE, SEEK_SET) < 0L) {
	pexit("put_block couldn't seek");
  }
  if (write(fd, buf, BLOCK_SIZE) != BLOCK_SIZE) {
	pexit("put_block couldn't write");
  }
}


/* Dummy routines to keep source file clean from #ifdefs */

flush()
{
  return;
}



cache_init()
{
  return;
}

#endif

/* Convert between 3-byte numbers and 4-byte numbers.  The 3-byte numbers
 * are character arrays in a standard order (low-middle-high).  The 4-byte
 * numbers are ordinary longs in the native (runtime) order.  Treat these
 * as 4-byte arrays for speed.
 */

#undef BYTE_0

#if CHIP == INTEL
#define BYTE_0 0
#define BYTE_1 1
#define BYTE_2 2
#define BYTE_3 3
#endif

#if CHIP == M68000
#define BYTE_0 3
#define BYTE_1 2
#define BYTE_2 1
#define BYTE_3 0
#endif

#ifndef BYTE_0
<<#error>> /* don't know the byte order for this processor */
#endif

PRIVATE long c3_i4(c3)
register char *c3;
{
  long i4;

  ((char *) &i4)[BYTE_0] = c3[0];
  ((char *) &i4)[BYTE_1] = c3[1];
  ((char *) &i4)[BYTE_2] = c3[2];
  ((char *) &i4)[BYTE_3] = 0;
  return(i4);
}

PRIVATE void i4_c3(c3, i4)
register char *c3;
long i4;
{
  c3[0] = ((char *) &i4)[BYTE_0];
  c3[1] = ((char *) &i4)[BYTE_1];
  c3[2] = ((char *) &i4)[BYTE_2];
}

/*
 * $PchId: mkfs2f.c,v 1.3 1995/11/27 22:06:03 philip Exp $
 */
