/*	mkfs 1.8 - Make a file system.			Author: Kees J. Bot
 *								27 Nov 1992
 *
 * This is just a forerunner to mkfs1, mkfs1f, mkfs2, mkfs2f, and mkswap.
 * (Albeit a smart one.)
 */

#define _MINIX_SOURCE

#define nil 0
#include <sys/types.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/mnttab.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <minix/cfg_public.h>
#include <minix/const.h>
#include <minix/type.h>
#include <minix/partition.h>
#include <fs/const.h>
#include <fs/type.h>
#include <fs/super.h>

#undef printf

#define arraysize(a)	(sizeof(a) / sizeof((a)[0]))
#define arraylimit(a)	((a) + arraysize(a))

char FSTAB[]=	"/etc/fstab";
char MTAB[]=	"/etc/mtab";

int tflag= 0, lflag= 0, dflag= 0;	/* -TDL */

void report(const char *label)
{
	if (label != nil && *label != 0)
		fprintf(stderr, "mkfs: %s: %s\n", label, strerror(errno));
	else
		fprintf(stderr, "mkfs: %s\n", strerror(errno));
}

void fatal(const char *label)
{
	report(label);
	exit(1);
}

#define last_char(block)	((off_t) (block) * BLOCK_SIZE + BLOCK_SIZE - 1)

unsigned long sizeup(char *device)
/* Determine the size of the device to make an FS on. */
{
	struct stat st;
	int fd, n;
	struct partition entry;
	unsigned long low, high, mid;
	char c;

	if ((fd= open(device, O_RDONLY)) < 0) fatal(device);

	/* No need to let mkfs* test the size of the device after sizeup(). */
	tflag= 1;

	/* Ask the device driver how big the device is. */
	if (ioctl(fd, DIOCGETP, &entry) >= 0) {
		close(fd);
		return div64u(entry.size, BLOCK_SIZE);
	}

	/* If this fails then do binary search. */
	low= 0;
	high= 512;

	for (;;) {
		if (lseek(fd, last_char(high), SEEK_SET) < 0) fatal(device);

		if ((n= read(fd, &c, 1)) < 0) fatal(device);

		if (n == 0) break;
		high*= 2;
	}

	while (low <= high && high != -1) {
		mid= (low + high) / 2;

		if (lseek(fd, last_char(mid), SEEK_SET) < 0) fatal(device);

		if ((n= read(fd, &c, 1)) < 0) fatal(device);

		if (n == 0) high= mid - 1; else low= mid + 1;
	}
	close(fd);

	return low;
}

void check_active(char *device, unsigned long *size)
{
	int fd;
	int n;
	struct super_block super;
	FILE *mtf;
	struct mnttab mte, look;

	if ((mtf= fopen(MTAB, "r")) == nil && errno != ENOENT)
		fatal(device);

	if (mtf != nil) {
		look.mnt_special= device;
		look.mnt_mountp= nil;
		look.mnt_fstype= nil;
		look.mnt_mntopts= nil;

		if (getmntany(mtf, &mte, &look) >= 0) {
			fprintf(stderr, "mkfs: %s is mounted on %s\n",
					device, mte.mnt_mountp);
			exit(1);
		}
		(void) fclose(mtf);
	}

	/* Check if the device contains a file system. */
	if ((fd= open(device, O_RDONLY)) < 0) fatal(device);

	if (lseek(fd, (off_t) SUPER_BLOCK * BLOCK_SIZE, SEEK_SET) < 0)
		fatal(device);

	if ((n= read(fd, (void *) &super, sizeof(super))) < 0) fatal(device);

	close(fd);

	if (n < sizeof(super)) {
		fprintf(stderr, "mkfs: %s: too small\n", device);
		exit(1);
	}

	switch (super.s_magic) {
	case SUPER_MAGIC:
		*size= super.s_nzones;
		break;
	case SUPER_REV:
		*size=    (((super.s_nzones >> 8) & 0xFF) << 0)
			| (((super.s_nzones >> 0) & 0xFF) << 8);
		break;
	case SUPER_V2:
		*size= super.s_zones;
		break;
	case SUPER_V2_REV:
		*size=    (((super.s_zones >> 24) & 0xFF) <<  0)
			| (((super.s_zones >> 16) & 0xFF) <<  8)
			| (((super.s_zones >>  8) & 0xFF) << 16)
			| (((super.s_zones >>  0) & 0xFF) << 24);
		break;
	default:
		*size= 0;
	}
}

char *get_type(char *device)
{
	FILE *fsf;
	struct mnttab mte, look;
	static char type[64];

	if ((fsf= fopen(FSTAB, "r")) == nil)
		fatal(device);

	look.mnt_special= device;
	look.mnt_mountp= nil;
	look.mnt_fstype= nil;
	look.mnt_mntopts= nil;

	if (getmntany(fsf, &mte, &look) < 0) {
		fprintf(stderr, "mkfs: FS type of %s unknown\n", device);
		exit(1);
	}
	strncpy(type, mte.mnt_fstype, sizeof(type) - 1);

	(void) fclose(fsf);

	return type;
}

char *mkproto(char *device, char *type, char *Size, char *Inodes, char *Bpi)
{
	unsigned long size, inodes, bpi;
	int ino_p_b= type[0] == '1' ? V1_INODES_PER_BLOCK : V2_INODES_PER_BLOCK;
	int pfd[2];
	static char proto[200];

	size= Size != nil ? strtoul(Size, (char **) nil, 0) : sizeup(device);

	/* 2 Gb is the absolute max (lseek...) */
	if (size >= 2 * 1024 * 1024L) {
		fprintf(stderr, "mkfs: block count of %lu is too big.\n", size);
		exit(1);
	}

	if (strcmp(type, "swap") == 0) {
		/* A swap device, a size is all we need. */
		sprintf(proto, "%lu", size);
		return proto;
	}

	if (Inodes != nil) {
		/* Number of inodes to make. */
		inodes= strtoul(Inodes, (char **) nil, 0);
	} else {
		if (Bpi != nil) {
			/* Number of bytes per inode. */
			bpi= strtoul(Bpi, (char **) nil, 0);
		} else {
			/* Let the number of bytes per inode depend on the
			 * size of the device.  Bigger devices are assumed to
			 * have bigger files.
			 */
			bpi= 3 * 1024;
			if (size >=   8 * 1024L) bpi= 4 * 1024L;
			if (size >=  16 * 1024L) bpi= 5 * 1024L;
			if (size >=  32 * 1024L) bpi= 6 * 1024L;
			if (size >=  64 * 1024L) bpi= 7 * 1024L;
			if (size >= 128 * 1024L) bpi= 8 * 1024L;
		}
		inodes= (size * 1024 + bpi - 1) / bpi;
	}

	/* Round the number of inodes up to the number that fit in a block. */
	inodes= (inodes + ino_p_b - 1) / ino_p_b * ino_p_b;

	/* Alas there are only 16 bits for an inode number. */
	if (inodes > MAX_INODE_NR) inodes= MAX_INODE_NR;

	/* Create a proto "file". */
	sprintf(proto, "\
boot\n\
%lu %lu\n\
d-1777 %d %d\n\
$\n",		size, inodes, geteuid(), getegid());

	printf(
	  "%s: V%c%s filesystem, %lu kilobytes, %lu inodes, %lu bytes/inode\n",
		device, type[0], type[1] != 0 ? " flex" : "",
		size, inodes, size * 1024 / inodes);
	return proto;
}


void usage(void)
{
	fprintf(stderr, "\
Usage: mkfs [-t type] [-i inodes] [-b bpi] [-p priority] [-TDL] device [size]\
\n");
	exit(1);
}

int numeric(char *num, int sign)
/* True if "num" is a number to strtol. */
{
	char *end;
	long n;

	n= strtoul(num, &end, 0);

	return (sign == '-' || n >= 0) && end != nil && end != num && *end == 0;
}

int main(int argc, char **argv)
{
	char *mkv[10], **mkp= mkv + 1;
	char mk0[10];
	char *device;
	char *type= nil, *priority= nil, *Size= nil, *Inodes= nil, *Bpi= nil;
	unsigned long fssize;
	int i= 1;
	int swap, plusswap= 0;
	struct stat st;

	/* Gather options. */
	while (i < argc && argv[i][0] == '-') {
		char *p= argv[i++] + 1;

		if (p[0] == '-' && p[1] == 0) break;

		while (*p != 0) {
			switch (*p++) {
			case 't':		/* mkfs -t type */
				if (*p == 0) {
					if (i == argc) usage();
					p= argv[i++];
				}
				type= p;
				p= "";
				break;
			case 'i':		/* mkfs -i inodes */
				if (Bpi != nil) usage();
				if (*p == 0) {
					if (i == argc) usage();
					p= argv[i++];
				}
				Inodes= p;
				if (!numeric(Inodes, '+')) usage();
				p="";
				break;
			case 'b':		/* mkfs -b bpi */
				if (Inodes != nil) usage();
				if (*p == 0) {
					if (i == argc) usage();
					p= argv[i++];
				}
				Bpi= p;
				if (!numeric(Bpi, '+')) usage();
				p="";
				break;
			case 'p':		/* mkswap -p priority */
				if (*p == 0) {
					if (i == argc) usage();
					p= argv[i++];
				}
				priority= p;
				if (!numeric(priority, '-')) usage();
				p="";
				break;
			case 'T':		/* mkfsxx -t */
				tflag= 1;
				break;
			case 'D':		/* mkfsxx -d */
				dflag= 1;
				break;
			case 'L':		/* mkfsxx -l */
				lflag= 1;
				break;
			default:
				usage();
			}
		}
	}

	/* The device name optionally followed by a size or proto file. */
	if (i == argc) usage();
	device= argv[i++];
	if (i != argc) Size= argv[i++];
	if (i != argc) usage();

	if (type == nil) {
		type= get_type(device);
	} else {
		if (strcmp(type, "+swap") == 0) {
			plusswap= 1;
			type++;
		}
	}

	swap= (strcmp(type, "swap") == 0);
	if (!swap
		&& strcmp(type, "1") != 0
		&& strcmp(type, "1f") != 0
		&& strcmp(type, "2") != 0
		&& strcmp(type, "2f") != 0
	) {
		fprintf(stderr, "mkfs: \"%s\": unknown FS type\n", type);
		exit(0);
	}

	check_active(device, &fssize);

	if (Size != nil && !numeric(Size, '+')) {
		if (swap) usage();

		/* Size is not a number, so it's a proto file. */
		if (stat(Size, &st) < 0) fatal(Size);
	} else {
		/* Got to make a proto "file". */
		Size= mkproto(device, type, Size, Inodes, Bpi);
	}

	/* mkswap makes a swap device, the rest goes with mkfs<type>. */
	if (swap) {
		unsigned long offset, size;
		unsigned a;
		static char swapoffset[20], swapsize[20];

		strcpy(mk0, "mkswap");
		mkv[0]= mk0;

		*mkp++= plusswap ? "-a" : "-f";

		if (priority != nil) {
			*mkp++= "-p";
			*mkp++= priority;
		}
		*mkp++= device;

		if (!plusswap) fssize= SUPER_BLOCK + 1;

		if (fssize == 0) {
			fprintf(stderr,
				"mkfs: %s: No file system to add swap to\n",
				device);
			exit(1);
		}
		offset= fssize;
		size= strtoul(Size, nil, 0);
		if (offset > size) {
			fprintf(stderr, "mkfs: %s: No room for swapspace\n",
				device);
			exit(1);
		}
		size -= offset;

		/* Align start of swap to the click (page) size. */
		if (CLICK_SIZE > BLOCK_SIZE
			&& (a= offset % (CLICK_SIZE / BLOCK_SIZE)) < size
		) {
			offset += a;
			size -= a;
		}

		sprintf(swapoffset, "%lu", offset);
		*mkp++= swapoffset;
		sprintf(swapsize, "%lu", size);
		*mkp++= swapsize;

		*mkp= nil;
		printf("%s: swapspace at offset %s, size %s kilobytes\n",
			device, swapoffset, swapsize);
	} else {
		static char opt[]= "-\0\0\0";
		char *f= opt+1;

		strcpy(mk0, "mkfs");
		strcat(mk0, type);
		mkv[0]= mk0;

		if (tflag || dflag || lflag) {
			*mkp++= opt;
			if (tflag) *f++= 't';
			if (dflag) *f++= 'd';
			if (lflag) *f++= 'l';
		}
		*mkp++= device;
		*mkp++= Size;
		*mkp= nil;
	}

	fflush(nil);

	execvp(mk0, mkv);

	fatal(mk0);
	exit(1);
}

/*
 * $PchId: mkfs.c,v 1.4 1995/11/27 22:10:52 philip Exp $
 */
