!	Doshead.s - DOS & BIOS support for boot.c	Author: Kees J. Bot
!
!
! This file contains the startup and low level support for the secondary
! boot program.  It contains functions for disk, tty and keyboard I/O,
! copying memory to arbitrary locations, etc.
!
! This runs under MS-DOS as a .COM file.  A .COM file is what Minix calls
! a common I&D executable, except that the first 256 bytes contains DOS
! thingies.
!
.sect .text; .sect .rom; .sect .data; .sect .bss

	o32	    =	  0x66	! This assembler doesn't know 386 extensions
	a32	    =	  0x67
	K_I386	    =	0x0001	! Call Minix in 386 mode
	K_RET	    =	0x0020	! Returns to the monitor on reboot
	K_INT86	    =	0x0040	! Requires generic INT support
	STACK	    =	 16384	! Number of bytes for the stack

	DS_SELECTOR =	   3*8	! Kernel data selector
	ES_SELECTOR =	   4*8	! Flat 4 Gb
	SS_SELECTOR =	   5*8	! Monitor stack
	CS_SELECTOR =	   6*8	! Kernel code
	MCS_SELECTOR=	   7*8	! Monitor code

! Imported variables and functions:
.extern _caddr, _daddr, _runsize, _edata, _end	! Runtime environment
.extern _k_flags				! Special kernel flags

.sect .text

.extern _boot, _printk				! Boot Minix, kernel printf

.define _PSP
_PSP:
	.space	256		! Program Segment Prefix

dosboot:
	cld			! C compiler wants UP
	xor	ax, ax		! Zero
	mov	di, _edata	! Start of bss is at end of data
	mov	cx, _end	! End of bss (begin of heap)
	sub	cx, di		! Number of bss bytes
	shr	cx, 1		! Number of words
 rep	stos			! Clear bss
	mov	sp, _end+STACK	! "chmem" to 16 kb

! Remember the current video mode for restoration on exit.
	movb	ah, 0x0F	! Get current video mode
	int	0x10
	andb	al, 0x7F	! Mask off bit 7 (no blanking)
	movb	(old_vid_mode), al
	movb	(cur_vid_mode), al

! One needs a 386 for all the games we want to play.
	call	_getprocessor
	cmp	ax, 386
	jb	not386

! A V86 straightjacket is also not comfy.
	smsw	ax		! Use old 286 instruction to get the MSW
	testb	al, 0x01	! Protected mode enabled?
	jz	notV86
not386:
	push	tellV86
	call	_printk
	jmp	quit
.sect .rom
tellV86:.ascii	"Either this is not at least a 386, "
	.ascii	"or there is a memory manager active\n\0"
.sect .text
notV86:

! Put a benign value into the device codes.  We'd rather resort to these tricks
! then add 'if (DOS)' tests to the generic code.  We may be running under DOS,
! but we don't have to like it.
	mov	(_device), 0x7F
	mov	(_heads), 128
	mov	(_sectors), 128

! Find out how much "low" memory there is available, where it starts and
! where it ends.  Later a chunk of extended memory will be found.  Minix
! expects these chunks to start at addresses 0 and 1M.  Since the first
! number of bytes in these chunks are in-use by DOS we use parts at the end
! of the extended memory chunk to hold bytes that should be at address 0 or
! 1M.  When it is time to call Minix the DOS stuff is exchanged for the Minix
! stuff.  When Minix returns to the monitor the pieces are changed back.
! We do not even inform the Monitor of this trickery, but hide the details
! in this file.
	mov	ax, _PSP+256		! Up to and including the PSP is DOS
	mov	dx, ds
	call	seg2abs
	mov	(mem_base+0), ax
	mov	(mem_base+2), dx	! Base of the low memory chunk
	xor	ax, ax
	mov	dx, (_PSP+2)		! First in-use segment above us
	call	seg2abs
	mov	(mem_size+0), ax	! Byte size of available low memory
	mov	(mem_size+2), dx


! Give C code access to the code segment, data segment and the size of this
! process.
	.data1	o32
	mov	ax, (mem_base)
	.data1	o32
	mov	(_caddr), ax
	.data1	o32
	mov	(_daddr), ax
	mov	ax, _end-256+STACK+16-1	! End of our address space + 16-1
	and	ax, 0xFFF0		! Round up to a segment
	mov	(_runsize+0), ax	! 32-bit size in bytes
	!mov	(_runsize+2), 0

! Grab the largest chunk of extended memory available.
	mov	ax, 0x4300	! XMS driver check
	int	0x2F
	cmpb	al, 0x80	! XMS driver exists?
	je	xmsthere
getext:				! No driver, so can use all ext memory directly
	call	_getprocessor
	cmp	ax, 486		! Assume 486s were the first to have >64M
	jb	smallext	! (It helps to be paranoid when using the BIOS)
bigext:
	mov	ax, 0xE801	! Code for get memory size for >64M
	int	0x15		! ax = mem < 16M per 1K, bx = mem > 16M per 64K
	jc	smallext
	cmp	ax, 15*1024	! No hole below 16M please, we want 15M
	jb	smallext
	mov	dx, 1024	! ax = mem < 16M in K
	mul	dx		! dx:ax = mem < 16M in bytes
	add	dx, bx		! dx:ax += mem > 16M (bx is in 64K units)
	jmp	gotext
smallext:
	movb	ah, 0x88	! Code for get extended memory size
	clc			! Carry will stay clear if call exists
	int	0x15		! Returns size (in K) in ax for AT's
	jc	gotxms		! Carry set? If so leave xms_size zero
	mov	dx, 1024
	mul	dx		! dx:ax = bytes of ext memory
gotext:	
	mov	(xms_size+0), ax	! XMS size in bytes
	mov	(xms_size+2), dx
	jmp	gotxms
xmsthere:
	mov	ax, 0x4310		! Get XMS driver address
	int	0x2F
	mov	(xms_driver+0), bx
	mov	(xms_driver+2), es
	push	ds
	pop	es
	movb	ah, 0x08		! Query free extended memory
	xorb	bl, bl
	callf	(xms_driver)
	testb	bl, bl
	jnz	xmserr
	mov	dx, ax			! dx = ax = Size of largest block in kb
	mov	(xms_size+0), ax
	shl	(xms_size+0), 10
	shr	ax, 16-10
	mov	(xms_size+2), ax	! Size in bytes
	movb	ah, 0x09 		! Allocate XMS block of size dx
	callf	(xms_driver)
	test	ax, ax
	jz	xmserr
	mov	(xms_handle), dx	! Save handle
	movb	ah, 0x0C		! Lock XMS block (handle in dx)
	callf	(xms_driver)
	test	ax, ax
	jz	xmserr
	mov	(xms_base+0), bx
	mov	(xms_base+2), dx	! Address of locked block
	sub	dx, 0x0010		! dx:bx = xms_base - 1M
	mov	(xms_inuse+0), bx
	mov	(xms_inuse+2), dx	! In-use XMS below xms_base
gotxms:
	.data1	o32
	mov	ax, (mem_base)
	.data1	o32
	sub	ax, (C1k5)		! DOS uses 1.5k to mem_base
	.data1	o32
	mov	(mem_inuse), ax		! mem_inuse = mem_base - 1.5k
	.data1	o32
	cmp	ax, (xms_size)		! Enough XMS to store DOS low memory?
	jbe	enoughxms
	push	tellbadxms
	call	_printk
	jmp	quit
.sect .rom
tellbadxms:
	.ascii	"Not enough extended memory available\n\0"
.sect .text
enoughxms:
	.data1	o32
	mov	ax, (xms_base)
	.data1	o32
	add	ax, (xms_size)		! eax = end of XMS
	.data1	o32
	sub	ax, (mem_inuse)		! Make room for in-use low memory
	.data1	o32
	mov	(mem_temp), ax		! mem<mem_base is temporarily put here
	.data1	o32
	sub	ax, (xms_inuse)		! Make room for in-use ext memory
	.data1	o32
	cmp	ax, (xms_base)		! in-use > free ext memory?
	jae	0f
	.data1	o32
	sub	ax, (xms_base)		! - (too much in-use)
	.data1	o32
	add	(xms_inuse), ax		! This part need not be mirrored
	.data1	o32
	mov	ax, (xms_base)
0:	.data1	o32
	mov	(xms_temp), ax		! mem<xms_base is temporarily put here


! Time to switch to a higher level language (not much higher)
	call	_boot

.define	_exit, __exit, ___exit		! Make various compilers happy
_exit:
__exit:
___exit:
	mov	dx, (xms_handle)
	cmp	dx, -1			! Is there an ext mem block in use?
	je	nohandle
	movb	ah, 0x0D		! Unlock extended memory block
	callf	(xms_driver)
	mov	dx, (xms_handle)
	movb	ah, 0x0A		! Free extended memory block
	callf	(xms_driver)
nohandle:
	call	restore_video
	pop	ax
	pop	ax			! Return code in al
	movb	ah, 0x4C		! Code for terminate with return code
	int	0x21

quit:					! exit(1)
	movb	al, 1
	push	ax
	call	_exit

xmserr:
	xorb	bh, bh
	push	bx
	push	tellxmserr
	call	_printk
	jmp	quit
.sect	.rom
tellxmserr:	.ascii	"Extended memory problem, error 0x%02x\n\0"
.sect	.text

! u32_t mon2abs(void *ptr)
!	Address in monitor data to absolute address.
.define _mon2abs
_mon2abs:
	mov	bx, sp
	mov	ax, 2(bx)	! ptr
	mov	dx, ds		! Monitor data segment
	!jmp	seg2abs

seg2abs:			! Translate dx:ax to the 32 bit address dx-ax
	push	cx
	movb	ch, dh
	movb	cl, 4
	shl	dx, cl
	shrb	ch, cl		! ch-dx = dx << 4
	add	ax, dx
	adcb	ch, 0		! ch-ax = ch-dx + ax
	movb	dl, ch
	xorb	dh, dh		! dx-ax = ch-ax
	pop	cx
	ret

abs2seg:			! Translate the 32 bit address dx-ax to dx:ax
	push	cx
	movb	ch, dl
	mov	dx, ax		! ch-dx = dx-ax
	and	ax, 0x000F	! Offset in ax
	movb	cl, 4
	shr	dx, cl
	shlb	ch, cl
	orb	dh, ch		! dx = ch-dx >> 4
	pop	cx
	ret

! void raw_copy(u32_t dstaddr, u32_t srcaddr, u32_t count)
!	Copy count bytes from srcaddr to dstaddr.  Don't do overlaps.
!	Also handles copying words to or from extended memory.  Memory below
!	mem_base or below xms_base is mapped to mem_temp and xms_temp without
!	the monitor knowledge.
.define _raw_copy
_raw_copy:
	push	bp
	mov	bp, sp
	sub	sp, 8		! Local variables
	push	si
	push	di		! Save C variable registers
copy:
	.data1	o32
	mov	cx, 12(bp)
	.data1	o32
	cmp	cx, (CFFF0)	! Don't copy more than about 64K
	jb	smallcopy
	.data1	o32
	mov	cx, (CFFF0)
smallcopy:
	test	cx, cx
	jz	copydone	! Count is zero, end copy
	lea	si, 4(bp)	! Destination address
	lea	di, -4(bp)
	call	addr_map	! *di = *si possibly mapped to temp area
	lea	si, 8(bp)	! Source address
	lea	di, -8(bp)
	call	addr_map	! *di = *si possibly mapped to temp area
	.data1	o32
	push	cx		! Save copying count (possibly made smaller)
	mov	ax, -4(bp)
	mov	dx, -2(bp)
	cmp	dx, 0x0010	! Copy to extended memory?
	jae	ext_copy
	cmp	-6(bp), 0x0010
	jae	ext_copy	! Copy from extended memory?
	call	abs2seg
	mov	di, ax
	mov	es, dx		! es:di = dstaddr
	mov	ax, -8(bp)
	mov	dx, -6(bp)
	call	abs2seg
	mov	si, ax
	mov	ds, dx		! ds:si = srcaddr
	shr	cx, 1		! Words to move
 rep	movs			! Do the word copy
	adc	cx, cx		! One more byte?
 rep	movsb			! Do the byte copy
	mov	ax, ss		! Restore ds and es from the remaining ss
	mov	ds, ax
	mov	es, ax
 	jmp	copyadjust
ext_copy:
	mov	(x_dst_desc+2), ax
	movb	(x_dst_desc+4), dl
	movb	(x_dst_desc+7), dh ! Set base of destination segment
	mov	ax, -8(bp)
	mov	dx, -6(bp)
	mov	(x_src_desc+2), ax
	movb	(x_src_desc+4), dl
	movb	(x_src_desc+7), dh ! Set base of source segment
	mov	si, x_gdt	! es:si = global descriptor table
	inc	cx
	shr	cx, 1		! Words to move (rounded up...)
	movb	ah, 0x87	! Code for extended memory move
	int	0x15
copyadjust:
	.data1	o32
	pop	cx		! Restore count
	.data1	o32
	add	4(bp), cx	! srcaddr += copycount
	.data1	o32
	add	8(bp), cx	! dstaddr += copycount
	.data1	o32
	sub	12(bp), cx	! count -= copycount
	jmp	copy		! and repeat
copydone:
	pop	di
	pop	si		! Restore C variable registers
	mov	sp, bp
	pop	bp
	ret

addr_map:
	.data1	o32
	mov	ax, (si)
	.data1	o32
	mov	(di), ax		! Assume no mapping
	.data1	o32
	cmp	ax, (C1k5)		! Address < 1.5k is not mapped
	jae	0f
	.data1	o32
	mov	ax, (C1k5)		! Don't copy beyond this address
	jmp	count_limit
0:	.data1	o32
	cmp	ax, (mem_base)		! Address < mem_base?
	jae	0f
	.data1	o32
	sub	ax, (C1k5)
	.data1	o32
	add	ax, (mem_temp)
	.data1	o32
	mov	(di), ax		! Map to addr - 1.5k + mem_temp
	.data1	o32
	mov	ax, (mem_base)		! Limit
	jmp	count_limit
0:	.data1	o32
	cmp	ax, (C1M)		! address < 1M is not mapped
	jae	0f
	.data1	o32
	mov	ax, (C1M)		! Don't copy beyond this address
	jmp	count_limit
0:	.data1	o32
	cmp	ax, (xms_base)		! Address < xms_base?
	jae	0f
	.data1	o32
	sub	ax, (C1M)
	.data1	o32
	add	ax, (xms_temp)
	.data1	o32
	mov	(di), ax		! Map to addr - 1M + xms_temp
	.data1	o32
	mov	ax, (xms_base)		! Limit
count_limit:
	.data1	o32
	sub	ax, (si)		! Copyable bytes in this chunk
	.data1	o32
	cmp	cx, ax
	jbe	0f
	mov	cx, ax			! cx = max(cx, copyable bytes)
0:	ret

! u16_t get_word(u32_t addr);
! void put_word(u32_t addr, u16_t word);
!	Read or write a 16 bits word at an arbitrary location.
.define	_get_word, _put_word
_get_word:
	push	bp
	mov	bp, sp
	push	0
	push	2
	push	6(bp)
	push	4(bp)
	lea	ax, 4(bp)
	mov	dx, ds
	call	seg2abs
	push	dx
	push	ax
	call	_raw_copy	! raw_copy(&4(bp), addr, 2)
	mov	ax, 4(bp)
	mov	sp, bp
	pop	bp
	ret

_put_word:
	push	bp
	mov	bp, sp
	push	0
	push	2
	lea	ax, 8(bp)
	mov	dx, ds
	call	seg2abs
	push	dx
	push	ax
	push	6(bp)
	push	4(bp)
	call	_raw_copy	! raw_copy(addr, &word, 2)
	mov	sp, bp
	pop	bp
	ret

! void relocate(void);
!	After the program has copied itself to a safer place, it needs to change
!	the segment registers.  Caddr has already been set to the new location.
.define _relocate
_relocate:
	pop	bx		! Return address
	mov	ax, (_caddr+0)
	mov	dx, (_caddr+2)
	call	abs2seg
	sub	dx, 256>>4	! Don't forget to account for the former PSP
	mov	ds, dx		! New data segments (common I&D)
	mov	es, dx
	mov	ss, dx
	push	dx		! New text segment
	push	bx		! Return offset of this function
	.data1	o32
	mov	ax, (_caddr)
	.data1	o32
	mov	(_daddr), ax	! New data address
	retf			! Relocate


! void *brk(void *addr)
! void *sbrk(size_t incr)
!	Cannot fail implementations of brk(2) and sbrk(3), so we can use
!	malloc(3).  They reboot on stack collision instead of returning -1.
.sect .data
	.align	2
break:	.data2	_end		! A fake heap pointer
.sect .text
.define _brk, __brk, _sbrk, __sbrk
_brk:
__brk:				! __brk is for the standard C compiler
	xor	ax, ax
	jmp	sbrk		! break= 0; return sbrk(addr);
_sbrk:
__sbrk:
	mov	ax, (break)	! ax= current break
sbrk:	push	ax		! save it as future return value
	mov	bx, sp		! Stack is now: (retval, retaddr, incr, ...)
	add	ax, 4(bx)	! ax= break + increment
	mov	(break), ax	! Set new break
	lea	dx, -1024(bx)	! sp minus a bit of breathing space
	cmp	dx, ax		! Compare with the new break
	jb	heaperr		! Suffocating noises
	pop	ax		! Return old break (0 for brk)
	ret
heaperr:push	nomem
	call	_printk
	call	quit
.sect .rom
nomem:	.ascii	"\nOut of memory\n\0"
.sect .text

! int dos_open(char *name);
!	Open file to use as the Minix virtual disk.  Store handle in vfd.
!	Returns 0 for success, otherwise the DOS error code.
.define _dos_open
_dos_open:
	push	bp
	mov	bp, sp
	mov	dx, 4(bp)	! ds:dx = Address of file name
	mov	ax, 0x3D02	! Open file read/write
	int	0x21
	jb	openerr
	mov	(vfd), ax	! File handle to open file
	xor	ax, ax
openerr:
	pop	bp
	ret

! int readsectors(u32_t bufaddr, u32_t sector, u8_t count)
! int writesectors(u32_t bufaddr, u32_t sector, u8_t count)
!	Read/write several sectors from/to the Minix virtual disk.  Count
!	must fit in a byte.  The external variable vfd is the file handle.
!	Returns 0 for success, otherwise the DOS error code.
!
.define _readsectors, _writesectors
_writesectors:
	push	bp
	mov	bp, sp
	movb	13(bp), 0x40	! Code for a file write
	jmp	rwsec
_readsectors:
	push	bp
	mov	bp, sp
	movb	13(bp), 0x3F	! Code for a file read
rwsec:	mov	dx, 8(bp)
	mov	bx, 10(bp)	! bx-dx = Sector number
	mov	cx, 9
mul512:	shl	dx, 1
	rcl	bx, 1		! bx-dx *= 512
	loop	mul512
	mov	cx, bx		! cx-dx = Byte position in file
	mov	bx, (vfd)	! bx = File handle
	mov	ax, 0x4200	! Lseek absolute
	int	0x21
	jb	rwerr
	mov	bx, (vfd)	! bx = File handle
	mov	ax, 4(bp)
	mov	dx, 6(bp)	! dx-ax = Address to transfer data to/from
	call	abs2seg
	mov	ds, dx
	mov	dx, ax		! ds:dx = Address to transfer data to/from
	xorb	cl, cl
	movb	ch, 12(bp)	! ch = Number of sectors to transfer
	shl	cx, 1		! cx = Number of bytes to transfer
	push	cx		! Save count
	movb	ah, 13(bp)	! Read or write
	int	0x21
	pop	cx		! Restore count
	push	es
	pop	ds		! Restore ds
	jb	rwerr
	cmp	ax, cx		! All bytes transferred?
	je	rwall
	mov	ax, 0x05	! The DOS code for "I/O error", but different
	jmp	rwerr
rwall:	call	wheel		! Display tricks
	xor	ax, ax
rwerr:	pop	bp
	ret

! int getchar(void), peekchar(void);
!	Read a character from the keyboard, or just look if there is one.
!	A carriage return is changed into a linefeed for UNIX compatibility.
.define _getchar, _peekchar
_peekchar:
	movb	ah, 0x01	! Keyboard status
	int	0x16
	jnz	getc		! Keypress?
	mov	ax, -1		! No key
	ret
_getchar:
	xorb	ah, ah		! Read character from keyboard
	int	0x16
getc:	cmpb	al, 0x0D	! Carriage return?
	jnz	nocr
	movb	al, 0x0A	! Change to linefeed
nocr:	xorb	ah, ah		! ax = al
	ret

! int putchar(int c);
!	Write a character in teletype mode.  The putc and putk synonyms
!	are for the kernel printk function that uses one of them.
!	Newlines are automatically preceded by a carriage return.
!
.define _putchar, _putc, _putk
_putchar:
_putc:
_putk:	mov	bx, sp
	movb	al, 2(bx)	! al = character to be printed
	testb	al, al		! 1.6.* printk adds a trailing null
	jz	nulch
	cmpb	al, 0x0A	! al = newline?
	jnz	putc
	movb	al, 0x20	! Erase wheel and do a carriage return
	call	plotc		! plotc(' ');
nodirt:	movb	al, 0x0D
	call	putc		! putc('\r')
	movb	al, 0x0A	! Restore the '\n' and print it
putc:	movb	ah, 0x0E	! Print character in teletype mode
	mov	bx, 0x0001	! Page 0, foreground color
	int	0x10		! Call BIOS VIDEO_IO
nulch:	ret

! |/-\|/-\|/-\|/-\|/-\	(playtime)
wheel:	mov	bx, (gp)
	movb	al, (bx)
	inc	bx		! al = *gp++;
	cmp	bx, glyphs+4
	jne	0f
	mov	bx, glyphs
0:	mov	(gp), bx	! gp= gp == glyphs + 4 ? glyphs : gp;
	!jmp	plotc
plotc:	movb	ah, 0x0A	! 0x0A = write character at cursor
	mov	bx, 0x0001	! Page 0, foreground color
	mov	cx, 0x0001	! Just one character
	int	0x10
	ret
.sect .data
	.align	2
gp:	.data2	glyphs
glyphs:	.ascii	"|/-\\"
.sect .text

! void reset_video(unsigned mode);
!	Reset and clear the screen.
.define _reset_video
_reset_video:
	mov	bx, sp
	mov	ax, 2(bx)	! Video mode
	mov	(cur_vid_mode), ax
	testb	ah, ah
	jnz	xvesa		! VESA extended mode?
	int	0x10		! Reset video (ah = 0)
	jmp	setcur
xvesa:	mov	bx, ax		! bx = extended mode
	mov	ax, 0x4F02	! Reset video
	int	0x10
setcur:	xor	dx, dx		! dl = column = 0, dh = row = 0
	xorb	bh, bh		! Page 0
	movb	ah, 0x02	! Set cursor position
	int	0x10
	ret

restore_video:			! To restore the video mode on exit
	movb	al, 0x20
	call	plotc		! Erase wheel
	mov	ax, (old_vid_mode)
	cmp	ax, (cur_vid_mode)
	je	vidok
	push	ax
	call	_reset_video
	pop	ax
vidok:	ret

! u32_t get_tick(void);
!	Return the current value of the clock tick counter.  This counter
!	increments 18.2 times per second.  Poll it to do delays.  Does not
!	work on the original PC, but works on the PC/XT.
.define _get_tick
_get_tick:
	xorb	ah, ah		! Code for get tick count
	int	0x1A
	mov	ax, dx
	mov	dx, cx		! dx:ax = cx:dx = tick count
	ret


! Functions used to obtain info about the hardware, like the type of video
! and amount of memory.  Boot uses this information itself, but will also
! pass them on to a pure 386 kernel, because one can't make BIOS calls from
! protected mode.  The video type could probably be determined by the kernel
! too by looking at the hardware, but there is a small chance on errors that
! the monitor allows you to correct by setting variables.

.define _get_bus		! returns type of system bus
.define _get_video		! returns type of display
.define _get_memsize		! returns amount of low memory in K
.define _get_ext_memsize	! returns amount of extended memory in K

! u16_t get_bus(void)
!	Return type of system bus, in order: (XT), AT, MCA.
_get_bus:
	mov	dx, 1		! Assume AT
	movb	ah, 0xC0	! Code for get configuration
	int	0x15
	jc	got_bus		! Carry clear and ah = 00 if supported
	testb	ah, ah
	jne	got_bus
 eseg	movb	al, 5(bx)	! Load feature byte #1
	testb	al, 0x02	! Test bit 1 - "bus is Micro Channel"
	jz	got_bus
	dec	dx		! It is MCA
got_bus:
	push	ds
	pop	es		! Restore es
	mov	ax, dx		! Return bus code
	ret

! u16_t get_video(void)
!	Return type of display, in order: MDA, CGA, mono EGA, color EGA,
!	mono VGA, color VGA.
_get_video:
	mov	ax, 0x1A00	! Function 1A returns display code
	int	0x10		! al = 1A if supported
	cmpb	al, 0x1A
	jnz	no_dc		! No display code function supported

	mov	ax, 2
	cmpb	bl, 5		! Is it a monochrome EGA?
	jz	got_video
	inc	ax
	cmpb	bl, 4		! Is it a color EGA?
	jz	got_video
	inc	ax
	cmpb	bl, 7		! Is it a monochrome VGA?
	jz	got_video
	inc	ax
	cmpb	bl, 8		! Is it a color VGA?
	jz	got_video

no_dc:	movb	ah, 0x12	! Get information about the EGA
	movb	bl, 0x10
	int	0x10
	cmpb	bl, 0x10	! Did it come back as 0x10? (No EGA)
	jz	no_ega

	mov	ax, 2
	cmpb	bh, 1		! Is it monochrome?
	jz	got_video
	inc	ax
	jmp	got_video

no_ega:	int	0x11		! Get bit pattern for equipment
	and	ax, 0x30	! Isolate color/mono field
	sub	ax, 0x30
	jz	got_video	! Is it an MDA?
	mov	ax, 1		! No it's CGA

got_video:
	ret

! u16_t get_memsize(void);
!	How much normal memory is there available?
_get_memsize:
!---
!.data1 o32;	push	(xms_inuse)
!.data1 o32;	push	(xms_temp)
!.data1 o32;	push	(xms_size)
!.data1 o32;	push	(xms_base)
!.data1 o32;	push	(mem_inuse)
!.data1 o32;	push	(mem_temp)
!.data1 o32;	push	(mem_size)
!.data1 o32;	push	(mem_base)
!	push	mem_tell
!	call	_printk
!	add	sp, 2+8*4
!.sect .rom
!mem_tell:
!	.ascii	"mem_base  = %08lX\n"
!	.ascii	"mem_size  = %08lX\n"
!	.ascii	"mem_temp  = %08lX\n"
!	.ascii	"mem_inuse = %08lX\n"
!	.ascii	"xms_base  = %08lX\n"
!	.ascii	"xms_size  = %08lX\n"
!	.ascii	"xms_temp  = %08lX\n"
!	.ascii	"xms_inuse = %08lX\n"
!	.data1	0
!.sect .text
!---
	mov	ax, (mem_size+0)
	mov	dx, (mem_size+2)
	shr	ax, 10
	shl	dx, 16-10
	or	ax, dx		! ax = mem_size >> 10
	ret

! u32_t get_ext_memsize(void);
!	How much extended memory is there available?
_get_ext_memsize:
	.data1	o32
	mov	ax, (xms_size)
	.data1	o32
	sub	ax, (mem_inuse)	! eax = xms_size - mem_inuse
	.data1	o32
	shr	ax, 10		! eax >>= 10
	.data1	o32
	mov	dx, ax
	.data1	o32
	shr	dx, 16		! dx-ax = free extended memory in K
	ret


! Function to leave the boot monitor and run Minix.
.define _minix

! void minix(u32_t koff, u32_t kcs, u32_t kds,
!					char *bootparams, size_t paramsize);
_minix:
	push	bp
	mov	bp, sp		! Pointer to arguments

	movb	ah, 0x68	! Flush modified file blocks
	mov	bx, (vfd)
	int	0x21
	mov	dx, 0x03F2	! Floppy motor drive control bits
	movb	al, 0x0C	! Bits 4-7 for floppy 0-3 are off
	outb	dx		! Kill the motors
	cli			! No more interruptions

  cseg	mov	(cs_real-2), cs	! Patch CS and DS into the instructions that
  cseg	mov	(ds_real-2), ds	! reload them when switching back to real mode

	mov	dx, cs		! Monitor cs
	xor	ax, ax		! dx:ax = Monitor code segment
	call	seg2abs
	mov	(p_mcs_desc+2), ax
	movb	(p_mcs_desc+4), dl

	mov	dx, ds		! Monitor ds
	mov	ax, p_gdt	! dx:ax = Global descriptor table
	call	seg2abs
	mov	(p_gdt_desc+2), ax
	movb	(p_gdt_desc+4), dl ! Set base of global descriptor table

	mov	dx, ss		! Monitor ss
	xor	ax, ax		! dx:ax = Monitor stack segment
	call	seg2abs		! Minix starts with the stack of the monitor
	mov	(p_ss_desc+2), ax
	movb	(p_ss_desc+4), dl

	mov	ax, 8(bp)
	mov	dx, 10(bp)	! Kernel cs (absolute address)
	mov	(p_cs_desc+2), ax
	movb	(p_cs_desc+4), dl

	mov	ax, 12(bp)
	mov	dx, 14(bp)	! Kernel ds (absolute address)
	mov	(p_ds_desc+2), ax
	movb	(p_ds_desc+4), dl ! Set base of kernel data segment

	call	real2prot	! Switch to protected mode
	call	exchange	! Exchange DOS stuff with Minix stuff

	test	(_k_flags), K_I386 ! Minix-386?
	jnz	minix386

! Call Minix in real mode.
minix86:
	call	prot2real	! Switch back to real mode

	push	18(bp)		! # bytes of boot parameters
	push	16(bp)		! address of boot parameters

	test	(_k_flags), K_RET ! Can the kernel return?
	jz	noret86
	push	cs
	push	ret86		! Monitor far return address
noret86:

	mov	ax, 8(bp)
	mov	dx, 10(bp)
	call	abs2seg
	push	dx		! Kernel code segment
	push	4(bp)		! Kernel code offset
	mov	ax, 12(bp)
	mov	dx, 14(bp)
	call	abs2seg
	mov	ds, dx		! Kernel data segment
	mov	es, dx		! Set es to kernel data too
	retf			! Make a far call to the kernel

! Call 386 Minix in 386 mode.
minix386:
	push	MCS_SELECTOR
	test	(_k_flags), K_INT86 ! Generic INT86 support?
	jz	0f
	push	int86		! Far address to INT86 support
	jmp	1f
0:	push	bios13		! Far address to BIOS int 13 support
1:
	push	0
	push	18(bp)		! 32 bit size of parameters on stack
	push	0
	push	16(bp)		! 32 bit address of parameters (ss relative)

	test	(_k_flags), K_RET ! Can the kernel return?
	jz	noret386
	push	MCS_SELECTOR
	push	ret386		! Monitor far return address
noret386:

	push	0
	push	CS_SELECTOR
	push	6(bp)
	push	4(bp)		! 32 bit far address to kernel entry point

	mov	ax, DS_SELECTOR
	mov	ds, ax		! Kernel data
	mov	ax, ES_SELECTOR
	mov	es, ax		! Flat 4 Gb

	.data1	o32		! Make a far call to the kernel
	retf

! Minix-86 returns here on a halt or reboot.
ret86:
	mov	8(bp), ax
	mov	10(bp), dx	! Return value
	call	real2prot
	call	exchange	! Unexchange Minix and DOS stuff
	call	prot2real
	jmp	return

! Minix-386 returns here on a halt or reboot.
ret386:
	.data1	o32
	mov	8(bp), ax	! Return value
	call	exchange	! Unexchange Minix and DOS stuff
	call	prot2real	! Switch to real mode

return:
	mov	sp, bp		! Pop parameters
	sti			! Can take interrupts again
	call	_get_video	! MDA, CGA, EGA, ...
	movb	dh, 24		! dh = row 24
	cmp	ax, 2		! At least EGA?
	jb	is25		! Otherwise 25 rows
	push	ds
	mov	ax, 0x0040	! BIOS data segment
	mov	ds, ax
	movb	dh, (0x0084)	! Number of rows on display minus one
	pop	ds
is25:
	xorb	dl, dl		! dl = column 0
	xorb	bh, bh		! Page 0
	movb	ah, 0x02	! Set cursor position
	int	0x10
	mov	(cur_vid_mode), -1  ! Minix may have messed things up

	mov	ax, 8(bp)
	mov	dx, 10(bp)	! dx-ax = return value from the kernel
	pop	bp
	ret			! Continue as if nothing happened

! Support function for Minix-386 to make a BIOS int 13 call (disk I/O).
bios13:
	mov	bp, sp
	call	exchange
	call	prot2real
	sti			! Enable interrupts

	mov	ax, 8(bp)	! Load parameters
	mov	bx, 10(bp)
	mov	cx, 12(bp)
	mov	dx, 14(bp)
	mov	es, 16(bp)
	int	0x13		! Make the BIOS call
	mov	8(bp), ax	! Save results
	mov	10(bp), bx
	mov	12(bp), cx
	mov	14(bp), dx
	mov	16(bp), es

	cli			! Disable interrupts
	call	real2prot
	call	exchange
	mov	ax, DS_SELECTOR	! Kernel data
	mov	ds, ax
	.data1	o32
	retf			! Return to the kernel

! Support function for Minix-386 to make an 8086 interrupt call.
int86:
	mov	bp, sp
	call	exchange
	call	prot2real
	sti			! Enable interrupts

	movb	al, 0xCD	! INT instruction
	movb	ah, 8(bp)	! Interrupt number?
	testb	ah, ah
	jnz	0f		! Nonzero if INT, otherwise far call
	push	cs
	push	intret+2	! Far return address
	.data1	o32
	push	12(bp)		! Far driver address
	mov	ax, 0x90CB	! RETF; NOP
0: cseg	mov	(intret), ax	! Patch `INT n' or `RETF; NOP' into code

	mov	ds, 16(bp)	! Load parameters
	mov	es, 18(bp)
	.data1	o32
	mov	ax, 20(bp)
	.data1	o32
	mov	bx, 24(bp)
	.data1	o32
	mov	cx, 28(bp)
	.data1	o32
	mov	dx, 32(bp)
	.data1	o32
	mov	si, 36(bp)
	.data1	o32
	mov	di, 40(bp)
	.data1	o32
	mov	bp, 44(bp)

intret:	int	0xFF		! Do the interrupt or far call

	.data1	o32		! Save results
	push	bp
	.data1	o32
	pushf
	mov	bp, sp
	add	bp, 2*4
	.data1	o32
	pop	8(bp)		! eflags
	mov	16(bp), ds
	mov	18(bp), es
	.data1	o32
	mov	20(bp), ax
	.data1	o32
	mov	24(bp), bx
	.data1	o32
	mov	28(bp), cx
	.data1	o32
	mov	32(bp), dx
	.data1	o32
	mov	36(bp), si
	.data1	o32
	mov	40(bp), di
	.data1	o32
	pop	44(bp)		! ebp

	cli			! Disable interrupts
	mov	ax, ss
	mov	ds, ax		! Restore monitor ds
	call	real2prot
	call	exchange
	mov	ax, DS_SELECTOR	! Kernel data
	mov	ds, ax
	.data1	o32
	retf			! Return to the kernel

! Exchange memory in-use by DOS by Minix stuff or vice-versa.  Smallest unit
! is a page (16 bytes), so copying words is ok.
exchange:
	push	si
	push	di
	.data1	o32
	xor	cx, cx
	mov	cx, 4
	.data1	o32
	mov	si, (C1k5)	! Low memory start address
	.data1	o32
	mov	bx, (mem_base)	! End address
	.data1	o32
	mov	di, (mem_temp)	! Temporary save area
	call	ex1		! Exchange low DOS memory
	.data1	o32
	mov	si, (C1M)	! Extended memory start address
	.data1	o32
	mov	bx, si
	.data1	o32
	add	bx, (xms_inuse)	! End address
	.data1	o32
	mov	di, (xms_temp)	! Temporary save area
	.data1	o32
	cmp	bx, di		! Any XMS in the temp area?
	jae	0f
	call	ex1		! Exchange extended DOS memory
0:	pop	di
	pop	si
	ret
ex1:
	mov	ax, ES_SELECTOR
	mov	ds, ax			! Flat 4 Gb
	jmp	0f
	.align	16
1:
	.data1	o32, a32, 0x8B, 0x06
	!mov	eax, (esi)		! Load a longword from each area
	.data1	o32, a32, 0x8B, 0x17
	!mov	edx, (edi)
	.data1	o32, a32, 0x89, 0x16
	!mov	(esi), edx		! Store exchanged
	.data1	o32, a32, 0x89, 0x07
	!mov	(edi), eax
	.data1	o32
	add	si, cx			! si += 4
	.data1	o32
	add	di, cx			! di += 4
0:	.data1	o32
	cmp	si, bx			! End reached?
	jb	1b
	mov	ax, SS_SELECTOR
	mov	ds, ax
	ret

! Switch from real to protected mode.
real2prot:
	lgdt	(p_gdt_desc)	! Global descriptor table
	.data1	0x0F,0x20,0xC0	! mov	eax, cr0
	orb	al, 0x01	! Set PE (protection enable) bit
	.data1	0x0F,0x22,0xC0	! mov	cr0, eax
	jmpf	MCS_SELECTOR:cs_prot ! Set code segment selector
cs_prot:
	mov	ax, SS_SELECTOR	! Set data selectors
	mov	ds, ax
	mov	es, ax
	mov	ss, ax

	movb	ah, 0xDF	! Code for A20 enable
	jmp	gate_A20

! Switch from protected to real mode.
prot2real:
	lidt	(p_idt_desc)	! Real mode interrupt vectors
	.data1	0x0F,0x20,0xC0	! mov	eax, cr0
	andb	al, ~0x01	! Clear PE bit
	.data1	0x0F,0x22,0xC0	! mov	cr0, eax
	jmpf	0xDEAD:cs_real	! Reload cs register
cs_real:
	mov	ax, 0xBEEF
ds_real:
	mov	ds, ax		! Reload data segment registers
	mov	es, ax
	mov	ss, ax

	movb	ah, 0xDD	! Code for A20 disable
	!jmp	gate_A20

! Enable (ah = 0xDF) or disable (ah = 0xDD) the A20 address line.
gate_A20:
	call	kb_wait
	movb	al, 0xD1	! Tell keyboard that a command is coming
	outb	0x64
	call	kb_wait
	movb	al, ah		! Enable or disable code
	outb	0x60
	call	kb_wait
	mov	ax, 25		! 25 microsec delay for slow keyboard chip
0:	out	0xED		! Write to an unused port (1us)
	dec	ax
	jnz	0b
	ret
kb_wait:
	inb	0x64
	testb	al, 0x02	! Keyboard input buffer full?
	jnz	kb_wait		! If so, wait
	ret

.sect .data
	.align	4

! Global descriptor tables.
	UNSET	= 0		! Must be computed

! For "Extended Memory Block Move".
x_gdt:
x_null_desc:
	! Null descriptor
	.data2	0x0000, 0x0000
	.data1	0x00, 0x00, 0x00, 0x00
x_gdt_desc:
	! Descriptor for this descriptor table
	.data2	6*8-1, UNSET
	.data1	UNSET, 0x00, 0x00, 0x00
x_src_desc:
	! Source segment descriptor
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x92, 0x00, 0x00
x_dst_desc:
	! Destination segment descriptor
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x92, 0x00, 0x00
x_bios_desc:
	! BIOS segment descriptor (scratch for int 0x15)
	.data2	UNSET, UNSET
	.data1	UNSET, UNSET, UNSET, UNSET
x_ss_desc:
	! BIOS stack segment descriptor (scratch for int 0x15)
	.data2	UNSET, UNSET
	.data1	UNSET, UNSET, UNSET, UNSET

! Protected mode descriptor table.
p_gdt:
p_null_desc:
	! Null descriptor
	.data2	0x0000, 0x0000
	.data1	0x00, 0x00, 0x00, 0x00
p_gdt_desc:
	! Descriptor for this descriptor table
	.data2	8*8-1, UNSET
	.data1	UNSET, 0x00, 0x00, 0x00
p_idt_desc:
	! Real mode interrupt descriptor table descriptor
	.data2	0x03FF, 0x0000
	.data1	0x00, 0x00, 0x00, 0x00
p_ds_desc:
	! Kernel data segment descriptor (4Gb flat)
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x92, 0xCF, 0x00
p_es_desc:
	! Physical memory descriptor (4Gb flat)
	.data2	0xFFFF, 0x0000
	.data1	0x00, 0x92, 0xCF, 0x00
p_ss_desc:
	! Monitor data segment descriptor (64Kb flat)
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x92, 0x00, 0x00
p_cs_desc:
	! Kernel code segment descriptor (4Gb flat)
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x9A, 0xCF, 0x00
p_mcs_desc:
	! Monitor code segment descriptor (64 kb flat) (unused)
	.data2	0xFFFF, UNSET
	.data1	UNSET, 0x9A, 0x00, 0x00

C1k5:		.data4	0x00000600	! Constant for address 1.5k
C1M:		.data4	0x00100000	! Constant for address 1M
CFFF0:		.data4	0x0000FFF0	! Constant for largest block to copy
xms_base:	.data4	0x00100000	! Base of extended memory
xms_handle:	.data2	-1		! Handle of allocated XMS block

.sect .bss
	.comm	mem_base, 4		! Free low memory base
	.comm	mem_size, 4		! Low memory size from 0 up
	.comm	mem_temp, 4		! 1.5k < X < mem_base temporary
	.comm	mem_inuse, 4		! mem_base - 1.5k
	.comm	xms_driver, 4		! Vector to XMS driver
	.comm	xms_size, 4		! Size of allocated XMS block
	.comm	xms_temp, 4		! 1M < X < xms_base temporary
	.comm	xms_inuse, 4		! xms_base - 1M
	.comm	old_vid_mode, 2		! Video mode at startup
	.comm	cur_vid_mode, 2		! Current video mode
	.comm	vfd, 2			! Virtual disk file handle
