/*
libc/os/minix/other/u_sleep.c

Created:	Mar 15, 1993 by Philip Homburg
Modified:	Oct 13, 1993 by Philip Homburg <philip@cs.vu.nl>
		- rewritten to use POSIX signals

Sleep until a certain moment, return EINTR if returning earlier 
*/

#define _MINIX_SOURCE
#define _POSIX_SOURCE

#include <minix/stubs.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>

static void alarm_dummy _ARGS(( int sig ));

int u_sleep(tvp)
struct timeval *tvp;
{
	sigset_t ss, ss_orig, ss_tmp;
	struct timeval curr_time, time_out, timer_orig;
	struct sigaction action, action_orig;
	int r, saved_errno;
	int restore_procmask;
	int restore_handler;
	int restore_timer;

	r= 0;
	restore_procmask= 0;
	restore_handler= 0;
	restore_timer= 0;

	/* Block all signals */
	if (r != -1) r= sigfillset(&ss);
	if (r != -1) r= sigemptyset(&ss_orig);
	if (r != -1) r= sigprocmask(SIG_BLOCK, &ss, &ss_orig);
	if (r != -1) restore_procmask= 1;

	time_out= *tvp;

	/* Cancel old timer, and get value of that timer */
	if (r != -1)
	{
		timer_orig.tv_sec= 0;
		timer_orig.tv_usec= 0;
		r= sysutime(UTIME_SETALARM, &timer_orig);
		if (r != -1)
			restore_timer= 1;
	}

	/* Register new signal handler */
	if (r != -1)
	{
		action.sa_handler= alarm_dummy;
		sigemptyset(&action.sa_mask);
		action.sa_flags= 0;
		r= sigaction(SIGALRM, &action, &action_orig);
		if (r != -1)
			restore_handler =1;
	}

	/* Set new timer */
	if (r != -1)
	{
		if (timer_orig.tv_sec != 0 && timer_orig.tv_sec <
			time_out.tv_sec)
		{
			time_out=timer_orig;
		}
		r= sysutime(UTIME_SETALARM, &time_out);
	}

	/* Wait until something happens */
	if (r != -1)
	{
		ss_tmp= ss_orig;
		sigdelset(&ss_tmp, SIGALRM);
		r= sigsuspend(&ss_tmp);
		if (r == -1 && errno == EINTR)
			r= 0;
	}

	if (r == -1)
		saved_errno= errno;

	if (restore_timer)
		sysutime(UTIME_SETALARM, &timer_orig);
	if (restore_handler)
		sigaction(SIGALRM, &action_orig, NULL);
	if (restore_procmask)
		sigprocmask(SIG_SETMASK, &ss_orig, NULL);

	if (r == -1)
	{
		errno= saved_errno;
		return -1;
	}

	/* Calculate if we should return EINTR or 0 */
	time_out= curr_time;
	sysutime(UTIME_TIMEOFDAY, &time_out);
	if (time_out.tv_sec > tvp->tv_sec ||
		(time_out.tv_sec == tvp->tv_sec &&
		time_out.tv_usec >= tvp->tv_usec))
	{
		return 0;
	}
	errno= EINTR;
	return -1;
}

static void alarm_dummy(sig)
int sig;
{
}

/*
 * $PchId: u_sleep.c,v 1.4 1995/11/27 20:17:48 philip Exp $
 */
