/*
nbio_select.c

Created:	11 April 1994 by Philip Homburg <philip@cs.vu.nl>
*/

#include "ansi.h"
#include "nbio_int.h"

static fwait_t fw;

static void select_to ARGS(( int sig ));

DEFUN
(int nbio_select, (nfds, readfds, writefds, exceptfds, timeout),
	int nfds AND
	u32_t *readfds AND
	u32_t *writefds AND
	u32_t *exceptfds AND
	struct timeval *timeout
)
{
	asio_fd_set_t fd_set;
	int i, j, k, r, res, err, fd, op;
	u32_t ready_set, new_set, tmp_set, m;
	fd_set_t new_read;
	fd_set_t new_write;
	fd_set_t new_except;
	int block;
	int alarm_set;
	struct timeval timer, oldtimer;
	struct sigaction sanew, saold;

	DPRINT(2, (stderr, "select(%d, &0x%x &0x%x &0x%x, %p)\n", nfds,
		readfds ? readfds[0] : 0,
		writefds ? writefds[0]: 0,
		exceptfds ? exceptfds[0]: 0,
		timeout));

	if (nfds < 0 || nfds > SET_SIZE)
	{
		errno= EINVAL;
		return -1;
	}

	/* Look for blocking/non-blocking */

	block= 0;
	for(;;)
	{
		fd_set= nbio_asio_fd_set; 
		fw.fw_flags= 0;
		fw.fw_bits= fd_set.afds_bits;
		fw.fw_maxfd= ASIO_FD_SETSIZE;

		alarm_set= 0;
		if (!block)
			fw.fw_flags |= FWF_NONBLOCK;
		else if (timeout != NULL)
		{
			DPRINT(5, (stderr,
				"select: timeout { %d, %d }\n",
				timeout->tv_sec, timeout->tv_usec));

			/* Calculate timeout */
			if (sysutime(UTIME_TIMEOFDAY, &timer) == -1)
				abort();
			timer.tv_sec += timeout->tv_sec;
			timer.tv_usec += timeout->tv_usec;
			while(timer.tv_usec >= 1000000)
			{
				timer.tv_sec++;
				timer.tv_usec -= 1000000;
			}

			/* Stop previous timer */
			oldtimer.tv_sec= oldtimer.tv_usec= 0;
			if (sysutime(UTIME_SETALARM, &oldtimer) == -1)
				abort();

			/* Register a new signal handler */
			sanew.sa_handler= select_to;
			sigemptyset(&sanew.sa_mask);
			sanew.sa_flags= 0;
			if (sigaction(SIGALRM, &sanew, &saold) == -1)
				abort();
			if (oldtimer.tv_sec != 0 &&
				oldtimer.tv_sec < timer.tv_sec)
			{
				timer= oldtimer;
			}
			DPRINT(5, (stderr,
				"select: setting alarm to { %d, %d }\n",
				timer.tv_sec, timer.tv_usec));
			if (sysutime(UTIME_SETALARM, &timer) == -1)
				abort();
			alarm_set= 1;
		}

		r= fwait(&fw);

		if (r == 0)
		{
			for(;;)
			{
				fd= fw.fw_fd;
				op= fw.fw_operation;
				res= fw.fw_result;
				err= fw.fw_errno;
				nbio_completed(fd, op);
				(nbio_ip_table[fd][op].func)
					(nbio_ip_table[fd][op].ref, res, err);
				if (!(fw.fw_flags & FWF_MORE))
					break;
				r= fwait(&fw);
				if (r == -1)
					abort();
			}
		}
		if (r == -1)
		{
			if (errno != EAGAIN && errno != EINTR)
				abort();
		}

		if (alarm_set)
		{
			timer.tv_sec= 0;
			timer.tv_usec= 0;
			if (sysutime(UTIME_SETALARM, &timer) == -1)
				abort();
			if (sigaction(SIGALRM, &saold, NULL) == -1)
				abort();
			if (oldtimer.tv_sec != 0)
			{
				if (sysutime(UTIME_SETALARM, &timer) == -1)
					abort();
			}
		}

		res= 0;

#define block(x_fds, x_all, x_block, new_x) \
	if (x_fds) \
	{ \
		for (i= 0, j= 0; j<nfds; i++, j += NFDBITS) \
		{ \
			ready_set= x_all.fds_bits[i] & ~x_block.fds_bits[i]; \
			tmp_set= x_fds[i] & ready_set; \
			new_set= 0; \
			if (tmp_set) \
			{ \
				for (m= 1, k= j; k<nfds; m <<= 1, k++) \
				{ \
					if (tmp_set & m) \
					{ \
						new_set |= m; \
						res++; \
					} \
				} \
			} \
			new_x.fds_bits[i]= new_set; \
		} \
	} else ((void)0)

		if (readfds)
		{
			DPRINT(2, (stderr, "select: readfds[0]= 0x%x\n",
				readfds[0]));
		}
		DPRINT(2, (stderr, "select: read_all[0]= 0x%x\n",
			nbio_read_all.fds_bits[0]));
		DPRINT(2, (stderr, "select: read_block[0]= 0x%x\n",
			nbio_read_block.fds_bits[0]));
		if (writefds)
		{
			DPRINT(2, (stderr, "select: writefds[0]= 0x%x\n",
				writefds[0]));
		}
		DPRINT(2, (stderr, "select: write_all[0]= 0x%x\n",
			nbio_write_all.fds_bits[0]));
		DPRINT(2, (stderr, "select: write_block[0]= 0x%x\n",
			nbio_write_block.fds_bits[0]));
		if (exceptfds)
		{
			DPRINT(2, (stderr, "select: exceptfds[0]= 0x%x\n",
				exceptfds[0]));
		}
		DPRINT(2, (stderr, "select: except_all[0]= 0x%x\n",
			nbio_except_all.fds_bits[0]));
		DPRINT(2, (stderr, "select: except_block[0]= 0x%x\n",
			nbio_except_block.fds_bits[0]));

		block(readfds, nbio_read_all, nbio_read_block, new_read);
		block(writefds, nbio_write_all, nbio_write_block, new_write);
		block(exceptfds, nbio_except_all, nbio_except_block,
			new_except);

		DPRINT(2, (stderr, "select: new_read[0]= 0x%x\n",
			new_read.fds_bits[0]));
		DPRINT(2, (stderr, "select: new_write[0]= 0x%x\n",
			new_write.fds_bits[0]));
		DPRINT(2, (stderr, "select: new_except[0]= 0x%x\n",
			new_except.fds_bits[0]));

		if (res || block)
			break;
		if (timeout != NULL && timeout->tv_sec == 0 &&
			timeout->tv_usec == 0)
		{
			break;
		}
		block= 1;
	}

	for (i= 0, j= 0; j<nfds; i++, j += NFDBITS)
	{
		if (readfds) readfds[i]= new_read.fds_bits[i];
		if (writefds) writefds[i]= new_write.fds_bits[i];
		if (exceptfds) exceptfds[i]= new_except.fds_bits[i];
	}

	DPRINT(5, (stderr, "select: %d %x %x %x\n", res,
		readfds ? readfds[0] : 0,
		writefds ? writefds[0] : 0,
		exceptfds ? exceptfds[0] : 0));
	return res;
}

DEFUN
(static void select_to, (sig),
	int sig
)
{
	DPRINT(2, (stderr, "select_to(%d)\n", sig));

	fw.fw_flags |= FWF_NONBLOCK;
}

/*
 * $PchId: nbio_select.c,v 1.3 1996/02/23 07:12:40 philip Exp $
 */
