/*
nbio_rw.c

Created:	April 14, 1994 by Philip Homburg <philip@cs.vu.nl>
*/

#include "ansi.h"
#include "nbio_int.h"

static void nbio_read_cb ARGS(( nbio_ref_t ref, int res, int err ));
static void nbio_write_cb ARGS(( nbio_ref_t ref, int res, int err ));

DEFUN
(ssize_t nbio_read, (fd, buf, size),
	int fd AND
	void *buf AND
	size_t size
)
{
	struct ip_table *tabp;
	int len, r, ret, s_errno;
	int offset;
	asio_fd_set_t fd_set;
	fwait_t fw;
	char *cbuf;

	DPRINT(2, (stderr, "nbio_read(%d,%p,%d)\n", fd, buf, size));

	cbuf= buf;
	if (fd < 0 || fd > SET_SIZE)
	{
		errno= EINVAL;
		return -1;
	}

	tabp= &nbio_ip_table[fd][ASIO_READ];
	if (tabp->buf == NULL)
	{
		tabp->bufsize= READ_BUFSIZE;
		if ((tabp->buf= malloc(tabp->bufsize)) == NULL)
		{
			errno= ENOMEM;
			return -1;
		}
		tabp->data_offset= 0;
		tabp->data_size= 0;
		tabp->inprogress= 0;
		tabp->err= 0;
		tabp->func= nbio_read_cb;
		tabp->ref.ref_int= fd;
	}

	offset= 0;
	if (tabp->inprogress)
	{
		DPRINT(5, (stderr, "nbio_read: EAGAIN\n"));
		errno= EAGAIN;
		return -1;
	}

	/* Copy any data left in the buffer */
	if (tabp->data_offset < tabp->data_size)
	{
		len= tabp->data_size-tabp->data_offset;
		if (len > size-offset)
			len= size-offset;
		DPRINT(5, (stderr, "nbio_read: copying %d bytes\n", len));

		memcpy(cbuf+offset, tabp->buf + tabp->data_offset, len);
		offset += len;
		tabp->data_offset += len;
		if (tabp->data_offset < tabp->data_size)
			return offset;
	}

	/* Try to read directly into the user's buffer. */
	ret= 0;
	s_errno= 0;
	while(offset < size)
	{
		r= read(fd, cbuf+offset, size-offset);
		if (r == -1 && errno == EINPROGRESS)
		{
			r= fcancel(fd, ASIO_READ);
			if (r == -1)
				abort();
			ASIO_FD_ZERO(&fd_set);
			ASIO_FD_SET(fd, ASIO_READ, &fd_set);
			fw.fw_flags= FWF_NONBLOCK;
			fw.fw_bits= fd_set.afds_bits;
			fw.fw_maxfd= ASIO_FD_SETSIZE;
			r= fwait(&fw);
			if (r == -1 || fw.fw_fd != fd ||
				fw.fw_operation != ASIO_READ)
			{
				abort();
			}
			r= fw.fw_result;
			errno= fw.fw_errno;
		}

		if (r > 0)
		{
			DPRINT(5, (stderr, "nbio_read: read %d bytes\n", r));
			offset += r;
			continue;
		}
		else if (r == 0)
		{
			DPRINT(0, (stderr, "nbio_read: read EOF\n"));
			break;
		}
		else
		{
			if (errno == EINTR)
			{
				DPRINT(5, (stderr, "nbio_read: EINTR\n"));
				errno= EAGAIN;
			}
			else
			{
				DPRINT(0, (stderr, "nbio_read: read error %s\n",
					strerror(errno)));
			}
			s_errno= errno;
			ret= -1;
			break;
		}
	}
	if (offset != 0)
		ret= offset;

	if (tabp->data_offset != tabp->data_size)
		abort();
	tabp->data_offset= 0;
	r= read(fd, tabp->buf, tabp->bufsize);
	if (r >= 0)
	{
		DPRINT(5, (stderr, "nbio_read: buffered %d bytes\n", r));
		tabp->data_size= r;
	}
	else if (r == -1 && errno == EINPROGRESS)
	{
		tabp->inprogress= 1;
		nbio_inprogress(fd, ASIO_READ, 1 /* read */,
			0 /* write */, 0 /* exception */);
	}
	else
	{
		DPRINT(0, (stderr, "nbio_read: read failed: %s\n",
			strerror(errno)));
	}
	errno= s_errno;
	return ret;
}

DEFUN
(ssize_t nbio_write, (fd, buf, size),
	int fd AND
	void *buf AND
	size_t size
)
{
	struct ip_table *tabp;
	int len, r, ret, s_errno;
	int offset;
	asio_fd_set_t fd_set;
	fwait_t fw;
	char *cbuf;

	DPRINT(2, (stderr, "nbio_write(%d,%p,%d)\n", fd, buf, size));

	cbuf= buf;
	if (fd < 0 || fd > SET_SIZE)
	{
		errno= EINVAL;
		return -1;
	}

	tabp= &nbio_ip_table[fd][ASIO_WRITE];
	if (tabp->buf == NULL)
	{
		tabp->bufsize= WRITE_BUFSIZE;
		if ((tabp->buf= malloc(tabp->bufsize)) == NULL)
		{
			errno= ENOMEM;
			return -1;
		}
		tabp->data_offset= 0;
		tabp->data_size= 0;
		tabp->inprogress= 0;
		tabp->err= 0;
		tabp->func= nbio_write_cb;
		tabp->ref.ref_int= fd;
	}

	offset= 0;

	if (tabp->err)
	{
		DPRINT(0, (stderr, "nbio_write: err %d\n", tabp->err));
		errno= tabp->err;
		return -1;
	}

	if (tabp->inprogress)
	{
		DPRINT(5, (stderr, "nbio_write: EAGAIN\n"));
		errno= EAGAIN;
		return -1;
	}

	/* Try to write directly out of the user's buffer. */
	ret= 0;
	s_errno= 0;
	while(offset < size)
	{
		r= write(fd, cbuf+offset, size-offset);
		if (r == -1 && errno == EINPROGRESS)
		{
			r= fcancel(fd, ASIO_WRITE);
			if (r == -1)
				abort();
			ASIO_FD_ZERO(&fd_set);
			ASIO_FD_SET(fd, ASIO_WRITE, &fd_set);
			fw.fw_flags= FWF_NONBLOCK;
			fw.fw_bits= fd_set.afds_bits;
			fw.fw_maxfd= ASIO_FD_SETSIZE;
			r= fwait(&fw);
			if (r == -1 || fw.fw_fd != fd ||
				fw.fw_operation != ASIO_WRITE)
			{
				abort();
			}
			r= fw.fw_result;
			errno= fw.fw_errno;
		}
		if (r > 0)
		{
			DPRINT(5, (stderr, "nbio_write: wrote %d bytes\n", r));
			offset += r;
			continue;
		}
		else if (r == 0)
			abort();
		else
		{
			if (errno == EINTR)
			{
				DPRINT(5, (stderr, "nbio_write: EINTR\n"));
				errno= EAGAIN;
			}
			else
			{
				DPRINT(1, (stderr,
					"nbio_write: write error: %s\n",
					strerror(errno)));
			}
			s_errno= errno;
			ret= -1;
			break;
		}
	}

	/* Copy any data to the buffer */
	if (offset < size)
	{
		len= tabp->bufsize;
		if (len > size-offset)
			len= size-offset;
		DPRINT(5, (stderr, "nbio_write: copying %d bytes\n", len));

		memcpy(tabp->buf, cbuf+offset, len);
		offset += len;
		tabp->data_offset= 0;
		tabp->data_size= len;
	}
	if (offset != 0)
		ret= offset;

	while (tabp->data_offset < tabp->data_size)
	{
		r= write(fd, tabp->buf+tabp->data_offset,
			tabp->data_size-tabp->data_offset);
		if (r > 0)
		{
			DPRINT(5, (stderr, "nbio_write: wrote %d bytes\n", r));
			tabp->data_offset += r;
			continue;
		}
		else if (r == -1 && errno == EINPROGRESS)
		{
			tabp->inprogress= 1;
			nbio_inprogress(fd, ASIO_WRITE, 0 /* read */,
				1 /* write */, 0 /* exception */);
		}
		else
		{
			DPRINT(0, (stderr, "nbio_write: write failed: %s\n",
				strerror(errno)));
			tabp->err= errno;
		}
		break;
	}

	errno= s_errno;
	return ret;
}

DEFUN
(static void nbio_read_cb, (ref, res, err),
	nbio_ref_t ref AND
	int res AND
	int err
)
{
	struct ip_table *tabp;

	DPRINT(2, (stderr, "nbio_read_cb(%d,%d,%d)\n", ref.ref_int, res, err));

	tabp= &nbio_ip_table[ref.ref_int][ASIO_READ];
	if (res > 0)
		tabp->data_size= res;
	tabp->inprogress= 0;
}

DEFUN
(static void nbio_write_cb, (ref, res, err),
	nbio_ref_t ref AND
	int res AND
	int err
)
{
	struct ip_table *tabp;
	int fd, r;

	DPRINT(2, (stderr, "nbio_write_cb(%d,%d,%d)\n", ref.ref_int, res, err));

	fd= ref.ref_int;
	tabp= &nbio_ip_table[fd][ASIO_WRITE];

	if (res > 0)
		tabp->data_offset += res;
	else if (res == 0)
		abort();
	else
	{
		tabp->err= err;
		return;
	}
	tabp->inprogress= 0;

	while (tabp->data_offset < tabp->data_size)
	{
		r= write(fd, tabp->buf+tabp->data_offset,
			tabp->data_size-tabp->data_offset);
		if (r > 0)
		{
			DPRINT(0, (stderr, "nbio_write_cb: wrote %d bytes\n",
				r));
			tabp->data_offset += r;
			continue;
		}
		else if (r == -1 && errno == EINPROGRESS)
		{
			tabp->inprogress= 1;
			nbio_inprogress(fd, ASIO_WRITE, 0 /* read */,
				1 /* write */, 0 /* exception */);
		}
		else
		{
			DPRINT(1, (stderr, "nbio_write_cb: write failed: %s\n",
				strerror(errno)));
			tabp->err= errno;
		}
		break;
	}
}

/*
 * $PchId: nbio_rw.c,v 1.3 1996/02/23 07:13:33 philip Exp $
 */
