/*
 * Initialize an MSDOS diskette.  Read the boot sector, and switch to the
 * proper floppy disk device to match the format on the disk.  Sets a bunch
 * of global variables.  Returns 0 on success, or 1 on failure.
 */

#include "sysincludes.h"
#include "msdos.h"
#include "stream.h"
#include "mtools.h"
#include "fsP.h"
#include "plain_io.h"
#include "xdf_io.h"
#include "buffer.h"

extern int errno;


#if !__minix
#define FULL_CYL
#endif

unsigned int num_clus;			/* total number of cluster */

/*
 * Read the boot sector.  We glean the disk parameters from this sector.
 */
static int read_boot(Stream_t *Stream, struct bootsector * boot)
{	
	/* read the first sector, or part of it */
	if (force_read(Stream, (char *) boot, 0, sizeof(struct bootsector)) !=
	    sizeof(struct bootsector))
		return -1;
	return 0;
}

static int fs_flush(Stream_t *Stream)
{
	DeclareThis(Fs_t);

	fat_write(This);
	return 0;
}

Class_t FsClass = {
	read_pass_through, /* read */
	write_pass_through, /* write */
	fs_flush, 
	0, /* free */
	0, /* set geometry */
	get_data_pass_through
};

static int get_media_type(Stream_t *St, struct bootsector *boot)
{
	int media;

	media = boot->descr;
	if(media < 0xf0){
		char temp[512];
		/* old DOS disk. Media descriptor in the first FAT byte */
		/* old DOS disk always have 512-byte sectors */
		if (force_read(St,temp,512,512) == 512)
			media = (unsigned char) temp[0];
		else
			media = 0;
	} else
		media += 0x100;
	return media;
}



Stream_t *find_device(char *drive, int mode, struct device *out_dev,
		      struct bootsector *boot,
		      char *name, int *media)
{
	char errmsg[80];
	Stream_t *Stream;
	struct device *dev;
#if __minix
	static char *devname;
	struct device onedevice[2];
	struct stat stbuf;

	free(devname);
	devname = safe_malloc((9 + strlen(drive)) * sizeof(devname[0]));
	strcpy(devname, "/dev/dosX");
	if (isupper(drive[0]) && drive[1] == 0) {
		/* single letter device name, use /dev/dos$drive */
		devname[8]= drive[0];
	} else
	if (strchr(drive, '/') == NULL) {
		/* a simple name, use /dev/$drive */
		strcpy(devname+5, drive);
	} else {
		/* a pathname, use as is. */
		strcpy(devname, drive);
	}
	if (stat(devname, &stbuf) != -1) {
		memset(onedevice, 0, sizeof(onedevice));
		onedevice[0].name = devname;
		onedevice[0].drive = drive;
		onedevice[1].name = NULL;
		onedevice[1].drive = NULL;
		dev = onedevice;
	} else {
		dev = devices;
	}
#else
	dev = devices;
#endif

	Stream = NULL;
	sprintf(errmsg, "Drive '%s:' not supported", drive);	
					/* open the device */
	for (; dev->name; dev++) {
		FREE(&Stream);
		if (strcmp(dev->drive, drive))
			continue;
		*out_dev = *dev;
		expand(dev->name,name);

#ifdef USE_XDF
		Stream = XdfOpen(out_dev, name, mode, errmsg);
#else
		Stream = 0;
#endif

		if (!Stream){
			Stream = SimpleFloppyOpen(out_dev, dev, name, mode,
						  errmsg);
		} else
			out_dev->use_2m = 0x7f;

		if( !Stream)
			continue;

		/* read the boot sector */
		if (read_boot(Stream, boot)){
			sprintf(errmsg,"init %s: read boot", drive);
			continue;
		}

		if((*media= get_media_type(Stream, boot)) <= 0xf0 ){
			sprintf(errmsg,"init %s: unknown media type", drive);
			continue;
		}

		/* set new parameters, if needed */
		if(SET_GEOM(Stream, out_dev, dev, *media, boot)){
			sprintf(errmsg, "Can't set disk parameters for %s: %s", 
				drive, strerror(errno));
			continue;
		}
		break;
	}

	/* print error msg if needed */	
	if ( dev->drive == 0 ){
		FREE(&Stream);
		fprintf(stderr,"%s\n",errmsg);
		return NULL;
	}
	return Stream;
}


Stream_t *fs_init(char *drive, int mode)
{
	int media,i;
	int disk_size = 0;	/* In case we don't happen to set this below */
	unsigned long tot_sectors;
	char name[EXPAND_BUF];
	int cylinder_size;
	struct device dev;

	struct bootsector boot0;
#define boot (&boot0)
	Fs_t *This;

	This = New(Fs_t);
	if (!This)
		return NULL;

	This->Direct = NULL;
	This->Next = NULL;
	This->refs = 1;
	This->Buffer = 0;
	This->Class = &FsClass;
	This->drive = drive;

	This->Direct = find_device(drive, mode, &dev, &boot0, name, &media);
	if(!This->Direct)
		return NULL;
	
	This->sector_size = WORD(secsiz);
	if(This->sector_size > MAX_SECTOR){
		fprintf(stderr,"init %s: sector size too big\n", drive);
		return NULL;
	}

	cylinder_size = dev.heads * dev.sectors;
	if (!tot_sectors) tot_sectors = dev.tracks * cylinder_size;

	if ((media & ~7) == 0xf8){
		i = media & 3;
		This->cluster_size = old_dos[i].cluster_size;
		tot_sectors = cylinder_size * old_dos[i].tracks;
		This->fat_start = 1;
		This->fat_len = old_dos[i].fat_len;
		This->dir_len = old_dos[i].dir_len;
		This->num_fat = 2;
		This->sector_size = 512;
		This->serialized = 0;
		This->fat_bits = 12;
	} else {
		/*
		 * all numbers are in sectors, except num_clus 
		 * (which is in clusters)
		 */
		tot_sectors = DWORD(bigsect);
		set_ulong(&tot_sectors, WORD(psect));
		This->cluster_size = boot0.clsiz; 		
		This->fat_start = WORD(nrsvsect);
		This->fat_len = WORD(fatlen);
		This->dir_len = WORD(dirents) * MDIR_SIZE / This->sector_size;
		This->num_fat = boot0.nfat;
		This->serialized = boot0.dos4 == 0x29;
		This->serial_number = DWORD(serial);
	}	
	This->dir_start = This->fat_start + This->num_fat * This->fat_len;

	if (!tot_sectors ){
		/* this should only happen on hard disks, so let's
		 * assume a 16 bit fat for now */
		tot_sectors = This->dir_start + This->dir_len + 
			This->fat_len * This->sector_size / 
				2 * This->cluster_size;
		tot_sectors -= (tot_sectors + DWORD(nhs)) % cylinder_size;
		fprintf(stderr,
			"Warning: disk size unknown, assuming %d Megabytes\n", 
			(int) (tot_sectors * This->sector_size / 1024 / 1024));
	}

	This->num_clus = (tot_sectors - This->dir_start - This->dir_len) /
		This->cluster_size;

#if !__minix	/* Strange check, MS-DOS isn't that picky. */
	if(!mtools_skip_check && (tot_sectors % dev.sectors)){
		fprintf(stderr,
			"Total number of sectors not a multiple of"
			" sectors per track!\n");
		cleanup_and_exit(1);
	}
#endif

	/* full cylinder buffering */
#ifdef FULL_CYL
	disk_size = (dev.tracks) ? cylinder_size : 512;
#else /* FULL_CYL */
	disk_size = (dev.tracks) ? dev.sectors : 512;
#endif /* FULL_CYL */

#if (defined sysv4 && !defined(solaris))
	/*
	 * The driver in Dell's SVR4 v2.01 is unreliable with large writes.
	 */
        disk_size = 0;
#endif /* (defined sysv4 && !defined(solaris)) */

#ifdef linux
	disk_size = cylinder_size;
#endif
	if (disk_size % 2)
		disk_size *= 2;

	if (disk_size)
		This->Next = buf_init(This->Direct,
				      disk_size * This->sector_size,
				      This->sector_size,
				      disk_size * This->sector_size);
	else
		This->Next = This->Direct;

	if (This->Next == NULL) {
		perror("init: allocate buffer");
		This->Next = This->Direct;
	}

	if (dev.use_2m & 0x7f)
		This->num_fat = 1;
	/* read the FAT sectors */
	if(fat_read(This, &boot0, dev.fat_bits)){
		FREE(&This->Next);
		Free(This->Next);
		return NULL;
	}
	return (Stream_t *) This;
}

/*
 * Get the amount of free space on the diskette
 */

long getfree(Stream_t *Stream)
{
	DeclareThis(Fs_t);
	register unsigned int i;
	long total;

	total = 0L;
	for (i = 2; i < This->num_clus + 2; i++) {
		/* if fat_decode returns zero */
		if (!This->fat_decode(This,i))
			total += This->cluster_size;
	}
	return(total * This->sector_size);
}
