/* cron - clock daemon.					Author: S.R. Sampson */
/* atrun - perform the work 'at' has squirreled away	Author: Jan Looyen */


/*
 * cron		clock daemon.
 *		Cron is the clock daemon.  It is typically started up from
 *		the system initialization file (/etc/rc or /etc/inittab) by
 *		the INIT program.
 *
 *		The superuser may start it by hand to test it, it will not
 *		ignore signals nor background itself to avoid surprises.
 *
 *		Cron runs all day, spending most of its time asleep.  Once
 *		a minute it wakes up and forks off a child which then
 *		examines /usr/lib/crontab to see if there are any commands
 *		to be executed.  The format of this table is the same as in
 *		UNIX, except that % is not allowed to indicate 'new line'.
 *
 * Usage:	/usr/sbin/cron
 *
 * Datafile:	/usr/lib/crontab
 *		Each crontab entry has six fields:
 *
 *		 minute hour day-of-month month day-of-week command
 *
 *		Each entry is checked in turn, and any entry matching the
 *		current time is executed.  The entry * matches anything.
 *
 * Version:	1.9	03/02/91
 *
 * Authors:	Steven R. Sampson, Fred van Kempen, Simmule Turner,
 *		Peter de Vrijer
 *
 * Changed 19 Mar 1994 by Kees J. Bot
 *		Simplyfied it a bit.
 *
 * Changed 16 Apr 1994 by Kees J. Bot
 *		Support for variable timeouts for commands scanning queues.
 *		Mail to root if a command produces output.  Atrun included.
 *		(So much for the simplification.)
 */
#define _POSIX_SOURCE	1
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <dirent.h>
#include <pwd.h>


char CRONTAB[] =	"/usr/lib/crontab";
char MAILER[] =		"/usr/lib/sendmail";
char MAILOPTS[] =	"-oi";
char ADMIN[] =		"root";
enum who { CRON, ATRUN };

#define MINUTE		60L
#define HOUR		(60 * MINUTE)
#define NEVER		((time_t) ((time_t) -1 > 0 ? ULONG_MAX : LONG_MAX))
#define SEPARATOR	" \t"
#define CRONSIZE	2048

typedef struct cron_entry {
  struct cron_entry	*next;
  char			*mn;
  char			*hr;
  char			*day;
  char			*mon;
  char			*wkd;
  char			*cmd;
  pid_t			pid;		/* pid of last started process */
  int			special;	/* uses special exit codes */
  time_t		started;	/* when started */
  time_t		again;		/* do not call before this time */
} cron_entry;


static char crontab[CRONSIZE];		/* memory for the entries	*/
static cron_entry *head, *entry_ptr;		/* crontab entry pointers	*/
static char any_min[3];			/* A "don't care" minute	*/
static struct tm *tm;
static time_t cur_time;
static time_t atrunagain;		/* Next time to look for AT jobs */
static int talking= 0;			/* Running a mailer? */
static int debug;			/* Debug mode? */

_PROTOTYPE(int main, (int argc, char **argv));
_PROTOTYPE(void assign, (cron_entry *entry, char *line));
_PROTOTYPE(void load_crontab, (void));
_PROTOTYPE(int match, (char *left, int right));
_PROTOTYPE(void wakeup, (void));
_PROTOTYPE(void catch, (int sig));
_PROTOTYPE(void open_mail, (char *receipient, char *subject, char *object));
_PROTOTYPE(void close_mail, (void));
_PROTOTYPE(void run_command, (char *user, enum who who, char *command));
_PROTOTYPE(time_t atrun, (void));


void open_mail(receipient, subject, object)
char *receipient;
char *subject;
char *object;
{
  /* Redirect standard output to a mail command. */
  int mfd[2];
  pid_t pid;

  talking= 1;

  if (pipe(mfd) < 0) {
	fprintf(stderr, "cron: pipe(): %s\n", strerror(errno));
	return;
  }
  if ((pid = fork()) < 0) {
	fprintf(stderr, "cron: fork(): %s\n", strerror(errno));
	close(mfd[0]);
	close(mfd[1]);
	return;
  }
  if (pid == 0) {
	dup2(mfd[0], 0);
	close(mfd[0]);
	close(mfd[1]);
	execl(MAILER, MAILER, MAILOPTS, receipient, (char *) NULL);
	fprintf(stderr, "cron: %s: %s\n", MAILER, strerror(errno));
	exit(1);
  }
  signal(SIGPIPE, SIG_IGN);
  dup2(mfd[1], 1);
  close(mfd[0]);
  printf("Subject: ");
  printf(subject, object);
  printf("\n\n");
  fflush(stdout);
}


void close_mail()
{
  /* Done talking to the mailer. */
  if (talking) {
	fflush((FILE *) NULL);
	dup2(2, 1);
	talking= 0;
	signal(SIGPIPE, SIG_DFL);
  }
}


/* Assign the field values to the current crontab entry in core. */
void assign(entry, line)
cron_entry *entry;
char *line;
{
  entry->mn	= strtok(line, SEPARATOR);
  if (strcmp(entry->mn, "?") == 0) entry->mn = any_min;
  entry->hr	= strtok( (char *)NULL, SEPARATOR);
  entry->day	= strtok( (char *)NULL, SEPARATOR);
  entry->mon	= strtok( (char *)NULL, SEPARATOR);
  entry->wkd	= strtok( (char *)NULL, SEPARATOR);
  entry->cmd	= strchr(entry->wkd, '\0') + 1;
  while (strchr(SEPARATOR, entry->cmd[0]) != NULL) entry->cmd++;
  entry->pid	= -1;
  entry->special= strcmp(entry->mn, "*") == 0
		&& strcmp(entry->hr, "*") == 0
		&& strcmp(entry->day, "*") == 0
		&& strcmp(entry->mon, "*") == 0
		&& strcmp(entry->wkd, "*") == 0;
  entry->again= 0;
}


/* Read the on-disk crontab file into core. */
void load_crontab()
{
  int len, pos;
  FILE *cfp;
  struct stat st;
  static struct stat oldst;

  entry_ptr = head;

  if (stat(CRONTAB, &st) < 0) {
	char *err= strerror(errno);

	st.st_ino = 0;
	if (errno != ENOENT && st.st_ino != oldst.st_ino) {
		/* Something bad happened to the crontab file. */
		fprintf(stderr, "cron: %s: %s\n", CRONTAB, err);
		open_mail(ADMIN, "Cron error", "");
		printf("Can't stat %s: %s\n", CRONTAB, err);
		close_mail();
	}
	oldst= st;
	goto foul;
  }

  if (st.st_atime == oldst.st_atime && st.st_mtime == oldst.st_mtime
				&& st.st_ctime == oldst.st_ctime) {
	/* No change in crontab. */
	return;
  }
  oldst= st;

  if ((cfp = fopen(CRONTAB, "r")) == NULL) {
	char *err= strerror(errno);

	fprintf(stderr, "cron: %s: %s\n", CRONTAB, err);
	open_mail(ADMIN, "Cron error", "");
	printf("Can't open %s: %s\n", CRONTAB, err);
	close_mail();
	goto foul;
  }

  len = pos = 0;

  while (fgets(&crontab[pos], CRONSIZE - pos, cfp) != NULL) {
	if (crontab[pos] == '#' || crontab[pos] == '\n') continue;
	len = strlen(&crontab[pos]);
	if (crontab[pos + len - 1] == '\n') {
		len--;
		crontab[pos + len] = '\0';
	}

	assign(entry_ptr, &crontab[pos]);

	if (entry_ptr->next == NULL) {
		entry_ptr->next = (cron_entry *)malloc(sizeof(cron_entry));
		entry_ptr->next->next = NULL;
	}
	entry_ptr = entry_ptr->next;
	pos += ++len;
	if (pos >= CRONSIZE) break;
  }
  (void) fclose(cfp);

foul:
  while (entry_ptr) {
	entry_ptr->mn = NULL;
	entry_ptr = entry_ptr->next;
  }
  (void) stat(CRONTAB, &oldst);
  atrunagain = 0;
  return;
}


/*
 * This routine will match the left string with the right number.
 *
 * The string can contain the following syntax *
 *	*		This will return 1 for any number
 *	x,y [,z, ...]	This will return 1 for any number given.
 *	x-y		This will return 1 for any number within
 *			the range of x thru y.
 */
int match(left, right)
register char *left;
register int right;
{
  register int n;
  register char c;

  n = 0;
  if (!strcmp(left, "*")) return(1);

  while ((c = *left++) && (c >= '0') && (c <= '9')) n = (n * 10) + c - '0';

  switch (c) {
	case '\0':
		return(right == n);
		/*NOTREACHED*/
		break;
	case ',':
		if (right == n) return(1);
		do {
			n = 0;
			while ((c = *left++) && (c >= '0') && (c <= '9'))
				n = (n * 10) + c - '0';

			if (right == n) return(1);
		} while (c == ',');
		return(0);
		/*NOTREACHED*/
		break;
	case '-':
		if (right < n) return(0);
		n = 0;
		while ((c = *left++) && (c >= '0') && (c <= '9'))
			n = (n * 10) + c - '0';
		return(right <= n);
		/*NOTREACHED*/
		break;
	default:
		break;
  }
  return(0);
}

void run_command(user, who, command)
char *user;
enum who who;
char *command;
{
  /* Run a command.  Mail its output if it generates any. */
  int cmdout[2];
  char buf[1024];
  int n;
  pid_t pid;

  if (debug) {
	fprintf(stderr, "%.24s  %s\n", asctime(tm), command);
	fflush(stderr);
  }

  if (pipe(cmdout) < 0) {
	fprintf(stderr, "cron: pipe(): %s\n", strerror(errno));
	exit(220);
  }
  if ((pid = fork()) < 0) {
	fprintf(stderr, "cron: fork(): %s\n", strerror(errno));
	exit(170);
  }

  if (pid != 0) {
	/* Let the parent run the command. */
	dup2(cmdout[1], 1);
	dup2(cmdout[1], 2);
	close(cmdout[0]);
	close(cmdout[1]);
	if (who == CRON) {
		execl("/bin/sh", "sh", "-c", command, (char *)NULL);
	} else {
		execl("/bin/sh", "sh", command, (char *)NULL);
	}
	fprintf(stderr, "cron: exec %s failed: %s\n", command, strerror(errno));
	exit(244);
  }
  close(cmdout[1]);

  /* Child reads output and mails it to someone who cares. */
  while ((n= read(cmdout[0], buf, sizeof(buf))) > 0) {
	if (who == CRON) {
		open_mail(user, "Output from cron job '%s'", command);
	} else {
		open_mail(user, "Output from at job '%s'", command);
	}
	fwrite(buf, sizeof(buf[0]), n, stdout);
  }
  exit(0);
}

/* Wakeup from the sleep(), and check for any work. */
void wakeup()
{
  cron_entry *this_entry;
  pid_t pid;

  for (this_entry = head; this_entry->next && this_entry->mn;
					this_entry = this_entry->next) {
	if (cur_time >= this_entry->again &&
	    match(this_entry->mn, tm->tm_min) &&
	    match(this_entry->hr, tm->tm_hour) &&
	    match(this_entry->day, tm->tm_mday) &&
	    match(this_entry->mon, tm->tm_mon + 1) &&
	    match(this_entry->wkd, tm->tm_wday)) {

		if (this_entry->special) {
			/* Special commands must be tested for existence or
			 * madness will result.
			 */
			struct stat dummy;
			char *p;
			int r;

			if ((p= strchr(this_entry->cmd, ' ')) != NULL) *p= 0;
			r= stat(this_entry->cmd, &dummy);
			if (p != NULL) *p= ' ';
			if (r < 0) {
				if (errno != ENOENT) {
					fprintf(stderr, "cron: %s: %s\n",
						this_entry->cmd,
						strerror(errno));
				}
				this_entry->again= cur_time + 10 * MINUTE;
				continue;
			}
		}

		fflush((FILE *) NULL);

		if ((pid = fork()) == 0)
			run_command(ADMIN, CRON, this_entry->cmd);

		if (this_entry->special) {
			this_entry->pid= pid;
			this_entry->started= cur_time;
			this_entry->again= cur_time + 10 * MINUTE;
		}
	}
  }
}


int expired;

void catch(sig)
int sig;
{
	signal(sig, catch);
	alarm(1);
	expired = 1;
}


int main(argc, argv)
int argc;
char *argv[];
{
  int status, r;
  int ticks;

  (void) chdir("/");

  /* A '?' in the minute field means: don't care when in this hour.  It
   * is not random though, it is just to keep different machines out of
   * each others hair.
   */
  sprintf(any_min, "%d", (int) (time((time_t *) NULL) % MINUTE));

  if (argc == 2 && strcmp(argv[1], "-d") == 0) {
	debug= 1;
  } else
  if (argc > 1) {
	fprintf(stderr, "Usage: cron [-d]\n");
	exit(1);
  }

  /* Wait one minute before doing anything. */
  if (!debug) sleep(60);

  time(&cur_time);
  if (debug) fprintf(stderr, "%.24s  Cron started\n", ctime(&cur_time));

  entry_ptr = (cron_entry *)malloc(sizeof(cron_entry));
  entry_ptr->next = NULL;
  head = entry_ptr;

  /* The endless loop. */
  while (1) {
	/* Sleep till the next full minute */
	expired = 0;
	signal(SIGALRM, catch);
	alarm((unsigned) (MINUTE - cur_time % MINUTE));

	do {
		if ((r = wait(&status)) == -1) {
			if (errno == ECHILD) pause();
			continue;
		}
		if (!WIFEXITED(status)) continue;

		ticks = WEXITSTATUS(status);

		for (entry_ptr = head; entry_ptr->next && entry_ptr->mn;
						entry_ptr = entry_ptr->next) {
			if (entry_ptr->special && entry_ptr->pid == r) {
				/* A special command exited. */
				time_t started, again;

				started= entry_ptr->started;
				if (160 <= ticks && ticks <= 219) {
					/* 0-59 minutes */
					again = started + (ticks-160) * MINUTE;
				} else
				if (220 <= ticks && ticks <= 243) {
					/* 1-24 hours. */
					again = started + (ticks-220+1) * HOUR;
				} else
				if (ticks == 244) {
					/* don't call me again. */
					again = NEVER;
				} else {
					/* any other exit code means 10 min. */
					again = started + 10 * MINUTE;
				}
				entry_ptr->again = again;
				if (debug) {
					fprintf(stderr,
						"%.24s  Next run of %s\n",
						ctime(&entry_ptr->again),
						entry_ptr->cmd);
				}
			}
		}
	} while (!expired);
	alarm(0);

	time(&cur_time);
	tm = localtime(&cur_time);
	load_crontab();
	wakeup();

	/* Atrun is done by cron, makes many things easier. */
	if (atrunagain <= cur_time) {
		if (debug) fprintf(stderr, "%.24s  AT job scan\n", asctime(tm));

		atrunagain = atrun();

		if (debug) fprintf(stderr, "%.24s  Next AT job scan\n",
							ctime(&atrunagain));
	}
  }
}


/* The code of atrun has been incorporated into cron to make it easier to
 * send the right person mail and other things.
 */

/*-------------------------------------------------------------------------*
 *	atrun scans directory /usr/spool/at for 'at' jobs to be executed.  *
 *	Finished jobs have been moved to directory /usr/spool/at/past.     *
 *-------------------------------------------------------------------------*/

#define MAX_DELAY	(24 * HOUR)	/* Maximum delay. */

time_t atrun()
{
  char realtime[15], thentime[15], procname[35], procpast[35];
  DIR *dir;
  struct dirent *entry;
  struct tm *p;
  struct stat sbuf;
  time_t clk, future;
  time_t delay = MAX_DELAY;
  int anything = 0;
  struct passwd *pw;

/*-------------------------------------------------------------------------*
 *	Compute real time,  move 'at' jobs whose filenames < real time to  *
 *	/usr/spool/at/past and start a sh for each job.			   *
 *-------------------------------------------------------------------------*/
  clk = cur_time;
  p = localtime(&clk);
  sprintf(realtime, "%02d.%03d.%02d%02d.00",
	p->tm_year % 100, p->tm_yday, p->tm_hour, p->tm_min);
  future = clk + delay;
  p = localtime(&future);
  sprintf(thentime, "%02d.%03d.%02d%02d.00",
	p->tm_year % 100, p->tm_yday, p->tm_hour, p->tm_min);

  if ((dir = opendir("/usr/spool/at")) != NULL) {
	while ((entry = readdir(dir)) != NULL) {
		if (entry->d_ino == 0
			|| entry->d_name[0] == '.'
			|| entry->d_name[0] == 'p') continue;

		if (strncmp(entry->d_name, realtime, 11) <= 0) {

			sprintf(procname, "/usr/spool/at/%s", entry->d_name);
			sprintf(procpast, "/usr/spool/at/past/%s", entry->d_name);

			fflush((FILE *) NULL);

			if (fork() == 0) {	/* code for child */
				if (rename(procname, procpast) == 0) {
					stat(procpast, &sbuf);
					setgid(sbuf.st_gid);
					setuid(sbuf.st_uid);
					pw= getpwuid(sbuf.st_uid);
					run_command(pw ? pw->pw_name : ADMIN,
						ATRUN,
						procpast);
				}
			}
		} else {
			anything = 1;
			while (strncmp(entry->d_name, thentime, 11) < 0) {
				/* Entry will be run soon, but when? */
				delay -= delay <= HOUR ? MINUTE : HOUR;
				future = clk + delay;
				p = localtime(&future);
				sprintf(thentime,
					"%02d.%03d.%02d%02d.00",
					p->tm_year % 100, p->tm_yday,
					p->tm_hour, p->tm_min);
			}
		}
	}
	(void) closedir(dir);
  }
  return anything ? future : NEVER;
}
