/*
 * Copyright (c) 1983, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)syslog.c	8.4 (Berkeley) 3/18/94";
#endif /* LIBC_SCCS and not lint */

#ifdef __minix
#define MINIX
#include <minix/stubs.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include <sys/uio.h>
#include <netdb.h>

#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

static int	LogFile = -1;		/* fd for log */
static int	connected;		/* have done connect */
static int	LogStat = 0;		/* status bits, set by openlog() */
static const char *LogTag = NULL;	/* string to tag the entry with */
static int	LogFacility = LOG_USER;	/* default facility code */
static int	LogMask = 0xff;		/* mask of priorities to be logged */
extern char	*__progname;		/* Program name, from crt0. */
#ifdef MINIX
static char	*LogDevice = NULL;	/* Which device */
#endif

#ifdef MINIX
static void	decimal __P((char *buf, unsigned n));
#endif

/*
 * syslog, vsyslog --
 *	print message on log file; output is intended for syslogd(8).
 */
void
#if __STDC__
syslog(int pri, const char *fmt, ...)
#else
syslog(pri, fmt, va_alist)
	int pri;
	char *fmt;
	va_dcl
#endif
{
	va_list ap;

#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	vsyslog(pri, fmt, ap);
	va_end(ap);
}

void
vsyslog(pri, fmt, ap)
	int pri;
	register const char *fmt;
	va_list ap;
{
	register int cnt;
	register char ch, *p, *t;
	time_t now;
	int fd, saved_errno;
	char *stdp, tbuf[2048], fmt_cpy[1024];
#ifdef MINIX
	struct tm *tm;
	static char month[12][4] =  {
		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
	};
	int i;
#endif

#define	INTERNALLOG	LOG_ERR|LOG_CONS|LOG_PERROR|LOG_PID
	/* Check for invalid bits. */
	if (pri & ~(LOG_PRIMASK|LOG_FACMASK)) {
		syslog(INTERNALLOG,
		    "syslog: unknown facility/priority: %x", pri);
		pri &= LOG_PRIMASK|LOG_FACMASK;
	}

	/* Check priority against setlogmask values. */
	if (!(LOG_MASK(LOG_PRI(pri)) & LogMask))
		return;

	saved_errno = errno;

	/* Set default facility if none specified. */
	if ((pri & LOG_FACMASK) == 0)
		pri |= LogFacility;

	/* Build the message. */
	(void)time(&now);
	p = tbuf + sprintf(tbuf, "<%d>", pri);
#ifndef MINIX
	p += strftime(p, sizeof (tbuf) - (p - tbuf), "%h %e %T ",
	    localtime(&now));
#else
	tm = localtime(&now);
	strcpy(p, month[tm->tm_mon]);
	p[3] = ' ';
	p[4] = ' ';
	decimal(p+6, tm->tm_mday);
	p[7] = '0';
	decimal(p+9, tm->tm_hour);
	p[9] = ':';
	p[10] = '0';
	decimal(p+12, tm->tm_min);
	p[12] = ':';
	p[13] = '0';
	decimal(p+15, tm->tm_sec);
	p += 16;
	*p = '\0';
#endif

	if (LogStat & LOG_PERROR)
		stdp = p;
	if (LogTag == NULL)
		LogTag = __progname;
	if (LogTag != NULL)
		p += sprintf(p, "%s", LogTag);
	if (LogStat & LOG_PID)
		p += sprintf(p, "[%d]", getpid());
	if (LogTag != NULL) {
		*p++ = ':';
		*p++ = ' ';
	}

	/* Substitute error message for %m. */
	for (t = fmt_cpy; ch = *fmt; ++fmt)
		if (ch == '%' && fmt[1] == 'm') {
			++fmt;
			t += sprintf(t, "%s", strerror(saved_errno));
		} else
			*t++ = ch;
	*t = '\0';

	p += vsprintf(p, fmt_cpy, ap);
#ifdef MINIX
	if (p[-1] != '\n')
	{
		*p++= '\n';
		*p= '\0';
	}
#endif
	cnt = p - tbuf;

	/* Output to stderr if requested. */
	if (LogStat & LOG_PERROR) {
#ifndef MINIX
		struct iovec iov[2];
		register struct iovec *v = iov;

		v->iov_base = stdp;
		v->iov_len = cnt - (stdp - tbuf);
		++v;
		v->iov_base = "\n";
		v->iov_len = 1;
		(void)writev(STDERR_FILENO, iov, 2);
#else
		write(STDERR_FILENO, stdp, cnt - (stdp-tbuf));
#endif
	}

	/* Get connected, output the message to the local logger. */
	if (!connected)
		openlog(LogTag, LogStat | LOG_NDELAY, 0);
#ifndef MINIX
	if (send(LogFile, tbuf, cnt, 0) >= 0)
		return;
#else
	for (i= 0; i<3; i++)
	{
		switch (i)
		{
		case 0:	break;
		case 1: LogDevice= _PATH_DAEMONLOG; break;
		case 2: LogDevice= _PATH_LOG; break;
		}
		if (i !=0)
		{
			if (LogFile != -1)
			{
				close(LogFile);
				LogFile= -1;
			}
			connected= 0;
			openlog(LogTag, LogStat | LOG_NDELAY, 0);
			if (LogFile == -1)
				continue;
		}
		if (write(LogFile, tbuf, cnt) >= 0)
		{
			if (i == 2)
			{
				LogDevice= NULL;
				connected= 0;
				close(LogFile);
				LogFile= -1;
			}
			return;
		}
	}
#endif

	/*
	 * Output the message to the console; don't worry about blocking,
	 * if console blocks everything will.  Make sure the error reported
	 * is the one from the syslogd failure.
	 */
	if (LogStat & LOG_CONS &&
	    (fd = open(_PATH_CONSOLE, O_WRONLY, 0)) >= 0) {
		(void)strcat(tbuf, "\r\n");
		cnt += 2;
		p = index(tbuf, '>') + 1;
		(void)write(fd, p, cnt - (p - tbuf));
		(void)close(fd);
	}
}

#ifdef MINIX
static void
decimal(buf, n)
	char *buf;
	unsigned n;
{
	*buf = ' ';
	do {
		*--buf = '0' + (n % 10);
	} while ((n /= 10) != 0);
}
#endif

#ifndef MINIX
static struct sockaddr SyslogAddr;	/* AF_UNIX address of local logger */
#endif

void
openlog(ident, logstat, logfac)
	const char *ident;
	int logstat, logfac;
{
	if (ident != NULL)
		LogTag = ident;
	LogStat = logstat;
	if (logfac != 0 && (logfac &~ LOG_FACMASK) == 0)
		LogFacility = logfac;

	if (LogFile == -1) {
#ifndef MINIX
		SyslogAddr.sa_family = AF_UNIX;
		(void)strncpy(SyslogAddr.sa_data, _PATH_LOG,
		    sizeof(SyslogAddr.sa_data));
		if (LogStat & LOG_NDELAY) {
			if ((LogFile = socket(AF_UNIX, SOCK_DGRAM, 0)) == -1)
				return;
			(void)fcntl(LogFile, F_SETFD, 1);
		}
#else
		if (LogDevice == NULL)
			LogDevice= _PATH_DAEMONLOG;
		LogFile= open(LogDevice, O_WRONLY | O_NONBLOCK);
		if (LogFile == -1)
		{
			return;
		}
		(void) fcntl(LogFile, F_SETFD,
					fcntl(LogFile, F_GETFD) | FD_CLOEXEC);
		(void) fcntl(LogFile, F_SETFL,
				fcntl(LogFile, F_GETFL) & ~O_NONBLOCK);
#endif
	}
	if (LogFile != -1 && !connected)
#ifndef MINIX
		if (connect(LogFile, &SyslogAddr, sizeof(SyslogAddr)) == -1) {
			(void)close(LogFile);
			LogFile = -1;
		} else
#endif
			connected = 1;
}

void
closelog()
{
	(void)close(LogFile);
	LogFile = -1;
	connected = 0;
}

/* setlogmask -- set the log mask level */
int
setlogmask(pmask)
	int pmask;
{
	int omask;

	omask = LogMask;
	if (pmask != 0)
		LogMask = pmask;
	return (omask);
}

/*
 * $PchId: syslog.c,v 1.3 1995/11/20 19:27:17 philip Exp $
 */
