/* The <fcntl.h> header is needed by the open() and fcntl() system calls,
 * which  have a variety of parameters and flags.  They are described here.  
 * The formats of the calls to each of these are:
 *
 *	open(path, oflag [,mode])	open a file
 *	fcntl(fd, cmd [,arg])		get or set file attributes
 * 
 */

#ifndef _FCNTL_H
#define _FCNTL_H

#include <minix/ansi.h>
#include <minix/posix.h>
#include <minix/types.h>
_C_BEGIN

/* These values are used for cmd in fcntl().  POSIX Table 6-1.  */
#define F_DUPFD            0	/* duplicate file descriptor */
#define F_GETFD	           1	/* get file descriptor flags */
#define F_SETFD            2	/* set file descriptor flags */
#define F_GETFL            3	/* get file status flags */
#define F_SETFL            4	/* set file status flags */
#define F_GETLK            5	/* get record locking information */
#define F_SETLK            6	/* set record locking information */
#define F_SETLKW           7	/* set record locking info; wait if blocked */

#ifdef _MINIX_SOURCE
#define F_FREESP          16	/* free storage space of a section of a file */
#define F_SCTTY		  17	/* set controlling tty */
#define F_REVOKE	  18	/* revoke access to a device */
#define F_SEEK            19	/* seek to a 64 bit position */
#endif

/* File descriptor flags used for fcntl().  POSIX Table 6-2. */
#define FD_CLOEXEC         1	/* close on exec flag for third arg of fcntl */

#ifdef _MINIX_SOURCE		/* Extra MINIX specific flags */
#define FD_ASYNCHIO	   2	/* enable asynchronous I/O on fd */
#endif /* _MINIX_SOURCE */

/* L_type values for record locking with fcntl().  POSIX Table 6-3. */
#define F_RDLCK            1	/* shared or read lock */
#define F_WRLCK            2	/* exclusive or write lock */
#define F_UNLCK            3	/* unlock */

/* Oflag values for open().  POSIX Table 6-4. */
#define O_CREAT        00100	/* creat file if it doesn't exist */
#define O_EXCL         00200	/* exclusive use flag */
#define O_NOCTTY       00400	/* do not assign a controlling terminal */
#define O_TRUNC        01000	/* truncate flag */

/* File status flags for open() and fcntl().  POSIX Table 6-5. */
#define O_APPEND       02000	/* set append mode */
#define O_NONBLOCK     04000	/* no delay */

/* File access modes for open() and fcntl().  POSIX Table 6-6. */
#define O_RDONLY           0	/* open(name, O_RDONLY) opens read only */
#define O_WRONLY           1	/* open(name, O_WRONLY) opens write only */
#define O_RDWR             2	/* open(name, O_RDWR) opens read/write */

/* Mask for use with file access modes.  POSIX Table 6-7. */
#define O_ACCMODE         03	/* mask for file access modes */

/* Struct used for locking.  POSIX Table 6-8. */
struct flock {
  short l_type;			/* type: F_RDLCK, F_WRLCK, or F_UNLCK */
  short l_whence;		/* flag for starting offset */
  _mnx_off_t l_start;		/* relative offset in bytes */
  _mnx_off_t l_len;		/* size; if 0, then until EOF */
  _mnx_pid_t l_pid;		/* process id of the locks' owner */
};


/* Function Prototypes. */
_PROTOTYPE( int creat, (const char *_path, _mnx_Mode_t _mode)		);
_PROTOTYPE( int fcntl, (int _filedes, int _cmd, ...)	  		);
_PROTOTYPE( int open,  (const char *_path, int _oflag, ...) 		);

#ifdef _MINIX_SOURCE	/* Extra MINIX specific calls */

#define ASIO_NOOP	(-1)	/* code for no or unknown operation */
#define ASIO_READ	0	/* code for a read operation */
#define ASIO_WRITE	1	/* code for a write operation */
#define ASIO_IOCTL	2	/* code for an ioctl */
#define ASIO_NR		3	/* number of operations */

typedef struct fwait
{
	int fw_flags;
	unsigned char (*fw_bits)[4];
	int fw_maxfd;

	int fw_result;
	int fw_errno;
	int fw_fd;
	int fw_operation;
} fwait_t;

#define FWF_NONBLOCK	1
#define FWF_MORE	2

_PROTOTYPE( int fcancel, (int _filedes, int _operation)			);
_PROTOTYPE( int fwait, (struct fwait *_fwp)				);

#ifndef ASIO_FD_SETSIZE
#define ASIO_FD_SETSIZE	64
#endif

typedef unsigned char asio_fd_mask;

#ifndef NBBY
#define NBBY	8
#endif

#define ASIO_NFDBITS	(sizeof(asio_fd_mask) * NBBY)

#ifndef howmany
#define howmany(x, y)	(((x)+((y)-1))/(y))
#endif

typedef struct asio_fd_set
{
	asio_fd_mask	afds_bits[howmany(ASIO_FD_SETSIZE, ASIO_NFDBITS)][4];
} asio_fd_set_t;

#define ASIO_FD_SET(n, o, p) ((p)->afds_bits[(n)/ASIO_NFDBITS][o] |= \
						(1 << ((n) % ASIO_NFDBITS)))
#define ASIO_FD_CLR(n, o, p) ((p)->afds_bits[(n)/ASIO_NFDBITS][o] &= \
						~(1 << ((n) % ASIO_NFDBITS)))
#define ASIO_FD_ISSET(n, o, p) ((p)->afds_bits[(n)/ASIO_NFDBITS][o] & \
						(1 << ((n) % ASIO_NFDBITS)))
#define ASIO_FD_ZERO(p)	memset((char *)(p), '\0', sizeof(*(p)))

#endif /* _MINIX_SOURCE */

_C_END
#endif /* _FCNTL_H */

/*
 * $PchId: fcntl.h,v 1.4 1995/11/18 07:29:06 philip Exp $
 */
