/*
 * C compiler
 * ==========
 *
 * Copyright 1989, 1990, 1991 Christoph van Wuellen.
 * Credits to Matthew Brandt.
 * All commercial rights reserved.
 *
 * This compiler may be redistributed as long there is no
 * commercial interest. The compiler must not be redistributed
 * without its full sources. This notice must stay intact.
 *
 * History:
 *
 * 1989   starting an 68000 C compiler, starting with material
 *        originally by M. Brandt
 * 1990   68000 C compiler further bug fixes
 *        started i386 port (December)
 * 1991   i386 port finished (January)
 *        further corrections in the front end and in the 68000
 *        code generator.
 *        The next port will be a SPARC port
 */

#include	"c.h"
#include	"expr.h"
#include	"gen.h"
#include	"cglbdec.h"

static          void fold_const();

dooper(node)
/*
 * dooper will execute a constant operation in a node and modify the node to
 * be the result of the operation.
 */
    struct enode  **node;
{
    struct enode   *ep = *node;
    enum e_node     type = ep->nodetype;

    ep->nodetype = en_icon;
    switch (type) {
      case en_add:
	ep->v.i = ep->v.p[0]->v.i + ep->v.p[1]->v.i;
	break;
      case en_sub:
	ep->v.i = ep->v.p[0]->v.i - ep->v.p[1]->v.i;
	break;
      case en_mul:
	ep->v.i = ep->v.p[0]->v.i * ep->v.p[1]->v.i;
	break;
      case en_div:
	if (ep->v.p[1]->v.i == 0) {
	    fprintf(stderr, "division by zero in constant node\n");
	    ep->nodetype = en_div;
	} else {
	    ep->v.i = ep->v.p[0]->v.i / ep->v.p[1]->v.i;
	}
	break;
      case en_mod:
	if (ep->v.p[1]->v.i == 0) {
	    fprintf(stderr, "mod by zero in constant node\n");
	    ep->nodetype = en_mod;
	} else {
	    ep->v.i = ep->v.p[0]->v.i % ep->v.p[1]->v.i;
	}
	break;
      case en_lsh:
	ep->v.i = ep->v.p[0]->v.i << ep->v.p[1]->v.i;
	break;
      case en_rsh:
	ep->v.i = ep->v.p[0]->v.i >> ep->v.p[1]->v.i;
	break;
      case en_and:
	ep->v.i = ep->v.p[0]->v.i & ep->v.p[1]->v.i;
	break;
      case en_or:
	ep->v.i = ep->v.p[0]->v.i | ep->v.p[1]->v.i;
	break;
      case en_xor:
	ep->v.i = ep->v.p[0]->v.i ^ ep->v.p[1]->v.i;
	break;
      case en_eq:
	ep->v.i = ep->v.p[0]->v.i == ep->v.p[1]->v.i;
	break;
      case en_ne:
	ep->v.i = ep->v.p[0]->v.i != ep->v.p[1]->v.i;
	break;
      case en_uminus:
	ep->v.i = -ep->v.p[0]->v.i;
	break;
      case en_compl:
	ep->v.i = ~ep->v.p[0]->v.i;
/*
 * This prevents, e.g. ~0x8000 to be 0xFFFF7FFF in 16-bit mode
 * (it should be 0x7FFF in this case
 */
        ep->v.i = strip_icon(ep->v.i, ep->etype);
	break;
      case en_not:
	ep->v.i = !ep->v.p[0]->v.i;
	break;
      default:
	fatal("DOOPER");
    }
}

int
pwrof2(i)
/*
 * return which power of two i is or -1.
 */
    long            i;
{
    int             p;
    long            q;
    q = 2;
    p = 1;
    while (q > 0) {
	if (q == i)
	    return p;
	q <<= 1l;
	++p;
    }
    return -1;
}

long
mod_mask(i)
/*
 * make a mod mask for a power of two.
 */
    long            i;
{
    long            m;
    m = 0;
    while (i--)
	m = (m << 1) | 1;
    return m;
}

opt0(node)
/*
 * opt0 - delete useless expressions and combine constants.
 *
 * opt0 will delete expressions such as x + 0, x - 0, x * 0, x * 1, 0 / x, x /
 * 1, x mod 0, etc from the tree pointed to by node and combine obvious
 * constant operations. It cannot combine name and label constants but will
 * combine icon type nodes.
 */
    struct enode  **node;
{
    struct enode   *ep;
    long            val, sc;
    enum e_node     typ;
    ep = *node;
    if (ep == 0)
	return;
    typ = ep->nodetype;
    switch (typ) {
      case en_ref:
      case en_fieldref:
      case en_ainc:
      case en_adec:
      case en_deref:
	opt0(&(ep->v.p[0]));
	break;
      case en_uminus:
      case en_not:
      case en_compl:
	opt0(&(ep->v.p[0]));
        if (ep->v.p[0]->nodetype == typ) {
            *node = ep->v.p[0]->v.p[0];
            break;
        }
	if (ep->v.p[0]->nodetype == en_icon)
	    dooper(node);
	break;
      case en_add:
      case en_sub:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
        if (ep->v.p[1]->nodetype == en_uminus) {
            ep->v.p[1] = ep->v.p[1]->v.p[0];
            ep->nodetype = typ = (typ == en_add) ? en_sub : en_add;
        }
        if (ep->v.p[0]->nodetype == en_uminus && typ == en_add) {
            ep->v.p[0] = ep->v.p[0]->v.p[0];
            swap_nodes(ep);
            ep->nodetype = typ = en_sub;
        }
	if (typ == en_add) {
	    if (ep->v.p[1]->nodetype == en_autocon)
		swap_nodes(ep);
	}
	if (ep->v.p[0]->nodetype == en_icon) {
	    if (ep->v.p[1]->nodetype == en_icon) {
		dooper(node);
		break;
	    }
	    if (ep->v.p[0]->v.i == 0) {
		if (typ == en_sub) {
		    ep->v.p[0] = ep->v.p[1];
		    ep->nodetype = typ = en_uminus;
		} else
		    *node = ep->v.p[1];
		break;
	    }
	} else if (ep->v.p[1]->nodetype == en_icon) {
	    if (ep->v.p[0]->nodetype == en_autocon && typ == en_add) {
		ep->v.p[0]->v.i += ep->v.p[1]->v.i;
		*node = ep->v.p[0];
		break;
	    }
	    if (ep->v.p[1]->v.i == 0) {
		*node = ep->v.p[0];
		break;
	    }
	}
	break;
      case en_mul:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	if (ep->v.p[0]->nodetype == en_icon) {
	    if (ep->v.p[1]->nodetype == en_icon) {
		dooper(node);
		break;
	    }
	    val = ep->v.p[0]->v.i;
	    if (val == 0) {
		*node = ep->v.p[0];
		break;
	    }
	    if (val == 1) {
		*node = ep->v.p[1];
		break;
	    }
	    sc = pwrof2(val);
	    if (sc != -1) {
		swap_nodes(ep);
		ep->v.p[1]->v.i = sc;
		ep->nodetype = en_lsh;
	    }
	} else if (ep->v.p[1]->nodetype == en_icon) {
	    val = ep->v.p[1]->v.i;
	    if (val == 0) {
		*node = ep->v.p[1];
		break;
	    }
	    if (val == 1) {
		*node = ep->v.p[0];
		break;
	    }
	    sc = pwrof2(val);
	    if (sc != -1) {
		ep->v.p[1]->v.i = sc;
		ep->nodetype = en_lsh;
	    }
	}
	break;
      case en_div:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	if (ep->v.p[0]->nodetype == en_icon) {
	    if (ep->v.p[1]->nodetype == en_icon) {
		dooper(node);
		break;
	    }
	    if (ep->v.p[0]->v.i == 0) {	/* 0/x */
		*node = ep->v.p[0];
		break;
	    }
	} else if (ep->v.p[1]->nodetype == en_icon) {
	    val = ep->v.p[1]->v.i;
	    if (val == 1) {	/* x/1 */
		*node = ep->v.p[0];
		break;
	    }
	    sc = pwrof2(val);
	    if (sc != -1) {
		ep->v.p[1]->v.i = sc;
		ep->nodetype = en_rsh;
	    }
	}
	break;
      case en_mod:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	if (ep->v.p[1]->nodetype == en_icon) {
	    if (ep->v.p[0]->nodetype == en_icon) {
		dooper(node);
		break;
	    }
	    sc = pwrof2(ep->v.p[1]->v.i);
	    if (sc != -1) {
		ep->v.p[1]->v.i = mod_mask(sc);
		ep->nodetype = en_and;
	    }
	}
	break;
      case en_and:
      case en_or:
      case en_xor:
      case en_eq:
      case en_ne:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	if (ep->v.p[0]->nodetype == en_icon &&
	    ep->v.p[1]->nodetype == en_icon)
	    dooper(node);
	break;
      case en_lsh:
      case en_rsh:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	if (ep->v.p[0]->nodetype == en_icon && ep->v.p[1]->nodetype == en_icon)
	    dooper(node);
	/* CVW: shift with zero shift count */
	else if (ep->v.p[1]->nodetype == en_icon && ep->v.p[1]->v.i == 0)
	    *node = ep->v.p[0];
	break;
      case en_cond:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
        if (ep->v.p[0]->nodetype == en_icon) {
	    if (ep->v.p[0]->v.i)
		*node=ep->v.p[1]->v.p[0];
	    else
		*node=ep->v.p[1]->v.p[1];
	}
	break;
      case en_land:
      case en_lor:
      case en_lt:
      case en_le:
      case en_gt:
      case en_ge:
      case en_asand:
      case en_asor:
      case en_asxor:
      case en_asadd:
      case en_assub:
      case en_asmul:
      case en_asdiv:
      case en_asmod:
      case en_asrsh:
      case en_aslsh:
      case en_fcall:
      case en_void:
      case en_assign:
	opt0(&(ep->v.p[0]));
	opt0(&(ep->v.p[1]));
	break;
	/* now handled in expr.c */
      case en_cast:
	opt0(&(ep->v.p[0]));
	if (ep->v.p[0]->nodetype == en_icon &&
	    ep->etype != bt_float &&
	    ep->etype != bt_double) {
	    ep->nodetype = en_icon;
	    ep->v.i = ep->v.p[0]->v.i;
	}
	break;

    }
}

static long
xfold(node)
/*
 * xfold will remove constant nodes and return the values to the calling
 * routines.
 */
    struct enode   *node;
{
    long            i;
    if (node == 0)
	return 0;
    switch (node->nodetype) {
      case en_icon:
	i = node->v.i;
	node->v.i = 0;
	return i;
      case en_add:
	return xfold(node->v.p[0]) + xfold(node->v.p[1]);
      case en_sub:
	return xfold(node->v.p[0]) - xfold(node->v.p[1]);
      case en_mul:
	if (node->v.p[0]->nodetype == en_icon)
	    return xfold(node->v.p[1]) * node->v.p[0]->v.i;
	else if (node->v.p[1]->nodetype == en_icon)
	    return xfold(node->v.p[0]) * node->v.p[1]->v.i;
	else {
	    fold_const(&node->v.p[0]);
	    fold_const(&node->v.p[1]);
	    return 0;
	}
	/*
	 * CVW: This seems wrong to me... case en_lsh: if(
	 * node->v.p[0]->nodetype == en_icon ) return xfold(node->v.p[1]) <<
	 * node->v.p[0]->v.i; else if( node->v.p[1]->nodetype == en_icon )
	 * return xfold(node->v.p[0]) << node->v.p[1]->v.i; else return 0;
	 */
      case en_uminus:
	return -xfold(node->v.p[0]);
      case en_lsh:
      case en_rsh:
      case en_div:
      case en_mod:
      case en_asadd:
      case en_assub:
      case en_asmul:
      case en_asdiv:
      case en_asmod:
      case en_and:
      case en_land:
      case en_or:
      case en_lor:
      case en_xor:
      case en_asand:
      case en_asor:
      case en_asxor:
      case en_void:
      case en_fcall:
      case en_assign:
      case en_eq:
      case en_ne:
      case en_lt:
      case en_le:
      case en_gt:
      case en_ge:
	fold_const(&node->v.p[0]);
	fold_const(&node->v.p[1]);
	return 0;
      case en_ref:
      case en_fieldref:
      case en_compl:
      case en_not:
      case en_deref:
	fold_const(&node->v.p[0]);
	return 0;
	/*
	 * This is not stricly legal: (long)(x+10) * 4l might not be the same
	 * as (long)(x) * 4l + 40l but it is the same as long as no overflows
	 * occur
	 */
      case en_cast:
#ifdef DONTDEF
	return xfold(node->v.p[0]);
#endif
	/*
	 * Well, sometimes I prefer purity to efficiency
	 * It is a matter of tast how you decide here....
	 */
	fold_const(&node->v.p[0]);
	return 0;
    }
    return 0;
}

static void
fold_const(node)
/*
 * reorganize an expression for optimal constant grouping.
 */
    struct enode  **node;
{
    struct enode   *ep;
    long            i;
    ep = *node;
    if (ep == 0)
	return;
    if (ep->nodetype == en_add) {
	if (ep->v.p[0]->nodetype == en_icon) {
	    ep->v.p[0]->v.i += xfold(ep->v.p[1]);
	    return;
	} else if (ep->v.p[1]->nodetype == en_icon) {
	    ep->v.p[1]->v.i += xfold(ep->v.p[0]);
	    return;
	}
    } else if (ep->nodetype == en_sub) {
	if (ep->v.p[0]->nodetype == en_icon) {
	    ep->v.p[0]->v.i -= xfold(ep->v.p[1]);
	    return;
	} else if (ep->v.p[1]->nodetype == en_icon) {
	    ep->v.p[1]->v.i -= xfold(ep->v.p[0]);
	    return;
	}
    }
    i = xfold(ep);
    if (i != 0) {
	i = strip_icon(i, (*node)->etype);
	ep = mk_icon(i);
	ep->etype = (*node)->etype;
	ep->esize = (*node)->esize;
	ep = mk_node(en_add, *node, ep);
	ep->etype = (*node)->etype;
	ep->esize = (*node)->esize;
	*node = ep;
    }
}

opt4(node)
/*
 * apply all constant optimizations.
 */
    struct enode  **node;
{
    /* opt0(node); */
    fold_const(node);
    opt0(node);
}
