#include	<stdio.h>
#include	"c.h"
#include	"expr.h"
#include	"gen.h"
#include	"cglbdec.h"

/*
 * 68000 C compiler
 *
 * Copyright 1984, 1985, 1986 Matthew Brandt. all commercial rights reserved.
 *
 * This compiler is intended as an instructive tool for personal use. Any use
 * for profit without the written consent of the author is prohibited.
 *
 * This compiler may be distributed freely for non-commercial use as long as
 * this notice stays intact. Please forward any enhancements or questions to:
 *
 * Matthew Brandt Box 920337 Norcross, Ga 30092
 *
 * This compiler has been enhanced and corrected at the end of 1989 by Christoph
 * van Wullen, who generated this version. Look at the file README.CVW for
 * further comments.
 */

/*
 * this module contains all of the code generation routines for evaluating
 * expressions and conditions.
 */


extern struct amode push[], pop[];


struct amode   *
mk_label(lab)
/*
 * construct a reference node for an internal label number.
 */
    int 	    lab;
{
    struct enode   *lnode;
    struct amode   *ap;
    lnode = mk_node(en_labcon, NIL_ENODE, NIL_ENODE);
    lnode->v.i = lab;
    ap = (struct amode *) xalloc((int) sizeof(struct amode));
    ap->mode = am_direct;
    ap->offset = lnode;
    return ap;
}

struct amode   *
mk_immed(i)
/*
 * mk_ a node to reference an immediate value i.
 */
    long	    i;
{
    struct amode   *ap;
    struct enode   *ep;
    ep = mk_icon((long) 0);
    ep->v.i = i;
    ap = (struct amode *) xalloc((int) sizeof(struct amode));
    ap->mode = am_immed;
    ap->offset = ep;
    return ap;
}

struct amode   *
mk_offset(node)
/*
 * mk_ a direct reference to a node.
 */
    struct enode   *node;
{
    struct amode   *ap;
    ap = (struct amode *) xalloc((int) sizeof(struct amode));
    ap->mode = am_direct;
    ap->offset = node;
    return ap;
}

mk_legal(ap, flags, size)
/*
 * mk_legal will coerce the addressing mode in ap1 into a mode that is
 * satisfactory for the flag word.
 */
    struct amode   *ap;
    int 	    flags;
    long	    size;
{
    struct amode   *ap2;
    if (((flags & F_VOL) == 0) || ap->tempflag) {
	switch (ap->mode) {
	case am_immed:
	    if (flags & F_IMMED)
		return;		/* mode ok */
	    break;
	case am_areg:
	    if (flags & F_AREG)
		return;
	    break;
	case am_dreg:
	    if (flags & F_DREG)
		return;
	    break;
	case am_ind:
	case am_indx:
	case am_indx2:
	case am_direct:
	case am_indx3:
	case am_ainc:
	    if (flags & F_MEM)
		return;
	    break;
	}
    }
    if ((flags & F_DREG) && (flags & F_AREG)) {
	/* decide, which mode is better */
	if (ap->mode == am_immed) {
	    if (isbyte(ap->offset))
		flags &= F_DREG;
	    else if (isshort(ap->offset) && size == 4)
		flags &= F_AREG;
	}
    }
    if (flags & F_DREG) {
	freeop(ap);		/* maybe we can use it... */
	ap2 = temp_data();	/* allocate to dreg */
	g_code(op_move, size, ap, ap2);
	ap->mode = am_dreg;
	ap->preg = ap2->preg;
	ap->deep = ap2->deep;
	ap->tempflag = 1;
	return;
    }
    if (size < 4)
	fprintf(stderr,"DIAG: illegal: size<4 --> An\n");
    freeop(ap);
    ap2 = temp_addr();
    g_code(op_move, size, ap, ap2);
    ap->mode = am_areg;
    ap->preg = ap2->preg;
    ap->deep = ap2->deep;
    ap->tempflag = 1;
}


int
isshort(node)
/*
 * return true if the node passed can be generated as a short offset.
 */
    struct enode   *node;
{
    return node->nodetype == en_icon &&
	(node->v.i >= -32768 && node->v.i <= 32767);
}

int
isbyte(node)
    struct enode   *node;
{
    return node->nodetype == en_icon &&
	(node->v.i >= -128 && node->v.i <= 127);
}




struct amode   *
g_index(node)
/*
 * generate code to evaluate an index node (^+) and return the addressing
 * mode of the result. This routine takes no flags since it always returns
 * either am_ind or am_indx. or am_direct It is called by g_deref
 */
    struct enode   *node;
{
    struct amode   *ap1, *ap2;
    if (node->v.p[0]->nodetype == en_tempref &&
	node->v.p[1]->nodetype == en_tempref &&
	(node->v.p[0]->v.i >= 8 || node->v.p[1]->v.i >= 8)) {
	/* both nodes are registers, one is address */
	if (node->v.p[0]->v.i < 8)
	    /* first node is data register */
	{
	    ap1 = g_expr(node->v.p[1], F_AREG);
	    ap1->sreg = node->v.p[0]->v.i;
	    ap1->mode = am_indx2;	/* 0(Ax,Dx) */
	    ap1->offset = mk_icon((long) 0);
	    return ap1;
	} else {
	    /* first node is address register */
	    ap1 = g_expr(node->v.p[0], F_AREG);
	    ap2 = g_expr(node->v.p[1], F_AREG | F_DREG);
	    if (ap2->mode == am_dreg) {
		/* 0(Ax,Dx) */
		ap1->mode = am_indx2;
		ap1->sreg = ap2->preg;
	    } else {
		/* 0(Ax,Ay) */
		ap1->mode = am_indx3;
		ap1->sreg = ap2->preg;
	    }
	    ap1->offset = mk_icon((long) 0);
	    return ap1;
	}
    }
    /* The general case (no tempref) */
    /* put address temprefs first place, data temprefs second place */
    if (node->v.p[1]->nodetype == en_tempref && node->v.p[1]->v.i >= 8)
	swap_nodes(node);
    else if (node->v.p[0]->nodetype == en_tempref && node->v.p[0]->v.i < 8)
	swap_nodes(node);

    ap1 = g_expr(node->v.p[0], F_AREG | F_IMMED);
    if (ap1->mode == am_areg) {
	ap2 = g_expr(node->v.p[1], F_AREG | F_DREG | F_IMMED);
	validate(ap1);
    } else {
	ap2 = ap1;
	ap1 = g_expr(node->v.p[1], F_AREG | F_IMMED);
	validate(ap2);
    }
    /*
     * possible combinations:
     *
     * F_IMMED + F_IMMED F_AREG  + F_IMMED F_AREG  + F_DREG F_AREG  + F_AREG
     */


    /* watch out for: tempref(addr) + temp_addr, tempref(addr + temp_data */

    if (ap1->mode == am_areg && ap1->preg > MAX_ADDR) {
	/* ap1 = tempref address register */
	if (ap2->mode == am_dreg) {
	    /* 0(Ax,Dy) */
	    ap1->mode = am_indx2;
	    ap1->sreg = ap2->preg;
	    ap1->deep = ap2->deep;
	    ap1->offset = mk_icon((long) 0);
	    return ap1;
	}
	if (ap2->mode == am_areg) {
	    /* 0(Ax,Ay) */
	    ap1->mode = am_indx3;
	    ap1->sreg = ap2->preg;
	    ap1->deep = ap2->deep;
	    ap1->offset = mk_icon((long) 0);
	    return ap1;
	}
	if (ap2->mode == am_immed && (!isshort(ap2->offset)))
	    /* we want to add to ap1 later... */
	    mk_legal(ap1, F_AREG | F_VOL, 4l);

    }
    /* watch out for: temp_addr + tempref(data) */

    if (ap1->mode == am_areg && ap2->mode == am_dreg && ap2->preg > MAX_DATA) {
	ap1->mode = am_indx2;
	ap1->sreg = ap2->preg;
	ap1->offset = mk_icon((long) 0);
	return ap1;
    }
    if (ap1->mode == am_immed && ap2->mode == am_immed) {
	ap1->offset = mk_node(en_add, ap1->offset, ap2->offset);
	ap1->mode = am_direct;
	return ap1;
    }
    if (ap2->mode == am_immed && isshort(ap2->offset)) {
	mk_legal(ap1, F_AREG, 4l);
	ap1->mode = am_indx;
	ap1->offset = ap2->offset;
	return ap1;
    }
    /* ap1 is volatile ... */
    g_code(op_add, 4l, ap2, ap1);	/* add left to address reg */
    ap1->mode = am_ind;		/* mk_ indirect */
    freeop(ap2);		/* release any temps in ap2 */
    return ap1;			/* return indirect */
}

struct amode   *
g_deref(node, flags)
/*
 * return the addressing mode of a dereferenced node.
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1;
    long	    size;
    size = node->esize;
    if (node->v.p[0]->nodetype == en_add) {
	ap1 = g_index(node->v.p[0]);
	mk_legal(ap1, flags, size);
	return ap1;
    }
    if (node->v.p[0]->nodetype == en_autocon) {
	if (node->v.p[0]->v.i >=-32768 && node->v.p[0]->v.i <32767) {
	    ap1 = (struct amode *) xalloc((int) sizeof(struct amode));
	    ap1->mode = am_indx;
	    ap1->preg = 6;
	    ap1->offset = mk_icon((long) node->v.p[0]->v.i);
	} else {
	    ap1 = temp_addr();
	    g_code(op_move,4l,mk_immed((long)node->v.p[0]->v.i),ap1);
	    g_code(op_add,4l,mk_areg(6),ap1);
	    ap1->mode = am_ind;
	}
	mk_legal(ap1, flags, size);
	return ap1;
    }
    /* special 68000 instructions */
    if (node->v.p[0]->nodetype==en_ainc
	&& node->v.p[0]->v.p[1]->v.i == size
	&& node->v.p[0]->v.p[0]->nodetype == en_tempref
	&& node->v.p[0]->v.p[0]->v.i >= 8
	&& !(flags & F_USES)) {
	/* (An)+ */
	ap1 = (struct amode *) xalloc( (int) sizeof(struct amode));
	ap1->mode = am_ainc;
	ap1->preg = node->v.p[0]->v.p[0]->v.i-8;
	mk_legal(ap1,flags,size);
	return ap1;
    }
    ap1 = g_expr(node->v.p[0], F_AREG | F_IMMED);	/* generate address */
    if (ap1->mode == am_areg) {
	ap1->mode = am_ind;
	mk_legal(ap1, flags, size);
	return ap1;
    }
    ap1->mode = am_direct;
    mk_legal(ap1, flags, size);
    return ap1;
}

struct amode   *
g_unary(node, flags, op)
/*
 * generate code to evaluate a unary minus or complement. float: unary minus
 * calls a library function
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap;
    int 	    i;
    switch (node->etype) {
    case bt_uchar:
    case bt_char:
    case bt_short:
    case bt_ushort:
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	ap = g_expr(node->v.p[0], F_DREG | F_VOL);
	g_code(op, node->esize, ap, NIL_AMODE);
	mk_legal(ap, flags, node->esize);
	return ap;
    case bt_float:
	if (op == op_neg) {
	    temp_inv();
	    i = push_param(node->v.p[0]);
	    call_library("fpneg");
	    return func_result(flags, i);
	}
    }
    fprintf(stderr,"DIAG: g_unary: illegal type or operation.\n");
    return 0;
}

struct amode   *
g_addsub(node, flags, op)
/*
 * generate code to evaluate a binary node and return the addressing mode of
 * the result.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;
    int 	    i;
    switch (node->etype) {
    case bt_uchar:
    case bt_char:
    case bt_short:
    case bt_ushort:
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	flags &= (F_DREG | F_AREG);
	ap1 = g_expr(node->v.p[0], F_VOL | flags);
	ap2 = g_expr(node->v.p[1], F_ALL);
	validate(ap1);		/* in case push occurred */
	g_code(op, node->esize, ap2, ap1);
	freeop(ap2);
	mk_legal(ap1, flags, node->esize);
	return ap1;
    case bt_float:
	temp_inv();
	i = push_param(node->v.p[1]);
	i += push_param(node->v.p[0]);
	switch (op) {
	case op_add:
	    call_library("fpadd");
	    break;
	case op_sub:
	    call_library("fpsub");
	    break;
	}
	return func_result(flags, i);
    }

    fprintf(stderr,"DIAG: g_addsub illegal type\n");
    return 0;
}

struct amode   *
g_xbin(node, flags, op)
/*
 * generate code to evaluate a restricted binary node and return the
 * addressing mode of the result.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;
    ap1 = g_expr(node->v.p[0], F_VOL | F_DREG);
    ap2 = g_expr(node->v.p[1], F_DREG);
    validate(ap1);		/* in case push occurred */
    g_code(op, node->esize, ap2, ap1);
    freeop(ap2);
    mk_legal(ap1, flags, node->esize);
    return ap1;
}

struct amode   *
g_ybin(node, flags, op)
/*
 * generate code to evaluate a restricted binary node and return the
 * addressing mode of the result.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;
    ap1 = g_expr(node->v.p[0], F_VOL | F_DREG);
    ap2 = g_expr(node->v.p[1], F_ALL & ~F_AREG);
    validate(ap1);		/* in case push occurred */
    g_code(op, node->esize, ap2, ap1);
    freeop(ap2);
    mk_legal(ap1, flags, node->esize);
    return ap1;
}

struct amode   *
g_shift(node, flags, op)
/*
 * generate code to evaluate a shift node and return the address mode of the
 * result.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;

    if (op == op_asl && node->v.p[1]->nodetype == en_icon
	&& node->v.p[1]->v.i >= 1
	&& node->v.p[1]->v.i <= 2) {
	int		i;
	ap1 = g_expr(node->v.p[0], F_VOL | (flags & (F_DREG | F_AREG)));
	for (i = 1; i <= node->v.p[1]->v.i; ++i)
	    g_code(op_add, node->esize, ap1, ap1);
    } else {
	ap1 = g_expr(node->v.p[0], F_DREG | F_VOL);
	ap2 = g_expr(node->v.p[1], F_DREG | F_IMMED);

	/* CVW: immedeate value only legal if 1<=const<=8 */

	if (ap2->mode == am_immed && ap2->offset->nodetype == en_icon
	    && (ap2->offset->v.i > 8 || ap2->offset->v.i < 1)) {
	    if (ap2->offset->v.i <= 0 || ap2->offset->v.i > 32)
		fprintf(stderr,"warning: invalid shift constant\n");
	    mk_legal(ap2, F_DREG, 1l);
	}
	validate(ap1);
	g_code(op, node->esize, ap2, ap1);
	freeop(ap2);
    }

    mk_legal(ap1, flags, node->esize);
    return ap1;
}

struct amode   *
g_div(node, flags)
/*
 * generate code to evaluate a divide operator
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2;
    int 	    i;
    switch (node->etype) {
    case bt_short:
    case bt_ushort:
	ap1 = g_expr(node->v.p[0], F_DREG | F_VOL);
	ap2 = g_expr(node->v.p[1], F_ALL);
	validate(ap1);
	if (node->etype == bt_short) {
	    g_code(op_ext, 4l, ap1, NIL_AMODE);
	    g_code(op_divs, 0l, ap2, ap1);
	} else {
	    g_code(op_and, 4l, mk_immed(65535l), ap1);
	    g_code(op_divu, 0l, ap2, ap1);
	}
	mk_legal(ap1, flags, 2l);
	freeop(ap2);
	return ap1;
    case bt_long:
    case bt_ulong:
    case bt_float:
	temp_inv();
	i = push_param(node->v.p[1]);
	i += push_param(node->v.p[0]);
	switch (node->etype) {
	case bt_long:
	    call_library("$ldiv");
	    break;
	case bt_ulong:
	    call_library("$uldiv");
	    break;
	case bt_float:
	    call_library("fpdiv");
	    break;
	}
	return func_result(flags, i);
    }
    fprintf(stderr,"DIAG: g_div: illegal type\n");
    return 0;
}

struct amode   *
g_mod(node, flags)
/*
 * generate code to evaluate a mod operator
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2;
    int 	    i;
    switch (node->etype) {
    case bt_short:
    case bt_ushort:
	ap1 = g_expr(node->v.p[0], F_DREG | F_VOL);
	ap2 = g_expr(node->v.p[1], F_ALL);
	validate(ap1);
	if (node->etype == bt_short) {
	    g_code(op_ext, 4l, ap1, NIL_AMODE);
	    g_code(op_divs, 0l, ap2, ap1);
	} else {
	    g_code(op_and, 4l, mk_immed(65535l), ap1);
	    g_code(op_divu, 0l, ap2, ap1);
	}
	g_code(op_swap, 0l, ap1, NIL_AMODE);
	mk_legal(ap1, flags, 2l);
	freeop(ap2);
	return ap1;
    case bt_long:
    case bt_ulong:
	temp_inv();
	i = push_param(node->v.p[1]);
	i += push_param(node->v.p[0]);
	if (node->etype == bt_long)
	    call_library("$lrem");
	else
	    call_library("$ulrem");
	return func_result(flags, i);
    }
    fprintf(stderr,"DIAG: g_mod: illegal type\n");
    return 0;
}

swap_nodes(node)
/*
 * exchange the two operands in a node.
 */
    struct enode   *node;
{
    struct enode   *temp;
    temp = node->v.p[0];
    node->v.p[0] = node->v.p[1];
    node->v.p[1] = temp;
}

struct amode   *
g_mul(node, flags)
/*
 * */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2;
    int 	    i;
    switch (node->etype) {
    case bt_short:
    case bt_ushort:
	if (node->v.p[0]->nodetype == en_icon)
	    swap_nodes(node);
	ap1 = g_expr(node->v.p[0], F_DREG | F_VOL);
	ap2 = g_expr(node->v.p[1], F_ALL);
	validate(ap1);
	if (node->etype == bt_short)
	    g_code(op_muls, 0l, ap2, ap1);
	else
	    g_code(op_mulu, 0l, ap2, ap1);
	freeop(ap2);
	mk_legal(ap1, flags, 2l);
	return ap1;
    case bt_ulong:
    case bt_long:
    case bt_pointer:
    case bt_float:
	temp_inv();
	i = push_param(node->v.p[1]);
	i += push_param(node->v.p[0]);
	if (node->etype == bt_ulong)
	    call_library("$ulmul");
	else if (node->etype == bt_float)
	    call_library("fpmult");
	else
	    call_library("$lmul");
	return func_result(flags, i);
    }
    fprintf(stderr,"DIAG: g_mul: illegal type\n");
    return 0;
}

struct amode   *
g_hook(node, flags)
/*
 * generate code to evaluate a condition operator node (?:)
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2;
    int 	    false_label, end_label;
    false_label = nextlabel++;
    end_label = nextlabel++;
    flags = (flags & (F_AREG | F_DREG)) | F_VOL;
    falsejp(node->v.p[0], false_label);
    node = node->v.p[1];
    ap1 = g_expr(node->v.p[0], flags);
    freeop(ap1);
    g_code(op_bra, 0l, mk_label(end_label), NIL_AMODE);
    g_label(false_label);
    ap2 = g_expr(node->v.p[1], flags);
    if (!equal_address(ap1, ap2)) {
	freeop(ap2);
	if (ap1->mode == am_dreg)
	    temp_data();
	else
	    temp_addr();
	g_code(op_move, node->v.p[1]->esize, ap2, ap1);
    }
    g_label(end_label);
    return ap1;
}

struct amode   *
g_asadd(node, flags, op)
/*
 * generate a plus equal or a minus equal node.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    int f;
    struct amode   *ap1, *ap2;
    switch (node->etype) {
    case bt_char:
    case bt_uchar:
    case bt_short:
    case bt_ushort:
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	if (flags & F_NOVALUE)
	    f = F_ALL;
	else
	    f = F_ALL | F_USES;
	ap1 = g_expr(node->v.p[0], f);
	ap2 = g_expr(node->v.p[1], F_DREG | F_IMMED);
	validate(ap1);
	g_code(op, node->esize, ap2, ap1);
	freeop(ap2);
	mk_legal(ap1, flags, node->esize);
	return ap1;
    case bt_float:
	if (op == op_add)
	    return as_fcall(node, flags, "fpadd");
	else
	    return as_fcall(node, flags, "fpsub");
    }
    fprintf(stderr,"DIAG: asadd: illegal type.\n");
    return 0;
}

struct amode   *
g_asxor(node, flags)
/*
 * generate an ^= node
 */
    struct enode   *node;
    int 	    flags;
{
    int f;
    struct amode   *ap1, *ap2;
    switch (node->etype) {
    case bt_char:
    case bt_uchar:
    case bt_short:
    case bt_ushort:
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	if (flags & F_NOVALUE)
	    f = F_ALL;
	else
	    f = F_ALL | F_USES;
	ap1 = g_expr(node->v.p[0], f);
	ap2 = g_expr(node->v.p[1], F_DREG | F_IMMED);
	validate(ap1);
	g_code(op_eor, node->esize, ap2, ap1);
	freeop(ap2);
	mk_legal(ap1, flags, node->esize);
	return ap1;
    }
    fprintf(stderr,"DIAG: asxor: illegal type.\n");
    return 0;
}

struct amode   *
g_aslogic(node, flags, op)
/*
 * generate a and equal or a or equal node.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    int f;
    struct amode   *ap1, *ap2, *ap3;
    if (flags & F_NOVALUE)
	f = F_ALL;
    else
	f = F_ALL | F_USES;
    ap1 = g_expr(node->v.p[0], f);
    ap2 = g_expr(node->v.p[1], F_DREG | F_IMMED);
    validate(ap1);
    if (ap1->mode != am_areg)
	g_code(op, node->esize, ap2, ap1);
    else {
	ap3 = temp_data();
	g_code(op_move, 4l, ap1, ap3);
	g_code(op, node->esize, ap2, ap3);
	g_code(op_move, node->esize, ap3, ap1);
	freeop(ap3);
    }
    freeop(ap2);
    mk_legal(ap1, flags, node->esize);
    return ap1;
}

struct amode   *
g_asshift(node, flags, op)
/*
 * generate shift equals operators.
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    int f;
    struct amode   *ap1, *ap2, *ap3;
    switch (node->etype) {
    case bt_uchar:
    case bt_char:
    case bt_ushort:
    case bt_short:
    case bt_ulong:
    case bt_long:
	if (flags & F_NOVALUE)
	    f = F_ALL;
	else
	    f = F_ALL | F_USES;
	ap1 = g_expr(node->v.p[0], f);
	if (ap1->mode != am_dreg) {
	    ap3 = temp_data();
	    g_code(op_move, node->esize, ap1, ap3);
	} else
	    ap3 = ap1;
	ap2 = g_expr(node->v.p[1], F_DREG | F_IMMED);

	/* CVW: immedeate value only legal if 1<=const<=8 */

	if (ap2->mode == am_immed && ap2->offset->nodetype == en_icon
	    && (ap2->offset->v.i > 8 || ap2->offset->v.i < 1)) {
	    if (ap2->offset->v.i <= 0)
		fprintf(stderr,"warning: non-positive shift constant\n");
	    mk_legal(ap2, F_DREG, 1l);
	}
	validate(ap3);
	g_code(op, node->esize, ap2, ap3);
	freeop(ap2);
	if (ap3 != ap1) {
	    g_code(op_move, node->esize, ap3, ap1);
	    freeop(ap3);
	}
	mk_legal(ap1, flags, node->esize);
	return ap1;
    }
    fprintf(stderr,"DIAG: g_asshift: illegal type\n");
    return 0;
}

struct amode   *
g_asmul(node, flags)
/*
 * generate a *= node.
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2, *ap3;
    enum e_op	    op = op_mulu;
    switch (node->etype) {
    case bt_char:
	ap1 = g_expr(node->v.p[0],F_ALL | F_USES);
	if (ap1->mode != am_dreg) {
	    ap2 = temp_data();
	    g_code(op_move,1l,ap1,ap2);
	    }
	else
	    ap2 = ap1;
	g_code(op_ext,2l,ap2,NIL_AMODE);
	ap3 = g_expr(node->v.p[1],F_DREG | F_IMMED);
	freeop(ap3);
	if (ap3->mode == am_dreg)
	    g_code(op_ext,2l,ap3,NIL_AMODE);
	g_code(op_muls,0l,ap3,ap2);
	if (ap2 != ap1) {
	    freeop(ap2);
	    g_code(op_move,1l,ap2,ap1);
	}
	return ap1;
    case bt_uchar:
	ap1 = g_expr(node->v.p[0],F_ALL | F_USES);
	if (ap1->mode != am_dreg) {
	    ap2 = temp_data();
	    g_code(op_move,1l,ap1,ap2);
	    }
	else
	    ap2 = ap1;
	g_code(op_and,2l,mk_immed(255l),ap2);
	ap3 = g_expr(node->v.p[1],F_DREG | F_IMMED);
	freeop(ap3);
	if (ap3->mode == am_dreg)
	    g_code(op_and,2l,mk_immed(255l),ap3);
	g_code(op_mulu,0l,ap3,ap2);
	if (ap2 != ap1) {
	    freeop(ap2);
	    g_code(op_move,1l,ap2,ap1);
	}
	return ap1;
    case bt_short:
	op = op_muls;
    case bt_ushort:
	ap1 = g_expr(node->v.p[0], F_ALL | F_USES);
	ap2 = g_expr(node->v.p[1], F_ALL);
	if (ap1->mode != am_dreg) {
	    ap3 = temp_data();
	    validate(ap1);
	    freeop(ap2);
	    freeop(ap3);
	    g_code(op_move, 2l, ap1, ap3);
	    g_code(op, 0l, ap2, ap3);
	    g_code(op_move, 2l, ap3, ap1);
	} else {
	    validate(ap1);
	    g_code(op_muls, 0l, ap2, ap1);
	    freeop(ap2);
	}
	return ap1;
    case bt_long:
	return as_fcall(node, flags, "$lmul");
    case bt_ulong:
    case bt_pointer:
	return as_fcall(node, flags, "$ulmul");
    case bt_float:
	return as_fcall(node, flags, "fpmult");
    }
    fprintf(stderr,"DIAG: asmul: illegal type\n");
    return 0;
}

struct amode   *
g_asdiv(node, flags)
/*
 * generate /= and %= nodes.
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2, *ap3;
    switch (node->etype) {
    case bt_short:
    case bt_ushort:
	ap1 = temp_data();
	ap2 = g_expr(node->v.p[0], F_ALL | F_USES);
	validate(ap1);
	g_code(op_move, 2l, ap2, ap1);
	ap3 = g_expr(node->v.p[1], F_ALL & ~F_AREG);
	validate(ap2);
	validate(ap1);
	if (node->etype == bt_short) {
	    g_code(op_ext, 4l, ap1, NIL_AMODE);
	    g_code(op_divs, 0l, ap3, ap1);
	} else {
	    g_code(op_and, 4l, mk_immed(65535l), ap1);
	    g_code(op_divu, 0l, ap3, ap1);
	}
	freeop(ap3);
	g_code(op_move, 2l, ap1, ap2);
	freeop(ap2);
	mk_legal(ap1, flags, 2l);
	return ap1;
    case bt_long:
	return as_fcall(node, flags, "$ldiv");
    case bt_ulong:
    case bt_pointer:
	return as_fcall(node, flags, "$uldiv");
    case bt_float:
	return as_fcall(node, flags, "fpdiv");
    }
    fprintf(stderr,"DIAG: asdiv: illegal type");
    return 0;
}

struct amode   *
g_asmod(node, flags)
/*
 * generate /= and %= nodes.
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2, *ap3;
    switch (node->etype) {
    case bt_short:
    case bt_ushort:
	ap1 = temp_data();
	ap2 = g_expr(node->v.p[0], F_ALL | F_USES);
	validate(ap1);
	g_code(op_move, 2l, ap2, ap1);
	ap3 = g_expr(node->v.p[1], F_ALL & ~F_AREG);
	validate(ap2);
	validate(ap1);
	if (node->etype == bt_short) {
	    g_code(op_ext, 4l, ap1, NIL_AMODE);
	    g_code(op_divs, 0l, ap3, ap1);
	} else {
	    g_code(op_and, 4l, mk_immed(65535l), ap1);
	    g_code(op_divu, 0l, ap3, ap1);
	}
	g_code(op_swap, 0l, ap1, NIL_AMODE);
	freeop(ap3);
	g_code(op_move, 2l, ap1, ap2);
	freeop(ap2);
	mk_legal(ap1, flags, 2l);
	return ap1;
    case bt_long:
	return as_fcall(node, flags, "$lrem");
    case bt_ulong:
    case bt_pointer:
	return as_fcall(node, flags, "$ulrem");
    }
    fprintf(stderr,"DIAG: asmod: illegal type");
    return 0;
}

structassign(ap1,ap2,size)
struct amode *ap1,*ap2;
long size;
{
long loop;
int rest;
long i;
struct amode *ap3;
int label;

    ap1->mode = am_ainc;
    ap2->mode = am_ainc;
    loop = size / 4;
    rest = size % 4;
    if (loop<=10) /* loop-unrolling */
	for (i=1;i<=loop;i++)
	    g_code(op_move,4l,ap1,ap2);
    else {
	loop--; /* for dbra */
	ap3=temp_data();
	freeop(ap3);
	label=nextlabel++;
	if (loop<=65535) { /* single loop */
	    g_code(op_move,2l,mk_immed(loop),ap3);
	    g_label(label);
	    g_code(op_move, 4l, ap1, ap2);
	    g_code(op_dbra,0l,ap3,mk_label(label));
	} else { /* extended loop */
	    g_code(op_move,4l,mk_immed(loop),ap3);
	    g_label(label);
	    g_code(op_move, 4l, ap1, ap2);
	    g_code(op_dbra,0l,ap3,mk_label(label));
	    g_code(op_sub,4l,mk_immed(65536l),ap3);
	    g_code(op_bhs,0l,mk_label(label),NIL_AMODE);
	}
    }
    if (rest >= 2) {
	rest -= 2;
	g_code(op_move, 2l, ap1, ap2);
    }
/* This cannot happen if the size of structures is always even */
    if (rest >= 1)
	g_code(op_move, 1l, ap1, ap2);
}

struct amode   *
g_assign(node, flags)
/*
 * generate code for an assignment node.
 */
    struct enode   *node;
    int 	    flags;
{
    int f;
    struct amode   *ap1, *ap2;
    /*
     * we may return the value of the right operand because it is already
     * cast to the assignment type
     *
     */
    if (flags & F_NOVALUE)
	f = F_ALL;
    else
	f = F_ALL | F_USES;
    if (node->etype == bt_struct || node->etype == bt_union) {
/* WORKLEFT: functions returning structs/unions */
	if (!lvalue(node->v.p[0]) || !lvalue(node->v.p[1]))
	    fprintf(stderr,"DIAG: struct assign\n");
	/* STRUCTASSIGN */
	/* C-TeX is based on structures whose lengths are 4 */
	if (node->esize == 1 || node->esize == 2 || node->esize == 4) {
	    ap1 = g_expr(node->v.p[0], f);
	    ap2 = g_expr(node->v.p[1], F_ALL);
	    validate(ap1);
	    freeop(ap2);
	    g_code(op_move, node->esize, ap1, ap2);
	    mk_legal(ap1, flags, node->esize);
	    return ap1;
	} else {
	    /* assign structure */
	    /* skip ref nodes */
	    ap1 = g_expr(node->v.p[1]->v.p[0], F_AREG | F_VOL);
	    ap2 = g_expr(node->v.p[0]->v.p[0], F_AREG | F_VOL);
	    validate(ap1);
	    structassign(ap1,ap2,node->esize);
	    freeop(ap2);
	    freeop(ap1);
	    if (!(flags & F_NOVALUE)) {
		ap1 = g_expr(node->v.p[1], F_ALL);
		mk_legal(ap1, flags, 4l);
	    }
	    return ap1;
	}
    } else
	/* (uns.) char, (uns.) short, (uns.) long, float */
    {
	ap1 = g_expr(node->v.p[1], f);
	ap2 = g_expr(node->v.p[0], F_ALL);
	validate(ap1);
	freeop(ap2);
	g_code(op_move, node->esize, ap1, ap2);
	mk_legal(ap1, flags, node->esize);
	return ap1;
    }
}

struct amode   *
g_aincdec(node, flags, op)
/*
 * generate an auto increment or decrement node. op should be either op_add
 * (for increment) or op_sub (for decrement).
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;
    switch (node->etype) {
    case bt_uchar:
    case bt_char:
    case bt_short:
    case bt_ushort:
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	if (flags & F_NOVALUE) {/* dont need result */
	    ap1 = g_expr(node->v.p[0], F_ALL);
	    g_code(op, node->esize, mk_immed((long)node->v.p[1]->v.i), ap1);
	    freeop(ap1);
	    return ap1;
	}
	if (flags & F_DREG)
	    ap1 = temp_data();
	else
	    ap1 = temp_addr();
	ap2 = g_expr(node->v.p[0], F_ALL | F_USES);
	validate(ap1);
	g_code(op_move, node->esize, ap2, ap1);
	g_code(op, node->esize, mk_immed((long)node->v.p[1]->v.i), ap2);
	freeop(ap2);
	return ap1;
    }
    fprintf(stderr,"DIAG: g_aincdec: illegal type or float\n");
    return 0;
}

int
push_param(ep)
/*
 * push the operand expression onto the stack. return the number of bytes
 * pushed
 */
    struct enode   *ep;
{
    struct amode   *ap,*ap1;
    long	    size = ep->esize;

    /* pushing of structures and unions */
    if (ep->etype == bt_struct || ep->etype==bt_union) {
	if (!lvalue(ep)) error(ERR_LVALUE);
	ep = ep->v.p[0];
	/* allocate stack space */
	g_code(op_sub, 4l, mk_immed(size), mk_areg(7));
	ap = g_expr(ep, F_AREG);
	ap1 =temp_addr();
	validate (ap);
	g_code(op_move, 4l, mk_areg(7), ap1);
	/* now, copy it on stack - the same as structassign */
	structassign(ap,ap1,size);
	freeop(ap);
	freeop(ap1);
	return size;
    }
    ap = g_expr(ep, F_ALL);

    /*
     * This is a hook for the peephole optimizer, which will convert lea
     * <ea>,An + pea (An) ==> pea <ea>
     */

    if (ap->mode == am_areg && size == 4 && ap->preg <= MAX_ADDR + 8) {
	ap->mode = am_ind;
	g_code(op_pea, 0l, ap, NIL_AMODE);
    }
    else
	g_code(op_move, size, ap, push);
    freeop(ap);
    return size;
}

int
g_parms(plist)
/*
 * push a list of parameters onto the stack and return the number of
 * parameters pushed.
 */
    struct enode   *plist;
{
    int 	    i;
    i = 0;
    while (plist != 0) {
	i += push_param(plist->v.p[0]);
	plist = plist->v.p[1];
    }
    return i;
}

struct amode   *
func_result(flags, bytes)
    int 	    flags, bytes;
{
    /*
     * saves a function call result in D0 it is assumed that flags contain
     * either F_DREG or F_AREG return value is the addressing mode of the
     * result bytes is the number of bytes to pop off the stack
     */
    struct amode   *ap;
    if (bytes != 0)
	/* adjust stack pointer */
	g_code(op_add, 4l, mk_immed((long) bytes), mk_areg(7));
    if (flags & F_DREG) {
	ap = temp_data();
	if (ap->preg != 0)
	    g_code(op_move, 4l, mk_dreg(0), ap);
    } else if (flags & F_AREG) {
	ap = temp_addr();
	g_code(op_move, 4l, mk_dreg(0), ap);
    } else {
	fprintf(stderr,"DIAG: func_result: illegal addressing mode\n");
	ap = 0;
    }
    return ap;
}

struct amode   *
as_fcall(node, flags, libname)
    struct enode   *node;
    int 	    flags;
    char	   *libname;
/* assignment operations with library calls */
{
    int 	    i;
    struct amode   *ap1, *ap2;
    long	    size;
    size = node->esize;
    temp_inv();
    i = push_param(node->v.p[1]);
    ap1 = g_expr(node->v.p[0], F_ALL | F_USES);
    g_code(op_move, size, ap1, push);
    i += size;
    temp_inv();
    call_library(libname);
    g_code(op_add, 4l, mk_immed((long) i), mk_areg(7));
    /* assign result */

    /* ap1 may depend on the pushed value of d0 e.g. 0(a4,d0.l) */
    if ((ap1->mode == am_dreg && ap1->preg == 0) ||
	(ap1->mode == am_indx2 && ap1->sreg == 0)) {
	ap2 = temp_data();
	freeop(ap2);
	g_code(op_move, size, mk_dreg(0), ap2);
	validate(ap1);
	g_code(op_move, size, ap2, ap1);
	mk_legal(ap1, flags, size);
	return ap1;
    } else
	/* ap1 can safely be validatet and loaded from D0 */
    {
	validate(ap1);
	g_code(op_move, size, mk_dreg(0), ap1);
	mk_legal(ap1, flags, size);
	return ap1;
    }
}

struct amode   *
g_fcall(node, flags)
/*
 * generate a function call node and return the address mode of the result.
 */
    struct enode   *node;
{
/* WORKLEFT: functions returning structs or unions */
    struct amode   *ap;
    int 	    i;
    /* push any used addr&data temps */
    temp_inv();
    i = g_parms(node->v.p[1]);	/* generate parameters */
    if (node->v.p[0]->nodetype == en_nacon ||
	node->v.p[0]->nodetype == en_labcon)
	g_code(op_jsr, 0l, mk_offset(node->v.p[0]), NIL_AMODE);
    else {
	ap = g_expr(node->v.p[0], F_AREG);
	ap->mode = am_ind;
	freeop(ap);
	g_code(op_jsr, 0l, ap, NIL_AMODE);
    }
    return func_result(flags, i);
}

struct amode   *
g_cast(ap, typ1, typ2, flags)
    struct amode   *ap;
    int 	    flags;
    enum e_bt	    typ1, typ2;
/*
 * generates code for a en_cast node
 *
 */
{
    if (flags & F_NOVALUE)
	return 0;

    switch (typ2) {
	/* switch: type to cast to */
    case bt_char:
    case bt_uchar:
	switch (typ1) {
	case bt_uchar:
	case bt_char:
	    mk_legal(ap, flags, 1l);
	    return ap;
	case bt_ushort:
	case bt_short:
	case bt_enum:
	    if (!g_offset(ap, 1))
		mk_legal(ap, F_DREG, 2l);
	    mk_legal(ap, flags, 1l);
	    return ap;
	case bt_ulong:
	case bt_long:
	case bt_pointer:
	    if (!g_offset(ap, 3))
		mk_legal(ap, F_DREG, 4l);
	    mk_legal(ap, flags, 1l);
	    return ap;
	case bt_float:
	    return g_cast(g_cast(ap, bt_float, bt_long, F_DREG),
			  bt_long, typ2, F_DREG);
	}
    case bt_ushort:
    case bt_short:
    case bt_enum:
	switch (typ1) {
	case bt_uchar:
	    mk_legal(ap, F_DREG | F_VOL, 1l);
	    g_code(op_and, 2l, mk_immed(255l), ap);
	    mk_legal(ap, flags, 2l);
	    return ap;
	case bt_char:
	    mk_legal(ap, F_DREG, 1l);
	    g_code(op_ext, 2l, ap, NIL_AMODE);
	    mk_legal(ap, flags, 2l);
	    return ap;
	case bt_short:
	case bt_ushort:
	case bt_enum:
	    mk_legal(ap, flags, 2l);
	    return ap;
	case bt_long:
	case bt_ulong:
	case bt_pointer:
	    if (!g_offset(ap, 2))
		mk_legal(ap, F_DREG, 4l);
	    mk_legal(ap, flags, 2l);
	    return ap;
	case bt_float:
	    return g_cast(g_cast(ap, bt_float, bt_long, F_DREG),
			  bt_long, typ2, F_DREG);
	}
    case bt_long:
    case bt_ulong:
    case bt_pointer:
	switch (typ1) {
	case bt_uchar:
	    mk_legal(ap, F_DREG | F_VOL, 1l);
	    g_code(op_and, 4l, mk_immed(255l), ap);
	    mk_legal(ap, flags, 4l);
	    return ap;
	case bt_char:
	    mk_legal(ap, F_DREG, 1l);
	    g_code(op_ext, 2l, ap, NIL_AMODE);
	    g_code(op_ext, 4l, ap, NIL_AMODE);
	    mk_legal(ap, flags, 4l);
	    return ap;
	case bt_ushort:
	    mk_legal(ap, F_DREG | F_VOL, 2l);
	    g_code(op_and, 4l, mk_immed(65535l), ap);
	    mk_legal(ap, flags, 4l);
	    return ap;
	case bt_short:
	case bt_enum:
	    mk_legal(ap, F_DREG | F_AREG, 2l);
	    if (ap->mode == am_dreg)
		g_code(op_ext, 4l, ap, NIL_AMODE);
	    mk_legal(ap, flags, 4l);
	    return ap;
	case bt_long:
	case bt_ulong:
	case bt_pointer:
	    mk_legal(ap, flags, 4l);
	    return ap;
	case bt_float:
	    /* library call */
	    freeop(ap);
	    temp_inv();
	    g_code(op_move, 4l, ap, push);
	    if (typ2 == bt_long)
		call_library("fpftol");
	    else
		call_library("fpftou");
	    return func_result(flags, 4);
	}
    case bt_float:
	switch (typ1) {
	case bt_char:
	case bt_uchar:
	case bt_short:
	case bt_ushort:
	case bt_enum:
	    ap = g_cast(ap, typ1, bt_long, F_ALL);
	case bt_long:
	case bt_ulong:
	case bt_pointer:
	    /* library call */
	    freeop(ap);
	    temp_inv();
	    g_code(op_move, 4l, ap, push);
	    if (typ1 == bt_long)
		call_library("fpltof");
	    else
		call_library("fputof");
	    return func_result(flags, 4);
	}
    }
    fprintf(stderr,"DIAG: g_cast: illegal combination\n");
    return ap;
}

int
g_offset(ap, off)
    struct amode   *ap;
    int 	    off;
/*
 * return true, if ap can be switched to address a location with a short
 * offset. typical application: cast long -> short: 8(a6) --> 10(a6) offset
 * is a small number (1,2 or 3)
 */
{
    switch (ap->mode) {
    case am_ind:
	ap->mode = am_indx;
	ap->offset = mk_icon((long) off);
	return 1;
    case am_indx:
	if (ap->offset->nodetype == en_icon &&
	    off + ap->offset->v.i <= 32767) {
	    ap->offset->v.i += off;
	    return 1;
	}
	break;
    case am_indx2:
    case am_indx3:
	if (ap->offset->nodetype == en_icon &&
	    off + ap->offset->v.i <= 127) {
	    ap->offset->v.i += off;
	    return 1;
	}
	break;
    case am_direct:
	ap->offset = mk_node(en_add, ap->offset,
			     mk_icon((long) off));
	return 1;
    }
    return 0;
}

struct amode   *
g_xmul(node, flags, op)
/*
 * performs a mixed-mode multiplication
 */
    struct enode   *node;
    int 	    flags;
    enum e_op	    op;
{
    struct amode   *ap1, *ap2;

    ap1 = g_expr(node->v.p[1], F_DREG | F_VOL);
    ap2 = g_expr(node->v.p[0], F_ALL & ~F_AREG);
    validate(ap1);

    g_code(op, 0l, ap2, ap1);
    freeop(ap2);
    mk_legal(ap1, flags, node->esize);
    return ap1;
}

struct amode   *
g_expr(node, flags)
/*
 * general expression evaluation. returns the addressing mode of the result.
 */
    struct enode   *node;
    int 	    flags;
{
    struct amode   *ap1, *ap2;
    int 	    lab0, lab1;
    if (node == 0) {
	fprintf(stderr,"DIAG - null node in g_expr.\n");
	return 0;
    }
    if (tst_const(node)) {
	ap1 = (struct amode *) xalloc((int) sizeof(struct amode));
	ap1->mode = am_immed;
	ap1->offset = node;
	mk_legal(ap1, flags, node->esize);
	return ap1;
    }
    switch (node->nodetype) {
    case en_autocon:
	ap1 = temp_addr();
	if (node->v.i >= -32768 && node->v.i<=32767) {
	    ap2 = (struct amode *) xalloc((int) sizeof(struct amode));
	    ap2->mode = am_indx;
	    ap2->preg = 6;		/* frame pointer */
	    ap2->offset = node;	/* use as constant node */
	    g_code(op_lea, 0l, ap2, ap1);
	} else {
	    g_code(op_move,4l,mk_immed((long)node->v.p[0]->v.i),ap1);
	    g_code(op_add,4l,mk_areg(6),ap1);
	    ap1->mode = am_ind;
	}
	mk_legal(ap1, flags, node->esize);
	return ap1;		/* return reg */
    case en_ref:
	return g_deref(node, flags);
    case en_tempref:
	ap1 = (struct amode *) xalloc((int) sizeof(struct amode));
	if (node->v.i < 8) {
	    ap1->mode = am_dreg;
	    ap1->preg = node->v.i;
	} else {
	    ap1->mode = am_areg;
	    ap1->preg = node->v.i - 8;
	}
	ap1->tempflag = 0;	/* not a temporary */
	mk_legal(ap1, flags, node->esize);
	return ap1;
    case en_uminus:
	return g_unary(node, flags, op_neg);
    case en_compl:
	return g_unary(node, flags, op_not);
    case en_add:
	return g_addsub(node, flags, op_add);
    case en_sub:
	return g_addsub(node, flags, op_sub);
    case en_and:
	return g_ybin(node, flags, op_and);
    case en_or:
	return g_ybin(node, flags, op_or);
    case en_xor:
	return g_xbin(node, flags, op_eor);
    case en_mul:
	/*
	 * special optimization possible if there are patterns matching the
	 * 68000 mulu, muls instructions.
	 */
	if (node->etype == bt_long || node->etype == bt_ulong ||
	    node->etype == bt_pointer) {
	    if (tst_ushort(node->v.p[0]) && tst_ushort(node->v.p[1]))
		return g_xmul(node, flags, op_mulu);
	    if (tst_short(node->v.p[0]) && tst_short(node->v.p[1]))
		return g_xmul(node, flags, op_muls);
	}
	return g_mul(node, flags);
    case en_div:
	return g_div(node, flags);
    case en_mod:
	return g_mod(node, flags);
    case en_lsh:
	return g_shift(node, flags, op_asl);
    case en_rsh:
	return g_shift(node, flags, op_asr);
    case en_asadd:
	return g_asadd(node, flags, op_add);
    case en_assub:
	return g_asadd(node, flags, op_sub);
    case en_asand:
	return g_aslogic(node, flags, op_and);
    case en_asor:
	return g_aslogic(node, flags, op_or);
    case en_aslsh:
	return g_asshift(node, flags, op_asl);
    case en_asrsh:
	return g_asshift(node, flags, op_asr);
    case en_asmul:
	return g_asmul(node, flags);
    case en_asdiv:
	return g_asdiv(node, flags);
    case en_asmod:
	return g_asmod(node, flags);
    case en_asxor:
	return g_asxor(node, flags);
    case en_assign:
	return g_assign(node, flags);
    case en_ainc:
	return g_aincdec(node, flags, op_add);
    case en_adec:
	return g_aincdec(node, flags, op_sub);
    case en_land:
    case en_lor:
    case en_eq:
    case en_ne:
    case en_lt:
    case en_le:
    case en_gt:
    case en_ge:
    case en_not:
	lab0 = nextlabel++;
	lab1 = nextlabel++;
	falsejp(node, lab0);
	ap1 = temp_data();
	g_code(op_moveq, 0l, mk_immed(1l), ap1);
	g_code(op_bra, 0l, mk_label(lab1), NIL_AMODE);
	g_label(lab0);
	g_code(op_clr, 4l, ap1, NIL_AMODE);
	g_label(lab1);
	return ap1;
    case en_cond:
	return g_hook(node, flags);
    case en_void:
	freeop(g_expr(node->v.p[0], F_ALL | F_NOVALUE));
	return g_expr(node->v.p[1], flags);
    case en_fcall:
	return g_fcall(node, flags);
    case en_cast:
	return g_cast(g_expr(node->v.p[0], F_ALL | F_USES),
		      node->v.p[0]->etype,
		      node->etype, flags);
    default:
	fprintf(stderr,"DIAG - uncoded node in g_expr.\n");
	return 0;
    }
}

tst_ushort(node)
    struct enode   *node;
/*
 * tests if node is a integer constant falling in the range of uns. short or
 * if node is cast from uns. short, uns. char or char.
 */
{
    enum e_bt	    type;

    if (node->nodetype == en_icon && 0 <= node->v.i && node->v.i <= 65535)
	return 1;

    if (node->nodetype == en_cast) {
	type = node->v.p[0]->etype;
	if (type == bt_ushort || type == bt_uchar || type == bt_char) {
	    node->etype = bt_ushort;
	    node->esize = 2;
	    return 1;
	}
    }
    return 0;
}

tst_short(node)
    struct enode   *node;
/*
 * tests if node is a integer constant falling in the range of short or if
 * node is cast from short.
 */
{

    if (node->nodetype == en_icon && -32768 <= node->v.i && node->v.i <= 32767)
	return 1;

    if (node->nodetype == en_cast && node->v.p[0]->etype == bt_short) {
	node->etype = bt_short;
	node->esize = 2;
	return 1;
    }
    return 0;
}

tst_const(node)
    struct enode   *node;
/*
 * tests if it is a constant node, that means either en_icon, en_nacon or
 * en_labcon, or sums or differences of such nodes
 */
{
    enum e_node     typ1 = node->nodetype;
    enum e_node     typ2;
    if (typ1 == en_icon || typ1 == en_nacon || typ1 == en_labcon
	|| typ1 == en_fcon)
	return 1;

    if (typ1 == en_add || typ1 == en_sub) {
	typ1 = node->v.p[0]->nodetype;
	typ2 = node->v.p[1]->nodetype;
	if (((typ1 == en_nacon || typ1 == en_labcon) && typ2 == en_icon) ||
	    ((typ2 == en_nacon || typ2 == en_labcon) && typ1 == en_icon))
	    return 1;
    }
    return 0;
}


g_compare(node)
/*
 * generate code to do a comparison of the two operands of node. returns 1 if
 * it was an unsigned comparison
 */
    struct enode   *node;
{
    struct amode   *ap1, *ap2, *ap3;
    int 	    i;
    switch (node->v.p[0]->etype) {
    case bt_uchar:
    case bt_char:
    case bt_ushort:
    case bt_short:
    case bt_pointer:
    case bt_long:
    case bt_ulong:
    case bt_enum:
	ap2 = g_expr(node->v.p[1], F_ALL);
	if (ap2->mode == am_immed)
	    ap1 = g_expr(node->v.p[0], F_ALL & ~F_IMMED);
	else
	    ap1 = g_expr(node->v.p[0], F_AREG | F_DREG);
	validate(ap2);
	/*
	 * sorry, no tst.l An on the 68000, but we can move to a data
	 * register if one is free
	 */
	if (ap1->mode == am_areg && node->v.p[1]->nodetype == en_icon
	    && node->v.p[1]->v.i == 0 && free_data()) {
	    ap3 = temp_data();
	    g_code(op_move, 4l, ap1, ap3);
	    /* tst.l ap3 not needed */
	    freeop(ap3);
	} else
	    g_code(op_cmp, node->v.p[0]->esize, ap2, ap1);
	freeop(ap1);
	freeop(ap2);
	if (node->v.p[0]->etype == bt_char ||
	    node->v.p[0]->etype == bt_short ||
	    node->v.p[0]->etype == bt_long)
	    return 0;
	return 1;
    case bt_float:
	temp_inv();
	i = push_param(node->v.p[1]);
	i += push_param(node->v.p[0]);
	call_library("fpcmp");
	g_code(op_add, 4l, mk_immed((long) i), mk_areg(7));
	return 0;
    }
    fprintf(stderr,"DIAG: g_compare: illegal type\n");
    return 0;
}

truejp(node, label)
/*
 * generate a jump to label if the node passed evaluates to a true condition.
 */
    struct enode   *node;
    int 	    label;
{
    struct amode   *ap1;
    int 	    lab0;
    if (node == 0)
	return;
    opt_compare(node);
    switch (node->nodetype) {
    case en_eq:
	g_compare(node);
	g_code(op_beq, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_ne:
	g_compare(node);
	g_code(op_bne, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_lt:
	g_compare(node) ?
	    g_code(op_blo, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_blt, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_le:
	g_compare(node) ?
	    g_code(op_bls, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_ble, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_gt:
	g_compare(node) ?
	    g_code(op_bhi, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_bgt, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_ge:
	g_compare(node) ?
	    g_code(op_bhs, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_bge, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_land:
	lab0 = nextlabel++;
	falsejp(node->v.p[0], lab0);
	truejp(node->v.p[1], label);
	g_label(lab0);
	break;
    case en_lor:
	truejp(node->v.p[0], label);
	truejp(node->v.p[1], label);
	break;
    case en_not:
	falsejp(node->v.p[0], label);
	break;
    default:
	ap1 = g_expr(node, F_DALT);
	g_code(op_tst, node->esize, ap1, NIL_AMODE);
	freeop(ap1);
	g_code(op_bne, 0l, mk_label(label), NIL_AMODE);
	break;
    }
}

falsejp(node, label)
/*
 * generate code to execute a jump to label if the expression passed is
 * false.
 */
    struct enode   *node;
    int 	    label;
{
    struct amode   *ap;
    int 	    lab0;
    if (node == 0)
	return;
    opt_compare(node);
    switch (node->nodetype) {
    case en_eq:
	g_compare(node);
	g_code(op_bne, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_ne:
	g_compare(node);
	g_code(op_beq, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_lt:
	g_compare(node) ?
	    g_code(op_bhs, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_bge, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_le:
	g_compare(node) ?
	    g_code(op_bhi, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_bgt, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_gt:
	g_compare(node) ?
	    g_code(op_bls, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_ble, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_ge:
	g_compare(node) ?
	    g_code(op_blo, 0l, mk_label(label), NIL_AMODE) :
	    g_code(op_blt, 0l, mk_label(label), NIL_AMODE);
	break;
    case en_land:
	falsejp(node->v.p[0], label);
	falsejp(node->v.p[1], label);
	break;
    case en_lor:
	lab0 = nextlabel++;
	truejp(node->v.p[0], lab0);
	falsejp(node->v.p[1], label);
	g_label(lab0);
	break;
    case en_not:
	truejp(node->v.p[0], label);
	break;
    default:
	if (node->etype == bt_float) {
	    int i;
	    temp_inv();
	    i = push_param(node);
	    call_library("fptst");
	    /* The pop-off does not change the condition codes */
	    g_code(op_add,4l,mk_immed((long)i),mk_areg(7));
	} else {
	    ap = g_expr(node, F_DALT);
	    g_code(op_tst, node->esize, ap, NIL_AMODE);
	    freeop(ap);
	}
	g_code(op_beq, 0l, mk_label(label), NIL_AMODE);
	break;
    }
}

opt_compare(node)
    struct enode   *node;
{
    /* temprefs should be the second operand to a cmp instruction */
    enum e_node     t = node->nodetype;
    struct enode   *scratch;
    if ((t == en_eq || t == en_ne || t == en_le || t == en_ge
	 || t == en_lt || t == en_gt)
	&& (node->v.p[1]->nodetype == en_tempref ||
	    node->v.p[0]->nodetype == en_icon)) {

	scratch = node->v.p[0];
	node->v.p[0] = node->v.p[1];
	node->v.p[1] = scratch;
	/* if you change the operands, change the comparison operator */
	switch (t) {
	case en_le:
	    node->nodetype = en_ge;
	    break;
	case en_ge:
	    node->nodetype = en_le;
	    break;
	case en_lt:
	    node->nodetype = en_gt;
	    break;
	case en_gt:
	    node->nodetype = en_lt;
	    break;
	}
    }
}
