/*                         letters.c
 * Original (C) 1991 by Larry Moss <lm03_cif@uhura.cc.rochester.edu>
 * Minix version by Wim `Blue Baron' van Dorst <baron@wiesje.hobby.nl>
 *                  version 1.0 October 1991
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sgtty.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <pwd.h>
#include <stdio.h>

/*************************** configurable stuff ***************************/
#define DICTIONARY 	"/usr/dict/words"
#define HIGHSCORES 	"/usr/games/letters.scr"
#define START_DELAY	3000 	/* Initial delay before words move 
				 * to the next line */
#define DELAY_CHANGE	250	/* Amount the delay gets reduced by for each
				 * level */
/************************ end of configurable stuff ************************/
#define PAUSE		10	/* Length of pause before reading keyboard 
				 * again. There has to be some pause.  */
#define LEVEL_CHANGE	15	/* Number of words to be completed before 
				 * level change */
#define ADDWORD		6	/* There is 1/ADDWORD chance that a new word 
				 * will appear while words are falling. */
#define BONUSLENGTH	10	/* length of words in bonus round */
#define MAXWORDSIZE	25	/* This should be set to at least the length 
				 * of the longest word in your dictionary */
#define SCREENLENGTH	23	/* Probably best to leave these so it's the */
#define SCREENWIDTH	80	/* same everywhere. Or else, anyone with an 
				 * xterminal is likely to get higher scores. */

#define NEW		1
#define ORIG		0
#define TRUE		1
#define FALSE		0
#define CURSOR_OFF 	"\033[?25l"	/* VT100 cursor off */
#define CURSOR_ON 	"\033[?25h"	/* VT100 cursor on */
#define BELL		"\007"		/* Ascii bell */

#define CTRL(c)		(c & 037)
#define HAS_CAP(str)	(*str)
#define clrdisp()	tputs(clear_screen, Lines, outc)
#define putp(str)	tputs(str, 0, outc)
#define gotoxy(c, l)	putp(tgoto(crsr_address, c, l))

int Lines;			/* screen size, for clrdisp */
int score = 0;			/* total score */
int typed_total = 0;		/* number of letters typed (all) */
int typed_level = 0;		/* number of letters typed (this level) */
int err_total = 0;		/* number of wrong letters typed */
int level = 1;			/* level playing at */
int levels_played = -1;		/* number of levels played sofar */
int word_count = 0;		/* number of word typed */
int lives = 3;			/* number of lives left */
long delay;			/* how long to wait */
int bonus = FALSE;		/* to determine if we're in a bonus round */
long cpm = 0;			/* characters per minute */
time_t readtime;		/* time the score file is read last */
time_t level_start = 0L;	/* time to start counting for this level */
char dictfile[64] = DICTIONARY;	/* the dictionary file to be used */
char *term_name;		/* name of the terminal type */
char crsr_home[64], clear_screen[64], crsr_address[128]; /* termcap */
char enter_standout_mode[64], exit_standout_mode[64]; 	/* termcap */
char enter_underline_mode[64], exit_underline_mode[64]; /* termcap */
char visible_cursor[64], invisible_cursor[64]; 		/* termcap */

struct score_rec {
	char name[9];
	int level, words, score;
} high_scores[10];

struct stat s_buf;

struct s_word {
	char word[MAXWORDSIZE];
	int posx;
	int posy;
	int length;
	int drop;
	int matches;
	struct s_word *nextword;
} *words, *lastword, *prev_word;


/* must be functions */
outc(c) { putchar(c); }
int (*ding) ();
bell() { putp(BELL); }
quiet() { }

/* read and update high score file for Letter Invaders */
read_scores()
{
	int i;
	FILE *fp;

	/* get the last modified time so we know later if we have to reread
	 * the scores before saving a new file. */
	if (stat(HIGHSCORES, &s_buf) == -1) {
		perror("stat");
		exit(1);
	}
	readtime = s_buf.st_mtime;

	if ((fp = fopen(HIGHSCORES, "r")) == NULL) {
		fprintf("Error openning high score list.\n");
		exit(1);
	}
	for (i = 0; i < 10; i++) {
		fscanf(fp, "%s%d%d%d", high_scores[i].name, 
			&high_scores[i].level, &high_scores[i].words,
			&high_scores[i].score);
	}
	fclose(fp);
}

write_scores()
{
	int i;
	FILE *fp;

	/* check to make sure the high score list has not been modified since
	 * we read it. */
	if (stat(HIGHSCORES, &s_buf) == -1) {
		perror("stat");
		exit(1);
	}
	if (s_buf.st_mtime > readtime)
		return -1;

	if ((fp = fopen(HIGHSCORES, "w")) == NULL) {
		fprintf("Error opening high score list to write.\n");
		exit(1);
	}
	for (i = 0; i < 10; i++) {
		fprintf(fp, "%s %d %d %d", high_scores[i].name,
			high_scores[i].level, high_scores[i].words,
			high_scores[i].score);
	}
	fclose(fp);
}

update_scores()
{
	int i, j;
	struct passwd *p;

	for (i = 0; i < 10; i++)
		if (score > high_scores[i].score) {
			for (j = 10; j > i; j--) {
				strcpy(high_scores[j].name, high_scores[j - 1].name);
				high_scores[j].words = high_scores[j - 1].words;
				high_scores[j].score = high_scores[j - 1].score;
				high_scores[j].level = high_scores[j - 1].level;
			}
			if ((p = getpwuid(getuid())) == NULL)
				strcpy(high_scores[i].name, "nobody");
			else
				strcpy(high_scores[i].name, p->pw_name);
			high_scores[i].score = score;
			high_scores[i].words = word_count;
			high_scores[i].level = level;
			if (write_scores() == -1) {
				read_scores();
				update_scores();
			}
			break;
		}
}

show_scores()
{
	int i;

	clrdisp();
	gotoxy(20, 5);
	highlight(1);
	printf("Top Ten Scores for Letter Invaders");
	highlight(0);
	fflush(stdout);

	gotoxy(21, 7); 
	if (HAS_CAP(enter_underline_mode) && HAS_CAP(exit_underline_mode))
		printf(enter_underline_mode);
	printf("  name     level   words   score");
	if (HAS_CAP(enter_underline_mode) && HAS_CAP(exit_underline_mode))
		printf(exit_underline_mode);
	fflush(stdout);

	for (i = 0; i < 10; i++) {
		gotoxy(19, 8 + i);
		printf("%3d %-10s%2d%9d%9d", i + 1, high_scores[i].name,
		       high_scores[i].level, high_scores[i].words,
		       high_scores[i].score);
	}

	printf("\n");
}

/* do non-blocking keyboard reads */
key_pressed()
{
	int chars_read; char keypressed; 

	chars_read = read(0, &keypressed, 1);
	if (chars_read == 1) {
		typed_total++;
		typed_level++;
		return ((int) keypressed);
	}
	return (-1);
}

/* Interrupt handler */
interrupt()
{
	setterm(ORIG);
	gotoxy(0, SCREENLENGTH);
	printf("\nfinal score = %u\n", score);
	printf("Thank you for playing Letter Invaders\n");
	highlight(0);
	exit(1);
}

/* Set the terminal to raw mode, turn off echo and prevent special
 * characters from affecting the input.  We will handle EOF and other fun
 * stuff our own way.  Backup copies of the current setup will be kept
 * to insure the terminal gets returned to its initial state. */
setterm(setting)
int setting;
{
	struct sgttyb termrec;
	static struct sgttyb old_termrec;

	if (setting == NEW) {
		signal(SIGINT, interrupt);

		ioctl(0, TIOCGETP, &termrec);
		old_termrec = termrec;
		termrec.sg_flags &= ~(ECHO | CRMOD);
		termrec.sg_flags |= CBREAK;
		ioctl(0, TIOCSETP, &termrec);

		fcntl(0, F_SETFL, O_NONBLOCK);
		if (HAS_CAP(visible_cursor) && HAS_CAP(invisible_cursor))
			printf(invisible_cursor);
		else printf(CURSOR_OFF);
	} else {
		ioctl(0, TIOCSETP, &old_termrec);
		fcntl(0, F_SETFL, O_ACCMODE);
		if (HAS_CAP(visible_cursor) && HAS_CAP(invisible_cursor))
			printf(visible_cursor);
		else printf(CURSOR_ON);
	}
}

/* move all words down 1 line. */
move()
{
	struct s_word *wordp, *next;
	int died = 0;

	for (wordp = words; wordp != NULL; wordp = next) {
		next = wordp->nextword;
		erase(wordp);
		wordp->posy += wordp->drop;

		if (wordp->posy >= SCREENLENGTH) {
			kill_word(wordp);
			died++;
		} else {
			putword(wordp);
		}
	}
	return died;
}

/* erase a word on the screen by printing the correct number of blanks */
erase(wordp)
struct s_word *wordp;
{
	int i;

	gotoxy(wordp->posx, wordp->posy);
	for (i = 0; i < wordp->length; i++)
		putchar(' ');
}

/* write the word to the screen with already typed letters highlighted */
putword(wordp)
struct s_word *wordp;
{
	int i;

	gotoxy(wordp->posx, wordp->posy);

	/* print the letters in the word that have so far been matched */
	highlight(1);
	for (i = 0; i < wordp->matches; i++)
		putchar(wordp->word[i]);
	highlight(0);

	/* print the rest of the word.  */
	for (i = wordp->matches; i < wordp->length; i++)
		putchar(wordp->word[i]);
}

/* clear the screen and redraw it */
redraw()
{
	clrdisp();
	status();
	fflush(stdout);
}

/* display the status line in inverse video */
status()
{
	int i;
	static char line[80];
	time_t curr_time;

	/* update the characters per minute */
	time(&curr_time);
	if (level_start != curr_time) 
		cpm = 60L * typed_level / (curr_time - level_start);

	sprintf(line, "Score: %-8uLevel: %-8uWords: %-8uLives: %-8dCPM: %-8ld",
		score, level, word_count, lives, cpm);

	/* fill the line with spaces */
	for (i = strlen(line); i < 79; i++)
		line[i] = ' ';

	highlight(1);
	gotoxy(0, SCREENLENGTH);
	fputs(line, stdout);
	highlight(0);
}

/* do stuff to change levels.  This is where special rounds can be stuck in. */
new_level()
{
	struct s_word *next, *wordp;
	time_t curr_time;

	/* if we're inside a bonus round we don't need to change anything
	 * else so just take us out of the bonus round and exit this routine */
	if (bonus == TRUE) {
		bonus = FALSE;
		banner("Bonus round finished");

		/* erase all existing words so we can go back to normal */
		for (wordp = words; wordp != NULL; wordp = next) {
			next = wordp->nextword;
			kill_word(wordp);
		}
		time(&curr_time);
		level_start = curr_time;
		typed_level = 0;
		status();
		return;
	}
	delay = START_DELAY - (level-1) * DELAY_CHANGE;

	/* no one should ever reach a level where there is no delay, but just
	 * to be safe ... */
	if (delay < PAUSE)
		delay = PAUSE;

	levels_played++;
	typed_level = 0;
	time(&curr_time);
	level_start = curr_time;

	/* If you start at a level other than 1, the level does not actually
	 * change until you've completed a number of levels equal to the
	 * starting level. */
	if (level <= levels_played)
		level++;

	if ((levels_played % 3 == 0) && (levels_played != 0)) {
		bonus = TRUE;

		/* erase all existing words so we can have a bonus round */
		for (wordp = words; wordp != NULL; wordp = next) {
			next = wordp->nextword;
			kill_word(wordp);
		}

		banner("Prepare for bonus words");
		lives++;
	}
	status();
}

char *bonusword()
{
	static char buf[BONUSLENGTH + 1];
	int i;

	for (i = 0; i < BONUSLENGTH; i++)
		buf[i] = (char) (rand() % 94) + 33;

	buf[BONUSLENGTH] = 0;

	return buf;
}

/* find a random word in a dictionary file as part of letters.  */
char *getword()
{
	char buf[80];
	long foo;
	static struct stat s_buf;
	static FILE *fp = NULL;

	/* This is stuff that only needs to get done once.  */
	if (fp == NULL) {
		/* open the dictionary file */
		if ((fp = fopen(dictfile, "r")) == NULL) {
			setterm(ORIG);
			fprintf(stderr, "\ncan't open file: %s.\n", dictfile);
			exit(1);
		}
		/* Get length of dictionary in bytes so we can pick a random
		 * entry in it.  */
		if (stat(dictfile, &s_buf) == -1) {
			perror("stat");
			exit(1);
		}
	}
	/* pick a random place in the large dictionary */
	fseek(fp, ((long) rand() * (long) rand()) % s_buf.st_size, 0);

	/* read until the end of a line, then read the next word.  */
	fscanf(fp, "%s%s", buf, buf);

	/* Since we're reading two words at a time it's possible to go past
	 * the end of the file.  If that happens, use the first word in the
	 * dictionary. */
	if (buf == NULL) {
		fseek(fp, 0L, 0);
		fscanf(fp, "%s", buf);
	}
	return buf;
}

/* allocate memory for a new word and get it all set up to use. */
struct s_word *newword(wordp)
struct s_word *wordp;
{
	struct s_word *nword;
	char *word, *getword(), *bonusword();
	int length;

	if (bonus == TRUE)
		word = bonusword();
	else
		word = getword();

	length = strlen(word);

	nword = (struct s_word *) malloc(sizeof(struct s_word) + length);
	if (nword == (struct s_word *) 0) {
		perror("\nmalloc");
		setterm(ORIG);
		exit(1);
	}
	strncpy(nword->word, word, length);
	nword->length = length;
	nword->drop = length > 6 ? 1 : length > 3 ? 2 : 3;
	nword->matches = 0;
	nword->posx = rand() % ((SCREENWIDTH - 1) - nword->length);
	nword->posy = 0;
	nword->nextword = NULL;

	if (wordp != NULL)
		wordp->nextword = nword;

	return nword;
}

/* look at the first characters in each of the words to find one which
 * one matches the amount of stuff typed so far */
struct s_word *searchstr(key, str, len)
char key, *str;
int len;
{
	struct s_word *wordp, *best;

	for (best = NULL, prev_word = NULL, wordp = words;
	     wordp != NULL;
	     prev_word = wordp, wordp = wordp->nextword) {
		if (wordp->length > len
		    && strncmp(wordp->word, str, len) == 0
		    && wordp->word[len] == key
		    && (!best || best->posy < wordp->posy))
			best = wordp;
	}

	return best;
}

/* look at the first character in each of the words to see if any match the
 * one that was typed. */
struct s_word *searchchar(key)
char key;
{
	struct s_word *wordp, *best;

	for (best = NULL, prev_word = NULL, wordp = words; wordp != NULL;
	     prev_word = wordp, wordp = wordp->nextword) {
		if (wordp->word[0] == key
		    && (!best || best->posy < wordp->posy))
			best = wordp;
	}
	return best;
}

kill_word(wordp)
struct s_word *wordp;
{
	struct s_word *temp, *prev = NULL;

	/* check to see if the current word is the first one on our list */
	if (wordp != words)
		for (prev = words, temp = words->nextword; temp != wordp;) {
			prev = temp;
			temp = temp->nextword;
		}

	if (prev != NULL) {
		prev->nextword = wordp->nextword;
	} else
		words = wordp->nextword;

	if (wordp->nextword != NULL)
		wordp->nextword = wordp->nextword->nextword;

	if (wordp == lastword)
		lastword = prev;

	free((char *) wordp);
}

/* momentarily display a banner message across the screen and eliminate any
 * random keystrokes from the last round */
banner(text)
char *text;
{
	/* display banner message */
	clrdisp();
	gotoxy((SCREENWIDTH - strlen(text)) / 2, 10);
	puts(text);
	fflush(stdout);
	sleep(1);
	clrdisp();

	/* flush keyboard */
	while (key_pressed() != -1);
}

/* used to get the actual terminal control string.  */
opt_cap(cap, buf)
char *cap, *buf;
{
	char *tgetstr();

	*buf = (char) NULL;
	return tgetstr(cap, &buf) != NULL;
}

/* call opt_cap to get control string.  report if the terminal lacks that
 * capability. */
get_cap(cap, buf)
char *cap, *buf;
{
	if (!opt_cap(cap, buf))
		fprintf(stderr, "TERMCAP entry for %s has no '%s' capability\n",
			term_name, cap);
}

/* set everything up. find the necessary strings to control the terminal. */
init_term()
{
	char tbuf[1024];

	/* get terminal type from the environment or have the user enter it */
	if ((term_name = (char *) getenv("TERM")) == NULL) {
		fprintf(stderr, "No TERM variable in environment\n");
		fprintf(stderr, "Enter terminal type to use: ");
		scanf("%s", term_name = (char *) malloc(30 * sizeof(char)));
	}

	/* get the termcap entry for the terminal above */
	if (tgetent(tbuf, term_name) <= 0) {
		fprintf(stderr, "Unknown terminal type: %s\n", term_name);
		exit(1);
	}

	get_cap("cm", crsr_address);
	if (!opt_cap("ho", crsr_home))
		strcpy(crsr_home, tgoto(crsr_address, 0, 0));

	get_cap("cl", clear_screen);

	Lines = tgetnum("li");

	get_cap("vi", invisible_cursor);
	get_cap("ve", visible_cursor);
	opt_cap("so", enter_standout_mode);
	opt_cap("se", exit_standout_mode);
	opt_cap("us", enter_underline_mode);
	opt_cap("ue", exit_underline_mode);
	
}

highlight(on)
int on;
{
	if (HAS_CAP(enter_standout_mode) && HAS_CAP(exit_standout_mode))
		putp(on ? enter_standout_mode : exit_standout_mode);
}

/* Here's the main routine of the actual game.  */
int game()
{
	char key;
	long i;
	int died;
	struct s_word *curr_word, *temp_word;

	/* look to see if we already have a partial match, if not set the
	 * current word pointer to the first word */
	for (curr_word = words; curr_word; curr_word = curr_word->nextword)
		if (curr_word->matches > 0)
			break;
	if (!curr_word)
		curr_word = words;

	while (curr_word->matches < curr_word->length) {
		for (i = 0; i < delay; i += PAUSE) {
			while ((curr_word->matches != curr_word->length) &&
			       ((key = key_pressed()) != -1)) {
				if (key == CTRL('L')) {
					redraw();
					continue;
				}
				if (key == CTRL('N')) {
					level++;
					delay = START_DELAY - (level-1) * DELAY_CHANGE;
					status();
					continue;
				}

				/* if a word has already been started. */
				if (curr_word->matches > 0) {
					/* does character match? */
					if (key == curr_word->word[curr_word->matches]) {
						gotoxy(curr_word->posx + curr_word->matches, curr_word->posy);
						highlight(1);
						putchar(curr_word->word[curr_word->matches]);
						highlight(0);
						gotoxy(SCREENWIDTH, SCREENLENGTH);
						fflush(stdout);
						curr_word->matches++;
						/* fill the word with characters to
						 * "explode" it. */
						if (curr_word->matches >= curr_word->length)
							for (i = 0; i < curr_word->length; i++)
								curr_word->word[(int) i] = '-';
						continue;
					/* is there another word matching the string typed sofar */
					} else if (temp_word = searchstr(key, curr_word->word, curr_word->matches)) {
						erase(temp_word);
						temp_word->matches = curr_word->matches;
						curr_word->matches = 0;
						putword(curr_word);
						curr_word = temp_word;
						curr_word->matches++;
					/* if not, then it is wrong */
					} else {
						bell();
						curr_word->matches = 0;
						err_total++;
					}
				/* No word was started yet, is there one starting with this key */
				} else if (temp_word = searchchar(key)) {
					erase(temp_word);
					curr_word->matches = 0;
					putword(curr_word);
					curr_word = temp_word;
					curr_word->matches++;
				/* if not then it is wrong */
				} else {
					bell();
					curr_word->matches = 0;
					err_total++;
				}
				erase(curr_word);
				putword(curr_word);
				gotoxy(SCREENWIDTH, SCREENLENGTH);
				fflush(stdout);
			}
  		}

		if ((rand() % ADDWORD) == 0) 
			lastword = newword(lastword);

		died = move();	/* NB: move may invalidate curr_word */

		if (died > 0) {
			if (bonus == TRUE) new_level();
			else lives -= died;

			if (lives < 0) lives = 0;
			status();
			gotoxy(SCREENWIDTH, SCREENLENGTH);
			fflush(stdout);
			if (lives <= 0)
				return 0; /* End */
			else {
				while (key_pressed() != -1) /**/ ;
				return 1; /* Go On */
			}
		} else {
			gotoxy(SCREENWIDTH, SCREENLENGTH);
			fflush(stdout);
		}
	}

	/* all letters in the word have been correctly typed. */

	/* erase the word */
	if (curr_word->length == curr_word->matches) {
		ding();
		erase(curr_word);
	}
	/* add on an appropriate score. */
	score += curr_word->length + (2 * level);
	word_count++;
	status();

	/* delete the completed word and revise pointers. */
	kill_word(curr_word);

	/* increment the level if it's time. */
	if (word_count % LEVEL_CHANGE == 0)
		new_level();

	return 1; /* Go On */
}

main(argc, argv)
int argc;
char *argv[];
{
	time_t starttime, endtime;
	extern char *optarg;
	int c;

	/* default bell sound */
	ding = quiet;

	/* initialize display stuff */
	init_term();		/* termcap stuff */

	/* check for options */
	while ((c = getopt(argc, argv, "f:qbhl:")) != EOF) {
		switch (c) {
		case 'f':
			strcpy(dictfile, optarg);
			break;
		case 'q':
			ding = quiet;
			break;
		case 'b':
			ding = bell;
			break;
		case 'h':
			read_scores();
			show_scores();
			exit(0);
			break;
		case 'l':
			level = atoi(optarg);
			if (START_DELAY-(level-1)*DELAY_CHANGE < PAUSE ) {
				fprintf(stderr, "You may not start at level %d\n", level);
				exit(0);
			}
			break;
		default:
			fprintf(stderr, "Usage: %s [-hqb] [-l startlevel] [-f dictionary]\n", argv[0]);
			exit(0);
		}
	}

	/* get stuff initialized */
	setterm(NEW);		/* signal stuff, keyboard stuff */
	srand(getpid());
	clrdisp();
	new_level();
	status();
	words = NULL;
	time(&starttime);

	for (;;) {
		/* allocate memory for the first word and then find a word if
		 * there are no others on the screen.  There must always be
		 * at least one word active. */
		if (words == NULL) {
			lastword = words = newword((struct s_word *) NULL);
			prev_word = NULL;
		}
		if (game() == 0) {
			/* all finished.  print score and clean up. */
			time(&endtime);
			gotoxy(0, SCREENLENGTH);
			fflush(stdout);
			setterm(ORIG);
			putchar('\n');
			break;
		}
	}

	read_scores();
	update_scores();
	show_scores();

	printf("\nfinal score: %u points\t", score);
	printf("(%ld CPM", 60L * typed_total / (endtime - starttime));
	if (typed_total) {
		printf(", %ld%% accurate", 
			100L * (typed_total - err_total) / typed_total);
	}
	printf(")\n");
	exit(0);
}
