/*
 * bill		Create a bill for network users.
 *		This file reads the MUGNET Resource Accounting record
 *		files, and creates bill messages of them, which can be
 *		mailed to the users of the MUGNET network.
 *
 * Usage:	bill [-d date] [-n hostname] [-v] [admin_file]
 *
 * Version:	1.00	03/30/91
 *
 * Author:	Fred N. van Kempen, MUGNET Technical Manager
 */
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


#define	NEWS	"/usr/lib/news/news.adm"		/* NEWS accounting */
#define MAIL	"/usr/lib/uucp/umail.adm"		/* MAIL accounting */


typedef struct _rec_ {
  struct _rec_	*next;
  char		*host;
  char		*dist;
  long		recv;
  long		sent;
} REC;
#define NILREC	((REC *)NULL)
#define RECSIZE	(sizeof(REC))


static char *Version = "@(#) bill 1.00 (03/30/91)";


char *opt_date = (char *)NULL;			/* date specified?	*/
char *opt_host = (char *)NULL;			/* host specified?	*/
int opt_v = 0;					/* verbose?		*/
REC *reclist = NILREC;				/* acounting list	*/


extern int getopt(), optind, opterr;
extern char *optarg;
extern long atol();


/* Find the record of a certain host. */
REC *rec_find(who)
char *who;
{
  register REC *rp;

  rp = reclist;
  while (rp != NILREC) {
	if (!strcmp(rp->host, who)) return(rp);
	rp = rp->next;
  }
  return(NILREC);
}


/* Create a new record in the tree. */
REC *rec_add(who)
char *who;
{
  register REC *rp;

  if (reclist == NILREC) {
	reclist = (REC *)malloc(RECSIZE);
	rp = reclist;
  } else {
	rp = reclist;
	while (rp->next != NILREC) rp = rp->next;
	rp->next = (REC *)malloc(RECSIZE);
	rp = rp->next;
  }
  if (rp == NILREC) {
	fprintf(stderr, "bill: out of memory!\n");
	return(NILREC);
  }
  rp->host = (char *)malloc(strlen(who) + 1);
  if (rp->host == (char *)NULL) {
	fprintf(stderr, "bill: out of memory!\n");
	return(NILREC);
  }
  strcpy(rp->host, who);
  rp->sent = (long) 0;
  rp->recv = (long) 0;
  rp->next = NILREC;
  return(rp);
}


/* Update the record of a certain host. */
void rec_upd(who, code, length, dist)
char *who;
char *code;
char *length;
char *dist;
{
  register REC *rp;

  if ((rp = rec_find(who)) == NILREC) {
	rp = rec_add(who);
	if (rp == NILREC) return;
  }

  if (!strcmp(code, "sent")) rp->sent += (long) atol(length);
    else if (!strcmp(code, "rcvd")) rp->recv += (long) atol(length);
    else fprintf(stderr, "bill: bad operation code \"%s\"\n", code);
}


/* Update all hosts affected by this record. */
void upd_host(host, code, length, dist)
char *host;
char *code;
char *length;
char *dist;
{
  register char *sp;

  sp = host;
  while (*sp != '\0') {
	while (*sp != '\0' && *sp != ' ') sp++;
	if (*sp != '\0') *sp++ = '\0';

	rec_upd(host, code, length, dist);

	host = sp;
  }
}


/* Read and process the NEWS accounting file. */
void do_news(fname)
char *fname;
{
  char buff[1024];
  register char *sp, *bp;
  register FILE *fp;
  char *host, *dist, *code, *length;
  long bytes;

  printf("  - reading NEWS accounts...\n");

  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "bill: cannot open %s\n", fname);
	return;
  }

  /* "date <space> time <tab> code <tab> dist <tab> length <tab> host \n" */
  while (fgets(buff, 1024, fp) != (char *)NULL) {
	if (buff[0] == '#' || buff[0] == '\n') continue;
	sp = buff;
	while(*sp != '\n' && *sp != '\t') sp++;
	*sp++ = '\0';
	if (opt_date != (char *)NULL) {
		if (strstr(buff, opt_date) == (char *)NULL) continue;
	}
	code = sp;
	while (*sp != '\n' && *sp != '\t') sp++;
	*sp++ = '\0';
	dist = sp;
	while (*sp != '\n' && *sp != '\t') sp++;
	*sp++ = '\0';
	length = sp;
	while (*sp != '\n' && *sp != '\t') sp++;
	*sp++ = '\0';
	host = sp;
	while (*sp != '\n') sp++;
	*sp = '\0';

	upd_host(host, code, length, dist);
  }
  fclose(fp);
}


/* Read and process the MAIL accounting file. */
void do_mail(fname)
char *fname;
{
  printf("  - reading MAIL accounts...\n");
}


/* Display a nice summary of all accounts. */
void do_summary()
{
  register FILE *fp;
  register REC *rp;
  long totsent, totrecv;

  printf("  - creating summary...\n");

  if ((fp = fopen("admin.sum", "w")) == (FILE *)NULL) fp = stdout;

  totsent = (long) 0;
  totrecv = (long) 0;
  rp = reclist;
  if (rp != NILREC) {
	fprintf(fp, "HOST        SENT        RECEIVED\n");
	while (rp != NILREC) {
		fprintf(fp, "%-8.8s    %8.8ld    %8.8ld\n",
			rp->host, (long) rp->sent, (long) rp->recv);
		totsent += rp->sent;
		totrecv += rp->recv;
		rp = rp->next;
	}
	fprintf(fp, "--------------------------------\n");
	fprintf(fp, "TOTAL     %10.10ld  %10.10ld\n", totsent, totrecv);

  } else fprintf(fp, "No records found.\n");

}


void usage()
{
  fprintf("Usage: bill [-v] [-n hostname] [-d date] [admin_file]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  register int c;
  register char *sp;

  printf("MUGNET Administrator - Biller\n\n");

  opterr = 0;
  while ((c = getopt(argc, argv, "d:n:v")) != EOF) switch(c) {
	case 'd':
		opt_date = optarg;
		break;
	case 'n':
		opt_host = optarg;
		break;
	case 'v':
		opt_v = 1;
		break;
	default:
		usage();
  }

  /* At most one more argument allowed. */
  sp = NEWS;
  if (optind != argc) {
	argc--;
	if (optind != argc) usage();
	sp = argv[optind];
  } else usage();

  do_news(sp);

  do_mail(MAIL);

  do_summary();
}
