/*
 * W-NEWS	A simple NEWS processing package for MINIX.
 *		The 'sendbatch' program is used to create new batches that
 *		can be transported to other systems via the UUCP network.
 *		It gathers all files in the OUTGOING queue, and creates
 *		batches of up to LIM bytes.  Optionally, the generated batch
 *		is compressed using the 'compress(1)' program.  The final
 *		result will be handed to 'uux(1)' for transportation.
 *
 * Usage:	sendbatch [-c] [-d] [-s maxsize] sysname
 *
 * Version:	3.00	03/30/91
 *
 * Author:	Fred N. van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include "wnews.h"


static char *Version = "@(#) sendbatch 3.00 (03/30/91)";


char *lgmonths[] = {			/* for debugging purposes	*/
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
char remote[128];			/* name of remote system	*/
char batchname[128];			/* path name of current batch	*/
int debug = 0;				/* debugging flag		*/
int opt_c = 0;				/* compress after generation	*/
long maxsize = (long) BATCHLIM;		/* maximum size of a batch	*/
long cursize = (long) 0;		/* current size of batch	*/
FILE *bfp = (FILE *)NULL;		/* FILE ptr of current batch	*/


extern int getopt(), optind, opterr;	/* the from getopt(3) package	*/
extern char *optarg;
extern FILE *popen();			/* should be in stdlib.h !	*/


/* Write something to the log files. */
void logmsg(err, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int err;
char *fmt;
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  time_t now;

  sprintf(lfn, "%s/%s", LIBDIR, LOGFILE);
  sprintf(efn, "%s/%s", LIBDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "sendbatch: cannot append(%s)\n", efn);
	return;
  }
  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "sendbatch: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }
  time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02.2d %02.2d:%02.2d  ", lgmonths[mydate->tm_mon],
			mydate->tm_mday, mydate->tm_hour, mydate->tm_min);
  if (err == 1) {
	fprintf(efp, buff);
	fprintf(efp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  } else {
	fprintf(lfp, buff);
	fprintf(lfp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	fprintf(stderr, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/* Send a completed, compressed news batch to a host via UUCP. */
int send_it(name)
char *name;
{
  char buff[256];
  char *args[8];
  struct stat stb;
  int i, pid, status;
  register char *sp;

  if (stat(name, &stb) < 0) {
	logmsg(1, "sendbatch: cannot stat(%s)\n", name);
	return(-1);
  }
  if (stb.st_size == 0L) return(0);	/* nothing to do! */

  sprintf(buff, "%s %s!rnews", UUX, remote);
  i = 0;
  sp = buff;
  do {
	args[i++] = sp;
	while (*sp && *sp != ' ' && *sp != '\t') sp++;
	if (*sp == ' ' || *sp == '\t') *sp++ = '\0';	
  } while (*sp != '\0');
  args[i] = (char *)NULL;

  if ((pid = fork()) == 0) {
	(void) close(0);
	(void) open(batchname, O_RDONLY);
	i = execv(args[0], args);
	exit(i);
  } else if (pid > 0) {
	while (wait(&status) != pid)
		    ;
  }
  if (pid < 0 || WEXITSTATUS(status) != 0) {
	logmsg(1, "sendbatch: cannot exec(%s,%s)\n", args[0], args[1]);
	return(-1);
  }
  return(0);
}


/* Create a batch of news articles. */
int cr_batch()
{
  strcpy(batchname, "/tmp/sb.XXXXXX");
  mktemp(batchname);

  if ((bfp = fopen(batchname, "w")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot create(%s)\n", batchname);
	return(-1);
  }
  cursize = 0L;
  return(0);
}


/*
 * Close up the current batch, and (optionally) run it through compress(1).
 * Then, ask UUX to send it to the UUCP host.
 */
int cl_batch()
{
  char buff[128];
  char *args[8];
  int i, pid, status;

  (void) fflush(bfp);
  (void) fclose(bfp);
  if (opt_c == 1) {
	strcpy(buff, "/tmp/sbZ.XXXXXX");
	mktemp(buff);
	args[0] = COMPRESS;
	args[1] = "-b13";
	args[2] = (char *)NULL;
	if ((pid = fork()) == 0) {
		(void) close(0);
		(void) close(1);
		(void) open(batchname, O_RDONLY);
		(void) unlink(buff);
		(void) creat(buff, 0644);
		(void) close(2);
		(void) open("/dev/null", O_RDWR);
		(void) write(1, COMPRHDR, strlen(COMPRHDR));
		i = execv(args[0], args);
		exit(i);
	} else if (pid > 0) {
		while (wait(&status) != pid)
			    ;
	}
	if (pid < 0 || ((status = WEXITSTATUS(status)) != 0 && status != 2)) {
		logmsg(1, "sendbatch: cannot exec(%s,%s)\n",
						args[0], args[1]);
		(void) unlink(buff);
		return(-1);
	}
	/* If compression rate was to low, compress exits with status 2 */
	if (status != 2) {
		(void) unlink(batchname);
		strcpy(batchname, buff);
	} else {
		logmsg(0, "sendbatch: compression rate to low, sent uncompressed\n");
	}
  }

  if ((status = send_it(batchname)) == 0) (void) unlink(batchname);
  return(status);
}


/* Add an article to the current batch. */
int add_art(name)
char *name;
{
  char buff[1024];
  char temp[128];
  struct stat stb;
  register FILE *fp;
  register int len;

  /* Fetch some file info. */
  if (stat(name, &stb) < 0) {
	logmsg(1, "sendbatch: cannot stat(%s)\n", name);
	return(0);
  }
  if ((fp = fopen(name, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", name);
	return(0);
  }

  /*
   * Check to see if the article still fits in the current batch.
   * Note, that it is FORCED to fit if the batch is stiil empty.
   * This is needed to prevent the program from looping over an
   * article that is larger than 'maxsize' bytes...
   */
  if (cursize > 0L && (cursize + stb.st_size) > maxsize) {
	if (cl_batch() < 0) {
		(void) fclose(fp);
		return(-1);
	}
	if (cr_batch() < 0) {
		(void) fclose(fp);
		return(-1);
	}
  }

  /* Add size of this article to current size. */
  cursize += stb.st_size;

  /* Create an article-separation header. */
  fprintf(bfp, "%s%ld\n", RNEWSHDR, (long) stb.st_size);

  /* Copy the article into the batch. */
  do {
	len = fread(buff, sizeof(char), 1024, fp);
	fwrite(buff, sizeof(char), len, bfp);
  } while (len > 0);
  (void) fclose(fp);

  return(0);
}


/* Scan the batch list file for articles, and add them to the batch. */
int mk_batch(who)
char *who;
{
  char fname[128], temp[128];
  char buff[1024];
  FILE *fp;
  register char *sp;
  register int i;

  strcpy(remote, who);

  sprintf(fname, "%s/%s", LIBDIR, SYSFILE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", fname);
	return(-1);
  }
  sp = (char *)NULL;
  do {
	if (fgets(buff, 1024, fp) == (char *)NULL) break;
	if (buff[0] == '0' || buff[0] == '\n' || buff[0] == '#') continue;
	if (!strncmp(buff, who, strlen(who))) sp = buff;
  } while (sp == (char *)NULL);
  (void) fclose(fp);

  if (sp == (char *)NULL) {
	logmsg(1, "sendbatch: unknown target %s\n", who);
	return(-1);
  }
  if ((sp = strrchr(buff, ':')) == (char *)NULL) {
	logmsg(1, "sendbatch: \"sys\" file format error\n");
	return(-1);
  }
  if (*++sp == '\n') sprintf(fname, "%s/%s", BATCHDIR, who);
    else strcpy(fname, sp);
  if ((sp = strchr(fname, '\n')) != (char *)NULL) *sp = '\0';
  sprintf(temp, "%s.work", fname);

  /* Does a batch file exist? */
  if (access(fname, 0) < 0) return(0);

  /* Does a work file exist? */
  if (access(temp, 0) < 0) {
	/* Try to link the batch list to a "work file". */
	i = 0;
	while (i < 10) {
		if (link(fname, temp) == 0) break;
		if (++i >= 10) {
			logmsg(1, "sendbatch: cannot link(%s,%s)\n",
							fname, temp);
			return(-1);
		}
		sleep(6);
	}
	(void) unlink(fname);
  }

  /* Open the batch list file. */
  if ((fp = fopen(temp, "r")) == (FILE *)NULL) {
	logmsg(1, "sendbatch: cannot open(%s)\n", temp);
	return(-1);
  }
  logmsg(0, "sendbatch: checking files for %s in %s\n", who, fname);

  /* Create the news batch. */
  if (cr_batch() < 0) {
	(void) fclose(fp);
	return(-1);
  }

  /* Now read article names from the list file. */
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;

	if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
	if (add_art(buff) < 0) {
		(void) fclose(fp);
		return(-1);
	}
  }
  (void) fclose(fp);

  /* Close up any pending batch file. */
  if (cursize > 0L) {
	if (cl_batch() < 0) return(-1);
  }
  (void) unlink(temp);
  return(0);
}


void usage()
{
  fprintf(stderr, "Usage: sendbatch [-c] [-d] [-s maxsize] sysname\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char name[128];
  register int i;

  opterr = 0;
  while ((i = getopt(argc, argv, "cds:")) != EOF) switch(i) {
	case 'c':
		opt_c = 1;
		break;
	case 'd':
		debug = 1;
		break;
	case 's':
		maxsize = atol(optarg);
		break;
	default:
		usage();
  }

  /* At lest one more argument needed. */
  if (optind >= argc) usage();

  (void) umask(033);			/* set protection to rw-r--r-- */
  if (opt_c == 1) maxsize *= 2;		/* adjust 'compressed' batch size */

  /* Check if we are already (or still?) running. */
  sprintf(name, "%s/%s", LIBDIR, BATLOCK);
  if (access(name, 0) == 0) exit(0);

  /* No, we are not yet busy. */
  (void) creat(name, 0);

  i = 0;
  while (optind < argc) {
	if (mk_batch(argv[optind++]) < 0) i = -1;
  }

  /* Remove the lockfile. */
  (void) unlink(name);

  exit(i);
}
