/*
 * W-NEWS	A simple NEWS processing package for MINIX.
 *		This program takes care of the 'user-interface' part
 *		of posting articles to a newsgroup.  It actually is
 *		a message composer (derived from the W-MAIL Message
 *		Editor), which hands the finished article to 'inews'
 *		for the actual posting.
 *
 * Usage:	pnews [-d] [ newsgroup ]
 *
 * Version:	3.00	03/30/91
 *
 * Author:	Fred N. van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <ctype.h>
#include <fcntl.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include "wnews.h"


typedef struct _distr_ {
  struct _distr_ *next;
  char		 *dist;
  char		 *name;
} DISTRIB;
#define NILDIST ((DISTRIB *)NULL)
#define DISTSIZ (sizeof(DISTRIB))


static char *Version = "@(#) pnews 3.00 (03/30/91)";


char artname[128];			/* path name of article file	*/
FILE *artfp;				/* file pointer of article file	*/
int debug = 0;				/* debugging flag		*/

char subject[128];			/* subject of the article	*/
char keywords[128];			/* some useful keywords		*/
char newsgroups[128];			/* the addressed newsgroups	*/
char distribution[128];			/* article's distribution	*/
char *dflt_newsg;			/* Given as argument		*/
char *e_edit;				/* line editor			*/
char *v_edit;				/* screen editor		*/
char *shell;				/* shell to use			*/
DISTRIB *ldistrib = NILDIST;		/* a list of possible distr's	*/


extern int getopt(), optind, opterr;	/* from getopt(3) package	*/
extern char *optarg;


/*
 * Fetch the contents of file 'name' and copy them to the 'tempfp' file.
 * If 'quote' is 0, prefix each copied line with "> ".  Also note, that
 * we *must* quote any line beginning with "From_".
 */
void fetch_file(name, quote, nchar, nlin)
char *name;
int quote;
off_t *nchar;
int *nlin;
{
  char buff[1024];
  register char *bp;
  off_t nchars, curr, limit;
  int nlines;
  register FILE *ifp;

  nchars = 0L;
  nlines = 0;

  if ((ifp = fopen(name, "r")) != (FILE *)NULL) {
	nchars = 0L;
	nlines = 0;
	while (fgets(buff, 1024, ifp) != (char *)NULL) {
		if (quote == 1 || !strncmp(buff, "From ", 5))
					fprintf(artfp, "> %s", buff);
		  else fprintf(artfp, "%s", buff);
		nchars += (off_t) strlen(buff);
		nlines++;
	}
	(void) fclose(ifp);
  } else fprintf(stderr, ": cannot open file \"%s\"\n", name);
  *nchar = nchars;
  *nlin = nlines;
}


/* Add the ".signature" file to the current message. */
void app_signature()
{
  char fname[128];
  char buff[1024];
  struct passwd *pw;
  FILE *fp;
  char *sp;

  sp = getenv("SIGNATURE");
  if (sp == (char *)NULL) {
	pw = getpwuid(getuid());
	if (pw == (struct passwd *)NULL) {
		fprintf(stderr, "pnews: cannot get home directory!\n");
		return;
	}
	sprintf(fname, "%s/%s", pw->pw_dir, SIGNATURE);
  } else strcpy(fname, sp);

  if ((fp = fopen(fname, "r")) != (FILE *)NULL) {
	while (fgets(buff, 1024, fp) != (char *)NULL) {
		fprintf(artfp, "%s", buff);
	}
	(void) fclose(fp);
  }
}


/* Run a program, while redirecting input if desired. */
int do_run(input, prg, args)
char *input;
char *prg;
char *args[];
{
  int status, pid, fd;

  if (*input != '\0') {
	if ((fd = open(input, O_RDONLY)) < 0) {
		fprintf(stderr, "pnews: couldn't re-open temp article!\n");
		return(-1);
	}
  } else fd = 0;
  args[0] = prg;
  if ((pid = fork()) == 0) {
	if (fd != 0) {
		(void) close(0);
		(void) dup(fd);
	}
	(void) execv(prg, args);
	fprintf(stderr, "pnews: couldn't execute %s\n", prg);
	exit(-1);
  } else if (pid > 0) {
	if (fd != 0) (void) close(fd);
	while (wait(&status) != pid)
		    ;
  }
  return(WEXITSTATUS(status));
}


/* 
 * Save the current article to file 'dead.article'.
 *
 * If the environmental variable "DEADART" is defined,
 * use its value as the file name. Otherwise, use DEADART
 * in the user's home directory.
 */
void dead_art()
{
  char buff[1024];
  char fname[128];
  struct passwd *pw;
  register FILE *fp;
  char *sp;
  char *home;

  if ((sp = getenv("DEADART")) == (char *)NULL) {
	if ((home = getenv("HOME")) == (char *)NULL) {
		if ((pw = getpwuid(getuid())) == (struct passwd *)NULL) {
			fprintf(stderr,
				"pnews: cannot get home directory!\n");
			return;
		}
		home = pw->pw_dir;    
	}    
	sprintf(fname, "%s/%s", home, DEADART);
  } else strcpy(fname, sp);

  if ((fp = fopen(fname, "w")) != (FILE *)NULL) {
	artfp = fopen(artname, "r");
	while (fgets(buff, 1024, artfp) != (char *)NULL) {
		fprintf(fp, "%s", buff);
	}
	(void) fclose(fp);
  }
  fprintf(stderr, "pnews: dumped article on file %s\n", fname);
}


/*
 * Perform a "tilde" action, like in "W-MAIL" style.
 * Return 1 if the article should not be sent after all.
 */
int do_tilde(text)
char *text;
{
  char *args[4];
  char buff[16];
  register char *sp;
  int nlin;
  off_t nchar;

  /* First off, kill the newline. */
  sp = strchr(text, '\n');
  if (sp != (char *)NULL) *sp = '\0';

  switch(*text) {
	case '?': 	/* editor help */
	case 'h':
		printf("\n  ** W-NEWS PostNews Article Editor Help **\n\n");
		printf("~?\t\tThis Help Screen\n");
		printf("~!\t\tEscape into Shell\n");
		printf("~a\t\tAppend .signature to article\n");
		printf("~e\t\tInvoke line editor on article\n");
		printf("~h\t\tSame as ? (Help Screen)\n");
		printf("~q\t\tQuit (with save) sending this article\n");
		printf("~r filename\tInsert file into article\n");
		printf("~R filename\tInsert quoted file into article\n");
		printf("~v\t\tInvoke visual editor on article\n");
		printf("~x\t\tExit (abort) sending this article\n");
		printf("\n");
		break;
	case '!':	/* shell escape */
		sp = ++text;
		while (*sp == ' ' || *sp == '\t') sp++;
		if (*sp != '\0') {
			args[1] = "-c";
			args[2] = sp;
			args[3] = (char *)NULL;
		} else 	args[1] = (char *)NULL;
		do_run("", shell, args);
		printf("\n[continue]\n");
		break;
	case 'a':	/* append SIGNATURE */
		app_signature();
		printf("[continue]\n");
		break;
	case 'e':	/* invoke Line Editor (ED) */
		(void) fclose(artfp);
		args[1] = artname;
		args[2] = (char *)NULL;
		do_run("", e_edit, args);
		artfp = fopen(artname, "a");
		printf("[continue]\n");
		break;
	case 'q':	/* quit (abort) sending article (save article) */
		(void) fclose(artfp);
		dead_art();
		/* FALL_THROUGH */
	case 'x':	/* exit this article (no save) */
		return(1);
	case 'r':	/* read and insert file */
		text++;
		while (*text == ' ' || *text == '\t') text++;
		fprintf(stderr, "\"%s\"", text);
		fetch_file(text, 0, &nchar, &nlin);
		fprintf(stderr, " %d/%ld\n", nlin, (long) nchar);
		printf("[continue]\n");
		break;
	case 'R':	/* read and insert quoted file */
		text++;
		while (*text == ' ' || *text == '\t') text++;
		fprintf(stderr, "\"%s\"", text);
		fetch_file(text, 1, &nchar, &nlin);
		fprintf(stderr, " %d/%ld\n", nlin, (long) nchar);
		printf("[continue]\n");
		break;
	case 'v':	/* invoke Visual Editor (VI) */
		(void) fclose(artfp);
		args[1] = artname;
		args[2] = (char *)NULL;
		do_run("", v_edit, args);
		artfp = fopen(artname, "a");
		printf("[continue]\n");
		break;
	default:
		printf("*** Unknown escape.  Please use ~? for help.\n");
		break;
  }
  return(0);
}


/* Test if newsgroup exists, by testing if its directory is present. */
int testgroup(s)
char *s;
{
  char buf[256];
  char subdir[256];
  char *p;
  struct stat st;
  
  if (*s == '\0') return(0);

  strcpy(subdir, s);
  for(p = subdir; *p; p++)
  	if (*p == '.') *p = '/';

  sprintf(buf, "%s/%s", SPOOLDIR, subdir);
  if (stat(buf, &st) < 0 || (st.st_mode & S_IFMT) != S_IFDIR) {
	fprintf(stderr, "pnews: unknown newsgroup %s\n", s);
  	return(-1);
  }
  return(0);
}


/* Check if a distribution is legal, and if it is, get some info. */
char *legal(dist)
char *dist;
{
  register DISTRIB *dp;

  dp = ldistrib;
  while (dp != NILDIST) {
	if (!strcmp(dp->dist, dist)) return(dp->name);
	dp = dp->next;
  }

  printf("Possible distributions are:\n");
  dp = ldistrib;
  while (dp != NILDIST) {
	printf(" %-16.16s - %s\n", dp->dist, dp->name);
	dp = dp->next;
  }
  return((char *)NULL);
}


/*
 * Create a article, including the Article-header and the final
 * signature.  This routine copies lines of text from 'stdin' to
 * 'tempfp', until an EOF is typed, or a line containing only a
 *  '.'.  Return 0 if all is OK, < 0 for errors or 1 for abort.
 */
int edit_art(infp)
FILE *infp;
{
  char buff[1024];
  register char *sp;
  char *dest;
  int c;

  /* Create the temporary article. */
  if ((artfp = fopen(artname, "w")) == (FILE *)NULL) {
	fprintf(stderr, "pnews: cannot create temp file %s\n", artname);
	return(-1);
  }

  /* An Article Header must contain some vital information.  Ask for it. */
  subject[0] = '\0';
  do {	/* we MUST get a Subject: line !!! */
	fprintf(stderr, "Subject: ");
	gets(subject);
  } while (subject[0] == '\0');

  fprintf(stderr, "Keywords: ");
  gets(keywords);

  fprintf(stderr, "Newsgroups [%s]:\n", dflt_newsg);
  strcpy(newsgroups, "");
  c = 0;
  do {
	fprintf(stderr, "> ");
	gets(buff);
	if (testgroup(buff) < 0) continue;
	if (buff[0] != '\0') {
		strcat(newsgroups, buff);
		strcat(newsgroups, ",");
	} else if (c == 0) {
		strcpy(newsgroups, dflt_newsg);
		strcat(newsgroups, ",");
	}
	c++;
  } while (buff[0] != '\0');
  if (newsgroups[0] != '\0') newsgroups[strlen(newsgroups) - 1] = '\0';

  do {
	fprintf(stderr, "Distribution [%s]: ", DFLT_DIST);
	gets(distribution);
	if (distribution[0] == '\0') strcpy(distribution, DFLT_DIST);
  } while ((dest = legal(distribution)) == (char *)NULL);

  if (strcmp(distribution, "local") && isatty(0)) {
	printf("This machine posts news %s.\n", dest);
	printf("Are you absolutely sure you want to do this [n]? ");
	fflush(stdout);
	gets(buff);
	if (*buff != 'y' && *buff != 'Y') exit(1);
  }
  fprintf(stderr, "\n");

  /* We now have enough information to set up the article header. */
  fprintf(artfp, "Subject: %s\n", subject);
  if (keywords[0] != '\0')
		fprintf(artfp, "Keywords: %s\n", keywords);
  fprintf(artfp, "Newsgroups: %s\n", newsgroups);
  fprintf(artfp, "Distribution: %s\n", distribution);
  fprintf(artfp, "\n");

  /*
   * We now have a complete header.  Add the article body.
   * If this stdIN is a terminal, we should process the in-
   * put in "editor" mode; if the stdIN is a file (redirecttion!)
   * then we can just copy the data to the temp. file.
   */
  c = 0;
  if (isatty(fileno(infp))) {
	while (c == 0) {
		if (fgets(buff, 1024, infp) == (char *)NULL) break;
		if (buff[0] == '.' && buff[1] == '\n') break;
		  else if (buff[0] == '~' && buff[1] != '~')
						c = do_tilde(&buff[1]);
	       	  else fprintf(artfp, "%s", buff);
	}
  } else {
	while (fgets(buff, 1024, infp) != (char *)NULL) {
		fprintf(artfp, "%s", buff);
	}
  }

  /* Only add signature if we are W-NEWS. */
#if HAVE_WALT
  if (isatty(fileno(infp))) app_signature();
#endif

  /* Close the article file. */
  if (ferror(artfp) || fclose(artfp)) {
	fprintf(stderr, "pnews: could not close temp article\n");
	c = -2;
  }
  return(c);
}


/* Read environment EDITOR, VISUAL and SHELL variables. */
void do_environ()
{
  char *getenv();
  char *e;
  
  e_edit  = (e = getenv("EDITOR")) == (char *)NULL ? E_EDIT : e;
  v_edit  = (e = getenv("VISUAL")) == (char *)NULL ? V_EDIT : e;
  shell   = (e = getenv("SHELL"))  == (char *)NULL ? SHELL  : e;
}


/* Read the 'distributions' file. */
void do_dists()
{
  char buff[128];
  char fname[128];
  FILE *fp;
  DISTRIB *dp;
  register char *sp;

  sprintf(fname, "%s/%s", LIBDIR, DISTRIBS);
  dp = ldistrib;
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "Warning: I cannot read the %s file!\n", fname);
	fprintf(stderr, "         Only \"local\" distribution allowed.\n");
	dp = (DISTRIB *)malloc(DISTSIZ);
	if (dp == NILDIST) {
		fprintf(stderr, "pnews: out of memory.\n");
		return;
	}
	dp->dist = (char *)malloc(6);
	dp->name = (char *)malloc(19);
	if (dp->dist == (char *)NULL || dp->name == (char *)NULL) {
		fprintf(stderr, "pnews: out of memory.\n");
		return;
	}
	strcpy(dp->dist, "local");
	strcpy(dp->name, "local to this site");
	return;
  }

  /* Now read the file and allocate the structures. */
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
	sp = buff;
	while (*sp && *sp != ' ' && *sp != '\t') sp++;
	*sp++ = '\0';
	while (*sp == ' ' || *sp == '\0') sp++;

	if (ldistrib == NILDIST) {
		ldistrib = (DISTRIB *)malloc(DISTSIZ);
		dp = ldistrib;
	} else {
		dp = ldistrib;
		while (dp->next != NILDIST) dp = dp->next;
		dp->next = (DISTRIB *)malloc(DISTSIZ);
		dp = dp->next;
	}
	if (dp == NILDIST) {
		fprintf(stderr, "pnews: out of memory.\n");
		(void) fclose(fp);
		return;
	}
	dp->next = NILDIST;
	dp->dist = (char *)malloc(strlen(buff) + 1);
	dp->name = (char *)malloc(strlen(sp) + 1);
	if (dp->dist == (char *)NULL || dp->name == (char *)NULL) {
		fprintf(stderr, "pnews: out of memory.\n");
		(void) fclose(fp);
		return;
	}
	strcpy(dp->dist, buff);
	strcpy(dp->name, sp);
  }
  (void) fclose(fp);
}


void usage()
{
  fprintf(stderr, "Usage: pnews [-d] [ newsgroup ]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char cmd[128];
  int st;
  char *args[4];

  opterr = 0;
  while ((st = getopt(argc, argv, "d")) != EOF) switch(st) {
	case 'd':
		debug = 1;
		break;
	default:
		usage();
  }

  /* We may have one more argument. */
  if (argc > optind) {
	argc--;
	if (optind == argc) {
		if (testgroup(dflt_newsg = argv[optind]) < 0) exit(1);
	} else usage();
  } else dflt_newsg = DFLT_NEWSG;

  (void) umask(0133);		/* set protection mask to -rw-r--r-- */

  do_environ();
  do_dists();

  strcpy(artname, "/tmp/pn.XXXXXX"); mktemp(artname);

  if ((st = edit_art(stdin)) == 0) {
	sprintf(cmd, "%s/inews", LIBDIR);
	fprintf(stderr, "Posting article...\n");
	st = 1;
	args[st++] = "-h";
	if (debug > 0) args[st++] = "-v";
	args[st] = (char *)NULL;

	if ((st = do_run(artname, cmd, args)) != 0) {
		fprintf(stderr, "Article could not be posted !\n");
		dead_art();
	} else fprintf(stderr, "Article has been posted.\n");
  }

  (void) unlink(artname);

  exit(st);
}
