/*
 * W-NEWS	A simple NEWS processing package for MINIX.
 *		This little program checks to see if the user has any
 *		news articles pending.  It can be used standalone, or
 *		it can be called from the /etc/profile startup file.
 *
 * Usage:	checknews [-nqvy] [ -N ng_list ]
 *
 * Version:	3.00	03/30/91
 *
 * Author:	Fred N. van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "wnews.h"


typedef struct _active_ {
  struct _active_ *next;
  char		  *name;
  unsigned int	  seq;
} ACT;
#define ACTSIZE (sizeof(ACT))
#define NIL_ACT ((ACT *)NULL)


static char *Version = "@(#) checknews 3.00 (03/30/91)";


int opt_n = 0;				/* "No News" is default		*/
int opt_y = 1;				/* "There is news" is default	*/
int opt_q = 0;				/* be quiet; just an exit value	*/
int opt_v = 0;				/* tell which groups have news	*/
ACT *active = NIL_ACT;			/* in-core "active" file	*/


extern int getopt(), optind, opterr;	/* from getopt(3) package	*/
extern char *optarg;
extern long atol();			/* should be in stdlib.h !	*/


/* Read the "active" file into core. */
void act_rd()
{
  char buff[128], fname[128];
  register ACT *ap, *lp;
  register FILE *fp;
  register char *sp, *bp;

  /* Open the "active" file for reading. */
  sprintf(fname, "%s/%s", LIBDIR, ACTIVE);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "checknews: cannot open(%s)\n", fname);
	return;
  }

  /* Now, read the file, and store it in core. */
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	sp = strchr(buff, ' ');
	*sp++ = '\0';
	bp = strchr(sp, ' ');
	*bp++= '\0';

	if (active == NIL_ACT) {
		active = (ACT *)malloc(ACTSIZE);
		ap = active;
	} else {
		ap = lp;
		ap->next = (ACT *)malloc(ACTSIZE);
		ap = ap->next;
	}
	if (ap == NIL_ACT) {
		fprintf(stderr, "checknews: out of memory\n");
		(void) fclose(fp);
		return;
	}
	lp = ap;
	ap->next = NIL_ACT;
	ap->name = (char *)malloc(strlen(buff) + 1);
	if (ap->name == (char *)NULL) {
		fprintf(stderr, "checknews: out of memory\n");
		(void) fclose(fp);
		return;
	}
	strcpy(ap->name, buff);
	ap->seq = (unsigned int) atol(sp);
  }
  (void) fclose(fp);
}


/* Find a newsgroup in the in-core "active" file. */
unsigned int ng_find(name)
char *name;
{
  register ACT *ap, *bp;

  ap = active;
  bp = active;
  while (ap != NIL_ACT) {
	if (!strcmp(ap->name, name)) {
	/* This is a dirty hack, we jus remove the found newsgroup
	   from the linked list. This should make it a lot faster */
		if (ap == active) {	/* This means the first one is the one we wanted */
			active = ap->next; /* We just remove it from the linked list */
		} else {
			bp->next = ap->next; /* Does the same for an group in the middle of the list */
		}
		return(ap->seq);
	}
	bp = ap;
	ap = ap->next;
  }
  return((unsigned int) 0);
}


/* Check if there is any news pending. */
int chk_news(ng_list)
char *ng_list;
{
  char buff[128], fname[128];
  struct passwd *pw;
  register ACT *ap;
  register FILE *fp;
  register char *sp, *bp;
  unsigned int seq, act;
  int found;

  act_rd();

  /* Open the ACTIVE file for reading. */
  pw = getpwuid(getuid());
  if (pw == (struct passwd *)NULL) {
	fprintf(stderr, "checknews: cannot find myself\n");
	return(-1);
  }
  sprintf(fname, "%s/%s", pw->pw_dir, NEWSRC);
  if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
	fprintf(stderr, "checknews: cannot open(%s)\n", fname);
	return(-1);
  }

  /* Now, read the file, and check the numbers. */
  found = 0;
  while (fgets(buff, 128, fp) != (char *)NULL) {
	if (buff[0] == '0' || buff[0] == '\n') continue;
	sp = buff;
	while (*sp && *sp != ':') sp++;
	*sp++ = '\0';
	while (*sp && *sp != '-') sp++;
	bp = ++sp;
	while (*sp && *sp >= '0' && *sp <= '9') sp++;
	*sp = '\0';
	seq = (unsigned int) atol(bp);

	if (ng_list != (char *)NULL) {
		if (strcmp(ng_list, buff)) continue;
	}
	if ((act = ng_find(buff)) > seq) {
		if (opt_v == 1) {
			if (!found++)
				printf("Newsgroup:                             new articles:\n");
			if ((seq+1)==act) {
				printf("%-20s                        %d\n", 
							buff, act);
			} else {
				printf("%-20s                   %5.5d - %d\n",
							buff, seq+1, act);
			}
		} else {
			(void) fclose(fp);
			return(1);
		}
	}
  }
  (void) fclose(fp);

  return((found == 0) ? 0 : 1);
}


void usage()
{
  fprintf(stderr, "Usage: checknews [-nqvy] [ -N ng_list ]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  register int c;
  int st;
  char *ng_list;

  ng_list = (char *)NULL;		/* all groups by default	*/

  opterr = 0;
  while ((c = getopt(argc, argv, "nqvyN:")) != EOF) switch(c) {
	case 'n':
		opt_n = 1;
		break;
	case 'q':
		opt_q = 1;
		break;
	case 'v':
		opt_v = 1;
		break;
	case 'y':
		opt_y = 1;
		break;
	case 'N':
		ng_list = (char *)malloc(strlen(optarg) + 1);
		if (ng_list == (char *)NULL) {
			fprintf(stderr, "checknews: out of memory\n");
			exit(-1);
		}
		strcpy(ng_list, optarg);
		break;
	default:
		usage();
  }

  /* No more arguments allowed. */
  if (optind != argc) usage();

  /* Check the news system. */
  st = 0;
  switch(chk_news(ng_list)) {
	case 0:		/* no news */
		if (opt_q == 0) {
			if (opt_n == 1) printf("No news.\n");
		} else st = 0;
		break;
	case 1:		/* there is news */
		if (opt_q == 0) {
			if (opt_y == 1 && opt_v == 0)
					printf("There is news.\n");
		} else st = 1;
		break;
	case -1:	/* error */
		st = -1;
		break;
  }

  exit(st);
}
