/* ptymgr.c by Michael Temari 11/04/92  a simple pty sessions manager */

#include <sys/types.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <sgtty.h>
#include <errno.h>

#define	COMMAND	1
#define	RUNNING	2
#define	EXIT	3

#define	MAX_SESS	16

int pty_fd[MAX_SESS];
int pty_pid[MAX_SESS];

char buff[1024];

int mode, curpty;

char escape = 0x1d;

struct sgttyb oldsgttyb, newsgttyb;
int keyflags;

char sessnum(a)
int a;
{
   return((a < 10) ? (a + '0') : (a - 10 + 'a'));
}

int keyon()
{
   /* Get current stdin flags */
   if (fcntl(fileno(stdin), F_GETFL, &keyflags) < 0) {
	fprintf(stderr, "fcntl error %d\n", errno);
	return(-1);
   }
   /* Turn on O_NONBLOCK */
   if (fcntl(fileno(stdin), F_SETFL, O_NONBLOCK) < 0) {
	fprintf(stderr, "fcntl error %d\n", errno);
	return(-1);
   }
   /* Get current stdin sgttyb */
   if (ioctl(fileno(stdin), TIOCGETP, &oldsgttyb) < 0) {
	fprintf(stderr, "ioctl error %d\n", errno);
	return(-1);
   }
   /* Set these settings */
   newsgttyb = oldsgttyb;
   newsgttyb.sg_flags |= RAW;
   newsgttyb.sg_flags &= ~(EVENP | ODDP | ECHO | XTABS | CRMOD);
   if (ioctl(fileno(stdin), TIOCSETP, &newsgttyb) < 0) {
	fprintf(stderr, "ioctl error %d\n", errno);
	return(-1);
   }

   return(0);
}

int keyoff()
{
   /* Reset sgttyb settings */
   if (ioctl(fileno(stdin), TIOCSETP, &oldsgttyb) < 0) {
	fprintf(stderr, "ioctl error %d\n", errno);
	return(-1);
   }
   /* Reset stdin flags */
   if (fcntl(fileno(stdin), F_SETFL, keyflags) < 0) {
	fprintf(stderr, "fcntl error %d\n", errno);
	return(-1);
   }
}

int doshell(x)
int x;
{
int i, pid;
char name[16];
struct sgttyb args;

#define SHELL1		"/bin/sh"
#define SHELL2		"/usr/bin/sh"

  if((pid = fork()) == 0) {
	for(i=0; i<MAX_SESS; i++)
		if(pty_fd[i] != -1)
			close(pty_fd[i]);
	close(0);
	close(1);
	close(2);
	sprintf(name, "/dev/ttyp%c", sessnum(x));

	if(open(name, O_RDWR) != 0) _exit(-1);
	if(dup(0)             != 1) _exit(-1);
	if(dup(1)             != 2) _exit(-1);

	/* Set line parameters. */

	if(ioctl(0, TIOCGETP, &args) < 0)
		_exit(-1);
	args.sg_ispeed = args.sg_ospeed = B9600;
	args.sg_flags = BITS8 | ECHO | XTABS | CRMOD;
	args.sg_erase = '\b';	/* CTRL-H (backspace) */
	args.sg_kill = '\030';	/* CTRL-X */
	if(ioctl(0, TIOCSETP, &args) < 0)
		_exit(-1);

	execl(SHELL1, SHELL1, (char *)0);
	execl(SHELL2, SHELL2, (char *)0);

	_exit(-1);
   }
   return(pid);
}

int dopty()
{
struct sgttyb args;
int i, j, fd, s, pid;
char name[16];

   for(i=0; i<MAX_SESS; i++)
	if(pty_fd[i] == -1) break;
   if(i >= MAX_SESS)
	return(-1);

   for(j = 0; j < 16; j++) {
	sprintf(name, "/dev/ptyp%c", sessnum(j));
	fd = open(name, O_RDWR + O_NONBLOCK);
	if(fd >= 0)
		break;
   }
   if(fd < 0)
	return(-1);
   pid = doshell(j);
   if(pid < 0) {
	close(fd);
	return(-1);
   }
   args.sg_ispeed = args.sg_ospeed = B9600;
   if(ioctl(fd, TIOCSETP, &args) < 0) {
	close(fd);
	kill(pid, SIGHUP);
	return(-1);
   }
   pty_fd[i] = fd;
   pty_pid[i] = pid;
   return(i);
}

void commands()
{
   printf("PTYMGR By: Michael Temari\r\n");
   printf("Commands\r\n");
   printf("?      - This text\r\n");
   printf("n      - Start new session\r\n");
   printf("s      - Shutdown current session\r\n");
   printf("0-9\r\n");
   printf("a-f    - Return to that session\r\n");
   printf("RETURN - Return to current session\r\n");
   printf("x      - Exit\r\n");
   printf("Command:");
}

void docommand(c)
char c;
{
int s, x, i, pid;

   if(c == escape) {
	if(curpty != -1) {
		write(pty_fd[curpty], &c, 1);
		mode = RUNNING;
	}
	return;
   }
   c = tolower(c);
   switch(c) {
	case '?':
		commands();
		break;
	case 'x':
		mode = EXIT;
		break;
	case '\r':
	case '\n':
		if(curpty != -1) {
			mode = RUNNING;
			fprintf(stderr, "Returning to session %c\r\n",
					sessnum(curpty));
		} else
			fprintf(stderr, "No current session\r\n");
		break;
	case 's':
		if(curpty == -1)
			fprintf(stderr, "No current session\r\n");
		else {
			mode = COMMAND;
			kill(pty_pid[curpty], SIGKILL);
			do {
				pid = wait((char *)NULL);
				for(i=0; i<MAX_SESS; i++) {
					if (pid == pty_pid[i])
						break;
				}
				if(i < MAX_SESS) {
					close(pty_fd[i]);
					pty_fd[i] = -1;
					fprintf(stderr,
					"Session %c has been shutdown\r\n",
							sessnum(i));
				}
			} while(pid != pty_pid[curpty]);
			curpty = -1;
		}
		break;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	case 'a':
	case 'b':
	case 'c':
	case 'd':
	case 'e':
	case 'f':
		x = ((c < 'a') ? (c - '0') : (10 + c - 'a'));
		if(pty_fd[x] != -1) {
			curpty = x;
			mode = RUNNING;
			fprintf(stderr, "Returning to session %c\r\n",
					sessnum(curpty));
		} else
			fprintf(stderr, "No such session\r\n");
		break;
	case 'n':
		s = dopty();
		if(s >= 0) {
			curpty = s;
			mode = RUNNING;
			printf("Session %c starting\r\n", sessnum(curpty));
		} else
			printf("Error starting new session\r\n");
		break;
   }
}

int main()
{
struct sgttyb args;
int status;
int c, i;

   curpty = -1;

   for(i=0; i<MAX_SESS; i++) pty_fd[i] = -1;

   if(keyon())
	exit(1);

   mode = COMMAND;
   commands();
   while(mode != EXIT) {
	status = read(fileno(stdin), &c, 1);
	if(status == 1) {
		if(mode == COMMAND)
			docommand(c);
		else
			if(c == escape)
				mode = COMMAND;
			else
				if(curpty != -1)
					write(pty_fd[curpty], &c, 1);
	}
	if(curpty != -1 && mode == RUNNING) {
		status = read(pty_fd[curpty], buff, sizeof(buff));
		if(status > 0)
				write(fileno(stdout), buff, status);
	}
   }
   args.sg_ispeed = args.sg_ospeed = B0;
   for(i=0; i<MAX_SESS; i++)
	if(pty_fd[i] >= 0) {
		if(ioctl(pty_fd[i], TIOCSETP, &args) < 0)
			printf("cannot ioctl %d\r\n", i);
		close(pty_fd[i]);
	}
   if(keyoff())
	exit();
}

