/* This file contains the procedures for the handling of select
 *
 * The entry points into this file are
 *   do_select:	perform the SELECT system call
 */

#include "fs.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <sys/types.h>
#include <time.h>
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"

FORWARD void sel_revive();

/* bitfields for minors 0..255 , in and out;
 * used to tell device task which channel to test
 */
PRIVATE fd_mask tty_bits[2][howmany(256, NFDBITS)];
PRIVATE int sel_id;


/*===========================================================================*
 *				sel_ack  				     *
 *===========================================================================*/
PUBLIC void sel_ack(who)
int who;
{
register struct inode *inoptr;
register struct fproc *lfp;
dev_t dev;

  /* message from task or clock */
  if (m.PROC_NR <= 0 || m.PROC_NR >= NR_PROCS ||
      (lfp = &fproc[m.PROC_NR])->fp_sel_id == 0) return;
  switch (who) {
    case TTY :
      dev = (4 << MAJOR) | (m.DEVICE << MINOR);
      break;
    case CLOCK :
      if (lfp->fp_sel_id != m.COUNT) return;
      sel_revive(m.PROC_NR, 0, 0);
      return;
    default :  
      return;
  }
  if (lfp->fp_sel_tty >= 0 && lfp->fs_tty == dev) {
      sel_revive(m.PROC_NR, lfp->fp_sel_tty, m.COUNT);
      return;
  }	
  for (inoptr = inode; inoptr < &inode[NR_INODES]; inoptr++)
    if (inoptr->i_sel_proc == m.PROC_NR &&
        (inoptr->i_mode & I_TYPE) == I_CHAR_SPECIAL &&
        (dev_t) inoptr->i_zone[0] == dev) {
		wake_select(inoptr, m.COUNT);
		break;
    }
}


/*===========================================================================*
 *				do_select				     *
 *===========================================================================*/
PUBLIC int do_select()
{
/* Perform the select(nd, in, out, ex, tv) system call. */
fd_set in, out, ex;
struct timeval tv, *tvp;
unsigned long time;
fd_mask mask, *ip = (fd_mask *)&in, *op = (fd_mask *)&out, *ep = (fd_mask *)&ex;
struct filp *filptr;
struct inode *inoptr;
unshort i_mode;
dev_t dev;
int oper, cum = 0, min, tty_cnt = 0;
register i;

  if (who <= 0) { /* message from task or clock */
	dont_reply = TRUE;
	sel_ack(who);
	return OK;
  } 
  /* else : user call SELECT */
  tvp = (struct timeval *) m.m1_i2;
  if (tvp) {
    if (rw_user(D, who, (vir_bytes) tvp, (vir_bytes) sizeof(struct timeval),
                (char *) &tv, FROM_USER) != OK) return EFAULT;
    time = tv.tv_sec * HZ + (tv.tv_usec * HZ) / 1000000;
    if (time >= MAX_P_LONG) time = MAX_P_LONG - 1L;
  } else time = MAX_P_LONG; /* indefinite timeout */
  if (m.m1_p1) {
    if (rw_user(D, who, (vir_bytes) m.m1_p1, (vir_bytes) sizeof(fd_set),
                (char *) &in, FROM_USER) != OK) return EFAULT;
  } else FD_ZERO(&in);
  if (m.m1_p2) {
    if (rw_user(D, who, (vir_bytes) m.m1_p2, (vir_bytes) sizeof(fd_set),
                (char *) &out, FROM_USER) != OK) return EFAULT;
  } else FD_ZERO(&out);
  if (m.m1_p3) {
    if (rw_user(D, who, (vir_bytes) m.m1_p3, (vir_bytes) sizeof(fd_set),
                (char *) &ex, FROM_USER) != OK) return EFAULT;
  } else FD_ZERO(&ex);

  fp->fp_sel_tty = -1;	/* assume: no /dev/tty */
  mask = 1;
  bzero((char *)tty_bits, howmany(256 * 2, NBBY));
  for (i = 0; i < m.m1_i1; i++) {
    oper  = (*ip & mask)? SEL_IN : 0;
    if (*op & mask) oper += SEL_OUT;
    if (*ep & mask) oper += SEL_EX;
    
    if (oper) { /* test this fd */
      if ((filptr = get_filp(i)) == NIL_FILP) {
        cum = err_code;
        break;
      }  
      inoptr = filptr->filp_ino;
      inoptr->i_sel_proc = who;
      inoptr->i_sel_fd = i;
      i_mode = inoptr->i_mode & I_TYPE;
      if (inoptr->i_pipe) {
        pipe_select(inoptr, &oper, filptr->filp_pos);
        if ((oper & SEL_IN ) == 0) *ip &= ~mask; /* reset in flag */
        if ((oper & SEL_OUT) == 0) *op &= ~mask; /* reset out flag */
        if ((oper & SEL_EX ) == 0) *ep &= ~mask; /* reset ex flag */
      } else if (i_mode == I_CHAR_SPECIAL) {
        /* cumulate requests of char special inodes in extra fields
         * and examine them later
         */
        dev = (dev_t) inoptr->i_zone[0];
        switch ((dev >> MAJOR) & BYTE) {
          case 5: /* /dev/tty */
            dev = fp->fs_tty; /* fall into */
            fp->fp_sel_tty = i;
          case 4: /* /dev/ttyXYZ */
            if (dev) {
              min = (dev >> MINOR) & BYTE;
              if (oper & SEL_IN ) FD_SET(min, (fd_set *)tty_bits[0]);
              if (oper & SEL_OUT) FD_SET(min, (fd_set *)tty_bits[1]);
              tty_cnt++;
            } else { /* /dev/tty = /dev/null */
              *ep &= ~mask;
            }
            break;
          default :
            *ep &= ~mask;
        }
      } else { /* normal file or block special */
        /* default : read, write OK, no exception */
        *ep &= ~mask;
      }
    }
    
    mask <<= 1;
    if (mask == 0) {
      mask = 1;
      ip++; op++; ep++;
    }
  } /* for */
  if (tty_cnt) { /* send request to TTY task */
    if (task_select(4, tty_bits, time ? who : 0) < 0) cum = EBADF;
  }  
    
  if (cum == 0 && tty_cnt) {  
    /* examine bitfields of outstanding char special devices */
    mask = 1;
    ip = (fd_mask *)&in; op = (fd_mask *)&out; ep = (fd_mask *)&ex;
    for (i = 0; i < m.m1_i1; i++) {
      oper  = (*ip & mask)? SEL_IN : 0;
      if (*op & mask) oper += SEL_OUT;
      if (*ep & mask) oper += SEL_EX;
      
      if (oper) { /* test this fd */
        if ((filptr = get_filp(i)) == NIL_FILP) {
          cum = err_code;
          break;
        }
        inoptr = filptr->filp_ino;
        i_mode = inoptr->i_mode & I_TYPE;
        if (i_mode == I_CHAR_SPECIAL) {
          dev = (dev_t) inoptr->i_zone[0];
          switch ((dev >> MAJOR) & BYTE) {
            case 5: /* /dev/tty */
              dev = fp->fs_tty; /* fall into */
            case 4: /* /dev/ttyXYZ */
              if (dev) {
                min = (dev >> MINOR) & BYTE;
                if (oper & SEL_IN ) {
                  if (FD_ISSET(min, (fd_set *)tty_bits[0]) == 0) *ip &= ~mask;
                }
                if (oper & SEL_OUT) {
                  if (FD_ISSET(min, (fd_set *)tty_bits[1]) == 0) *op &= ~mask;
                }
                *ep &= ~mask;  /* no exception */
                break;
              } 
          }
        } /* others than char special are allready done */
        /* count active descriptors */
        if (*ip & mask) cum++;
        if (*op & mask) cum++;
        if (*ep & mask) cum++;
      }
    
      mask <<= 1;
      if (mask == 0) {
        mask = 1;
        ip++; op++; ep++;
      }
    } /* for */
  } /* else : no char special in list */
   
  if (cum || time == 0L) { 
    /* remove select marks */
    for (inoptr = inode; inoptr < &inode[NR_INODES]; inoptr++)
      if (inoptr->i_sel_proc == who) inoptr->i_sel_proc = 0;
    if (m.m1_p1) {
      if (rw_user(D, who, (vir_bytes) m.m1_p1, (vir_bytes) sizeof(fd_set),
                  (char *) &in, TO_USER) != OK) cum = EFAULT;
    }
    if (m.m1_p2) {
      if (rw_user(D, who, (vir_bytes) m.m1_p2, (vir_bytes) sizeof(fd_set),
                  (char *) &out, TO_USER) != OK) cum = EFAULT;
    }
    if (m.m1_p3) {
      if (rw_user(D, who, (vir_bytes) m.m1_p3, (vir_bytes) sizeof(fd_set),
                  (char *) &ex, TO_USER) != OK) cum = EFAULT;
    }
    return(cum);
  } else {
    fp->fp_sel_in = (vir_bytes) m.m1_p1;
    fp->fp_sel_out = (vir_bytes) m.m1_p2;
    fp->fp_sel_ex = (vir_bytes) m.m1_p3;
    if (++sel_id == 0) sel_id++;
    fp->fp_sel_id = sel_id;
    suspend(XSELECT);
    if (time < MAX_P_LONG) clock_select(time);
  }
}

/*===========================================================================*
 *				sel_revive				     *
 *===========================================================================*/
void sel_revive(proc_nr, fdn, oper)
int proc_nr;
int fdn;
int oper;
{
struct fproc *lfp;
fd_set fset;
vir_bytes vir_set = (vir_bytes) 0;
struct inode *inoptr;

  /* remove select marks */
  for (inoptr = inode; inoptr < &inode[NR_INODES]; inoptr++)
    if (inoptr->i_sel_proc == proc_nr) inoptr->i_sel_proc = 0;
  if (proc_nr < INIT_PROC_NR || proc_nr >= NR_PROCS) return;
  FD_ZERO(&fset);
  lfp = &fproc[proc_nr];
  if (lfp->fp_sel_id == 0) return;
  lfp->fp_sel_id = 0;
  if (lfp->fp_sel_in && oper != SEL_IN) {
    if (rw_user(D, proc_nr, lfp->fp_sel_in, (vir_bytes) sizeof(fd_set),
                (char *) &fset, TO_USER) != OK) return;
  }
  if (lfp->fp_sel_out && oper != SEL_OUT) {
    if (rw_user(D, proc_nr, lfp->fp_sel_out, (vir_bytes) sizeof(fd_set),
                (char *) &fset, TO_USER) != OK) return;
  }
  if (lfp->fp_sel_ex && oper != SEL_EX) {
    if (rw_user(D, proc_nr, lfp->fp_sel_ex, (vir_bytes) sizeof(fd_set),
                (char *) &fset, TO_USER) != OK) return;
  }
  if (oper) {
    FD_SET(fdn, &fset);
    switch (oper) {
      case SEL_IN :
        if (lfp->fp_sel_in) vir_set = lfp->fp_sel_in;
        break;
      case SEL_OUT :
        if (lfp->fp_sel_out) vir_set = lfp->fp_sel_out;
        break;
      case SEL_EX :
        if (lfp->fp_sel_ex) vir_set = lfp->fp_sel_ex;
    }
    if (vir_set) {
      if (rw_user(D, proc_nr, vir_set, (vir_bytes) sizeof(fd_set),
                  (char *) &fset, TO_USER) != OK) return;
    }              
  }
  revive( proc_nr, oper ? 1 : 0); /* number of ready fdescs */
}


/*===========================================================================*
 *				wake_select				     *
 *===========================================================================*/
void wake_select(ip, oper)  /* asserts that i_sel_proc != 0 */
struct inode *ip;
int oper;
{
struct fproc *lfp;
int i;

  if (ip->i_sel_proc < INIT_PROC_NR || ip->i_sel_proc >= NR_PROCS) return;
  lfp = &fproc[ip->i_sel_proc];
  switch (oper) {
    case SEL_IN :
      if (!lfp->fp_sel_in) return;
      break;
    case SEL_OUT :
      if (!lfp->fp_sel_out) return;
      break;
    case SEL_EX :
      if (!lfp->fp_sel_ex) return;
  }
  sel_revive(ip->i_sel_proc, ip->i_sel_fd, oper);
}
