/* ctime.c  */

/* 
 * ctime() and asctime() by someone unknown
 * gmtime(), localtime() and mktime() by ERS
 * DST handling by Wim `Blue Baron' van Dorst <baron@wiesje.hobby.nl>
 */

#include <lib.h>
#include <time.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#define TOINT(X)          (X - '0')
#define LEAPYEAR(X)       (X) % 4 == 0 && (X) % 100 != 0 || (X) % 400 == 0
#define SECS_PER_MIN      (60L)
#define SECS_PER_HOUR     (60*SECS_PER_MIN)
#define SECS_PER_DAY      (24*SECS_PER_HOUR)
#define SECS_PER_YEAR     (365*SECS_PER_DAY)
#define SECS_PER_LEAPYEAR (SECS_PER_DAY + SECS_PER_YEAR)

PRIVATE char _timebuf[26];
PRIVATE char *_day[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
PRIVATE char *_month[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                          "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
PRIVATE int _days_per_mth[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
PRIVATE struct tm _the_time;
time_t timezone = -1;
PRIVATE char _dst[80] = {'\0'};
char *tzname[2];
PRIVATE char _tznamebuf[80] = {"GMT                                     DST"};
PRIVATE _PROTOTYPE(time_t tzoffset, (char *s, int *hasdst));
PRIVATE _PROTOTYPE(time_t indst, (_CONST struct tm * t));


/* 
 * Convert <time> structure value to a string.  The same format, and
 * the same internal buffer, as for ctime() is used for this function. 
 */
char *asctime(time)
register _CONST struct tm *time;
{
	if (time == (struct tm *) NULL)
		strcpy(_timebuf, "??? ??? ?? ??:??:?? ????\n");
	else
		sprintf(_timebuf, "%.3s %.3s%3d %02d:%02d:%02d %04d\n",
		  	_day[time->tm_wday], _month[time->tm_mon], 
			time->tm_mday, time->tm_hour, time->tm_min, 
			time->tm_sec, 1900 + time->tm_year);

	return (_timebuf);
}


/*
 * Convert <rawtime> to a string.  A 26 character fixed field string
 * is created from the raw time value.  The following is an example
 * of what this string might look like:
 *              "Wed Jul 08 18:43:07 1987\n\0"
 * A 24-hour clock is used, and due to a limitation in the ST system
 * clock value, only a resolution of 2 seconds is possible.  A pointer
 * to the formatted string, which is held in an internal buffer, is
 * returned.
 */
char *ctime(rawtime)
_CONST time_t *rawtime;
{
	return (asctime(localtime(rawtime)));
}


/*
 * mktime: take a time structure representing the local time (such as is
 * returned by localtime() and convert it into the standard representation
 * (as seconds since midnight Jan. 1 1970, GMT).
 */
time_t mktime(t)
_CONST struct tm *t;
{
	time_t s;
	int y;

	y = t->tm_year - 70;
	if (y < 0)		/* year before 1970 */
		return (time_t) - 1;
	s = y * SECS_PER_YEAR  + ((y + 1) / 4) * SECS_PER_DAY;
	_days_per_mth[1] = (LEAPYEAR(1970 + y)) ? 29 : 28;
	for (y = 0; y < t->tm_mon; y++)
		s += SECS_PER_DAY * _days_per_mth[y];
	s += (t->tm_mday - 1) * SECS_PER_DAY;
	s += t->tm_hour * SECS_PER_HOUR;
	s += t->tm_min * SECS_PER_MIN;
	s += t->tm_sec;

	/* Now adjust for the time zone and possible daylight savings time */
	if (timezone == -1)
		tzset();
	s += timezone;
	if (*_dst)
		s -= indst(t);

	return s;
}


/*
 * take a (time_t) and convert it to a (struct tm *) gmtime, indicating
 * the GMT, not the local time (so no timezone expansion).
 */
struct tm *gmtime(t)
_CONST time_t *t;
{
	struct tm *stm = &_the_time;
	time_t time = *t;
	int year, mday, i;

	if (time < 0)		/* negative times are bad */
		return 0;
	stm->tm_wday = ((time / SECS_PER_DAY) + 4) % 7;

	year = 70;
	for (;;) {
		if (time < SECS_PER_YEAR)
			break;
		if (LEAPYEAR(1900 + year)) {
			if (time < SECS_PER_LEAPYEAR)
				break;
			else
				time -= SECS_PER_LEAPYEAR;
		} else
			time -= SECS_PER_YEAR;
		year++;
	}
	stm->tm_year = year;
	mday = stm->tm_yday = time / SECS_PER_DAY;
	_days_per_mth[1] = (LEAPYEAR(1900 + year)) ? 29 : 28;
	for (i = 0; mday >= _days_per_mth[i]; i++)
		mday -= _days_per_mth[i];
	stm->tm_mon = i;
	stm->tm_mday = mday + 1;
	time = time % SECS_PER_DAY;
	stm->tm_hour = time / SECS_PER_HOUR;
	time = time % SECS_PER_HOUR;
	stm->tm_min = time / SECS_PER_MIN;
	stm->tm_sec = time % SECS_PER_MIN;
	stm->tm_isdst = 0;

	return stm;
}


/* given a standard time, convert it to a local time */
struct tm *localtime(t)
_CONST time_t *t;
{
	struct tm *stm;
	time_t offset;
	time_t x;

	if (timezone == -1)
		tzset();
	offset = *t - timezone;
	stm = gmtime(&offset);
	if (!stm)		/* check for illegal time */
		return stm;
	stm->tm_isdst = 0;

	if (*_dst && (x = indst(stm))) {
		offset += x;
		stm = gmtime(&offset);
		stm->tm_isdst = 1;
	}
	return stm;
}


/* set the timezone and dst flag to the local rules */
void tzset()
{
	timezone = tzoffset(getenv("TZ"), _dst);
}


/*
 * determine the difference, in seconds, between the given time zone
 * and Greenwich Mean. As a side effect, the character pointer hasdst
 * is set to the DST string in the TZ, if there is one.
 */
PRIVATE time_t tzoffset(s, hasdst)
char *s;
char *hasdst;
{
	time_t off = 0;
	int x, sgn = 1;
	char *ptr = _tznamebuf;

	*hasdst = '\0';		
	tzname[0] = _tznamebuf;
	_tznamebuf[3] = '\0';
	tzname[1] = _tznamebuf + 40;

	if (!s || !*s)
		return (time_t) 0;			/* Assume GMT */

	while (isalpha(*s)) {
		*ptr++ = *s++;				/* Name */
	}
	*ptr++ ='\0';

	if (*s == '-') {				/* Sign */
		sgn = -1;
		s++;
	}

	for (x = 0; isdigit(*s); s++)			/* Hours */
		x = 10 * x + TOINT(*s);
	off = x * SECS_PER_HOUR;
	if (*s == ':') {				/* Minutes */
		for (s++, x = 0; isdigit(*s); s++)
			x = 10 * x + TOINT(*s);
		off += (x * SECS_PER_MIN);
		if (*s == ':') {			/* Seconds */
			for (s++, x = 0; isdigit(*s); s++)
				x = 10 * x + TOINT(*s);
			off += x;
		}
	}

	if (isalpha(*s)) {				/* DST string */
		tzname[1] = ptr;
		while (isalpha(*s)) {
			*ptr++ = *s;
			*hasdst++ = *s++;
		}
		*ptr = '\0';
		while (*hasdst++ = *s++);
	}

	return (time_t) (sgn * off);
}


/*
 * time_t indst(struct tm *)
 * Given a tm struct representing the local time (GMT adapted for the time
 * zone), determine whether DST is currently in effect, and which offset.
 *
 * Time zones (TZ) are given as strings of the form (spaces only for clarity)
 *
 *    Std Offset [Dst [Offset] [,Startdate[/Time], Enddate[/Time]]]
 *
 * where:
 * Std and Dst three or more bytes that are the designation for the standard
 *             (std) or summer (dst) time zone. Only std is required; if dst
 *             is missing, then summer time doen not apply in this locale.
 *
 * Offset and  Indicates the value one must add to the local time to arrive
 * Time        at Coordinated Universal Time. The offset has the form
 *                hh[:mm[:ss]]
 *             The hours shall be required and may be a single digit. If no
 *             offset follows dst, summer time is assumed to be one hour ahead
 *             of standard time. If preceded by a "-", the time zone shall be
 *             east of the Prime meridian; otherwise it shall be west. Default
 *             DST offset will be 1 hour, default switch Time will be 02:00:00
 *
 * date        The format of the date shall be one of the following:
 *             Jn      Julian day (1 <= n <= 365)
 *                     leapdays are NOT counted, so March 1 is always day 60
 *             n       Zero-based Julian day (0 <= n <= 365)
 *                     leapdays are counted, so March 1 is day 61 in leapyears
 *             Mm.n.d  The d-th day (0<=d<=6) of week n (1 <=n<=5) of month
 *                     m (1<=m<=12), where week 5 means 'the last d day in
 *                     month m' which may occur in either the fourth or
 *                     fifth week. Week 1 is the first week in which d'th
 *                     day occurs. Day zero is sunday.
 *
 * In case no correct TZ specification is given no DST is assumed
 * to be in effect. No indication of fault is given.
 *
 * In case no explicit rule is given the US rules will be used, as is
 * represented by ,M4.5.0,M10.5.0 prior to 1987 and ,M4.1.0,M10.5.0
 * in 1987 and later, although this was not applicable in 1974.
 *
 * Examples    EST5EDT would represent the N. American Eastern time zone
 *             CST6CDT would represent the N. American Central time zone
 *             NFLD3:30NFLD would represent Newfoundland time (one and a
 *                     half hours ahead of Eastern).
 *             OZCST-9:30 would represent the Australian central time zone.
 *                     (which, so I hear, doesn't have DST).
 *             MET-1:00:00MEDST1:00:00,M3.5.0/2:00:00,M9.5.0/2:00:00
 *                     represents the most elaborate way to express the
 *                     Middle European time, with DST starting the last
 *                     Sunday in March lasting to the last Sunday in
 *                     September. Note that
 *             MET-1MEDST,M3.5.0,M9.5.0 will do just as fine.
 *
 * See for more explanations on the TZ definition the IEEE Std 1003.1-1988
 * Posix Standard, paragraphs 8.1.1 "Extensions to Time Functions" on 142-143.
 *
 * Important:  Time specification will be ignored, and is hardcoded to
 *             02:00:00 for reasons of simplicity.
 *
 * Important:  This function only works properly on the Nothern Hemisphere,
 *             for reasons of simplicity. The code can be easily adapted to
 *             proper behaviour on the Southern Hemisphere, by swopping the
 *             pieces for the 'Starting date of the rule' and the 'Ending
 *             date of the rule'. The needed code is already in.
 */
PRIVATE time_t indst(t)
_CONST struct tm *t;
{
	char *s = _dst;				/* DST string */
	time_t off = (time_t) 0;		/* offset for DST */
	time_t x = 0;				/* counter */
	int sgn = 1;				/* sign for DST offset */
	int ri, rmonth, rweek, rday, rtotal;	/* variables rule handling */

	while (isalpha(*s))
		s++;			/* name */

	if (*s == '-') {			/* Negative Dst? Odd. */
		sgn = -1;
		s++;
	}

	/* Calculate the DST offset */
	if (!isdigit(*s)) {
		off = SECS_PER_HOUR;		/* Default */
	} else {
		for (x = 0; isdigit(*s); s++)
			x = 10 * x + TOINT(*s);
		off = x * SECS_PER_HOUR;
		if (*s == ':') {
			for (s++, x = 0; isdigit(*s); s++)
				x = 10 * x + TOINT(*s);
			off += x * SECS_PER_MIN;
			if (*s == ':') {
				for (s++, x = 0; isdigit(*s); s++)
					x = 10 * x + TOINT(*s);
				off += x;
			}
		}
	}

	/* Use the US rule as default */
	if (!(*s)) {
		if (t->tm_year < 87)
			strcpy(s, ",M4.5.0,M10.5.0");
		else
			strcpy(s, ",M4.1.0,M10.5.0");
	}

	/* Starting date of the rule */
	ri = 0;
	switch (*++s) {

	/* Julian day. No leap days counted */
	case 'J':
		s++;
		if (LEAPYEAR(1900 + t->tm_year))
			ri = 1;
		/* fall through */

	/* Zero-based Julian day, leap days counted */
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
		for (rday = 0; isdigit(*s); s++)
			rday = rday * 10 + TOINT(*s);
		if (ri && rday > 59)
			rday--;
		if (t->tm_yday < rday) {
			return (time_t) 0;
		} else if (t->tm_yday > rday) {
			while (*s && *s != ',') s++;
			break;
		} else if (t->tm_hour < 2) {
			return (time_t) 0;
		}
		while (*s &&*s != ',') s++;
		break;

	/* Mmonth.week.day */
	case 'M':
		for (s++, rmonth = 0; isdigit(*s); s++)
			rmonth = rmonth * 10 + TOINT(*s);
		rmonth--;
		if (t->tm_mon < rmonth) {
			return (time_t) 0;
		} else if (t->tm_mon > rmonth) {
			while (*s && *s != ',') s++;
			break;
		}
		for (s++, rweek = 0; isdigit(*s); s++)
			rweek = rweek * 10 + TOINT(*s);
		for (s++, rday = 0; isdigit(*s); s++)
			rday = rday * 10 + TOINT(*s);
		for (ri = 70; ri < t->tm_year; ri++) {
			rtotal += 365;
			if (LEAPYEAR(1900 + ri))
				rtotal++;
		}
		_days_per_mth[1] = (LEAPYEAR(1900 + t->tm_year)) ? 29 : 28;
		for (ri = 0; ri < rmonth; ri++)
			rtotal += _days_per_mth[ri];
		rday = (rday - rtotal % 7 + 10) % 7 + 7 * rweek - 6;
		if (rweek == 5 && rday > _days_per_mth[rmonth])
			rday -= 7;
		if (t->tm_mday < rday) {
			return (time_t) 0;
		} else if (t->tm_mday > rday) {
			while (*s && *s != ',') s++;
			break;
		} else if (t->tm_hour < 2) {
			return (time_t) 0;
		}
		while (*s && *s != ',') s++;
		break;

	default:
		return (time_t) 0;
	}


	/* Ending date of the rule */
	ri = 0;
	switch (*++s) {

	/* Julian day. No leap days counted */
	case 'J':
		s++;
		if (LEAPYEAR(1900 + t->tm_year))
			ri = 1;
		/* fall through */

	/* Zero-based Julian day, leap days counted */
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
		for (rday = 0; isdigit(*s); s++)
			rday = rday * 10 + TOINT(*s);
		if (ri && rday > 59)
			rday--;
		if (t->tm_yday > rday) {
			return (time_t) 0;
		} else if (t->tm_yday < rday) {
			while (*s && *s != ',') s++;
			break;
		} else if (t->tm_hour >= 2) {
			return (time_t) 0;
		}
		while (*s && *s != ',') s++;
		break;

	/* Mmonth.week.day */
	case 'M':
		for (s++, rmonth = 0; isdigit(*s); s++)
			rmonth = rmonth * 10 + TOINT(*s);
		rmonth--;
		if (t->tm_mon > rmonth) {
			return (time_t) 0;
		} else if (t->tm_mon < rmonth) {
			while (*s && *s != ',') s++;
			break;
		}
		for (s++, rweek = 0; isdigit(*s); s++)
			rweek = rweek * 10 + TOINT(*s);
		for (s++, rday = 0; isdigit(*s); s++)
			rday = rday * 10 + TOINT(*s);
		for (ri = 70; ri < t->tm_year; ri++) {
			rtotal += 365;
			if (LEAPYEAR(1900 + ri))
				rtotal++;
		}
		_days_per_mth[1] = (LEAPYEAR(1900 + t->tm_year)) ? 29 : 28;
		for (ri = 0; ri < rmonth; ri++)
			rtotal += _days_per_mth[ri];
		rday = (rday - rtotal % 7 + 10) % 7 + 7 * rweek - 6;
		if (rweek == 5 && rday > _days_per_mth[rmonth])
			rday -= 7;
		if (t->tm_mday > rday) {
			return (time_t) 0;
		} else if (t->tm_mday < rday) {
			while (*s && *s != ',') s++;
			break;
		} else if (t->tm_hour >= 2) {
			return (time_t) 0;
		}
		while (*s && *s != ',') s++;
		break;

	default:
		return (time_t) 0;
	}

	/* Yes, we are in Daylight Saving Time */
	return off;
}


/* return difference between two time_t types  -- ERS */
double difftime(t1, t2)
time_t t1, t2;
{
	return (double) (t2 - t1);
}
