/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the external clients and servers.
 *
 * Version:	@(#)nproc.c		1.00	07/08/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <arpa/internet.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>

#include "mbuf.h"
#include "timer.h"
#include "icmp.h"
#include "netuser.h"
#include "tcp.h"
#include "inetd.h"
#include "nproc.h"
#include "udp.h"


struct udp_hdr {
  int size;
  struct socket lsocket;
  struct socket fsocket;
};


static char buffer[4096];
struct nproc nproc[NNPROC];
int16 lport = 1001;

#if HAVE_SELECT
extern fd_set fds_in_use;
#endif

_PROTOTYPE( void chg_state, (struct tcb *, char, char)			);
_PROTOTYPE( void udp_in, (struct socket *, int)				);


static void udp_in(lsocket, cnt)
struct socket *lsocket;
int cnt;
{
  struct socket fsocket;
  struct mbuf *bp;
  char *p;
  int i, total, size;
  struct udp_hdr udp_hdr;

  if (recv_udp(lsocket, &fsocket, &bp) <= 0) return;

  for(i = 0; i < NNPROC; i++) {
	if (nproc[i].type == UDP_PTCL &&
	   nproc[i].lsocket.address == lsocket->address &&
	   nproc[i].lsocket.port    == lsocket->port)
		break;
  }

  if (i >= NNPROC) {
	(void) free_p(bp);
	return;
  }
  p = buffer;
  udp_hdr.size = len_mbuf(bp);
  memcpy(&udp_hdr.lsocket, lsocket, sizeof(struct socket));
  memcpy(&udp_hdr.fsocket, &fsocket, sizeof(struct socket));

  memcpy(p, &udp_hdr, sizeof(udp_hdr));
  p = p + sizeof(udp_hdr);
  total = dqdata(bp, p, udp_hdr.size);
  if (total != udp_hdr.size) return;
  total = total + sizeof(udp_hdr);
  if (write(nproc[i].fdout, buffer, total) != total)
	freenproc(i);
}


static void chg_state(tcb, old, new)
struct tcb *tcb;
char old, new;
{
  extern char *psocket();
  int i, j;

  for(i = 0; i < NNPROC; i++)
	if (nproc[i].type == TCP_PTCL && tcb == nproc[i].tcb) break;

  switch (new) {
	case ESTABLISHED:
#if HAVE_DEBUG
		rprintf(2, "Client %d: Established ", i);
		rprintf(2, "%-23s", psocket(&tcb->conn.local));
		rprintf(2, "%-23s\r\n", psocket(&tcb->conn.remote));
		rprintf(2, "\r\n");
#endif
		break;
	case CLOSE_WAIT:
		if(i < NNPROC) {
			if(nproc[i].fdout != -1) {
				(void) close(nproc[i].fdout);
				nproc[i].fdout = -1;
			}
		}
		close_tcp(tcb);
		break;
	case TIME_WAIT:
		/* This is so we can free up fd's quicker Michael Temari */
		if(i < NNPROC) {
			if(nproc[i].fdin != -1) {
				(void) close(nproc[i].fdin);
#if HAVE_SELECT
				FD_CLR(nproc[i].fdin, &fds_in_use);
#endif
				nproc[i].fdin = -1;
			}
			if(nproc[i].fdout != -1) {
				(void) close(nproc[i].fdout);
				nproc[i].fdout = -1;
			}
		}
		break;
	case CLOSED:
		if(i < NNPROC)
			freenproc(i);
		else
			del_tcp(tcb);
#if HAVE_DEBUG
		rprintf(2, "Client %d: Closed\r\n", i);
#endif
		break;
#if HAVE_DEBUG
	default:
		rprintf(2, "Client %d: old=%d new=%d\r\n", i, old, new);
		break;
#endif
  }
}


void in_char(tcb, cnt)
struct tcb *tcb;
int16 cnt;
{
  struct mbuf *bp;
  int i;

  if (recv_tcp(tcb, &bp, cnt) > 0) {
	for(i = 0; i < NNPROC; i++) {
		if (nproc[i].type == TCP_PTCL &&
			nproc[i].tcb == tcb) break;
	}
	if (i < NNPROC) {
		if(data_in(bp, nproc[i].fdout))
			if(tcb->state == LISTEN)
				freenproc(i);
			else
				close_tcp(nproc[i].tcb);
	} else
		(void) free_p(bp);
  }
}


void send_okay(tcb, cnt)
struct tcb *tcb;
int16 cnt;
{
  int i;

  for(i = 0; i < NNPROC; i++)
	if (nproc[i].type == TCP_PTCL &&
		nproc[i].tcb == tcb) break;
  if (i < NNPROC) {
	if(nproc[i].sendok >= 0)
		nproc[i].sendok = cnt;
  }
}


static int data_in(bp, fdout)
struct mbuf *bp;
int fdout;
{
int okay;

  okay = 1;
  while(bp != NULLBUF) {
	if(okay) {
		if(write(fdout, bp->data, bp->cnt) != bp->cnt)
			okay = 0;
	}
	bp = free_mbuf(bp);
  }
  return(1 - okay);
}


void initnproc()
{
  int i;

  for(i = 0; i < NNPROC; i++)
	nproc[i].type = FREE;
}


/* find a free nproc entry */
int ffnproc()
{
  int i;

  for(i = 0; i < NNPROC; i++) {
	if(nproc[i].type == FREE) {
		nproc[i].sendok = 0;
		nproc[i].tcb = NULLTCB;
		nproc[i].fdin = -1;
		nproc[i].fdout = -1;
		nproc[i].pid = -1;
		return(i);
	}
  }
  return(-1);
}

static int icmp_use = 0;

void icmp_proc(bp)
struct mbuf *bp;
{
int i;
int len;
int s;

   if(!icmp_use)
	return;
   len = len_mbuf(bp);
   for(i = 0; i < NNPROC; i++) {
	if(nproc[i].type == ICMP_PTCL) {
		s = write(nproc[i].fdout, &len, sizeof(len));
		if(s == sizeof(len)) {
			s = write(nproc[i].fdout, bp->data, len);
			if(s == len)
				continue;
		}
		freenproc(i);
	}
   }
}

void freenproc(i)
int i;
{
  int pid, pstat;

  if (i < 0 || i >= NNPROC) return;
  if (nproc[i].type == FREE) return;

  switch(nproc[i].type) {
	case ICMP_PTCL:
		icmp_use--;
		break;
	case TCP_PTCL:
		if(nproc[i].tcb != NULLTCB)
			del_tcp(nproc[i].tcb);
		break;
	case UDP_PTCL:
		del_udp(&nproc[i].lsocket);
		break;
  }
  if (nproc[i].fdin != -1) {
	(void) close(nproc[i].fdin);
#if HAVE_SELECT
	FD_CLR(nproc[i].fdin, &fds_in_use);
#endif
  }
  if (nproc[i].fdout != -1) (void) close(nproc[i].fdout);
  if (nproc[i].pid != -1) (void) kill(nproc[i].pid, SIGKILL);

  nproc[i].type = FREE;
  if (nproc[i].pid != -1) {
	pid = wait(&pstat);
#if HAVE_DEBUG
	rprintf(2, "nproc: Pid %d Status %04x\n", pid, pstat);
#endif
  }
}


int cnproc(protocol, localport, port, addr, flags)
int protocol;
int16 *localport;
int16 *port;
int32 *addr;
int32 flags;
{
  int i;
  struct socket lsocket, fsocket;
  struct tcb *tcb;

  i = ffnproc();
  if (i < 0) return(i);

  nproc[i].type = protocol;

  lsocket.address = ip_addr;
  if (*localport == 0) {
	lsocket.port = lport++;
	*localport = lsocket.port;
  } else lsocket.port = *localport;
  fsocket.address = *addr;
  fsocket.port = *port;

  switch(protocol) {
	case ICMP_PTCL:
		icmp_use++;
		break;
	case UDP_PTCL:
		if (open_udp(&lsocket, udp_in)) {
			freenproc(i);
			return(-1);
		}
		nproc[i].lsocket = lsocket;
		break;
	case TCP_PTCL:
		tcb = open_tcp(&lsocket, &fsocket,
				(flags == 1?TCP_ACTIVE:TCP_PASSIVE),
				0, in_char, send_okay, chg_state, 0,
				(int *)NULL);
		nproc[i].tcb = tcb;
		if (tcb == NULLTCB || tcb->state == CLOSED) {
			/* we need to set tcb to NULL since if we */
			/* don't find a route to get out then the tcb */
			/* has already been deleted. M. Temari 02/20/93 */
			nproc[i].tcb = NULLTCB;
			freenproc(i);
			return(-1);
		}
		break;
	default:
		freenproc(i);
		return(-1);
  }
  return(i);
}


void DOnproc()
{
  int i, cnt;
  struct mbuf *bp;
  struct udp_hdr udp_hdr;

  for(i = 0; i < NNPROC; i++)
	switch(nproc[i].type) {
	   case ICMP_PTCL:
		if (nproc[i].sendok <= 0) {
			nproc[i].sendok++;
			break;
		}
		while((cnt = read(nproc[i].fdin, buffer, sizeof(buffer)) > 0)) ;
		if(cnt == 0) {
#if HAVE_DEBUG
			fprintf(stderr, "nproc: %d close for icmp\n", i);
#endif
			freenproc(i);
		}
		break;
	   case TCP_PTCL:
		if (nproc[i].sendok > 0) {
			while((cnt = read(nproc[i].fdin, buffer,
				min(sizeof(buffer), nproc[i].sendok))) > 0) {
				nproc[i].sendok -= cnt;
				bp = qdata(buffer, cnt);
				send_tcp(nproc[i].tcb, bp);
				if (nproc[i].sendok <= 0) break;
			}
			if (cnt == 0) {
#if HAVE_DEBUG
				fprintf(stderr, "nproc: %d close for write\n", i);
#endif
				if(nproc[i].tcb->state == LISTEN)
					freenproc(i);
				else {
					nproc[i].sendok = -1;
					close_tcp(nproc[i].tcb);
					(void) close(nproc[i].fdin);
#if HAVE_SELECT
					FD_CLR(nproc[i].fdin, &fds_in_use);
#endif
					nproc[i].fdin = -1;
				}
			}
		}
		break;
	   case UDP_PTCL:
		if (nproc[i].sendok <= 0) {
			nproc[i].sendok++;
			break;
		}
		cnt = read(nproc[i].fdin, &udp_hdr, sizeof(udp_hdr));
		if (cnt == 0) freenproc(i);
		if (cnt != sizeof(udp_hdr)) break;
		if (udp_hdr.size > 0) {
			cnt = read(nproc[i].fdin, buffer, udp_hdr.size);
			if (cnt == 0) freenproc(i);
			if (cnt != udp_hdr.size) break;
			bp = qdata(buffer, udp_hdr.size);
		} else bp = NULLBUF;
		send_udp(&udp_hdr.lsocket, &udp_hdr.fsocket,
			0, 0, bp, (int16)udp_hdr.size,
			(int16)0, 0);
		break;
	}
}
