/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the various server processes for
 *		the TCP and UDP ports.  It also handles the process
 *		of reading the "inetd.conf" file.
 *
 * Version:	@(#)inetd.c		1.00	07/12/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <sys/types.h>
#include <arpa/internet.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>

#include "mbuf.h"
#include "timer.h"
#include "icmp.h"
#include "netuser.h"
#include "tcp.h"
#include "inetd.h"
#include "nproc.h"


struct servers servers[NSERVERS];


_PROTOTYPE( void new_state, (struct tcb *, char, char)			);
_PROTOTYPE( int start_proc, (int, char *, char *)			);
_PROTOTYPE( int srv_start, (char **argv)				);
_PROTOTYPE( void config, (void)						);
_PROTOTYPE( void srv_init, (void)					);


static void new_state(tcb, old, new)
struct tcb *tcb;
char old, new;
{
  extern char *psocket();
  int i, j;

  for(i = 0; i < NNPROC; i++)
	if (nproc[i].type == TCP_PTCL && tcb == nproc[i].tcb) break;

  switch(new) {
	case ESTABLISHED:
		i = ffnproc();
		if (i < 0) {
			rprintf(2, "Too many server processes\r\n");
			close_tcp(tcb);
			return;
		}
		nproc[i].type = TCP_PTCL;
		nproc[i].tcb = tcb;
		for(j = 0; j < NSERVERS; j++)
			if(servers[j].type == TCP_PTCL &&
			   servers[j].port == tcb->conn.local.port) break;
		if (j >= NSERVERS) {
			rprintf(2, "Server process could not be found!\r\n");
			close_tcp(tcb);
			return;
		}
		if (start_proc(i, servers[j].program, servers[j].args)) {
			rprintf(2, "Could not start server process!\r\n");
			close_tcp(tcb);
			return;
		}
#if HAVE_DEBUG
		rprintf(2, "Server: Established ");
		rprintf(2, "%-23s", psocket(&tcb->conn.local));
		rprintf(2, "%-23s\r\n", psocket(&tcb->conn.remote));
		rprintf(2, "\r\n");
#endif
		break;
	case CLOSE_WAIT:
		close_tcp(tcb);
		break;
	case CLOSED:
		freenproc(i);
#if HAVE_DEBUG
		rprintf(2, "Server: Closed\r\n");
#endif
		break;
#if HAVE_DEBUG
	default:
		rprintf(2, "Server: old=%d new=%d\r\n", old, new);
		break;
#endif
  }
}


static int start_proc(entry, program, args)
int entry;
char *program;
char **args;
{
  int pfd1[2], pfd2[2], i;

  if (pipe(pfd1) < 0) {
	rprintf(2, "Error creating pipe1 for server (%d)\n", errno);
	return(-1);
  }
  if (pipe(pfd2) < 0) {
	rprintf(2, "Error creating pipe2 for server (%d)\n", errno);
	close(pfd1[0]); close(pfd1[1]);
	return(-1);
  }

  if ((nproc[entry].pid = fork()) < 0) {
	rprintf(2, "Error forking for server (%d)\n", errno);
	close(pfd1[0]); close(pfd1[1]);
	close(pfd2[0]); close(pfd2[1]);
	return(-1);
  }

  if (nproc[entry].pid == 0) {
	close(0); close(1); close(2);
	dup(pfd1[0]); dup(pfd2[1]); dup(1);
	closefiles();
	execv(program, args);
	rprintf(2, "Error execv program (%d)\n", errno);
  }
  close(pfd1[0]); close(pfd2[1]);
  nproc[entry].fdin = pfd2[0]; nproc[entry].fdout = pfd1[1];
  if (fcntl(nproc[entry].fdin, F_SETFL, O_NONBLOCK) < 0)
	rprintf(2, "fcntl error server pipe (%d)\n", errno);
  return(0);
}


static int srv_start(argv)
char *argv[];
{
  struct socket lsocket;
  int i, j, type, size;
  int16 port;
  char *p, **v;

  if (strcmp("tcp", argv[1])) {
	rprintf(2, "Protocol must be tcp\r\n");
	return(0);
  } else type = TCP_PTCL;
  port = atoi(argv[2]);
  if (port <= 0) {
	rprintf(2, "Bad port %s\r\n", argv[2]);
	return(0);
  }

  i = NSERVERS;
  for(j = 0;j < NSERVERS; j++) {
	if(servers[j].type == FREE) {
		if(i >= NSERVERS) i = j;
	} else {
		if (servers[j].type == type && servers[j].port == port) {
			rprintf(2,
				"Server already running on %s port %s\r\n",
							argv[1], argv[2]);
			return(0);
		}
	}
  }
  if (i >= NSERVERS) {
	rprintf(2, "Too many server processes\r\n");
	return(0);
  }

  lsocket.address = ip_addr;
  lsocket.port = servers[i].port = port;

  /* Fill in the server's ARGV array. */
  servers[i].program = (char *)malloc(strlen(argv[3])+1);
  strcpy(servers[i].program, argv[3]);
  size = 2 * sizeof(char *);
  servers[i].args = (char *)malloc(size);
  v = (char **)servers[i].args;
  v[0] = servers[i].program;
  v[1] = NULLCHAR;

  /* See above: we are always in TCP mode. */
  servers[i].type = type;
  servers[i].tcb = open_tcp(&lsocket, NULLSOCK, TCP_PASSIVE, 0,
			in_char, send_okay, new_state, 0, (int *)NULL);

#if HAVE_DEBUG
  if (debug == 1) rprintf(2, "Server started on %s port %s %s\r\n",
					argv[1], argv[2], argv[3]);
#endif
  return(1);
}


static void srv_init()
{
  int i;

  for(i = 0; i < NSERVERS; i++)
	servers[i].type = FREE;
}


/* Read and process the CFG file.  This will disappear very soon! */
static void config()
{
  char buff[128];
  char *argv[32];
  int argc, i;
  register FILE *fp;
  register char *sp;

  sprintf(buff, "%s/%s", TNET_DIR, TNET_CFG);
  if ((fp = fopen(buff, "r")) != NULLFILE) {
	while(fgets(buff, sizeof(buff), fp) != NULLCHAR) {
		if ((sp = strchr(buff, '\r')) != NULLCHAR) *sp = '\0';
		if ((sp = strchr(buff, '\n')) != NULLCHAR) *sp = '\0';
		if (buff[0] == '#' || buff[0] == '\0') continue;

		/* Break up the command line. */
		sp = buff; i = 0;
		argv[i] = sp;
		while (*sp && i < 32) {
			while (*sp && (*sp == ' ' || *sp == '\t')) sp++;
			argv[i++] = sp;
			while (*sp && *sp != ' ' && *sp != '\t') sp++;
			if (*sp != '\0') *sp++ = '\0';
		}
		while (i < 32) argv[i++] = (char *)NULL;

		/* Now decode the command line. */
		if (!strcmp(argv[0], "echo")) ec_start();
		  else if (!strcmp(argv[0], "discard")) di_start();
		  else if (!strcmp(argv[0], "server")) srv_start(argv);
		  else rprintf(2, "Syntax error: %s\n", argv[0]);
	}
	(void) fclose(fp);
  }
}


void inetd()
{
  srv_init();
  config();
}
