/*
 * slip.drv	A program for TNET, which reads SLIP encapsulated IP
 *		datagrams from a serial port, and sends them to TNET
 *		for further processing.
 *
 *		UNIX RS-232 support functions.  This file takes care of
 *		opening and setting up the seriel line, and it takes
 *		care of allocating the buffers and initializing their
 *		control structures.
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <sgtty.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "slip.h"


#define TTY_BUFSIZE	1024		/* size of one serial buffer	*/


static char *in_buff, *out_buff;	/* line input/output buffers	*/
static char *in_ptr, *out_ptr;
static int in_size, out_size;
static int in_cnt, out_cnt;
static struct sgttyb sl_sgtty;		/* saved TTY device state	*/
#ifndef _CSLIP
static int tty_fd;			/* TTY file descriptor		*/
#else
int tty_fd;			/* TTY file descriptor		*/
#endif
static struct {
  char *speed;
  int  code;
} allbauds[] = {			/* table of usable baud rates	*/
  { "50",		B50	},
  { "75",		B75  	},	
  { "110",		B110	},
  { "300",		B300	},
  { "600",		B600	},
  { "1200",		B1200	},
  { "2400",		B2400	},
  { "4800",		B4800	},
  { "9600",		B9600	},
#ifdef B19200
  { "19200",		B19200	},
#endif
#ifdef B38400
  { "38400",		B38400	},
#endif
#ifdef B115200
  { "115200",		B115200	},
#endif
#ifdef EXTA
  { "exta",		EXTA	},
#endif
#ifdef EXTB
  { "extb",		EXTB	},
#endif
  { (char *)NULL,	0	}
};


_PROTOTYPE( int set_baud, (char *what)					);


/* Set the desired baud rate. */
static int set_baud(what)
register char *what;
{
  register int i;

  i = 0;
  if (!strcmp(what, "(default)")) return(i);

  while (allbauds[i].speed != (char *)NULL) {
	if (!strcmp(allbauds[i].speed, what)) return(allbauds[i].code);
	i++;
  }
  return(-EINVAL);
}


/* Initialize the serial line. */
int tty_init(name, baud)
char *name;
char *baud;
{
  struct sgttyb sgtty;
  int fd, i;

  /* Try opening the TTY device. */
  if (*name != '\0') {
	fd = open(name, O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "SLIP: open(%s, RW) == %d\n", name, errno);
		return(-errno);
	}
  } else {
	fprintf(stderr, "SLIP: no terminal device given???\n");
	return(-errno);
  }
  if (opt_v) fprintf(stderr, "TTY = %s (%d) ", name, fd);

  /* Save the current TTY state, and set new speed and flags. */
  if (ioctl(fd, TIOCGETP, &sl_sgtty) < 0) {
	fprintf(stderr, "SLIP: cannot ioctl(%d, GETP): %d\n", fd, errno);
	return(-errno);
  }
  memcpy((char *) &sgtty, (char *) &sl_sgtty, sizeof(struct sgttyb));
  sgtty.sg_flags = RAW;
#ifdef CS8
  sgtty.sg_flags |= CS8;
#endif
#ifdef BITS8
  sgtty.sg_flags |= BITS8;
#endif

  /* Fetch the baud rate, and set it if needed. */
  if ((i = set_baud(baud)) < 0) return(i);
  if (i != 0) {
	sgtty.sg_ispeed = i;
	sgtty.sg_ospeed = i;
  }
  if (ioctl(fd, TIOCSETP, &sgtty) < 0) {
	fprintf(stderr, "SLIP: cannot ioctl(%d, SETP): %d\n", fd, errno);
	return(-errno);
  }
  if (opt_v) fprintf(stderr, "SPEED=%s (%d)\n", baud, i);

  /* Size and allocate the I/O buffers. */
#ifndef _CSLIP
  in_size = sl_bufs * TTY_BUFSIZE;
#else
  in_size = 1;
#endif
  out_size = in_size;
  in_buff = (char *) malloc(in_size);
  out_buff = (char *) malloc(out_size);
  if (in_buff == (char *)NULL || out_buff == (char *)NULL) {
	fprintf(stderr, "SLIP: cannot allocate (%d, %d) buffers (%d)\n",
						in_size, out_size, errno);
	return(-ENOMEM);
  }
  in_cnt = 0; out_cnt = 0;
  in_ptr = in_buff; out_ptr = out_buff;
  if (opt_v) fprintf(stderr, "IBUF=%d(%d) OBUF=%d(%d)\n",
			in_size, sl_bufs, out_size, sl_bufs);
  out_size -= 4; /* safety */
  tty_fd = fd;
  return(0);
}


/* Restore the terminal line. */
void tty_stop()
{
  (void) ioctl(tty_fd, TIOCSETP, &sl_sgtty);
}


/* Read one character (byte) from the SLIP link. */
int tty_getc()
{
  register int s;

  if (in_cnt == 0) {
	s = read(tty_fd, in_buff, in_size);
	if (s <= 0) return(-1);
	in_cnt = s;
	in_ptr = in_buff;
  }

  s = ((int) *in_ptr) & 0xFF;
  in_ptr++;
  in_cnt--;
  return(s);
}


/* Write one character (byte) to the SLIP link. */
int tty_putc(c)
register int c;
{
  register int s;

  if ((out_cnt == out_size) || (c == -1)) {
	s = write(tty_fd, out_buff, out_cnt);
	out_cnt = 0;
	out_ptr = out_buff;
	if (s < 0) return(-1);
  }

  if (c != -1) {
	*out_ptr = (char) c;
	out_ptr++;
	out_cnt++;
  }

  return(0);
}
