/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This module handles telnet option processing.
 *
 * Author:	Michael Temari, <temari@temari.ae.ge.com>  01/13/93
 *
 */
#include <sys/types.h>
#include <string.h>
#include <sgtty.h>
#include <fcntl.h>
#include <stdio.h>
#include "telnet.h"


#include "arpa/telnet.h"

extern int opt_t;

#define	LASTTELOPT	TELOPT_SGA

static int TelROpts[LASTTELOPT+1];
static int TelLOpts[LASTTELOPT+1];

static char *WHAT[] = {
	"WILL", "WONT", "DO  ", "DONT" };

static char *OPTION[LASTTELOPT+1] = {
	"BINARY", "ECHO", "RCP", "SGA" };

static int TelOpts;

static int ThisOpt;

static int telfdout;

static void dowill(), dowont(), dodo(), dodont(), respond(), dodebug();

static struct sgttyb oldtty, newtty, savetty;

extern int opt_d;

static void doraw()
{
   if(!opt_t) {
	newtty.sg_flags &= ~ECHO;
	newtty.sg_flags |= CBREAK;
	newtty.sg_flags |= RAW;
	ioctl(1, TIOCSETP, &newtty);
   }
}

static void docooked()
{
   if(!opt_t) {
	newtty.sg_flags |= ECHO;
	newtty.sg_flags |= CBREAK;
	newtty.sg_flags &= ~RAW;
	ioctl(1, TIOCSETP, &newtty);
   }
}

void tel_init()
{
int i;

   if(!opt_t) {
	ioctl(1, TIOCGETP, &oldtty);
	newtty = oldtty;
   }
   for(i = 0; i <= LASTTELOPT; i++) {
	TelROpts[i] = 0;
	TelLOpts[i] = 0;
   }
   TelOpts = 0;
   if(!opt_t)
	docooked();
}

void tel_stop()
{
   if(!opt_t)
	ioctl(1, TIOCSETP, &oldtty);
}

void tel_save()
{
   ioctl(1, TIOCGETP, &savetty);
   ioctl(1, TIOCSETP, &oldtty);
}

void tel_rest()
{
   ioctl(1, TIOCSETP, &savetty);
}

void telopt(fdout, what, option)
int fdout;
int what;
int option;
{
char buf[3];
int len;

   buf[0] = IAC;
   buf[1] = what;
   buf[2] = option;
   len = 0;

   switch(what) {
	case DO:
		if(option <= LASTTELOPT) {
			TelROpts[option] = 1;
			len = 3;
		}
		break;
	case DONT:
		if(option <= LASTTELOPT) {
			TelROpts[option] = 1;
			len = 3;
		}
		break;
	case WILL:
		if(option <= LASTTELOPT) {
			TelLOpts[option] = 1;
			len = 3;
		}
		break;
	case WONT:
		if(option <= LASTTELOPT) {
			TelLOpts[option] = 1;
			len = 3;
		}
		break;
   }
   if(len > 0) {
	(void) write(fdout, buf, len);
	if(opt_d)
		dodebug("SENT:", what, option);
   }
}

int tel_in(fdout, telout, buffer, len)
int fdout;
int telout;
unsigned char *buffer;
int len;
{
unsigned char *p, *p2;
int size, got_iac;

   telfdout = telout;
   p = buffer;

   while(len > 0) {
	while(len > 0 && TelOpts) {
		DoTelOpt(fdout, (int)*p);
		p++;
		len--;
	}
	if(len == 0) break;
	size = 0; p2 = p; got_iac = 0;
	while(len--) {
		if(*p == IAC) {
			got_iac = 1;
			break;
		}
		p++;
		size++;
	}
	if(size > 0)
		write(fdout, p2, size);
	if(got_iac) {
		TelOpts = 1;
		p++;
	}
   }
}

int tel_out(fdout, buf, size)
int fdout;
char *buf;
int size;
{
char *p;
int got_iac, len;

   p = buf;
   while(size > 0) {
	buf = p;
	got_iac = 0;
	if((p = (char *)memchr(buf, IAC, size)) != (char *)NULL) {
		got_iac = 1;
		p++;
	} else
		p = buf + size;
	len = p - buf;
	if(len > 0)
		(void) write(fdout, buf, len);
	if(got_iac)
		(void) write(fdout, p - 1, 1);
	size = size - len;
   }
}

int DoTelOpt(fdout, c)
int fdout;
int c;
{
   if(TelOpts == 1) {
	switch(c) {
		case WILL:
		case WONT:
		case DO:
		case DONT:
			ThisOpt = c;
			TelOpts++;
			break;
		case IAC:
			write(fdout, &c, 1);
			TelOpts = 0;
			break;
		default:
			TelOpts = 0;
	}
	return(TelOpts);
   }
   if(opt_d)
	dodebug("RCVD:", ThisOpt, c);
   switch(ThisOpt) {
	case WILL:
		dowill(c);
		break;
	case WONT:
		dowont(c);
		break;
	case DO:
		dodo(c);
		break;
	case DONT:
		dodont(c);
		break;
	default:
		TelOpts = 0;
		return(0);
   }
   
   TelOpts = 0;
   return(1);
}

static void dowill(c)
int c;
{
int ack;

   switch(c) {
	case TELOPT_BINARY:
	case TELOPT_ECHO:
	case TELOPT_SGA:
		if(TelROpts[c] == 1)
			return;
		if(c == TELOPT_ECHO)
			doraw();
		TelROpts[c] = 1;
		ack = DO;
		break;
	default:
		ack = DONT;
   }
   respond(ack, c);
}

static void dowont(c)
int c;
{
   if(c <= LASTTELOPT) {
	if(TelROpts[c] == 0)
		return;
	if(c == TELOPT_ECHO)
		docooked();
	TelROpts[c] = 0;
   }
   respond(DONT, c);
}

static void dodo(c)
int c;
{
int ack;

   switch(c) {
	case TELOPT_BINARY:
	case TELOPT_SGA:
		if(TelLOpts[c] == 1)
			return;
		TelLOpts[c] = 1;
		ack = WILL;
		break;
	default:
		ack = WONT;
   }
   respond(ack, c);
}

static void dodont(c)
int c;
{
   if(c <= LASTTELOPT) {
	if(TelLOpts[c] == 0)
		return;
	TelLOpts[c] = 0;
   }
   respond(WONT, c);
}

static void respond(ack, option)
int ack, option;
{
unsigned char c[3];

   c[0] = IAC;
   c[1] = ack;
   c[2] = option;
   if(opt_d)
	dodebug("SENT:", ack, option);
   write(telfdout, c, 3);
}

static void dodebug(msg, what, option)
char *msg;
int what, option;
{
   fprintf(stderr, "%s ", msg);
   if(what >= WILL && what <= DONT)
	fprintf(stderr, "%s ", WHAT[what - WILL]);
   else
	fprintf(stderr, "%03d  ", what);
   if(option <= LASTTELOPT)
	fprintf(stderr, "%s", OPTION[option]);
   else
	fprintf(stderr, "%03d", option);
   fprintf(stderr, "\n");
}
