/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file contains an implementation of the SMTP
 *		protocol (see RFC-821) for sending electronic mail
 *		across the network.
 *
 * Usage:	smtp [-dv] host
 *
 * Version:	@(#)smtp.c	1.00	07/02/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/wait.h>
#include <arpa/internet.h>
#include <arpa/inet.h>
#include <inet/socket.h>
#include <tnet/tnet.h>
#include <tnet/client.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>


#define SMTP_CFG	"smtp.cfg"
#define SMTP_PORT	25
#define SITELEN		8
#define SPOOLDIR	"/usr/spool/uucp"


static char *Version = "@(#) smtp 1.00 (07/02/92)";


char hostname[128];			/* my full hostname		*/


static char line[1024];
char Cfile[128];
char Dfile[128];
char Mfile[128];
char from[128];
char to[128];
int opt_d = 0;				/* debugging output flag	*/


extern int getopt(), optind, opterr;
extern char *optarg;


_PROTOTYPE( int DOgetreply, (FILE *)					);
_PROTOTYPE( int DOhellomsg, (FILE *)					);
_PROTOTYPE( int DOcommand, (FILE *, FILE *, char *)			);
_PROTOTYPE( int DOhelo, (FILE *, FILE *, char *)			);
_PROTOTYPE( int DOmail, (FILE *, FILE *)				);
_PROTOTYPE( int DOrcpt, (FILE *, FILE *)				);
_PROTOTYPE( int DOdata, (FILE *, FILE *)				);
_PROTOTYPE( int DOsend, (FILE *, FILE *, FILE *)			);
_PROTOTYPE( int DOdot, (FILE *, FILE *)					);
_PROTOTYPE( int DOquit, (FILE *, FILE *)				);
_PROTOTYPE( int readcfg, (FILE *, char *, char *)			);
_PROTOTYPE( int DOfinish, (void)					);
_PROTOTYPE( int getto, (FILE *)						);
_PROTOTYPE( int getfrom, (FILE *)					);
_PROTOTYPE( int getwork, (DIR **, char *, FILE **)			);
_PROTOTYPE( int smtp, (char *, char *)					);


int DOgetreply(fpin)
FILE *fpin;
{
  register char *sp;

  do {
	if (fgets(line, sizeof(line), fpin) == (char *)NULL) return(-1);
  } while(line[3] == '-');

  /* Remove the network's CR from the line. */
  if ((sp = strchr(line, '\r')) != (char *)NULL) {
	*sp++ = '\n';
	*sp = '\0';
  }

  if (opt_d == 1) fprintf(stderr, ">>> %s", line);
  return(atoi(line));
}


int hellomsg(fpin)
FILE *fpin;
{
  return(DOgetreply(fpin));
}


int DOcommand(fpin, fpout, command)
FILE *fpin, *fpout;
char *command;
{
  register char *sp;

  fprintf(fpout, command);
  (void) fflush(fpout);
  if (opt_d == 1) {
	/* Remove the network's CR from the line. */
	if ((sp = strchr(command, '\r')) != (char *)NULL) {
		*sp++ = '\n';
		*sp = '\0';
	}
	fprintf(stderr, "<<< %s", command);
  }
  return(DOgetreply(fpin));
}


int DOhelo(fpin, fpout, host)
FILE *fpin, *fpout;
char *host;
{
  char command[128];

  sprintf(command, "HELO %s\r\n", host);
  return(DOcommand(fpin, fpout, command));
}


int DOmail(fpin, fpout)
FILE *fpin, *fpout;
{
  char command[128];

  sprintf(command, "MAIL From:%s\r\n", from);
  return(DOcommand(fpin, fpout, command));
}


int DOrcpt(fpin, fpout)
FILE *fpin, *fpout;
{
  char command[128];

  sprintf(command, "RCPT To:%s\r\n", to);
  return(DOcommand(fpin, fpout, command));
}


int DOdata(fpin, fpout)
FILE *fpin, *fpout;
{
  return(DOcommand(fpin, fpout, "DATA\r\n"));
}


int DOsend(fpin, fpout, fpmail)
FILE *fpin, *fpout, *fpmail;
{
  int firstline = 1;		/* don't send uucp From line */

  while(fgets(line, sizeof(line), fpmail) != (char *)NULL) {
	if (!firstline) {
		line[strlen(line) - 1] = '\0';
		if (line[0] == '.')
			fprintf(fpout, ".%s\r\n", line);
		else
			fprintf(fpout, "%s\r\n", line);
		if (opt_d == 1)
			fprintf(stderr, "<<< %s\n", line);
	}
	firstline = 0;
  }
  (void) fflush(fpout);
  (void) fclose(fpmail);
  return(0);
}


int DOdot(fpin, fpout)
FILE *fpin, *fpout;
{
  return(DOcommand(fpin, fpout, ".\r\n"));
}


int DOquit(fpin, fpout)
FILE *fpin, *fpout;
{
  return(DOcommand(fpin, fpout, "QUIT\r\n"));
}


int readcfg(fpcfg, hostp, relayp)
FILE *fpcfg;
char *hostp;
char *relayp;
{
  char buff[256];
  register char *p;
  char *hp, *rp;

  while(1) {
	if (fgets(buff, 256, fpcfg) == (char *)NULL) return(0);
	if (buff[0] != '#' && buff[0] != '\n')
		break;
  }
  if ((p = strchr(buff, '\r')) != (char *)NULL) *p = '\0';

  p = buff;
  hp = hostp; rp = relayp;
  while(*p && isspace(*p)) p++;
  while(*p && !isspace(*p)) *hp++ = *p++;
  *hp = '\0';
  while(*p && isspace(*p)) p++;
  while(*p && !isspace(*p)) *rp++ = *p++;
  *rp = '\0';
  if (*relayp == '\0') strcpy(relayp, hostp);

  return(1);
}


int DOfinish()
{
  (void) unlink(Cfile);
  (void) unlink(Dfile);
  (void) unlink(Mfile);
  return(1);
}


int getto(fp)
FILE *fp;
{
  char *p, *t;

  to[0] = '\0';
  while(fgets(line, sizeof(line), fp) != (char *)NULL) {
	if (!strncmp(line, "C rmail ", 8)) {
		line[strlen(line) - 1] = '\0';
		t = to;
		p = line + 8;
		*t++ = '<';
		while(*p && isspace(*p)) p++;
		while(*p && !isspace(*p)) *t++ = *p++;
		*t++ = '>';
		break;
	}
  }
  *t = '\0';
  if (!*to) return(0);
    else return(1);
}


int getfrom(fpmail)
FILE *fpmail;
{
  char *p, *t;

  from[0] = '\0';
  while(fgets(line, sizeof(line), fpmail) != (char *)NULL) {
	if (!strncmp(line, "From: ", 6)) {
		line[strlen(line) - 1] = '\0';
		t = from;
		if ((p = strchr(line + 6, '<')) != (char *)NULL) {
			do {
				*t++ = *p++;
			} while(*p && *p != '>');
			*t++ = '>';
			break;
		}
		p = line + 6;
		*t++ = '<';
		while(*p && isspace(*p)) p++;
		while(*p && !isspace(*p)) *t++ = *p++;
		*t++ = '>';
		break;
	}
  }
  *t = '\0';
  (void) rewind(fpmail);
  if (!*from) return(0);
    else return(1);
}


int getwork(spooldir, site, fpmail)
DIR **spooldir;
char *site;
FILE **fpmail;
{
  FILE *fpC, *fpD;
  struct dirent *dp;
  int len, status;
  char junk[64], name[128];

  if (*spooldir == (DIR *)NULL) {
	if ((*spooldir = opendir(SPOOLDIR)) == (DIR *)NULL) return(0);
  }
  strcpy(name, "C.");
  strncat(name, site, SITELEN);
  len = strlen(name);

LOOP:
  Cfile[0] = '\0';
  Dfile[0] = '\0';
  Mfile[0] = '\0';
  status = 0;

  while((dp = readdir(*spooldir)) != (struct dirent *)NULL) {
	if (!strncmp(dp->d_name, name, len)) {
		sprintf(Cfile, "%s/%s", SPOOLDIR, dp->d_name);
		status = 1;
		break;
	}
  }
  if (!status) {
	(void) closedir(*spooldir);
	return(0);
  }
  if (opt_d == 1) fprintf(stderr, "Cfile: %s\n", Cfile);
  if ((fpC = fopen(Cfile, "r")) == (FILE *)NULL) goto LOOP;

  (void) fgets(line, sizeof(line), fpC);
  (void) sscanf(line, "%s %s", junk, name);
  sprintf(Mfile, "%s/%s", SPOOLDIR, name);

  (void) fgets(line, sizeof(line), fpC);
  (void) sscanf(line, "%s %s", junk, name);
  sprintf(Dfile, "%s/%s", SPOOLDIR, name);

  (void) fclose(fpC);
  if ((fpD = fopen(Dfile, "r")) == (FILE *)NULL) goto LOOP;

  if (!getto(fpD)) {
	(void) fclose(fpD);
	goto LOOP;
  }
  (void) fclose(fpD);
  if ((*fpmail = fopen(Mfile, "r")) == (FILE *)NULL) goto LOOP;

  if (!getfrom(*fpmail)) {
	(void) fclose(*fpmail);
	goto LOOP;
  }
  return(1);
}


/*
 * Call a host using the SMTP protocol.
 */
int smtp(host, relay)
char *host;
char *relay;
{
  char buff[128];
  u_long rmt_addr;
  u_short rmt_port;
  struct hostent *hp;
  struct servent *sp;
  FILE *fpin, *fpout;
  FILE *fpmail;
  DIR *DIRspool;
  int fd_in, fd_out, s;
  int OK, status;

  DIRspool = (DIR *)NULL;
  if (!getwork(&DIRspool, host, &fpmail)) return(0);

  if (opt_d == 1) fprintf(stderr, "Resolving %s...", relay);

  if ((hp = gethostbyname(relay)) == (struct hostent *)NULL) {
	fprintf(stderr, "Unknown host %s!\n", relay);  
	return(-1);
  } else {
	memcpy((char *) &rmt_addr, (char *) hp->h_addr, hp->h_length);
	strcpy(relay, hp->h_name);
  }

  /* Now, to which port must we connect? */
  if ((sp = getservbyname("smtp", "tcp")) == (struct servent *)NULL) {
	fprintf(stderr, "SMTP port unknown, using %d\n", SMTP_PORT);
	rmt_port = SMTP_PORT;
  } else rmt_port = sp->s_port;

  strcpy(buff, inet_ntoa(rmt_addr));
  if (opt_d == 1) fprintf(stderr, " trying %s:%d...\n", buff, rmt_port);

  s = client(TCP_PTCL, 0, rmt_port, rmt_addr, &fd_in, &fd_out);
  if (s != 0) {
	fprintf(stderr, "Connection failed: status = %d\n", s);
	return(s);
  }
  fpin = fdopen(fd_in, "r");
  fpout = fdopen(fd_out, "w");

  status = hellomsg(fpin);
  if (status != 220) {
	fprintf(stderr, "INTRO expect error (%d): ", status);
	fprintf(stderr, "%s\n", line);
	(void) fclose(fpmail);
	(void) fclose(fpin);
	(void) fclose(fpout);
	return(-1);
  }

  status = DOhelo(fpin, fpout, hostname);
  if (status == 250) {
	do {
		status = DOmail(fpin, fpout);
		if (status != 250) {
			fprintf(stderr, line);
			(void) fclose(fpmail);
			continue;
		}
		status = DOrcpt(fpin, fpout);
		if (status != 250) {
			fprintf(stderr, line);
			(void) fclose(fpmail);
			continue;
		}
		status = DOdata(fpin, fpout);
		if (status != 354) {
			fprintf(stderr, line);
			(void) fclose(fpmail);
			continue;
		}
		status = DOsend(fpin, fpout, fpmail);
		if (status) {
			fprintf(stderr, line);
			(void) fclose(fpmail);
			continue;
		}
		status = DOdot(fpin, fpout);
		if (status != 250) {
			fprintf(stderr, line);
			(void) fclose(fpmail);
			continue;
		}

		status = DOfinish();
		(void) fclose(fpmail);
	} while(getwork(&DIRspool, host, &fpmail));
  }

  (void) DOquit(fpin, fpout);
  (void) fclose(fpin); (void) fclose(fpout);

  if (opt_d == 1) fprintf(stderr, "SMTP: all work done for %s\n", host);
  return(0);
}


void usage()
{
  fprintf(stderr, "Usage: smtp [-dv]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char **argv;
{
  char buf1[128], buf2[64];
  register int c;
  FILE *fpcfg;

  opterr = 0;
  while ((c = getopt(argc, argv, "dv")) != EOF) switch(c) {
	case 'd':
	case 'v':
		opt_d = 1;
		break;
	default:
		usage();
  }

  /* No more arguments allowed. */
  if (optind != argc) usage();

  /* Ask the system what our hostname is. */
  if(gethostname(hostname, 128) < 0)
	strcpy(hostname, "unknown");
  if (opt_d == 1) fprintf(stderr, "Hostname is %s\n", hostname);

  /* Open the SMTP config file. */
  sprintf(buf1, "%s/%s", TNET_DIR, SMTP_CFG);
  if ((fpcfg = fopen(buf1, "r")) == (FILE *)NULL) {
	fprintf(stderr, "SMTP: cannot open cfg file (%s), status=%d\n",
							buf1, errno);
	return(-1);
  }

  /* Start reading the SMTP config file and transfer all jobs. */
  while(readcfg(fpcfg, buf2, buf1)) {
	if (opt_d == 1) fprintf(stderr, "Check host %s (%s)\n", buf2, buf1);
	c = smtp(buf2, buf1);
	if (c < 0) fprintf(stderr, "SMTP: problems with %s, status=%d\n",
						buf1, errno);
  }

  if (opt_d == 1) fprintf(stderr, "SMTP Done.\n");
  return(0);
}
