/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file contains the maintenace commands needed to
 *		set up TNET, and to keep it running.  It has commands
 *		for managing the ARP and ROUTE tables, it can attach
 *		and detach interfaces, and the like.
 *
 * Usage:	netcom [-dv] [command]
 *
 * Version:	@(#)netcom.c	1.00	07/11/92
 *				2.00    01/14/93
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 *              Michael Temari, <temari@temari.ae.ge.com>
 */
#include <sys/types.h>
#include <arpa/internet.h>
#include <arpa/inet.h>
#include <inet/socket.h>
#include <tnet/tnet.h>
#include <tnet/client.h>
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#define EXTRN
#include "netcom.h"


static char *Version = "@(#) netcom 2.00 (01/14/93)";


_PROTOTYPE( int getargs, (char *string, char *arguments[])		);
_PROTOTYPE( int do_halt, (int argc, char **argv)			);
_PROTOTYPE( int do_pause, (int argc, char **argv)			);
_PROTOTYPE( int do_restart, (int argc, char **argv)			);
_PROTOTYPE( void netcom, (void)						);
_PROTOTYPE( void usage, (char *name)					);


static struct _cmd_ {
  char *name;
  int (*func)();
} commands[] = {
  { "help",	do_help		},
  { "?",	do_help		},
  { "halt",	do_halt		},
  { "pause",	do_pause	},
  { "restart",	do_restart	},
  { "ip",	do_ip		},
  { "tcp",	do_tcp		},
  { "attach",	do_attach	},
  { "detach",	do_detach	},
  { "ifconfig",	do_ifconfig	},
  { "arp",	do_arp		},
  { "route",	do_route	},
  { "netstat",	do_stat		},
  { "",         (int (*)()) 0	}
};


extern int getopt(), optind, opterr;
extern char *optarg;


/* Split the input string into multiple fields. */
static int getargs(string, arguments)
char *string;
char *arguments[];
{
  register int i;
  register char *sp;
  int argc;

  sp = string; i = 0;
  arguments[i] = sp;
  while (*sp && i < 32) {
        while (*sp && (*sp == ' ' || *sp == '\t')) sp++;
        arguments[i++] = sp;
        while (*sp && *sp != ' ' && *sp != '\t') sp++;
        if (*sp != '\0') *sp++ = '\0';
  }
  argc = i;
  while (i < 32) arguments[i++] = (char *)NULL;
  return(argc);
}


/* Perform the "HALT" command. */
static int do_halt(argc, argv)
int argc;
char *argv[];
{
  RMTHDR hdr;
  int st, netin;

  if (argc > 0) {
	printf("Usage: halt\n\n");
	return(-1);
  }

  hdr.c_opcode = RMT_EXIT;
  hdr.c_length = 0;

  st = tn_rlink(&hdr, (char *)NULL, 0, 10, &netin);
  if (st < 0) return(st);
  (void) close(netin);
  st = hdr.c_opcode;
  if (st != 0) printf("HALT failed; error code is %d\n", st);
    else printf("TNET has been shut down.\n");
  return(0);
}


/* Perform the "PAUSE" command. */
static int do_pause(argc, argv)
int argc;
char *argv[];
{
  RMTHDR hdr;
  int st, netin;

  if (argc > 0) {
	printf("Usage: pause\n\n");
	return(-1);
  }

  hdr.c_opcode = RMT_STOP;
  hdr.c_length = 0;

  st = tn_rlink(&hdr, (char *)NULL, 0, 10, &netin);
  if (st < 0) return(st);
  (void) close(netin);
  st = hdr.c_opcode;
  if (st != 0) printf("PAUSE failed; error code is %d\n", st);
    else printf("TNET is now in PAUSE mode.\n");
  return(0);
}


/* Perform the "RESTART" command. */
static int do_restart(argc, argv)
int argc;
char *argv[];
{
  RMTHDR hdr;
  int st, netin;

  if (argc > 0) {
	printf("Usage: restart\n\n");
	return(-1);
  }

  hdr.c_opcode = RMT_START;
  hdr.c_length = 0;

  st = tn_rlink(&hdr, (char *)NULL, 0, 10, &netin);
  if (st < 0) return(st);
  (void) close(netin);
  st = hdr.c_opcode;
  if (st != 0) printf("RESTART failed; error code is %d\n", st);
    else printf("TNET is running again.\n");
  return(0);
}


/* Run an interactive NETCOM shell. */
static void netcom()
{
  char cline[128];
  char *argv[32];
  int argc, i;
  int running;
  register char *sp;

  printf("%s\n\n", Version+5);
  running = 1;
  do {
	printf("netcom> ");
	fflush(stdout);
        if (fgets(cline, 128, stdin) == (char *)NULL) break;
	if ((sp = strchr(cline, '\n')) != (char *)NULL) *sp = '\0';
	if (cline[0] == '#' || cline[0] == '\0') continue;

	if ((argc = getargs(cline, argv)) == 0) continue;

	/* Now, check which command it is. */
	if (strcmp(argv[0], "exit") != 0) {
		i = 0;
		while (commands[i].name[0] != '\0') {
			if (!strcmp(commands[i].name, argv[0])) break;
			i++;
		}
		if (commands[i].name[0] != '\0') {
			(*commands[i].func)(--argc, &argv[1]);
		} else printf("? Unknown command (%s).\n", argv[0]);
	} else running = 0;
  } while(running);

  printf("NETCOM says goodbye...\n");
}


void usage(name)
char *name;
{
  fprintf(stderr, "Usage: netcom [-dv] [command]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char command[1024];
  register int c;

  if ((progname = strrchr(argv[0], '/')) != (char *)NULL) progname++;
    else progname = argv[0];

  opterr = 0;
  opt_d = 0;
  while ((c = getopt(argc, argv, "dv")) != EOF) switch(c) {
	case 'd':
	case 'v':
		opt_d = 1;
		break;
	default:
		usage(progname);
  }

  /* If no more arguments, start an interactive shell. */
  if ((!strcmp(progname, "netcom") && (optind == argc)) ||
      (argc > optind && !strcmp(argv[optind], "netcom"))) {
	netcom();
	return(0);
  }

  /* One or more arguments are given.  */
  if (strcmp(progname, "netcom")) {
	c = 0;
	while (commands[c].name[0] != '\0') {
		if (!strcmp(commands[c].name, progname)) break;
		c++;
	}
	if (commands[c].name[0] != '\0') {
		(*commands[c].func)((argc - optind), &argv[optind]);
	} else printf("? Bad link to NETCOM (%s).\n", progname);
  } else {
	c = 0;
	while (commands[c].name[0] != '\0') {
		if (!strcmp(commands[c].name, argv[optind])) break;
		c++;
	}
	if (commands[c].name[0] != '\0') {
		(*commands[c].func)((argc - optind - 1), &argv[++optind]);
	} else printf("? Unknown NETCOM command (%s).\n", argv[optind]);
  }
  return(0);
}
