/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file takes care of communicating with the TNET
 *		kernel.  It is the lowest part of the TNET library,
 *		since it is called by all other functions that want
 *		to do something with TNET.
 *
 * Version:	@(#)tnet/rmtlink.c	1.00		07/11/92
 *
 * Authors:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 *		Michael Temari, <temari@temari.ae.ge.com>
 */
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <inet/in.h>
#include <inet/socket.h>
#include <tnet/tnet.h>
#include <tnet/client.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>


struct {
  RMTHDR	header;
  char		data[1024];
} block;


static int timed_out;			/* tells if the READ failed	*/


/* This function catches the ALRM signal when the timer expires. */
static int my_catch()
{
  (void) signal(SIGALRM, SIG_IGN);
  timed_out = 1;
  return(-1);
}


/*
 * This functions performs the transaction with the TNET COMMAND channel.
 * The first parameter is a pointer to the message header (which has to
 * be filled in by the called, except for the three "system" fields, which
 * we will do here).  The second parameter is a pointer to a buffer in
 * which any user data can be found, or where any result user data can be
 * stashed.  The third parameter tells how large this buffer is.  When
 * sending a message, this size tells whether there is any user data to
 * be sent at all.  When receiving the result, this is the maximum amount
 * of data that can be passed to the caller.  The remaining bytes of the
 * result data are read from the channel, but discarded completely.
 *
 * The actual number of "return data" bytes is returned.
 */
int tn_rlink(ptr, bufp, buflen, timeout, fd_in)
RMTHDR *ptr;
char *bufp;
int buflen;
int timeout;
int *fd_in;
{
  char my_pipe[32];
  char my_name[128], tnet[128];
  int fdin, fdout;
  int size, s, i;

  /* Fill in the rest of the message. */
  ptr->c_pid = getpid();
  ptr->c_uid = geteuid();

  /* Create a unique name for my return channel. */
  sprintf(my_pipe, TNET_CIN, ptr->c_pid);
#ifdef DEBUG
  fprintf(stderr, "LNK: my_pipe=\"%s\"\n", my_pipe);
#endif
  strncpy(ptr->c_return, my_pipe, 14);

  /* Create my return channel and open it for reading. */
  sprintf(my_name, "%s/%s/%s", TNET_DIR, TNET_PDIR, my_pipe);
#ifdef DEBUG
  fprintf(stderr, "LNK: my_name=\"%s\"\n", my_name);
#endif
  if (mkfifo(my_name, 0666) == -1 && errno != EEXIST) return(ECLIXPIPE);
  if (chmod(my_name, 0666) < 0) {
	(void) unlink(my_name);
	return(ECLIXPIPE);
  }

  /* Create the command block. */
  memcpy((char *) &block.header, (char *) ptr, RMTHDR_LEN);
  i = RMTHDR_LEN;
  if (ptr->c_length > 0) {
	memcpy((char *) block.data, bufp, ptr->c_length);
	i += ptr->c_length;
  }
  
  /* Open the TNET COMMAND channel. */
  sprintf(tnet, "%s/%s", TNET_DIR, TNET_PIPE);
#ifdef DEBUG
  fprintf(stderr, "LNK: tnet=\"%s\"\n", tnet);
#endif
  if ((fdout = open(tnet, O_WRONLY)) < 0) {
	(void) close(fdin);
	return(ECLITNETP);
  }

  /* Send the command and its data to the TNET COMMAND channel. */
  s = write(fdout, (char *) &block, i);
#ifdef DEBUG
  fprintf(stderr, "LNK: CMD(%d, %d) == %d\n", fdout, i, s);
#endif
  if (s != i) {
	(void) close(fdin);
	return(ECLIWCMD);
  }

  /* Close the COMMAND channel. */
  (void) close(fdout);

  /* Set an alarm timer, if any is needed. */
  (void) signal(SIGALRM, my_catch);
  timed_out = 0;
  if (timeout > 0) (void) alarm(timeout);
#ifdef DEBUG
  fprintf(stderr, "LNK: TMR(%d) SET\n", timeout);
#endif

  /* Open the return channel. */
  fdin = open(my_name, O_RDONLY);
  (void) alarm(0);
#ifdef DEBUG
  fprintf(stderr, "LNK: TMR CLR\n");
#endif
  (void) unlink(my_name);
  if (fdin < 0) return(ECLIRETPIPE);

  /* Set an alarm timer, if any is needed. */
  timed_out = 0;
  if (timeout > 0) (void) alarm(timeout);
#ifdef DEBUG
  fprintf(stderr, "LNK: TMR(%d) SET\n", timeout);
#endif

  /* Start a READ on the return channel. */
  s = read(fdin, (char *) ptr, RMTHDR_LEN);
#ifdef DEBUG
  fprintf(stderr, "LNK: READ(%d, %d) == %d ; TIMEOUT=%d\n",
				fdin, RMTHDR_LEN, s, timed_out);
#endif

  /* We returned. Did we time-out??? */
  if (timed_out == 1) {
	(void) close(fdin);
	return(ETIMEDOUT);
  }

  /* No, so the result is valid. */
  (void) alarm(0);
#ifdef DEBUG
  fprintf(stderr, "LNK: TMR CLR\n");
#endif
  if (s != RMTHDR_LEN) {
	(void) close(fdin);
	return(ECLIRSTATUS);
  }

  /* Now, read any result user data. */
#ifdef DEBUG
  fprintf(stderr, "LNK: RETURN LEN == %d\n", ptr->c_length);
#endif
  if ((size = ptr->c_length) > 0) {
	if ((buflen == 0) || (bufp == (char *)NULL)) {
		(void) close(fdin);
		return(EFAULT);
	}
	i = (buflen > size) ? size : buflen;
#ifdef DEBUG
	fprintf(stderr, "LNK: RETURN MAX == %d\n", i);
#endif
	timed_out = 0;
	if (timeout > 0) (void) alarm(timeout);
#ifdef DEBUG
	fprintf(stderr, "LNK: RETURN TMR(%d) SET\n", timeout);
#endif

	s = read(fdin, (char *) bufp, i);
#ifdef DEBUG
	fprintf(stderr, "LNK: RETURN READ(%d, %d) == %d ; TIMEOUT=%d\n",
						fdin, i, s, timed_out);
#endif
	if (timed_out == 1) {
		(void) close(fdin);
		return(ETIMEDOUT);
	}
	(void) alarm(0);
	if (s != i) {
		(void) close(fdin);
		return(ECLIRSTATUS);
	}

	/* Pick up the remaining stuff we cannot stash to the caller. */
	size = i;
	if (i < ptr->c_length) {
		i = (ptr->c_length - i);
		while(i > 0) {
#ifdef DEBUG
			fprintf(stderr, "LNK: RET: LEFT == %d\n", i);
#endif
			timed_out = 0;
			if (timeout > 0) (void) alarm(timeout);
#ifdef DEBUG
			fprintf(stderr, "LNK: RET TMR(%d) SET\n",
								timeout);
#endif
			s = read(fdin, (char *) tnet, (i > 128) ? 128 : i);
#ifdef DEBUG
			fprintf(stderr, "LNK: RET RD(%d)==%d TIMEOUT=%d\n",
							fdin, s, timed_out);
#endif
			if (timed_out == 1) {
				(void) close(fdin);
				return(ETIMEDOUT);
			}
			(void) alarm(0);
			if (s != i) {
				(void) close(fdin);
				return(ECLIRSTATUS);
			}
			i -= 128;
		}
	}
  }
#ifdef DEBUG
  fprintf(stderr, "LNK: DONE == %d\n", size);
#endif

  /* Stash the newly acquired file descriptor and return. */
  if (fd_in != (int *)NULL) *fd_in = fdin;
    else (void) close(fdin);

  return(size);
}
