/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles lookups in the /etc/hosts database.
 *		It is used to map from host/domain names to Internet
 *		addresses.
 *
 * Usage:	If the NETWORKED macro is defined at compile time,
 *		this file will also try to ask a name server to do
 *		the work.
 *
 * Version:	@(#)inet/gethostent.c	1.00		07/11/92
 *
 * Authors:	Original taken from BSD 4.3/TAHOE.
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <arpa/internet.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <inet/socket.h>
#include <inet/in.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <resolv.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define	MAXALIASES	35
#define	MAXADDRS	35

#ifdef NETWORKED
#define	MAXPACKET	1024

#define AF_UNSPEC	(-1)

typedef union {
  HEADER	hdr;
  unsigned char	buf[MAXPACKET];
} querybuf;

typedef union {
  long		al;
  char		ac;
} align;
#endif


#if _MINIX
#   define _gethtbyname	_ghbname
#   define _gethtbyaddr	_ghbaddr
#endif

static char		*h_addrptrs[MAXADDRS + 1];
static struct hostent	host;
static char		*aliases[MAXALIASES];
static char		hostbuf[BUFSIZ+1];
static struct in_addr	hx_addr;
static FILE		*hostf = (FILE *)NULL;
static char		*hstname = _PATH_HOSTS;
static char		hostaddr[MAXADDRS];
static char		*h_addrs[2];
static int		stayopen = 0;
       int		h_errno;


_PROTOTYPE( int strcamecmp, (char *, char *)				);
_PROTOTYPE( struct hostent *_gethtent, (void)				);
_PROTOTYPE( void _sethtent, (int)					);
_PROTOTYPE( void _endhtent, (void)					);
_PROTOTYPE( struct hostent *_gethtbyname, (char *)			);
_PROTOTYPE( struct hostent *_gethtbyaddr, (char *, int, int)		);


static int strcasecmp(s1, s2)
register char *s1, *s2;
{
  char buf1[128];
  char buf2[128];
  register char *sp;

  sp = buf1;
  while (*s1) {
	if (*s1 >= 'a' && *s1 <= 'z') *sp++ = *s1 - 'a' + 'A';
	  else *sp++ = *s1;
	*s1++;
  }
  *sp = '\0';
  sp = buf2;
  while (*s2) {
	if (*s2 >= 'a' && *s2 <= 'z') *sp++ = *s2 - 'a' + 'A';
	  else *sp++ = *s2;
	*s2++;
  }
  *sp = '\0';
  return(strcmp(buf1, buf2));
}


#ifdef NETWORKED
static struct hostent *getanswer(answer, anslen, iquery)
querybuf *answer;
int anslen;
int iquery;
{
  register HEADER *hp;
  register unsigned char *cp;
  register int n;
  unsigned char *eom;
  char *bp, **ap;
  int type, class, buflen, ancount, qdcount;
  int haveanswer, getclass = C_ANY;
  char **hap;

  eom = answer->buf + anslen;

  /*
   * find first satisfactory answer
   */
  hp = &answer->hdr;
  ancount = ntohs(hp->ancount);
  qdcount = ntohs(hp->qdcount);
  bp = hostbuf;
  buflen = sizeof(hostbuf);
  cp = answer->buf + sizeof(HEADER);
  if (qdcount) {
	if (iquery) {
		if ((n = dn_expand((char *)answer->buf, eom,
					     cp, bp, buflen)) < 0) {
			h_errno = NO_RECOVERY;
			return((struct hostent *)NULL);
		}
		cp += n + QFIXEDSZ;
		host.h_name = bp;
		n = strlen(bp) + 1;
		bp += n;
		buflen -= n;
	} else cp += dn_skipname(cp, eom) + QFIXEDSZ;
	while (--qdcount > 0)
		cp += dn_skipname(cp, eom) + QFIXEDSZ;
  } else if (iquery) {
	if (hp->third & THIRD_AA) h_errno = HOST_NOT_FOUND;
	  else h_errno = TRY_AGAIN;
	return((struct hostent *)NULL);
  }
  ap = aliases;
  *ap = (char *)NULL;
  host.h_aliases = aliases;
  hap = h_addrptrs;
  *hap = (char *)NULL;
  host.h_addr_list = h_addrptrs;
  haveanswer = 0;
  while (--ancount >= 0 && cp < eom) {
	if ((n = dn_expand((char *)answer->buf, eom, cp, bp, buflen)) < 0)
		break;
	cp += n;
	type = _getshort(cp);
 	cp += sizeof(u_short);
	class = _getshort(cp);
 	cp += sizeof(u_short) + sizeof(unsigned long);
	n = _getshort(cp);
	cp += sizeof(u_short);
	if (type == T_CNAME) {
		cp += n;
		if (ap >= &host.h_aliases[MAXALIASES-1]) continue;
		*ap++ = bp;
		n = strlen(bp) + 1;
		bp += n;
		buflen -= n;
		continue;
	}
	if (iquery && type == T_PTR) {
		if ((n = dn_expand((char *)answer->buf, eom,
					    cp, bp, buflen)) < 0) {
			cp += n;
			continue;
		}
		cp += n;
		host.h_name = bp;
		return(&host);
	}
	if (iquery || type != T_A)  {
		cp += n;
		continue;
	}
	if (haveanswer) {
		if (n != host.h_length) {
			cp += n;
			continue;
		}
		if (class != getclass) {
			cp += n;
			continue;
		}
	} else {
		host.h_length = n;
		getclass = class;
		host.h_addrtype = (class == C_IN) ? AF_INET : AF_UNSPEC;
		if (!iquery) {
			host.h_name = bp;
			bp += strlen(bp) + 1;
		}
	}

#if 0	/* sizeof(int) == sizeof(long) */
	bp += sizeof(align) - ((unsigned long)bp % sizeof(align));
#else	/* sizeof(int) != sizeof(long) */
	bp += sizeof(align) - ((unsigned int)bp % sizeof(align));
#endif

	if (bp + n >= &hostbuf[sizeof(hostbuf)]) break;

	bcopy(cp, *hap++ = bp, n);
	bp +=n;
	cp += n;
	haveanswer++;
  }
  if (haveanswer) {
	*ap = (char *)NULL;
	*hap = (char *)NULL;
	host.h_addr = h_addrptrs[0];
	return(&host);
  } else {
	h_errno = TRY_AGAIN;
	return((struct hostent *)NULL);
  }
}
#endif /* NETWORKED */


static struct hostent *_gethtent()
{
  char *p;
  register char *cp, **q;

  if (hostf == (FILE *)NULL &&
      (hostf = fopen(hstname, "r")) == (FILE *)NULL)
					return((struct hostent *)NULL);
again:
  if ((p = fgets(hostbuf, BUFSIZ, hostf)) == (char *)NULL)
					return((struct hostent *)NULL);
  if (*p == '#') goto again;
  cp = strpbrk(p, "#\n");
  if (cp == (char *)NULL) goto again;
  *cp = '\0';
  cp = strpbrk(p, " \t");
  if (cp == (char *)NULL) goto again;
  *cp++ = '\0';
  host.h_addr_list = h_addrs;
  host.h_addr = hostaddr;
  *((unsigned long *)host.h_addr) = inet_addr(p);
  host.h_length = sizeof (unsigned long);
  host.h_addrtype = AF_INET;
  while (*cp == ' ' || *cp == '\t') cp++;
  host.h_name = cp;
  q = host.h_aliases = aliases;
  cp = strpbrk(cp, " \t");
  if (cp != (char *)NULL) *cp++ = '\0';
  while (cp && *cp) {
	if (*cp == ' ' || *cp == '\t') {
		cp++;
		continue;
	}
	if (q < &aliases[MAXALIASES - 1]) *q++ = cp;
	cp = strpbrk(cp, " \t");
	if (cp != (char *)NULL) *cp++ = '\0';
  }
  *q = (char *)NULL;

  return(&host);
}


static void _sethtent(f)
int f;
{
  if (hostf == (FILE *)NULL) hostf = fopen(hstname, "r");
    else rewind(hostf);
  stayopen |= f;
}


static void _endhtent()
{
  if (hostf && !stayopen) {
	(void) fclose(hostf);
	hostf = (FILE *)NULL;
  }
}


static struct hostent *_gethtbyname(name)
char *name;
{
  register struct hostent *p;
  register char **cp;
	
  _sethtent(0);
  while (p = _gethtent()) {
	if (strcasecmp(p->h_name, name) == 0) break;
	for (cp = p->h_aliases; *cp != 0; cp++)
		if (strcasecmp(*cp, name) == 0) goto found;
  }
found:
  _endhtent();
  return(p);
}


static struct hostent *_gethtbyaddr(addr, len, type)
char *addr;
int len, type;
{
  register struct hostent *p;

  _sethtent(0);
  while (p = _gethtent()) {
	if (p->h_addrtype == type && !bcmp(p->h_addr, addr, len)) break;
  }
  _endhtent();
  return(p);
}


void sethostent(stayopen)
int stayopen;
{
#ifdef NETWORKED
  if (stayopen) _res.options |= RES_STAYOPEN | RES_USEVC;
#else
  _sethtent(stayopen);
#endif
}


void endhostent()
{
#ifdef NETWORKED
  _res.options &= ~(RES_STAYOPEN | RES_USEVC);
  _res_close();
#else
  _sethtent(stayopen);
#endif
}


void sethostfile(name)
char *name;
{
  if (name == (char *)NULL) hstname = _PATH_HOSTS;
    else hstname = name;
}


struct hostent *gethostbyname(name)
char *name;
{
#ifdef NETWORKED
  querybuf buf;
#endif
  register char *cp;
  int n;

  /*
   * disallow names consisting only of digits/dots, unless they end in a dot.
   */
  if (isdigit(name[0]))
	for (cp = name;; ++cp) {
		if (!*cp) {
			if (*--cp == '.') break;
			/*
			 * All-numeric, no dot at the end.
			 * Fake up a hostent as if we'd actually
			 * done a lookup.  What if someone types
			 * 255.255.255.255?  The test below will
			 * succeed spuriously... ???
			 */
			if ((hx_addr.s_addr = inet_addr(name)) == -1) {
				h_errno = HOST_NOT_FOUND;
				return((struct hostent *)NULL);
			}
			host.h_name = name;
			host.h_aliases = aliases;
			aliases[0] = (char *)NULL;
			host.h_addrtype = AF_INET;
			host.h_length = sizeof(unsigned long);
			h_addrptrs[0] = (char *)&hx_addr;
			h_addrptrs[1] = (char *)NULL;
			host.h_addr_list = h_addrptrs;
			return(&host);
		}
		if (!isdigit(*cp) && *cp != '.') break;
	}

#ifdef NETWORKED
	if ((n = res_search(name, C_IN, T_A, buf.buf, sizeof(buf))) < 0) {
		if (errno == ECONNREFUSED)
#endif
			return(_gethtbyname(name));
#ifdef NETWORKED
		  else return((struct hostent *)NULL);
	}
  return(getanswer(&buf, n, 0));
#else
  return((struct hostent *)NULL);
#endif
}


struct hostent *gethostbyaddr(addr, len, type)
char *addr;
int len, type;
{
  int n;
#ifdef NETWORKED
  querybuf buf;
  char qbuf[MAXDNAME];
#endif
  register struct hostent *hp;
	
  if (type != AF_INET) return((struct hostent *)NULL);
#ifdef NETWORKED
  (void)sprintf(qbuf, "%u.%u.%u.%u.in-addr.arpa",
	((unsigned)addr[3] & 0xff),
	((unsigned)addr[2] & 0xff),
	((unsigned)addr[1] & 0xff),
	((unsigned)addr[0] & 0xff));
  n = res_query(qbuf, C_IN, T_PTR, (char *)&buf, sizeof(buf));
  if (n < 0) {
	if (errno == ECONNREFUSED)
#endif
		return(_gethtbyaddr(addr, len, type));
#ifdef NETWORKED
	return((struct hostent *)NULL);
  }
  hp = getanswer(&buf, n, 1);
  if (hp == NULL) return((struct hostent *)NULL);
  hp->h_addrtype = type;
  hp->h_length = len;
  h_addrptrs[0] = (char *)&hx_addr;
  h_addrptrs[1] = (char *)0;
  hx_addr = *(struct in_addr *)addr;
  return(hp);
#endif
}
