/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the external clients and servers.
 *
 * Version:	@(#)nproc.c		1.00	07/08/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <sys/wait.h>
#include <arpa/internet.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>

#include "mbuf.h"
#include "timer.h"
#include "icmp.h"
#include "netuser.h"
#include "tcp.h"
#include "inetd.h"
#include "nproc.h"
#include "udp.h"


struct udp_hdr {
  int size;
  struct socket lsocket;
  struct socket fsocket;
};


static char buffer[4096];
struct nproc nproc[NNPROC];
int16 lport = 1001;


_PROTOTYPE( void in_char, (struct tcb *, int16)				);
_PROTOTYPE( void send_okay, (struct tcb *, int16)			);
_PROTOTYPE( void chg_state, (struct tcb *, char, char)			);
_PROTOTYPE( void data_in, (struct mbuf *, int)				);
_PROTOTYPE( void udp_in, (struct socket *, int)				);


static void udp_in(lsocket, cnt)
struct socket *lsocket;
int cnt;
{
  struct socket fsocket;
  struct mbuf *bp;
  char *p;
  int i, total, size;
  struct udp_hdr udp_hdr;

  if (recv_udp(lsocket, &fsocket, &bp) <= 0) return;

  for(i = 0; i < NNPROC; i++) {
	if (nproc[i].type == UDP_PTCL &&
	   nproc[i].lsocket.address == lsocket->address &&
	   nproc[i].lsocket.port    == lsocket->port)
		break;
  }

  if (i >= NNPROC) {
	free_p(bp);
	return;
  }
  p = buffer;
  udp_hdr.size = len_mbuf(bp);
  memcpy(&udp_hdr.lsocket, lsocket, sizeof(struct socket));
  memcpy(&udp_hdr.fsocket, &fsocket, sizeof(struct socket));

  memcpy(p, &udp_hdr, sizeof(udp_hdr));
  p = p + sizeof(udp_hdr);
  total = dqdata(bp, p, udp_hdr.size);
  if (total != udp_hdr.size) return;
  total = total + sizeof(udp_hdr);
  if (write(nproc[i].fdout, buffer, total) != total)
	rprintf(2, "udp_in: write error!\n");
}


static void chg_state(tcb, old, new)
struct tcb *tcb;
char old, new;
{
  int i, j;

  for(i = 0; i < NNPROC; i++)
	if (nproc[i].type == TCP_PTCL && tcb == nproc[i].tcb) break;

  switch (new) {
	case ESTABLISHED:
		nproc[i].type = TCP_PTCL;
#if HAVE_DEBUG
		rprintf(2, "Client %d: Established\r\n", i);
#endif
		break;
	case CLOSE_WAIT:
		close_tcp(tcb);
		break;
	case CLOSED:
		del_tcp(tcb);
		freenproc(i);
#if HAVE_DEBUG
		rprintf(2, "Client %d: Closed\r\n", i);
#endif
		break;
#if HAVE_DEBUG
	default:
		rprintf(2, "Client %d: old=%d new=%d\r\n", i, old, new);
		break;
#endif
  }
}


static void in_char(tcb, cnt)
struct tcb *tcb;
int16 cnt;
{
  struct mbuf *bp;
  int i;

  if (recv_tcp(tcb, &bp, cnt) > 0) {
	for(i = 0; i < NNPROC; i++)
		if (nproc[i].type == TCP_PTCL &&
			nproc[i].tcb == tcb) break;
	if (i < NNPROC) data_in(bp, nproc[i].fdout);
  }
}


static void send_okay(tcb, cnt)
struct tcb *tcb;
int16 cnt;
{
  int i;

  for(i = 0; i < NNPROC; i++)
	if (nproc[i].type == TCP_PTCL &&
		nproc[i].tcb == tcb) break;
  if (i < NNPROC) {
	nproc[i].sendok = cnt;
  }
}


static void data_in(bp, fdout)
struct mbuf *bp;
int fdout;
{
  while(bp != NULLBUF) {
	(void) write(fdout, bp->data, bp->cnt);
	bp = free_mbuf(bp);
  }
}


void initnproc()
{
  int i;

  for(i = 0; i < NNPROC; i++)
	nproc[i].type = FREE;
}


/* find a free nproc entry */
int ffnproc()
{
  int i;

  for(i = 0; i < NNPROC; i++) {
	if(nproc[i].type == FREE) {
		nproc[i].sendok = 0;
		nproc[i].fdin = -1;
		nproc[i].fdout = -1;
		nproc[i].pid = -1;
		return(i);
	}
  }
  return(-1);
}


void freenproc(i)
int i;
{
  int pid, pstat;

  if (i < 0 || i >= NNPROC) return;
  if (nproc[i].type == FREE) return;

  switch(nproc[i].type) {
	case TCP_PTCL:
		close_tcp(nproc[i].tcb);
		break;
	case UDP_PTCL:
		del_udp(&nproc[i].lsocket);
		break;
  }
  if (nproc[i].fdin != -1) (void) close(nproc[i].fdin);
  if (nproc[i].fdout != -1) (void) close(nproc[i].fdout);
  if (nproc[i].pid != -1) (void) kill(nproc[i].pid, SIGKILL);

  nproc[i].type = FREE;
  if (nproc[i].pid != -1) {
	pid = wait(&pstat);
#if HAVE_DEBUG
	rprintf(2, "nproc: Pid %d Status %04x\n", pid, pstat);
#endif
  }
}


int cnproc(protocol, localport, port, addr, flags)
int protocol;
int16 *localport;
int16 *port;
int32 *addr;
int32 flags;
{
  int i;
  struct socket lsocket, fsocket;
  struct tcb *tcb;

  i = ffnproc();
  if (i < 0) return(i);

  lsocket.address = ip_addr;
  if (*localport == 0) {
	lsocket.port = lport++;
	*localport = lsocket.port;
  } else lsocket.port = *localport;
  fsocket.address = *addr;
  fsocket.port = *port;

  switch(protocol) {
	case UDP_PTCL:
		if (open_udp(&lsocket, udp_in)) {
			freenproc(i);
			return(-1);
		}
		nproc[i].type = UDP_PTCL;
		nproc[i].lsocket = lsocket;
		nproc[i].tcb = NULLTCB;
		break;
	case TCP_PTCL:
		tcb = open_tcp(&lsocket, &fsocket,
				(flags == 1?TCP_ACTIVE:TCP_PASSIVE),
				0, in_char, send_okay, chg_state, 0,
				(int *)NULL);
		if (tcb == NULLTCB || tcb->state == CLOSED) {
			freenproc(i);
			return(-1);
		}
		nproc[i].type = TCP_PTCL;
		nproc[i].tcb = tcb;
		break;
	default:
		return(-1);
  }
  return(i);
}


void DOnproc()
{
  int i, cnt;
  struct mbuf *bp;
  struct udp_hdr udp_hdr;

  for(i = 0; i < NNPROC; i++)
	switch(nproc[i].type) {
	   case TCP_PTCL:
		if (nproc[i].sendok) {
			while((cnt = read(nproc[i].fdin, buffer,
				min(sizeof(buffer), nproc[i].sendok))) > 0) {
				nproc[i].sendok -= cnt;
				bp = qdata(buffer, cnt);
				send_tcp(nproc[i].tcb, bp);
				if (!nproc[i].sendok) break;
			}
			if (cnt == 0) freenproc(i);
		}
		break;
	   case UDP_PTCL:
		if (!nproc[i].sendok) {
			nproc[i].sendok = 1;
			break;
		}
		cnt = read(nproc[i].fdin, &udp_hdr, sizeof(udp_hdr));
		if (cnt == 0) freenproc(i);
		if (cnt != sizeof(udp_hdr)) break;
		if (udp_hdr.size > 0) {
			cnt = read(nproc[i].fdin, buffer, udp_hdr.size);
			if (cnt == 0) freenproc(i);
			if (cnt != udp_hdr.size) break;
			bp = qdata(buffer, udp_hdr.size);
		} else bp = NULLBUF;
		send_udp(&udp_hdr.lsocket, &udp_hdr.fsocket,
			0, 0, bp, (int16)udp_hdr.size,
			(int16)0, 0);
		break;
	}
}
