/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the Address Resolution Protocol (ARP)
 *		functions. ARP sits between IP and Level 2, mapping IP
 *		to Level 2 addresses for all outgoing datagrams.
 *
 * Version:	@(#)arp.c		1.00	07/12/92
 *
 * Authorss:	Original by Phil Karn KA9Q.
 *		Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#if HAVE_ETHER

#include <stdio.h>

#include "mbuf.h"
#include "timer.h"
#include "iface.h"
#include "eth.h" 
#include "arp.h"


char eth_bdcst[] = {0xff,0xff,0xff,0xff,0xff,0xff}; /* Ether broadcast addr*/ 
char myethad[6] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05 };


/* Table of ARP hardware types */
struct arp_type arp_type[] = {
	0,
	0,
	0,
	NULLCHAR,

	/* 10 megabit Ethernet */
	6,			/* Ethernet address length */
	0x800,			/* Ethernet type field for IP */
	0x806,			/* Ethernet type field for ARP */
	eth_bdcst,		/* Ethernet broadcast address */
};
#define	NTYPES	2


/* Hash table headers */
struct arp_tab *arp_tab[ARPSIZE];

struct arp_stat arp_stat;

char *arptypes[] = {
	NULLCHAR,
	"Ethernet",
	"Exp Ethernet",
	"AX.25",
	"Pronet",
	"Chaos"
};


/* Resolve an IP address to a hardware address; if not found,
 * initiate query and return NULLCHAR.  If an address is returned, the
 * interface driver may send the packet; if NULLCHAR is returned,
 * res_arp() will have saved the packet on its pending queue,
 * so no further action (like freeing the packet) is necessary.
 */
char *
res_arp(interface,hardware,target,bp)
struct interface *interface;	/* Pointer to interface block */
int16 hardware;		/* Hardware type */
int32 target;		/* Target IP address */
struct mbuf *bp;	/* IP datagram to be queued if unresolved */
{
	struct arp_tab *arp_lookup(),*arp_add();
	void arp_output();
	register struct arp_tab *arp;

	if((arp = arp_lookup(hardware,target)) != NULLARP && arp->state == ARP_VALID)
		return arp->hw_addr;
	/* Create an entry and put the datagram on the
	 * queue pending an answer
	 */
	arp = arp_add(target,hardware,NULLCHAR,0);
	enqueue(&arp->pending,bp);
	arp_output(interface,hardware,target);
	return NULLCHAR;
}
/* Handle incoming ARP packets. This is almost a direct implementation of
 * the algorithm on page 5 of RFC 826, except for:
 * 1. Outgoing datagrams to unresolved addresses are kept on a queue
 *    pending a reply to our ARP request.
 * 2. The names of the fields in the ARP packet were made more mnemonic.
 */
void
arp_input(interface,bp)
struct interface *interface;
struct mbuf *bp;
{
	struct arp arp;
	struct arp_tab *arp_lookup();
	struct arp_type *at;
	struct mbuf *htonarp();
	char *myaddr;
	int merge_flag = 0;

#if HAVE_TRACE
	if((interface->trace & 0xf0))
		arp_dump(bp);
#endif
	arp_stat.recv++;
	if(ntoharp(&arp,bp) == -1)	/* Convert into host format */
		return;
	if(arp.hardware >= NTYPES){
		/* Unknown hardware type, ignore */
		arp_stat.badtype++;
		return;
	}
	at = &arp_type[arp.hardware];
	if(arp.protocol != at->iptype){
		/* Unsupported protocol type, ignore */
		arp_stat.badtype++;
		return;
	}
	if(arp.hwalen > MAXHWALEN || arp.pralen != sizeof(int32)){
		/* Incorrect protocol addr length (different hw addr lengths
		 * are OK since AX.25 addresses can be of variable length)
		 */
		arp_stat.badlen++;
		return;
	}
	/* If this guy is already in the table, update its entry */
	if(arp_lookup(arp.hardware,arp.sprotaddr) != NULLARP){
		arp_add(arp.sprotaddr,arp.hardware,arp.shwaddr,arp.hwalen & 0xff);
		merge_flag = 1;
	}
	/* See if we're the address they're looking for */
	if(arp.tprotaddr == ip_addr){
		if(merge_flag == 0)
			arp_add(arp.sprotaddr,arp.hardware,arp.shwaddr,arp.hwalen & 0xff);

		if(arp.opcode == ARP_REQUEST){
			/* Swap sender's and target's (us) hardware and protocol
			 * fields, and send the packet back as a reply
			 */
			bcopy(arp.shwaddr,arp.thwaddr,arp.hwalen);
			switch(arp.hardware){
			case ARP_ETHER:
				myaddr = myethad;
				break;
			default:
				myaddr = NULLCHAR;
				break;
			}
			bcopy(myaddr,arp.shwaddr,at->hwalen);
			arp.tprotaddr = arp.sprotaddr;
			arp.sprotaddr = ip_addr;
			arp.opcode = ARP_REPLY;
			bp = htonarp(&arp);
			(*interface->output)(interface,arp.thwaddr,myaddr,at->arptype,bp);
			arp_stat.inreq++;
		} else {
			arp_stat.replies++;
		}
	}
}
/* Add an IP-addr / hardware-addr pair to the ARP table */
static
struct arp_tab *
arp_add(ip_addr,hardware,hw_addr,hw_alen)
int32 ip_addr;	/* IP address, host order */
int16 hardware;	/* Hardware type */
char *hw_addr;	/* Hardware address, if known; NULLCHAR otherwise */
int16 hw_alen;	/* Length of hardware address */
{
	struct arp_tab *arp_lookup();
	void arp_drop();
	struct mbuf *bp,*dequeue();
	register struct arp_tab *ap;
	register struct arp_type *at;
	unsigned hashval,arp_hash();

	at = &arp_type[hardware];
	if((ap = arp_lookup(hardware,ip_addr)) == NULLARP){
		/* New entry */
		if((ap = (struct arp_tab *)calloc(1,sizeof(struct arp_tab))) == NULLARP)
			return NULLARP;
		ap->timer.func = arp_drop;
		ap->timer.arg = (int *)ap;
		ap->hardware = hardware;
		ap->ip_addr = ip_addr;

		/* Put on head of hash chain */
		hashval = arp_hash(hardware,ip_addr);
		ap->prev = NULLARP;
		ap->next = arp_tab[hashval];
		arp_tab[hashval] = ap;
		if(ap->next != NULLARP){
			ap->next->prev = ap;
		}
	}
	if(hw_addr == NULLCHAR){
		/* Await response */
		ap->state = ARP_PENDING;
     		ap->timer.start = PENDTIME;
	} else {
		/* Response has come in, update entry and run through queue */
		ap->state = ARP_VALID;
		ap->timer.start = ARPLIFE;
		if((ap->hw_addr = (char *) malloc(hw_alen)) == NULLCHAR){
			free((char *)ap);
			return NULLARP;
		}
		bcopy(hw_addr,ap->hw_addr,hw_alen);

		while((bp = dequeue(&ap->pending)) != NULLBUF)

		ip_route(bp,0); /* pass on to IP */  
	}
	start_timer(&ap->timer);
	return ap;
}

/* Remove an entry from the ARP table */
static
void
arp_drop(ap)
register struct arp_tab *ap;
{
	unsigned arp_hash();

	if(ap == NULLARP)
		return;
	stop_timer(&ap->timer);	/* Shouldn't be necessary */
	if(ap->next != NULLARP)
		ap->next->prev = ap->prev;
	if(ap->prev != NULLARP)
		ap->prev->next = ap->next;
	else
		arp_tab[arp_hash(ap->hardware,ap->ip_addr)] = ap->next;
	if(ap->hw_addr != NULLCHAR)
		free(ap->hw_addr);
	free_q(&ap->pending);
	free((char *)ap);
}

/* Look up the given IP address in the ARP table */
static
struct arp_tab *
arp_lookup(hardware,ip_addr)
int16 hardware;
int32 ip_addr;
{
	unsigned arp_hash();
	register struct arp_tab *ap;

	for(ap = arp_tab[arp_hash(hardware,ip_addr)]; ap != NULLARP; ap = ap->next){
		if(ap->ip_addr == ip_addr && ap->hardware == hardware)
			break;
	}
	return ap;
}
/* Send an ARP request to resolve IP address target_ip */
static
void
arp_output(interface,hardware,target)
struct interface *interface;
int16 hardware;
int32 target;
{
	char *myaddr;
	struct arp arp;
	struct mbuf *bp,*htonarp();
	struct arp_type *at;

	at = &arp_type[hardware];
	if(interface->output == NULLFP)
		return;
	
	arp.hardware = hardware;
	arp.protocol = at->iptype;
	arp.hwalen = at->hwalen;
	arp.pralen = sizeof(int32);
	arp.opcode = ARP_REQUEST;
	switch(hardware){

	case ARP_ETHER:
		myaddr = myethad;
		break;

	default:
		myaddr = NULLCHAR;
		break;
	}
	bcopy(myaddr,arp.shwaddr,at->hwalen);
	arp.sprotaddr = ip_addr;
	bzero(arp.thwaddr,at->hwalen);
	arp.tprotaddr = target;
	bp = htonarp(&arp);
	(*interface->output)(interface,at->bdcst,myaddr,at->arptype,bp);
	arp_stat.outreq++;
}

/* Hash a {hardware type, IP address} pair */
static
unsigned
arp_hash(hardware,ip_addr)
int16 hardware;
int32 ip_addr;
{
	register unsigned hashval;

	hashval = hardware;
	hashval ^= hiword(ip_addr);
	hashval ^= loword(ip_addr);
	hashval %= ARPSIZE;
	return hashval;
}		
/* Copy a host format arp structure into mbuf for transmission */
static
struct mbuf *
htonarp(arp)
register struct arp *arp;
{
	struct mbuf *bp;
	register char *buf;

	if(arp == (struct arp *)NULL)
		return NULLBUF;
	if((bp = alloc_mbuf(sizeof(struct arp))) == NULLBUF)
		return NULLBUF;

	buf = bp->data;

	*(int16 *)buf = htons(arp->hardware);
	buf += sizeof(int16);

	*(int16 *)buf = htons(arp->protocol);
	buf += sizeof(int16);

	*buf++ = arp->hwalen;

	*buf++ = arp->pralen;

	*(int16 *)buf = htons(arp->opcode);
	buf += sizeof(int16);

	bcopy(arp->shwaddr,buf,arp->hwalen);
	buf += arp->hwalen;

	*(int32 *)buf = htonl(arp->sprotaddr);
	buf += sizeof(int32);

	bcopy(arp->thwaddr,buf,arp->hwalen);
	buf += arp->hwalen;

	*(int32 *)buf = htonl(arp->tprotaddr);
	buf += sizeof(int32);

	bp->cnt = buf - bp->data;
	return bp;
}
/* Convert an incoming ARP packet into a host-format structure */
static
int
ntoharp(arp,bp)
register struct arp *arp;
struct mbuf *bp;
{
	if(arp == (struct arp *)NULL || bp == NULLBUF)
		return -1;

	pullup(&bp,(char *)&arp->hardware,sizeof(int16));
	arp->hardware = ntohs(arp->hardware);

	pullup(&bp,(char *)&arp->protocol,sizeof(int16));
	arp->protocol = ntohs(arp->protocol);

	pullup(&bp,(char *)&arp->hwalen,sizeof(char));

	pullup(&bp,(char *)&arp->pralen,sizeof(char));

	pullup(&bp,(char *)&arp->opcode,sizeof(int16));
	arp->opcode = ntohs(arp->opcode);

	pullup(&bp,arp->shwaddr,arp->hwalen);

	pullup(&bp,(char *)&arp->sprotaddr,sizeof(int32));
	arp->sprotaddr = ntohl(arp->sprotaddr);

	pullup(&bp,arp->thwaddr,arp->hwalen);

	pullup(&bp,(char *)&arp->tprotaddr,sizeof(int32));
	arp->tprotaddr = ntohl(arp->tprotaddr);

	free_p(bp);
	return 0;
}


int doarp(fdout)
int fdout;
{
  register int i;
  extern struct arp_stat arp_stat;
  register struct arp_tab *ap;
  char e[20];

  rprintf(fdout, "received %u badtype %u reqst in %u ",
	 arp_stat.recv,arp_stat.badtype,arp_stat.inreq);
  rprintf(fdout, "replies %u reqst out %u\r\n",
	 arp_stat.replies,arp_stat.outreq);

  rprintf(fdout, "IP addr         Type            Addr");
  rprintf(fdout, "                Time Queue\r\n");
  for(i=0;i<ARPSIZE;i++){
	for(ap = arp_tab[i];ap != NULLARP;ap = ap->next){
		rprintf(fdout, "%-16s",inet_ntoa(htonl(ap->ip_addr)));
		rprintf(fdout, "%-16s",arptypes[ap->hardware]);
		if(ap->state == ARP_VALID){
			switch(ap->hardware){
				case ARP_ETHER:
					pether(e,ap->hw_addr);
					break;
				/* MORE HW'S SHOULD FOLLOW HERE */
				default:
					e[0] = '\0';
					break;
			}
			rprintf(fdout, "%-20s",e);
		} else {
			rprintf(fdout, "[unknown]           ");
		}
		rprintf(fdout, "%-5ld",ap->timer.count*(long)MSPTICK/1000);
		if(ap->state == ARP_PENDING)
			rprintf(fdout, "%-5u",len_q(ap->pending));
		rprintf(fdout, "\r\n");
	}
  }
  return(0);
}


#if HAVE_TRACE
/* Dump ARP packets (incomplete) */
static char *hwtypes[] = {
	"",
	"10 Mb Ethernet",
	NULLCHAR,
};
#define	NHWTYPES 2
arp_dump(bp)
struct mbuf *bp;
{
	struct mbuf *tbp;
	struct arp arp;

	dup_p(&tbp,bp,0,len_mbuf(bp));
	ntoharp(&arp,tbp);
	if(arp.hardware < NHWTYPES)
		rprintf(2, "ARP: hwtype %s",hwtypes[arp.hardware]);
	else
		rprintf(2, "ARP: hwtype %u",arp.hardware);
	rprintf(2, " prot 0x%x hwlen %u prlen %u",
		arp.protocol,arp.hwalen,arp.pralen);
	switch(arp.opcode){
	case ARP_REQUEST:
		rprintf(2, " op REQUEST");
		break;
	case ARP_REPLY:
		rprintf(2, " op REPLY");
		break;
	default:
		rprintf(2, " op %u",arp.opcode);
		break;
	}
	rprintf(2, "\r\n");
}
#endif

#endif	/* HAVE_ETHER */
