
/*
 * slip.drv	A program for TNET, which reads SLIP encapsulated IP
 *		datagrams from a serial port, and sends them to TNET
 *		for further processing.
 *
 *		Modem dialing functions.  This module expects that
 *		the modem understands the basic HAYES commands when
 *		dialing in "manual" (i.e. non-script) mode. The
 *		used commands are defined in the macros below.
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <setjmp.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "slip.h"


/* Here are some HAYES modem commands. */
#define OK_STR		"OK"			/* command was OK	*/
#define RSET_STR	"AT Z"			/* reset the modem	*/
#define INIT_STR	"AT E0 Q0 V1 X1"	/* setup for dialing	*/
#define DIAL_STR	"AT D"			/* start a dial		*/
#define CONN_STR	"CONNECT"		/* we are connected	*/
#define DLAY_TIME	2			/* command delay time	*/
#define CONN_TIME	45			/* connecting delay time*/


static int timeout;


_PROTOTYPE( void TimeOut, (void)					);
_PROTOTYPE( char cvt_char, (char c)					);
_PROTOTYPE( void do_send, (char *what)					);
_PROTOTYPE( int do_wait, (char *what)					);
_PROTOTYPE( int do_manual, (char *what)					);


static void TimeOut()
{
  timeout = 1;
}


/* Convert a C-style backslash sequence to ASCII. */
static char cvt_char(c)
char c;
{
  switch(c) {
	case '\0':
		return('\\');
		break;
	case 'a':
		return('\a');
		break;
	case 'b':
		return('\b');
		break;
	case 'f':
		return('\f');
		break;
	case 'n':
		return('\n');
		break;
	case 'r':
		return('\r');
		break;
	case 't':
		return('\t');
		break;
	case 'v':
		return('\v');
		break;
	case '\\':
		return('\\');
		break;
	case '\'':
		return('\'');
		break;
	case '"':
		return('\"');
		break;
	case '?':
		return('\?');
		break;
	default:
		return('?');
  }
  /* NOTREACHED */
  return('?');
}


/* Send a string to the serial driver. */
static void do_send(what)
register char *what;
{
  register char c;

  while(*what) {
	if(*what == '\\') {
		what++;
		c = cvt_char(*what);
		tty_putc((int) c);
		if(!*what) break;
	} else {
		c = *what;
		tty_putc((int) c);
	}
	what++;
  }
  tty_putc(-1);
}


/* Wait for a certain string, with time-out. */
static int do_wait(what)
char *what;
{
  char c, c2, *p, *xp;
  int howlong;
  void (*oldsig)();

  p = what;
  while (*p && *p != ' ' && *p != '\t') p++;
  if (*p) {
	*p++ = '\0';
	xp = p;
 	howlong = atoi(what);
  } else {
 	xp = what;
	howlong = 0;
  }

  if (howlong == 0) howlong = 1;
  timeout = 0;
  oldsig = signal(SIGALRM, TimeOut);
  alarm(howlong);

  if (*p == '\0') {	/* just plain wait */
	while(1) {
		(void) tty_getc();
		if (timeout) break;
	}
	alarm(0);
	(void) signal(SIGALRM, oldsig);
  	return(0);
  } else {
	while(!timeout && *p != '\0') {
		c = tty_getc();
		c = (c & 0x7F);
		if (timeout) break;
		if (*p == '\\') c2 = cvt_char(*++p);
		  else c2 = *p;
		if (c2 != c) p = xp;
		  else p++;
	}
  }
  alarm(0);
  (void) signal(SIGALRM, oldsig);
  return(timeout);
}


static int do_manual(what)
char *what;
{
  char buff[128];
  int st;

  /* Reset the modem. */
  sprintf(buff, "%s\r\n", RSET_STR);
  do_send(buff);
  (void) do_wait("1");

  /* Setup to dial out. */
  sprintf(buff, "%s\r\n", INIT_STR);
  do_send(buff);
  sprintf(buff, "%d %s", DLAY_TIME, OK_STR);
  st = do_wait(buff);
  if (st != 0) return(st);

  /* Dial the phone number. */
  sprintf(buff, "%s %s\r\n", DIAL_STR, what);
  do_send(buff);
  sprintf(buff, "%d %s", CONN_TIME, CONN_STR);
  st = do_wait(buff);

  /* Clean out any garbage. */
  (void) do_wait("1");

  return(st);
}


int do_dial(telscript)
char *telscript;
{
  char buff[128];
  register FILE *fp;
  register char *sp;
  int st;

  /* Return successfully if no dialing is needed. */
  if (*telscript == '\0') return(0);

  if (opt_v == 1) fprintf(stderr, "Dialing %s\n", telscript);

  if ((fp = fopen(telscript, "r")) != (FILE *)NULL) {
	while(fgets(buff, 128, fp) != (char *)NULL) {
		if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
		if (opt_v) fprintf(stderr, "%s>> %s\n", telscript, buff);
		if (buff[0] == '#' || buff[0] =='\0') continue;
		sp = buff;
		while (*sp && *sp != ' ' && *sp != '\t') sp++;
		if (*sp) *sp++ = '\0';
		  else sp = "\0";
		if (!strcmp(buff, "send")) do_send(sp);
		  else if (!strcmp(buff, "wait")) {
			st = do_wait(sp);
			if (st != 0) break;
		} else fprintf(stderr, "DIAL: syntax error: %s %s\n",
								buff, sp);
	}
	(void) fclose(fp);
  } else st = do_manual(telscript);

  return(st);
}
