/*
 * ether.drv	A program for TNET, which reads IEEE Ethernet encapsu-
 *		lated IP datagrams from an Ethernet driver, and sends
 *		them to TNET for further processing.
 *
 *		ETHER protocol module.
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <setjmp.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <arpa/inet.h>
#include "ether.h"


static int eth_fd = -1;			/* file descriptor of ETHER	*/
static char *arp_hws[] = {
  "(none)", "10MBps Ethernet",
  "Exp. Ethernet", "AX.25",
  "ProNET", "Chaos"
};


_PROTOTYPE( char *pether, (char *)					);
_PROTOTYPE( void arp_dump, (char *)					);
_PROTOTYPE( void eth_dump, (char *, int)				);


/* Convert an Ethernet address into ASCII. */
static char *pether(addr)
char *addr;
{
  static char buff[20];

  sprintf(buff, "%02x:%02x:%02x:%02x:%02x:%02x",
	addr[0] & 0xFF, addr[1] & 0xFF, addr[2] & 0xFF,
	addr[3] & 0xFF, addr[4] & 0xFF, addr[5] & 0xFF);
  return(buff);
}


/* Dump the contents of an ARP or RARP packet. */
static void arp_dump(p)
char *p;
{
  short int hw, prot;
  short int hwl, prl;
  short int op;
  long ipaddr;
  char *hwaddr;

  hw = ntohs(*(short int *) p); p += sizeof(short int);
  prot = ntohs(*(short int *) p); p += sizeof(short int);
  hwl = (int) *(char *) p & 0xFF; p++;
  prl = (int) *(char *) p & 0xFF; p++;
  op = ntohs(*(short int *) p); p += sizeof(short int);

  fprintf(stderr, "ARP: HW=%d (%s), PROT=0x%04.4x\n", hw, arp_hws[hw], prot);
  fprintf(stderr, "     HWLEN=%d  IPLEN=%d  OPCODE=%d\n", hwl, prl, op);
  hwaddr = pether(p); p += hwl;
  ipaddr = *(long *) p; p += prl;
  fprintf(stderr, "     Source HW=%s  IP=%s\n", hwaddr, inet_ntoa(ipaddr));
  hwaddr = pether(p); p += hwl;
  ipaddr = *(long *) p; p += prl;
  fprintf(stderr, "     Target HW=%s  IP=%s\n", hwaddr, inet_ntoa(ipaddr));
}


/* Dump an Ethernet frame. */
static void eth_dump(ptr, len)
char *ptr;
int len;
{
  char *scan;
  int i;

  scan = ptr;
  len -= 14;
  fprintf(stderr, "%s->", pether(scan));
  scan += 6;
  fprintf(stderr, "%s type ", pether(scan));
  scan += 6;
  i = ntohs(*(int *) scan);
  fprintf(stderr, "0x%04.4x (%u) len %u\n", i, i, len);
  scan += 2;

  if (i == 0x0806) arp_dump(scan);
    else if (i == 0x0800) ip_dump(scan, len);
}


/* Initialize the Ethernet device. */
int dev_init(name)
char *name;
{
  int i;

  /* Try opening the ETHER device. */
  if (*name != '\0') {
	eth_fd = open(name, O_RDWR);
	if (eth_fd < 0) {
		fprintf(stderr, "ETHER: open(%s, RW) == %d\n", name, errno);
		return(-errno);
	}
  } else {
	fprintf(stderr, "ETHER: no ethernet device given???\n");
	return(-errno);
  }
  if (opt_v) fprintf(stderr, "ETHER = %s (%d) ", name, eth_fd);

  return(0);
}


/*
 * This is the input half of the ETHER driver.
 */
void in_eth(fd)
int fd;
{
  int s, n;
  char *buffer;

  /* Allocate a buffer for the decoded IP datagrams. */
  buffer = (char *) malloc(ec_mtu + 16);
  if (buffer == (char *)NULL) {
	fprintf(stderr, "ETHER: IN: cannot allocate my buffer!\n");
	return;
  }

  while(1) {
	setjmp(env);
	n = read(eth_fd, buffer + sizeof(int), ec_mtu + 16);
	if (n == 0) break;
	if (n < 0 || n > ec_mtu) continue;

	/* We now have a full Ethernet frame in the buffer. */
	if (opt_v) fprintf(stderr, "ETHER: Frame LEN=%d recv.\n", n);

	/* Dump contents of datagram if needed. */
	if (opt_d) eth_dump(buffer + sizeof(int), n);

	*(int *)buffer = n;
	s = write(fd, buffer, sizeof(int) + n);
	if (s != (sizeof(int) + n)) {
		fprintf(stderr, "ETHER: IN: dgram write error %d(%d)\n",
							errno, fd);
	}
  }
}


/*
 * This is the ETHER datagram sender.
 * We start an endless loop, reading the "ETHER header" from the
 * I/O input channel.  This header is simply a word which tells
 * us the total length of the IP datagram which we have to send
 * to the ETHERNET channel.
 */
void out_eth(fd)
int fd;
{
  int s, n;
  char *buffer;

  /* Allocate a buffer for the outgoing ETHER frame. */
  buffer = (char *) malloc(ec_mtu);
  if (buffer == (char *)NULL) {
	fprintf(stderr, "ETHER: OUT: cannot allocate my buffer!\n");
	return;
  }

  while(1) {
	setjmp(env);
	s = read(fd, (char *) &n, sizeof(int));
	if (s == 0) return;
	if (s != sizeof(int)) {
		fprintf(stderr, "ETHER OUT: dghdr read error %d(%d)\n",
							errno, fd);
		continue;
	}

	/* Read in the entire datagram. */
	s = read(fd, buffer, n);
	if (s != n) {
		fprintf(stderr, "ETHER OUT: dgram read error %d(%d)\n",
							errno, fd);
		continue;
	}

	/* We now have a full Ethernet frame in the buffer. */
	if (opt_v) fprintf(stderr, "ETHER: Frame LEN=%d sent.\n", n);

	/* Dump contents of datagram if needed. */
	if (opt_d) eth_dump(buffer, n);

	(void) write(eth_fd, buffer, n);
  }
}
