/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file contains an implementation of the NNTP
 *		protocol for accessing USENET news article servers.
 *
 * Usage:	nntp [-dv] [server]
 *
 * Version:	@(#)nntp.c	1.00	07/02/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <arpa/internet.h>
#include <arpa/inet.h>
#include <inet/in.h>
#include <inet/socket.h>
#include <tnet/tnet.h>
#include <tnet/client.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>


#define NNTP_CFG	"nntp.cfg"
#define NNTP_PORT	119
#define MAIL		"/usr/bin/mail"


static char *Version = "@(#) nntp 1.00 (07/02/92)";


static char line[1024];
static char config[1024];
static char user[64];
long newdate;
static char group[128];
static char article[128];
static char newarticle[128];
int opt_d = 0;				/* debugging output flag	*/


extern int getopt(), optind, opterr;
extern char *optarg;


_PROTOTYPE( int DOgetline, (FILE *)					);
_PROTOTYPE( int DOgetreply, (FILE *)					);
_PROTOTYPE( int DOtext, (FILE *)					);
_PROTOTYPE( int DOhellomsg, (FILE *)					);
_PROTOTYPE( int DOcommand, (FILE *, FILE *, char *)			);
_PROTOTYPE( int DOnewgroups, (FILE *, FILE *, long)			);
_PROTOTYPE( int DOgroup, (FILE *, FILE *, char *)			);
_PROTOTYPE( int DOarticle, (FILE *, FILE *, char *)			);
_PROTOTYPE( int DOnext, (FILE *, FILE *)				);
_PROTOTYPE( int DOquit, (FILE *, FILE *)				);
_PROTOTYPE( int readcfg, (FILE *, FILE *)				);
_PROTOTYPE( int DOexec, (char *)					);
_PROTOTYPE( int DOmain, (char *, int *, char *)				);
_PROTOTYPE( int headerok, (void)					);
_PROTOTYPE( int DOsend, (FILE *, char *, int)				);


int DOgetline(fpin)
FILE *fpin;
{
  int len;

  if (fgets(line, sizeof(line), fpin) == (char *)NULL) return(-1);
  len = strlen(line);
  if (line[len - 2] == '\r') {
	line[len - 2] = '\n';
	line[len - 1] = '\0';
	len--;
  }
  if (line[0] == '.' && len == 2) return(1);
    else return(0);
}


int DOgetreply(fpin)
FILE *fpin;
{
  int len;

  do {
	if (fgets(line, sizeof(line), fpin) == (char *)NULL) return(-1);
  } while(line[3] == '-');
  len = strlen(line);
  if (line[len - 1] == '\r') {
	line[len - 1] = '\n';
	line[len] = '\0';
  }
  return(atoi(line));
}


int DOtext(fpin)
FILE *fpin;
{
  return(DOgetline(fpin));
}


int hellomsg(fpin)
FILE *fpin;
{
  return(DOgetreply(fpin));
}


int DOcommand(fpin, fpout, command)
FILE *fpin, *fpout;
char *command;
{
  fprintf(fpout, "%s\r\n", command);
  fflush(fpout);
  return(DOgetreply(fpin));
}


int DOnewgroups(fpin, fpout, date)
FILE *fpin, *fpout;
long date;
{
  char command[80];

  sprintf(command, "newgroups %06ld 000000 GMT", date);
  return(DOcommand(fpin, fpout, command));
}


int DOgroup(fpin, fpout, group)
FILE *fpin, *fpout;
char *group;
{
  char command[200];

  sprintf(command, "group %s", group);
  return(DOcommand(fpin, fpout, command));
}


int DOarticle(fpin, fpout, id)
FILE *fpin, *fpout;
char *id;
{
  char command[200];
  int s;
  char *p, *a;

  sprintf(command, "article %s", id);
  s = DOcommand(fpin, fpout, command);
  if (s == 220) {
	p = line + 4; a = newarticle;
	while(*p && isspace(*p)) p++;
	while(*p && !isspace(*p)) *a++ = *p++;
	*a = '\0';
  } else strcpy(newarticle, article);
  return(s);
}


int DOnext(fpin, fpout)
FILE *fpin, *fpout;
{
  return(DOcommand(fpin, fpout, "next"));
}


int DOquit(fpin, fpout)
FILE *fpin, *fpout;
{
  return(DOcommand(fpin, fpout, "quit"));
}


int readcfg(fpcfg, fpnew)
FILE *fpcfg, *fpnew;
{
  char *p, *u, *g, *a;

  while(1) {
	if (fgets(config, sizeof(config), fpcfg) == (char *)NULL) return(0);
	if (config[0] == '#' || config[0] == '\n') {
		fprintf(fpnew, config);
		(void) fflush(fpnew);
	} else break;
  }
  config[strlen(config) - 1] = '\0';
  if (!isspace(config[0])) {
	p = config; u = user;
	while(*p && !isspace(*p)) *u++ = *p++;
	*u = '\0';
	newdate = atol(p);
	return(1);
  }
  p = config; g = group; a = article;
  while(*p && isspace(*p)) p++;
  while(*p && !isspace(*p)) *g++ = *p++;
  *g = '\0';
  while(*p && isspace(*p)) p++;
  while(*p && !isspace(*p)) *a++ = *p++;
  *a = '\0';
  return(2);
}


int DOexec(subject)
char *subject;
{
  char subline[128];
  int pfd[2], pid, i;

  if (pipe(pfd) == -1) return(-1);
  sprintf(subline, "-s%s", subject);
  switch(pid = fork()) {
	case -1:
		return(-1);
		break;
	case 0:
		(void) close(0);
		(void) dup(pfd[0]);
		(void) close(pfd[0]);
		(void) close(pfd[1]);
		(void) execlp(MAIL, "mail", subline, user, (char *)NULL);
		break;
	default:
		(void) close(pfd[0]);
		return(pfd[1]);
  }
  /* NOTREACHED */
}


int DOmail(buff, fdout, subject)
char *buff;
int *fdout;
char *subject;
{
  char temp[128];

  if (*fdout == -1) {
	*fdout = DOexec(subject);
#ifdef DEBUG
	if (*fdout >= 0) {
		sprintf(temp, "Article Trace Number: %s\n", newarticle);
		write(*fdout, temp, strlen(temp));
	}
#endif
  }
  if (*fdout >= 0) write(*fdout, buff, strlen(buff));
  return(*fdout);
}


int headerok()
{
  int status = 0;

  if (!strncmp(line, "From:", 5)) status = 1; else
  if (!strncmp(line, "Newsgroups:", 11)) status = 1; else
  if (!strncmp(line, "Subject:", 8)) status = 1; else
  if (!strncmp(line, "Message-ID:", 11)) status = 1; else
  if (!strncmp(line, "Date:", 5)) status = 1; else
  if (!strncmp(line, "Reply-To:", 9)) status = 1;
  return(status);
}


int dosend(fpin, subject, flag)
FILE *fpin;
char *subject;
int flag;		/* 0=flush, 1=article, 2=text */
{
  int pid, pstat;
  int status;
  int headers;
  int mailit;
  int fdmail = -1;

  if (flag == 1) headers = 1;
    else headers = 0;

  while((status = DOtext(fpin)) == 0) {
	if (flag == 1 && strlen(line) == 1) headers = 0;
	if (headers) mailit = headerok();
	  else mailit = 1;
	if (flag && mailit) {
		status = DOmail(line, &fdmail, subject);
		if (status < 0) break;
	}
  }
  (void) close(fdmail);
  if (flag) {
	pid = wait(&pstat);
#ifdef DEBUG
	fprintf(stderr, "Pid: %d  Status=%04x\n", pid, pstat);
#endif
  }
  return(status);
}


/*
 * Call an NNTP USENET server.
 */
int nntp(host)
char *host;
{
  char newname[128];
  char name[128];
  char temp[128];
  char subject[64];
  u_long rmt_addr;
  u_short rmt_port;
  struct tm *tm;
  struct hostent *hp;
  struct servent *sp;
  FILE *fpin, *fpout;
  FILE *fpcfg, *fpnew;
  char *p;
  time_t now;
  int fd_in, fd_out, s;
  int OK, status;

  sprintf(name, "%s/%s", TNET_DIR, NNTP_CFG);
  sprintf(newname, "%s/NEW%s", TNET_DIR, NNTP_CFG);
  if ((fpcfg = fopen(name, "r")) == (FILE *)NULL) {
	fprintf(stderr, "Could not open netnews configuration file\n");
	return(-1);
   }
   if((fpnew = fopen(newname, "w")) == (FILE *)NULL) {
	fprintf(stderr, "Could not open netnews new config file\n");
	return(-1);
   }

  if(opt_d == 1) fprintf(stderr, "Resolving %s...", host);
  if ((hp = gethostbyname(host)) == (struct hostent *)NULL) {
	fprintf(stderr, "Unknown host %s!\n", host);  
	fclose(fpcfg);
	fclose(fpnew);
	return(-1);
  } else {
	memcpy((char *) &rmt_addr, (char *) hp->h_addr, hp->h_length);
	strcpy(host, hp->h_name);
  }

  /* Now, to which port must we connect? */
  if ((sp = getservbyname("nntp", "tcp")) == (struct servent *)NULL) {
	fprintf(stderr, "NNTP port is unknown????\n");
	fclose(fpcfg);
	fclose(fpnew);
	return(-1);
  } else rmt_port = sp->s_port;

  strcpy(temp, inet_ntoa(rmt_addr));
  if (opt_d == 1) fprintf(stderr, " trying %s:%d...\n", temp, rmt_port);

  s = client(TCP_PTCL, 0, rmt_port, rmt_addr, &fd_in, &fd_out);
  if (s != 0) {
	fprintf(stderr, "Connection failed: status = %d\n", s);
	fclose(fpcfg);
	fclose(fpnew);
	return(-1);
  }
   fpin = fdopen(fd_in, "r");
   fpout = fdopen(fd_out, "w");

   status = hellomsg(fpin);
   if(status == 200 || status == 201)
	OK = 1;
   else {
	OK = 0;
	status = -1;
   }
   while(OK) {
	status = readcfg(fpcfg, fpnew);
	if(status == 0) {
		OK = 0;
		continue;
	}
	if(status == 1) {
		if(newdate == 0L) {
			sprintf(subject, "All News Groups");
			status = DOcommand(fpin, fpout, "list");
		} else {
			sprintf(subject,
				"New News Groups as of %06ld GMT", newdate);
			status = DOnewgroups(fpin, fpout, newdate);
		}
		(void)time(&now);
		tm = gmtime(&now);
		newdate = tm->tm_year * 10000L +
			  (tm->tm_mon + 1L) * 100L +
			  tm->tm_mday;
		status = dosend(fpin, subject, 2);
		fprintf(fpnew, "%s\t%06ld\n", user, newdate);
		fflush(fpnew);
		continue;
	}

	status = DOgroup(fpin, fpout, group);
	if(status != 211) {
		OK = 0;
		continue;
	}
	status = DOarticle(fpin, fpout, article);
	if(status == 220) {
#ifdef DEBUG
		if(strlen(article) == 0)
			fprintf(stderr, "User: %s Group: %s Article: %s New: %s\n",
				user, group, article, newarticle);
#endif
		status = dosend(fpin, group, (strlen(article)?0:1));
		if(status < 0) {
			OK = 0;
			continue;
		}
	}
	while((status = DOnext(fpin, fpout)) == 223) {
		status = DOarticle(fpin, fpout, "");
#ifdef DEBUG
		fprintf(stderr, "User: %s Group: %s Article: %s New: %s\n",
			user, group, article, newarticle);
#endif
		status = dosend(fpin, group, 1);
		if(status < 0) {
			OK = 0;
			break;
		}
	}
	if(status < 0) continue;
	fprintf(fpnew, "\t%s\t%s\n", group, newarticle);
	fflush(fpnew);
   }

   fclose(fpcfg); fclose(fpnew);

   if(status >= 0) {
	unlink(name);
	rename(newname, name);
	status = DOquit(fpin, fpout);
   }

   printf("Netnews closing status cause %d\n", status);
   printf("%s", line);

   fclose(fpin); fclose(fpout);
   close(fd_in); close(fd_out);

   return(0);
}

void usage()
{
  fprintf(stderr, "Usage: nntp [-dv] [server]\n");
  exit(-1);
}

int main(argc, argv)
int argc;
char *argv[];
{
  char host[128];
  register int c;

  opterr = 0;
  while ((c = getopt(argc, argv, "dv")) != EOF) switch(c) {
	case 'd':
	case 'v':
		opt_d = 1;
		break;
	default:
		usage();
  }

  if (optind != argc && optind != (argc - 1)) usage();

  /* If an argument is given, use that as the NNTP server. */
  if (optind != argc) strcpy(host, argv[optind]);
    else gethostname(host, 128);

  return(nntp(host));
}
