#include "kernel.h"
#include <minix/com.h>
#include "ether.h"

/* #define DEBUG 1 */

#if (ETHERNET == ETH_WD8003) || \
    (ETHERNET == ETH_3C503)  || \
    (ETHERNET == ETH_NE1000) || \
    (ETHERNET == ETH_NE2000)

/* GLOBAL VARIABLES */
PUBLIC unsigned sm_rstop_ptr=SM_RSTOP_PG;
PUBLIC unsigned endcfg=ENDCFG_FT01|ENDCFG_BMS;

/* LOCAL VARIABLES */
PRIVATE unsigned char rcv_hdr[RCV_HDR_SIZE];
PRIVATE unsigned rxcr_bits=ENRXCR_BCST;

/* LOCAL FUNCTIONS */
FORWARD void recv_overrun();
FORWARD int recv_frame();
FORWARD void isr_tx();
FORWARD void isr_stat();
FORWARD phys_bytes rcv_frm();

/*===========================================================================*
 *				recv_pkt				     * 
 *===========================================================================*/
PUBLIC int recv_pkt()
{
/* Receive packet routine.
 * Returns OK if packet received.
 */
  int res;

  output(EN0_IMR, 0);		/* no interrupts please */
  res=recv_frame(FALSE);	/* see if any packets are available
				 * and if so, do process them
				 */
  output(EN0_IMR, ENISR_ALL);	/* allow interrupts */ 
  return res;
}

/*===========================================================================*
 *				send_pkt				     * 
 *===========================================================================*/
PUBLIC int send_pkt(pkt_buf, pkt_length)
phys_bytes pkt_buf;	/* packet buffer */
vir_bytes pkt_length;	/* packet length */
{
/* Transmit packet routine.
 * Returns OK if packet sent or error code otherwise.
 */
  int res, timeout=TRANSMIT_TIMEOUT;
  vir_bytes real_pkt_length;

  while((input(EN_CCMD)&ENC_TRANS) &&	/* is transmitter still running? */
	timeout--)			/* avoid infinite loop */
	;				/* wait for transmitter to finish */
  if (timeout == 0)		/* TX is stuck? */
  {				/* should count these error timeouts */
  /* count_out_err(); */
  /* maybe need to add recovery logic here */
  }
  real_pkt_length=pkt_length;	/* store real packet length */
  if (pkt_length > GIANT)	/* is this packet too large? */
    return(EFBIG);
  if (pkt_length < RUNT)	/* is this frame long enough? */
    pkt_length=RUNT;		/* stretch frame to minimum allowed */
  output(EN0_ISR, ENISR_RDC);	/* clear remote interrupt int. */
  output(EN0_TCNTLO, pkt_length & BYTE); /* tell card low byte of TX count */
  output(EN0_TCNTHI, pkt_length >> 8);	 /* tell card high byte of TX count */
  res=block_output((phys_bytes) pkt_buf,	 /* src: packet buffer */
		   (vir_bytes) SM_TSTART_PG<<8,	 /* dst: offset in s.m. */
		   (vir_bytes) real_pkt_length); /* how many bytes to copy? */
  if (res != OK)		/* error copying packet? */
    return(EIO);
  output(EN0_TPSR, SM_TSTART_PG);	/* start the transmitter */
  output(EN_CCMD, ENC_TRANS|ENC_NODMA|ENC_START);	/* ditto */
  return OK;
}

/*===========================================================================*
 *				set_8390_eaddr				     * 
 *===========================================================================*/
PUBLIC void set_8390_eaddr()
{
 /* copy our Ethernet address from curr_hw_addr into the 8390 */
  register int i;
 
  lock();					/* protect from unwanted irq */
  output(EN_CCMD, ENC_NODMA|ENC_PAGE1);		/* switch to page 1 */
  for (i=0 ; i<sizeof(Eth_addr) ; i++)		/* set ethernet addr */
    output(EN1_PHYS+i, curr_hw_addr.ea[i]);
  output(EN_CCMD, ENC_NODMA|ENC_PAGE0);		/* restore to page 0 */
  unlock();
}

/*===========================================================================*
 *				reset_board				     * 
 *===========================================================================*/
PUBLIC void reset_board()
{
  int timeout=RESET_TIMEOUT;

  reset_8390();
  output(EN_CCMD, ENC_STOP|ENC_NODMA);		/* stop the 8390 */
  while(!(input(EN0_ISR)&ENISR_RESET) && timeout--)
	;				/* wait for board to reset */
}

/*===========================================================================*
 *				recv					     * 
 *===========================================================================*/
PUBLIC void recv()
{
/* Interrupt handling routine. Actually, not just receive, but all
 * interrupts come here. Upon exit, the interrupt will be acknowledged.
 */
  register unsigned isr;

  ram_enable();
  output(EN0_IMR, 0);		/* no interrupts please */
  for (isr=input(EN0_ISR) ; isr & ENISR_ALL ; isr=input(EN0_ISR))
  {
#ifdef DEBUG
printf("=> isr=%02x (OVR=%02x, RX=%02x, RX_ERR=%02x, COUNT=%02x)\n",
	isr, isr & ENISR_OVER, isr & ENISR_RX, 
	isr & ENISR_RX_ERR, isr & ENISR_COUNTERS);
#endif
    if (isr & ENISR_OVER)
    {
#ifdef DEBUG
      printf("calling recv_overrun()\n");
#endif
      recv_ovrerrun();
#ifdef DEBUG
      printf("recv_overrun() completed\n");
#endif
    }
    else if (isr & (ENISR_RX|ENISR_RX_ERR))
    {
#ifdef DEBUG
      printf("calling recv_frame()\n");
#endif
      recv_frame(TRUE);
#ifdef DEBUG
      printf("recv_frame() completed\n");
#endif
    }
    else if (isr & (ENISR_TX|ENISR_TX_ERR))
    {
#ifdef DEBUG
      printf("calling isr_tx()\n");
#endif
      isr_tx();
#ifdef DEBUG
      printf("isr_tx() completed\n");
#endif
    }
    else if (isr & ENISR_COUNTERS)
    {
#ifdef DEBUG
      printf("calling isr_stat()\n");
#endif
      isr_stat();
#ifdef DEBUG
      printf("isr_stat() completed\n");
#endif
    }
  }
  output(EN0_IMR, ENISR_ALL);	/* allow interrupts */ 
}

/*===========================================================================*
 *				recv_overrun				     * 
 *===========================================================================*/
PRIVATE void recv_overrun()
{
/* ENISR_OVER handling routine.
 * No statistics for now.
 */
  unsigned curpag, boundary, frm_size;
  int timeout=RESET_TIMEOUT;
  vir_bytes sm_offset;
  phys_bytes rcv_hdr_phys;

  output(EN_CCMD, ENC_STOP|ENC_NODMA);	/* write "stop" to command register */
  output(EN_CCMD, ENC_PAGE1|ENC_NODMA);	/* switch to page 1 */
  curpag=input(EN1_CURPAG);		/* get current page */
  output(EN_CCMD, ENC_PAGE0|ENC_NODMA);	/* back to page 0 */
  boundary=input(EN0_BOUNDARY);		/* get memory page number */
  boundary++;				/* page + 1 */
#ifdef DEBUG
printf("curpag = %d, boundary = %d\n", curpag, boundary);
#endif
  if (boundary==sm_rstop_ptr)		/* wrapped around ring? */
  {
    boundary=SM_RSTART_PG;		/* yes, wrap the page pointer */
#ifdef DEBUG
printf("boundary reset to %d\n", boundary);
#endif
  }
  if (boundary!=curpag)			/* check if buffer empty */
  {					/* no, do remove frame */
    sm_offset=boundary<<8;	/* make a byte address, e.g. page 0x46
				 * becomes 0x4600 into buffer 
				 */
    rcv_hdr_phys=numap(ETHER, (vir_bytes) rcv_hdr, (vir_bytes) RCV_HDR_SIZE);
    block_input((vir_bytes) sm_offset, 		/* get frame header */
		(phys_bytes) rcv_hdr_phys, 
		(vir_bytes) RCV_HDR_SIZE);
    frm_size=(rcv_hdr[EN_RBUF_SIZE_HI]<<8)+	/* extract size of frame */
	      rcv_hdr[EN_RBUF_SIZE_LO];
    frm_size-=EN_RBUF_NHDR;			/* less the header stuff */
#ifdef DEBUG
printf("rcv_hdr[] is:  STAT=%02x  NXT_PG=%02x  SIZE_LO=%02x,  SIZE_HI=%02x  (size=%d)\n",
	(unsigned) rcv_hdr[EN_RBUF_STAT],
	(unsigned) rcv_hdr[EN_RBUF_NXT_PG],
	(unsigned) rcv_hdr[EN_RBUF_SIZE_LO],
	(unsigned) rcv_hdr[EN_RBUF_SIZE_HI],
	frm_size
	);
#endif
  /* Internal 8390 chip logic that also issued overwrite interrupt protects 
   * frames previously received and stored in shared memory from unwanted
   * erasing by next incoming packet. Chip will reject such frames until
   * there is enough space in buffer to accept a frame.
   * Unfortunately, tests revealed that sometimes, especially on heavily
   * loaded networks, it may happen that first frame in receive ring buffer
   * (the one pointed to by Boundary Pointer) becomes overwritten.
   * These situations are quite rare, but may lead to a crash of the whole
   * system. Thus we should detect them and take some* recovery action.
   * Because of total destruction of receive ring, all frames
   * stored there are lost. Board should be stopped and re-initialized.
   */
    if (rcv_hdr[EN_RBUF_NXT_PG] < SM_RSTART_PG ||	/* wrong link */
        rcv_hdr[EN_RBUF_NXT_PG] >= sm_rstop_ptr ||	/* ... or ... */
	frm_size < RUNT || frm_size > GIANT )		/* frame size? */
    {
#ifdef DEBUG
printf("recovery logic activated in recv_overrun()\n");
#endif
      etopen();		/* re-initialize board */
      return;		/* quit immediately */
    }
    if (rcv_hdr[EN_RBUF_STAT] & ENRSR_RXOK)	/* is this frame any good? */
      rcv_frm(boundary);			/* yes, go accept it */
    boundary=rcv_hdr[EN_RBUF_NXT_PG];	/* get pointer to next frame */
    boundary--;			/* back up one page */
#ifdef DEBUG
printf("new boundary is %d\n", boundary);
#endif
    if (boundary < SM_RSTART_PG)	/* did it wrap? */
    {
      boundary=sm_rstop_ptr;		/* yes, back to end of ring */
      boundary--;
#ifdef DEBUG
printf("new boundary reset to %d\n", boundary);
#endif
    }
    output(EN0_BOUNDARY, boundary);	/* set new boundary */
  }
  output(EN0_RCNTLO, 0);		/* clear byte count registers */
  output(EN0_RCNTHI, 0);		/* ... */
  while(!(input(EN0_ISR)&ENISR_RESET) && timeout--)
	;				/* wait for board to reset */
  output(EN0_TXCR, ENTXCR_LOOP);	/* put transmitter in loopback mode */
  output(EN_CCMD, ENC_START|ENC_NODMA);	/* start the chip running again */
  output(EN0_TXCR, 0);			/* clear the loopback bit */
  output(EN0_ISR, ENISR_OVER);		/* ACK */
/* count_in_err(); */
}

/*===========================================================================*
 *				recv_frame				     * 
 *===========================================================================*/
PRIVATE int recv_frame(hard_int)
int hard_int;	/* TRUE if called by interrupt handler */
{
/* ENISR_RX and ENISR_RX_ERR handling routine.
 * It is also used as a standard receive routine.
 * Return OK if frame received.
 *
 * No statistics for now.
 */
  unsigned curpag, boundary, frm_size;
  vir_bytes sm_offset;
  phys_bytes rcv_hdr_phys, res;

  res=DISCARD_PKT;
  while (res == DISCARD_PKT)		/* loop to process all the frames */
  {
    if (hard_int)		/* processing as a part of int handler? */
      output(EN0_ISR, ENISR_RX|ENISR_RX_ERR);	/* ACK */
    output(EN_CCMD,
	   ENC_NODMA|ENC_PAGE1|ENC_START);	/* switch to page 1 regs */
    curpag=input(EN1_CURPAG);			/* hold current page */
    output(EN_CCMD,
	   ENC_NODMA|ENC_PAGE0|ENC_START);	/* back to page 0 */
    boundary=input(EN0_BOUNDARY);		/* get boundary page number */
    boundary++;			/* step boundary from last used page */
#ifdef DEBUG
printf("curpag = %d, boundary = %d\n", curpag, boundary);
#endif
    if (boundary==sm_rstop_ptr)	/* wrapped around ring? */
    {
      boundary=SM_RSTART_PG;	/* yes, wrap to first RX page */
#ifdef DEBUG
printf("boundary reset to %d\n", boundary);
#endif
    }
    if (boundary==curpag)	/* read all the frames? */
      break;			/* yes, finished them all */
  /* now do receive frame! */
    sm_offset=boundary<<8;	/* make a byte address, e.g. page 0x46
				 * becomes 0x4600 into buffer 
				 */
    rcv_hdr_phys=numap(ETHER, (vir_bytes) rcv_hdr, (vir_bytes) RCV_HDR_SIZE);
    block_input((vir_bytes) sm_offset, 		/* get frame header */
		(phys_bytes) rcv_hdr_phys, 
		(vir_bytes) RCV_HDR_SIZE);
    frm_size=(rcv_hdr[EN_RBUF_SIZE_HI]<<8)+	/* extract size of frame */
	      rcv_hdr[EN_RBUF_SIZE_LO];
    frm_size-=EN_RBUF_NHDR;			/* less the header stuff */
#ifdef DEBUG
printf("rcv_hdr[] is:  STAT=%02x  NXT_PG=%02x  SIZE_LO=%02x,  SIZE_HI=%02x  (size=%d)\n",
	(unsigned) rcv_hdr[EN_RBUF_STAT],
	(unsigned) rcv_hdr[EN_RBUF_NXT_PG],
	(unsigned) rcv_hdr[EN_RBUF_SIZE_LO],
	(unsigned) rcv_hdr[EN_RBUF_SIZE_HI],
	frm_size
	);
#endif
  /* Check for frame overwrite case. See comment in recv_overrrun() above.
   * Board should be stopped and re-initialized.
   */
    if (rcv_hdr[EN_RBUF_NXT_PG] < SM_RSTART_PG ||	/* wrong link */
        rcv_hdr[EN_RBUF_NXT_PG] >= sm_rstop_ptr ||	/* ... or ... */
	frm_size < RUNT || frm_size > GIANT )		/* frame size? */
    {
#ifdef DEBUG
printf("recovery logic activated in recv_frame()\n");
#endif
      etopen();		/* re-initialize board */
      return EAGAIN;	/* quit immediately with error code */
    }
    if (rcv_hdr[EN_RBUF_STAT] & ENRSR_RXOK)	/* good frame? */
    {						/* yes, go accept it */
      res=rcv_frm(boundary);
      if (res == LEAVE_PKT)	/* leave packet? (because not received) */
        break;			/* yes, exit loop immediately and
				 * don't update Boundary Pointer
				 */
    }
    else
    {						/* oops, bad frame */
     /* soft_rx_errors++; */
    }
    boundary=rcv_hdr[EN_RBUF_NXT_PG];	/* start of next frame */
    boundary--;			/* make previous page for new boundary */
    if (boundary < SM_RSTART_PG)	/* wrap around the bottom? */
    {
      boundary=sm_rstop_ptr;		/* yes. */
      boundary--;
    }
    output(EN0_BOUNDARY, boundary);	/* set new boundary */
  } /* loop */
  output(EN_CCMD, ENC_NODMA|ENC_PAGE0|ENC_START);

  return (res != LEAVE_PKT && res != DISCARD_PKT) ? OK : EAGAIN;
}

/*===========================================================================*
 *				isr_tx					     * 
 *===========================================================================*/
PRIVATE void isr_tx()
{
/* ENISR_TX and ENISR_TX_ERR handling routine.
 * No statistics for now.
 */
  if ( input(EN0_ISR) & ENISR_TX )
  {
    output(EN0_ISR, ENISR_TX);		/* ACK */
  }
  else	/* must be ENISR_TX_ERR */
  {
    output(EN0_ISR, ENISR_TX_ERR);	/* ACK */
/* count_out_err(); */
  }
}

/*===========================================================================*
 *				isr_stat				     * 
 *===========================================================================*/
PRIVATE void isr_stat()
{
/* ENISR_COUNTER handling routine.
 * We have to read counters to clear them and to clear the interrupt.
 */
  input(EN0_COUNTER0);			/* ignore contents */
  input(EN0_COUNTER1);			/* ... */
  input(EN0_COUNTER2);			/* ... */
  output(EN0_ISR, ENISR_COUNTERS);	/* ACK */
}

/*===========================================================================*
 *				rcv_frm					     * 
 *===========================================================================*/
PRIVATE phys_bytes rcv_frm(page_no)
unsigned page_no;	/* page number of the frame header in shared memory */
{
/* Do the work of copying out a receive frame.
 * Returns:
 *   LEAVE_PKT   - if packet should be left in shared memory
 *   DISCARD_PKT - if packet should be discarded (skipped) from s.m.
 *     other     - packet was copied and now it can be discarded
 */
  unsigned frm_size, pkt_type, pkt_class;
  vir_bytes sm_offset;
  phys_bytes dst_addr;

/* software multicast filter stuff (if any) would go just here */

  frm_size=(rcv_hdr[EN_RBUF_SIZE_HI]<<8)+	/* extract size of frame */
	    rcv_hdr[EN_RBUF_SIZE_LO];
  frm_size-=EN_RBUF_NHDR;			/* less the header stuff */
  pkt_type=(rcv_hdr[EN_RBUF_NHDR+2*EADDR_LEN]<<8)+	
	    rcv_hdr[EN_RBUF_NHDR+2*EADDR_LEN+1];
  pkt_class=BLUEBOOK;			/* determine packet type and class */
  dst_addr=recv_find((unsigned) frm_size, 
		     (unsigned) pkt_type,
		     (unsigned) pkt_class);
				/* see if anybody wants this frame
				 * for now, we expect BlueBook packets only
				 */

  switch (dst_addr)	  	/* what did recv_find answer? */
  {
    case LEAVE_PKT:	/* answer 1: there is no read request for 
			 * any packet, so please leave it in
			 * board's shared memory
			 */
      break;

    case DISCARD_PKT:	/* answer 2: we don't like this particular
			 * packet, so please discard it from s. m.
 			 */
      break;

    default:		/* answer 3: a valid pointer to rcv buffer */
      sm_offset=page_no<<8;		/* page to start from */
      sm_offset+=EN_RBUF_NHDR;		/* skip the header stuff */
      block_input(sm_offset, dst_addr, (vir_bytes) frm_size);
      recv_copy(dst_addr, frm_size);	/* data copied! */
      break;
  }

  return dst_addr;	/* report what have been done */
}

/*===========================================================================*
 *				etopen					     * 
 *===========================================================================*/
PUBLIC void etopen()		/* initialize interface */
{
/* Step 1: reset and stop the 8390 */
  reset_board();

/* Step 2: init Data Configuration Register */
  output(EN0_DCFG, endcfg);

/* Step 3: clear Remote Byte Counter Registers */
  output(EN0_RCNTLO, 0x00);
  output(EN0_RCNTHI, 0x00);

/* Step 4: set receiver to monitor mode */
  output(EN0_RXCR, ENRXCR_MON);

/* Step 5: place 8390 into loopback mode */
  output(EN0_TXCR, ENTXCR_LOOP);

/* Step 6: do anything special that the card needs */
  init_card();

/* Step 7: re-init endcfg in case they put it into word mode */
  output(EN0_DCFG, endcfg);

/* Step 8: init EN0_STARTPG to same value as EN0_BOUNDARY */
  output(EN0_STARTPG, SM_RSTART_PG);
  output(EN0_BOUNDARY, SM_RSTART_PG);
  output(EN0_STOPPG, sm_rstop_ptr);

/* Step 9: write all 1's to all bits of EN0_ISR to clear pending interrupts */
  output(EN0_ISR, 0xff);

/* Step 10: init EN0_IMR as desired */
  output(EN0_IMR, ENISR_ALL);

/* Step 11: init the Ethernet address */
  set_8390_eaddr();

/* Step 12: program EN_CCMD for page 1 */
  output(EN_CCMD, ENC_PAGE1|ENC_NODMA|ENC_STOP);

/* Step 13: program Current Page Register to a value of Boundary Pointer + 1 */
  output(EN1_CURPAG, SM_RSTART_PG+1);

/* Step 14: program EN_CCMD back to page 0, and start it */
  output(EN_CCMD, ENC_NODMA|ENC_START);
  output(EN0_TXCR, 0);		/* set transmiter mode to normal */
  output(EN0_RXCR, rxcr_bits);	/* tell it what frames to accept */
  ram_enable();
}

#endif /* ETHERNET - no code after this line ! */
