#define ITABLE __section__ (".itable")

/* 16 bit free running timer support */

typedef void (*ifunc) ();

void reset_handler ();
void timer_handler ();
void inc_handler ();
void coldstart_handler ();

struct itable_struct
{
  ifunc coldstart;
  ifunc reserved1;
  ifunc reserved2;
  /* external interrupts */
  ifunc nmi;
  ifunc irq0;
  ifunc irq1;
  ifunc irq2;
  ifunc irq3;
  ifunc irq4;
  ifunc irq5;
  ifunc irq6;
  ifunc irq7;
  /* 16 bit timer */
  ifunc icia;
  ifunc icib;
  ifunc icic;
  ifunc icid;
  ifunc ocia;
  ifunc ocib;
  ifunc fovi;

  /* 8 bit timer 0 */
  ifunc cmi0a;
  ifunc cmi0b;
  ifunc ovi0;
  /* 8 bit timer 1 */
  ifunc cmi1a;
  ifunc cmi1b;
  ifunc ovi1;

  /* Dual port ram */
  ifunc mrei;
  ifunc mwei;

  /* Serial interface */
  ifunc eri;
  ifunc rxi;
  ifunc txi;

  /* A/D converter */
  ifunc adi;
};

const ITABLE
struct itable_struct itable =
{
  coldstart_handler,            /* reset */
  reset_handler,                /* other */
  reset_handler,                /* other */
  inc_handler,                  /* nmi */
  reset_handler,                /* irq 0 */
  reset_handler,
  reset_handler,
  reset_handler,
  reset_handler,
  reset_handler,
  reset_handler,
  reset_handler,
 /* 16 bit timer */
  reset_handler,
  reset_handler,
  reset_handler,
  reset_handler,
  timer_handler,
  reset_handler,
  reset_handler,
 /* 8 bit timer 0 */
  reset_handler,
  reset_handler,
  reset_handler,

 /* 8 bit timer 1 */
  reset_handler,
  reset_handler,
  reset_handler,

 /* Dual port ram */
  reset_handler,
  reset_handler,

 /* Serial line */
  reset_handler,
  reset_handler,
  reset_handler,

 /* A/D */
  reset_handler,

};

typedef union
  {
    char val;
    struct
      {
	unsigned int bit7:1;
	unsigned int bit6:1;
	unsigned int bit5:1;
	unsigned int bit4:1;
	unsigned int bit3:1;
	unsigned int bit2:1;
	unsigned int bit1:1;
	unsigned int bit0:1;
      }
    bits;
  }

eight_bit_type;

struct port6_struct
  {
    /* 0xffb9 - port 6 data direction register */
    eight_bit_type ddr;
    /* 0xffba dummy */
    eight_bit_type empty;
    /* 0xffbb - port 6 data register */
    eight_bit_type dr;
  };

struct timer_struct
  {

    /* 0xff90 Timer Control Register */
    struct
      {
	unsigned int iciae:1;
	unsigned int icibe:1;
	unsigned int icice:1;
	unsigned int icide:1;
	unsigned int ociae:1;
	unsigned int ocibe:1;
	unsigned int ovie:1;
	unsigned int dummy:1;
      }
    tier;

    /* 0xff91 Timer Control/Status Register */
    struct
      {
	unsigned int icfa:1;
	unsigned int icfb:1;
	unsigned int icfc:1;
	unsigned int icfd:1;
	unsigned int ocfa:1;
	unsigned int ocfb:1;
	unsigned int ovf:1;
	unsigned int cclra:1;
      }
    tcsr;

/* 0xff92 Free-Running counter */
    unsigned int frc;

    /* 0xff94 Output Compare Register A/B */
    unsigned int ocr;

    /* 0xff96 Timer Control Register */
    struct
      {
	unsigned int iedga:1;
	unsigned int iedgb:1;
	unsigned int iedgc:1;
	unsigned int iedgd:1;
	unsigned int bufea:1;
	unsigned int bufeb:1;
	unsigned int cks1:1;
	unsigned int cks0:1;
      }
    tcr;

    /* 0xff97 Timer Output Compare Control Register */
    struct
      {
	unsigned int d1:1;
	unsigned int d2:1;
	unsigned int d3:1;
	unsigned int ocrs:1;
	unsigned int oea:1;
	unsigned int oeb:1;
	unsigned int olvla:1;
	unsigned int olvlb:1;
      }
    tocr;

    /* 0xff98 Input Capture register */
    unsigned int icra;

  };



static struct timer_struct volatile *const otimer = (struct timer_struct *) (0xff90);
static struct port6_struct volatile *const port6 = (struct port6_struct *) (0xffb9);

#define timer  ( ( struct timer_struct *) (0xff90))



void
init_timer ()
{
  timer->tier.ociae = 1;        /* interrupt when counter reaches reg a */

  timer->tocr.ocrs = 0;         /* select counter A */
  timer->tocr.oea = 0;          /* don't output it */
  timer->tocr.oeb = 0;          /* don't output B either */

   timer->tcr.cks1 = 1;         /* Divide system clock by 32 */
  timer->tcr.cks0 = 0;

  timer->ocr = 100;             /* Count to 100 before interrupt */
  timer->tcsr.cclra = 1;        /* clear frc when match */
  timer->frc = 0;               /* Initialize free running counter */
}

volatile int ledcount;

static void 
delay ()
{
  int j;

  for (j = 0; j < 1000; j++)
    asm ("nop");
}

static void
init_leds ()
{
  int i;

  port6->ddr.val = 0xff;
  port6->dr.val = ledcount = -1;

  for (i = 0; i < 100; i++)
    {
      port6->dr.val = ++ledcount;
      delay ();
    }
}
unsigned int speed = 32;

#pragma interrupt

void
timer_handler ()
{
  static int count;

  count++;

  if (count > speed)
    {
      count = 0;
      timer->frc = 0;           /* reset counter */
      /* Count 100 interrupts then inc light */
      port6->dr.val = ++ledcount;
    }
}

#pragma interrupt

void
inc_handler ()
{
  /* Add one to the leds */
  port6->dr.val = ++ledcount;

  speed *= 2;
  if (speed > 20000)
    speed = 1;

}

#pragma interrupt

void
reset_handler ()
{
  port6->dr.val = ++ledcount;
  coldstart_handler ();
}

void
disable_interrupts ()
{
  asm ("orc #0x80,ccr");
}

void
enable_interrupts ()
{
  asm ("andc #0x7f,ccr");
}


entry ()
{
  init_leds ();
  init_timer ();
  enable_interrupts ();

  while (1)
    {
      asm ("nop");
    }
}


void 
coldstart_handler ()
{
  asm ("mov.w #_end+0x400,r7");
  init_leds ();
  entry ();
}

