#ifndef lint
static char sccsid[] = "@(#)j0.c	1.2	(ucb.beef)	10/2/89";
#endif	/* !defined(lint) */
/* 
 *  This packet computes zero-order Bessel functions of the first and
 *    second kind (j0 and y0), for real arguments x, where 0 < x <= XMAX
 *    for y0, and |x| <= XMAX for j0.  It contains three function-type
 *    subprograms, j0, y0 and caljy0.  The calling statements for the
 *    primary entries are:
 * 
 *            y = j0(x)
 *    and
 *            y = y0(x),
 * 
 *    where the entry points correspond to the functions j0(x) and y0(x),
 *    respectively.  The routine caljy0() is intended for internal packet
 *    use only, all computations within the packet being concentrated in
 *    this one routine.  The function subprograms invoke  caljy0  with
 *    the statement
 *
 *            result = caljy0(x,jint);
 *
 *    where the parameter usage is as follows:
 * 
 *       Function                  Parameters for caljy0
 *        call              x             result          jint
 * 
 *       j0(x)        |x| <= XMAX         j0(x)           0
 *       y0(x)      0 < x <= XMAX         y0(x)           1
 * 
 *    The main computation uses unpublished minimax rational
 *    approximations for x <= 8.0, and an approximation from the 
 *    book  Computer Approximations  by Hart, et. al., Wiley and Sons, 
 *    New York, 1968, for arguments larger than 8.0   Part of this
 *    transportable packet is patterned after the machine-dependent
 *    FUNPACK program for j0(x), but cannot match that version for
 *    efficiency or accuracy.  This version uses rational functions
 *    that are theoretically accurate to at least 18 significant decimal
 *    digits for x <= 8, and at least 18 decimal places for x > 8.  The
 *    accuracy achieved depends on the arithmetic system, the compiler,
 *    the intrinsic functions, and proper selection of the machine-
 *    dependent constants.
 * 
 *********************************************************************
 * 
 *  Explanation of machine-dependent constants
 * 
 *    XINF   = largest positive machine number
 *    XMAX   = largest acceptable argument.  The functions sin(), floor()
 *             and cos() must perform properly for  fabs(x) <= XMAX.
 *             We recommend that XMAX be a small integer multiple of
 *             sqrt(1/eps), where eps is the smallest positive number
 *             such that  1+eps > 1. 
 *    XSMALL = positive argument such that  1.0-(x/2)**2 = 1.0
 *             to machine precision for all  fabs(x) <= XSMALL.
 *             We recommend that  XSMALL < sqrt(eps)/beta, where beta
 *             is the floating-point radix (usually 2 or 16).
 * 
 *      Approximate values for some important machines are
 * 
 *                           eps      XMAX     XSMALL      XINF  
 * 
 *   CDC 7600      (S.P.)  7.11E-15  1.34E+08  2.98E-08  1.26E+322
 *   CRAY-1        (S.P.)  7.11E-15  1.34E+08  2.98E-08  5.45E+2465
 *   IBM PC (8087) (S.P.)  5.96E-08  8.19E+03  1.22E-04  3.40E+38
 *   IBM PC (8087) (D.P.)  1.11D-16  2.68D+08  3.72D-09  1.79D+308
 *   IBM 195       (D.P.)  2.22D-16  6.87D+09  9.09D-13  7.23D+75
 *   UNIVAC 1108   (D.P.)  1.73D-18  4.30D+09  2.33D-10  8.98D+307
 *   VAX 11/780    (D.P.)  1.39D-17  1.07D+09  9.31D-10  1.70D+38
 * 
 *********************************************************************
 *********************************************************************
 * 
 *  Error Returns
 * 
 *   The program returns the value zero for  x > XMAX, and returns
 *     -XINF when y0 is called with a negative or zero argument.
 * 
 * 
 *  Intrinsic functions required are:
 * 
 *      fabs, floor, cos, log, sin, sqrt
 * 
 *
 *   Latest modification: June 2, 1989
 * 
 *   Author: W. J. Cody
 *           Mathematics and Computer Science Division 
 *           Argonne National Laboratory
 *           Argonne, IL 60439
 */

 #include "fpumath.h"

					/* Machine-dependent constants */
#if defined(vax) || defined(tahoe)
#define	XMAX	(double)1.07e9
#define	XSMALL	(double)9.31e-10
#define	XINF	(double)1.70e38
#else	/* defined(vax) || defined(tahoe) */
#define	XMAX	(double)2.68e8
#define	XSMALL	(double)3.72e-9
#define	XINF	MAXFLOAT
#endif	/* defined(vax) || defined(tahoe) */
					/* Mathematical constants */
#define	EIGHT	(double)8
#define	CONS	(double)(-1.1593151565841244881e-1) /* ln(.5)+Euler's gamma */
#define	FIVE5	(double)5.5
#define	FOUR	(double)4
#define	ONE	(double)1
#define	ONEOV8	(double)0.125
#define	PI2	(double)6.3661977236758134308e-1
#define	P17	(double)1.716e-1
#define	SIXTY4	(double)64
#define	THREE	(double)3
#define	TWOPI	(double)6.2831853071795864769e0
#define	TWOPI1	(double)6.28125
#define	TWOPI2	(double)1.9353071795864769253e-03
#define	TWO56	(double)256
#define	ZERO	(double)0
					/* Zeroes of Bessel functions */
#define	XJ0	(double)2.4048255576957727686e0
#define	XJ1	(double)5.5200781102863106496e0
#define	XY0	(double)8.9357696627916752158e-1
#define	XY1	(double)3.9576784193148578684e0
#define	XY2	(double)7.0860510603017726976e0
#define	XJ01	(double)616
#define	XJ02	(double)(-1.4244423042272313784e-3)
#define	XJ11	(double)1413
#define	XJ12	(double)5.4686028631064959660e-4
#define	XY01	(double)228.0
#define	XY02	(double)2.9519662791675215849e-3
#define	XY11	(double)1013
#define	XY12	(double)6.4716931485786837568e-4
#define	XY21	(double)1814
#define	XY22	(double)1.1356030177269762362e-4

/*
 * Coefficients for rational approximation to ln(x/a)
 */
static double PLG[] = {
	-2.4562334077563243311e01,
	 2.3642701335621505212e02,
	-5.4989956895857911039e02,
	 3.5687548468071500413e02,
};
static double QLG[] = {
	-3.5553900764052419184e01,
	 1.9400230218539473193e02,
	-3.3442903192607538956e02,
	 1.7843774234035750207e02,
};

/*
 * Coefficients for rational approximation of
 * j0(x) / (x**2 - XJ0**2),  XSMALL  <  |x|  <=  4.0
 */
static double PJ0[] = {
	 6.6302997904833794242e06,
	-6.2140700423540120665e08,
	 2.7282507878605942706e10,
	-4.1298668500990866786e11,
	-1.2117036164593528341e-01,
	 1.0344222815443188943e02,
	-3.6629814655107086448e04,
};
static double QJ0[] = {
	4.5612696224219938200e05,
	1.3985097372263433271e08,
	2.6328198300859648632e10,
	2.3883787996332290397e12,
	9.3614022392337710626e02,
};

/*
 * Coefficients for rational approximation of
 * j0(x) / (x**2 - XJ1**2),  4.0  <  |x|  <=  8.0
 */
static double PJ1[] = {
	 4.4176707025325087628e03,
	 1.1725046279757103576e04,
	 1.0341910641583726701e04,
	-7.2879702464464618998e03,
	-1.2254078161378989535e04,
	-1.8319397969392084011e03,
	 4.8591703355916499363e01,
	 7.4321196680624245801e02,
};
static double QJ1[] = {
	 3.3307310774649071172e02,
	-2.9458766545509337327e03,
	 1.8680990008359188352e04,
	-8.4055062591169562211e04,
	 2.4599102262586308984e05,
	-3.5783478026152301072e05,
	-2.5258076240801555057e01,
};

/*
 * Coefficients for rational approximation of
 *   (y0(x) - 2 LN(x/XY0) j0(x)) / (x**2 - XY0**2),
 *       XSMALL  <  |x|  <=  3.0
 */
static double PY0[] = {
	 1.0102532948020907590e04,
	-2.1287548474401797963e06,
	 2.0422274357376619816e08,
	-8.3716255451260504098e09,
	 1.0723538782003176831e11,
	-1.8402381979244993524e01,
};
static double QY0[] = {
	6.6475986689240190091e02,
	2.3889393209447253406e05,
	5.5662956624278251596e07,
	8.1617187777290363573e09,
	5.8873865738997033405e11,
};

/*
 * Coefficients for rational approximation of
 *   (y0(x) - 2 LN(x/XY1) j0(x)) / (x**2 - XY1**2),
 *       3.0  <  |x|  <=  5.5
 */
static double PY1[] = {
	-1.4566865832663635920e04,
	 4.6905288611678631510e06,
	-6.9590439394619619534e08,
	 4.3600098638603061642e10,
	-5.5107435206722644429e11,
	-2.2213976967566192242e13,
	 1.7427031242901594547e01,
};
static double QY1[] = {
	8.3030857612070288823e02,
	4.0669982352539552018e05,
	1.3960202770986831075e08,
	3.4015103849971240096e10,
	5.4266824419412347550e12,
	4.3386146580707264428e14,
};

/*
 * Coefficients for rational approximation of
 *   (y0(x) - 2 LN(x/XY2) j0(x)) / (x**2 - XY2**2),
 *       5.5  <  |x|  <=  8.0
 */
static double PY2[] = {
	 2.1363534169313901632e04,
	-1.0085539923498211426e07,
	 2.1958827170518100757e09,
	-1.9363051266772083678e11,
	-1.2829912364088687306e11,
	 6.7016641869173237784e14,
	-8.0728726905150210443e15,
	-1.7439661319197499338e01,
};
static double QY2[] = {
	8.7903362168128450017e02,
	5.3924739209768057030e05,
	2.4727219475672302327e08,
	8.6926121104209825246e10,
	2.2598377924042897629e13,
	3.9272425569640309819e15,
	3.4563724628846457519e17,
};

/*
 * Coefficients for Hart,s approximation,  |x| > 8.0
 */
static double P0[] = {
	3.4806486443249270347e03,
	2.1170523380864944322e04,
	4.1345386639580765797e04,
	2.2779090197304684302e04,
	8.8961548424210455236e-01,
	1.5376201909008354296e02,
};
static double Q0[] = {
	3.5028735138235608207e03,
	2.1215350561880115730e04,
	4.1370412495510416640e04,
	2.2779090197304684318e04,
	1.5711159858080893649e02,
};
static double P1[] = {
	-2.2300261666214198472e01,
	-1.1183429920482737611e02,
	-1.8591953644342993800e02,
	-8.9226600200800094098e01,
	-8.8033303048680751817e-03,
	-1.2441026745835638459e00,
};
static double Q1[] = {
	1.4887231232283756582e03,
	7.2642780169211018836e03,
	1.1951131543434613647e04,
	5.7105024128512061905e03,
	9.0593769594993125859e01,
};

static double
#if defined(__STDC__) || defined(__GNUC__)
caljy0(double x,int jint)
#else
caljy0(x,jint)
double x;
int jint;
#endif
{
	int i;
	double resj,down,up,xden,xnum,w,wsq,z,zsq;

	if (jint && x <= ZERO)		/* Check for error conditions */
		return -XINF;
#define	ax x
	else if ((ax = fabs(x)) > XMAX)
		return ZERO;
/*
 * Calculate j0 or y0 for |x|  >  8.0
 */
	if (ax > EIGHT) {
		z = EIGHT/ax;
		w = ax/TWOPI;
		w = floor(w)+ONEOV8;
		w = (ax-w*TWOPI1)-w*TWOPI2;
		zsq = z*z;
		xnum = P0[4]*zsq+P0[5];
		xden = zsq+Q0[4];
		up = P1[4]*zsq+P1[5];
		down = zsq+Q1[4];
		for (i = 0; i <= 3; i++) {
			xnum = xnum*zsq+P0[i];
			xden = xden*zsq+Q0[i];
			up = up*zsq+P1[i];
			down = down*zsq+Q1[i];
		}
#define	r0 xnum
#define	r1 up
		r0 = xnum/xden;
		r1 = up/down;
		return sqrt(PI2/ax)*(jint ? r0*sin(w)+z*r1*cos(w) :
			r0*cos(w)-z*r1*sin(w));
#undef	r1
#undef	r0
	}
	if (ax <= XSMALL)
		return jint ? PI2*(log(ax)+CONS) : ONE;
/*
 * Calculate j0 for appropriate interval, preserving
 *    accuracy near the zero of j0
 */
	zsq = ax*ax;
	if (ax <= FOUR) {
		xnum = (PJ0[4]*zsq+PJ0[5])*zsq+PJ0[6];
		xden = zsq+QJ0[4];
		for (i = 0; i <= 3; i++) {
			xnum = xnum*zsq+PJ0[i];
			xden = xden*zsq+QJ0[i];
		}
#define	prod resj
		prod = ((ax-XJ01/TWO56)-XJ02)*(ax+XJ0);
	}
	else {
		wsq = ONE-zsq/SIXTY4;
		xnum = PJ1[6]*wsq+PJ1[7];
		xden = wsq+QJ1[6];
		for (i = 0; i <= 5; i++) {
			xnum = xnum*wsq+PJ1[i];
			xden = xden*wsq+QJ1[i];
		}
		prod = (ax+XJ1)*((ax-XJ11/TWO56)-XJ12);
	}
#define	result resj
	result = prod*xnum/xden;
#undef	prod
	if (!jint)
		return result;
/*
 * Calculate y0.  First find  resj = pi/2 ln(x/xn) j0(x),
 *   where xn is a zero of y0
 */
#define	xy z
	if (ax <= THREE) {
		up = (ax-XY01/TWO56)-XY02;
		xy = XY0;
	}
	else if (ax <= FIVE5) {
		up = (ax-XY11/TWO56)-XY12;
		xy = XY1;
	}
	else {
		up = (ax-XY21/TWO56)-XY22;
		xy = XY2;
	}
	down = ax+xy;
	if (fabs(up) < P17*down) {
		w = up/down;
		wsq = w*w;
		xnum = PLG[0];
		xden = wsq+QLG[0];
		for (i = 1; i <= 3; i++) {
			xnum = xnum*wsq+PLG[i];
			xden = xden*wsq+QLG[i];
		}
		resj = PI2*result*w*xnum/xden;
	}
	else
		resj = PI2*result*log(ax/xy);
#undef	xy
#undef	result
/*
 * Now calculate y0 for appropriate interval, preserving
 *    accuracy near the zero of y0
 */
	if (ax <= THREE) {
		xnum = PY0[5]*zsq+PY0[0];
		xden = zsq+QY0[0];
		for (i = 1; i <= 4; i++) {
			xnum = xnum*zsq+PY0[i];
			xden = xden*zsq+QY0[i];
		}
	}
	else if (ax <= FIVE5) {
#undef	ax
		xnum = PY1[6]*zsq+PY1[0];
		xden = zsq+QY1[0];
		for (i = 1; i <= 5; i++) {
			xnum = xnum*zsq+PY1[i];
			xden = xden*zsq+QY1[i];
		}
	}
	else {
		xnum = PY2[7]*zsq+PY2[0];
		xden = zsq+QY2[0];
		for (i = 1; i <= 6; i++) {
			xnum = xnum*zsq+PY2[i];
			xden = xden*zsq+QY2[i];
		}
	}
	return resj+up*down*xnum/xden;
}

/* 
 *  This subprogram computes approximate values for Bessel functions
 *    of the first kind of order zero for arguments  |x| <= XMAX
 *    (see comments heading caljy0).
 */
float
j0f(float x)
{
	return ((float)caljy0(x,0));
}

/* 
 *  This subprogram computes approximate values for Bessel functions
 *    of the second kind of order zero for arguments 0 < x <= XMAX
 *    (see comments heading caljy0).
 */
float
y0f(float x)
{
	return ((float)caljy0(x,1));
}
