/*
 * testnfs -- mount ourselves as a user-mode NFS server,
 * passing all requests to an real NFS server on another hosts,
 * while logging each request and reply.
 *
 * Very useful for debugging NFS.
 */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: nfstrace.c,v 1.5 1994/07/31 13:12:13 tml Exp $";

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>

#include <sys/time.h>
#define NFSCLIENT
#include <sys/mount.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <sys/param.h>

#include <netdb.h>

#include <netinet/in.h>

#include <rpc/xdr.h>
#include <nfs/nfs.h>
#include <rpc/rpc.h>
#include <rpc/pmap_prot.h>

#include "mount.h"

#include "util.h"

#ifdef M_NEWTYPE
#define M_BITS M_NEWTYPE
#define NFS_MOUNT_TYPE "nfs"
#else
#define M_BITS 0
#define NFS_MOUNT_TYPE MOUNT_NFS
#endif

#ifdef __hpux
#define MOUNT_SYSCALL vfsmount
#else
#define MOUNT_SYSCALL mount
#endif

static char *progname;
static char mountpoint[1024];
char hostname[MAXHOSTNAMELEN];

static SVCXPRT *xprt;
CLIENT *clntp;

struct timeval timeout;

extern CLIENT *clntudp_create();

static void mount_server P((char *server, char *dir));
static void usage P((void));

#if __STDC__
int
main(int argc,
     char **argv)
#else
int
main(argc, argv)
     int argc;
     char **argv;
#endif
{
  int c;
  extern char *optarg;
  extern int optind;
  
  progname = strrchr(argv[0], '/');
  if (progname)
    progname++;
  else
    progname = argv[0];

  gethostname(hostname, sizeof(hostname));
  
  timeout.tv_sec = 5;
  timeout.tv_usec = 0;

  while ((c = getopt(argc, argv, "v")) != EOF)
    switch (c)
      {
      case 'h':
	/* Help.  */
	usage();
	exit(0);
	
      case 'v':
	/* Print version.  */
	fprintf(stderr, "%s\n", RCS_Id);
	exit(0);
      }

  if (optind != argc - 2)
    usage();
  
  openlog(progname, LOG_PID|LOG_CONS, LOG_DAEMON);

  mount_server(argv[optind], argv[optind+1]);

#if 0
  signal(SIGINT, terminate);
  signal(SIGTERM, terminate);
#endif

  svc_run();
  return 1;
}

#if __STDC__
static void
usage(void)
#else
static void
usage()
#endif  
{
  fprintf(stderr, "Usage: %s [ options ] server mountpoint\n\
  Options are:\n\
    -h          print this text\n\
    -v          print program version\n\
",
	  progname);
  exit(2);
}

#if __STDC__
static void
mount_server(char *server,
	     char *dir)
#else
static void
mount_server(server, dir)
     char *server;
     char *dir;
#endif
{
  CLIENT *clp;
  struct hostent *hp;
  static struct sockaddr_in saddr, myaddr;
  static struct nfs_args nfs_args;
  struct timeval pertry, try;
  enum clnt_stat clnt_stat;
  static fhstatus fs_fh;
  static char fs_hostname[100];
  char *host, *fs, *p;
  char namebuf[256];
  int pid;
  int so;

  extern void nfs_program_2();

  if ((p = strchr(server, ':')) == NULL)
    {
      host = server;
      fs = "/";
    }
  else
    {
      *p = 0;
      host = server;
      fs = p + 1;
    }

  if ((xprt = svcudp_create(RPC_ANYSOCK)) == NULL)
    {
      fprintf(stderr, "Cannot create UDP service.\n");
      exit(1);
    }

  if (!svc_register(xprt, NFS_PROGRAM, NFS_VERSION, nfs_program_2, 0))
    {
      fprintf(stderr, "Unable to register service.\n");
      exit(1);
    }

  if ((hp = gethostbyname(host)) == NULL)
    {
      fprintf(stderr, "%s: unknown host '%s'\n", progname, host);
      exit(1);
    }

  memcpy(&saddr.sin_addr, hp->h_addr, hp->h_length);

  if ((so = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
    {
      perror("socket");
      exit(1);
    }

  if (bindresvport(so, (struct sockaddr_id *)NULL) == -1)
    {
      perror("bindresvport");
      exit(1);
    }
  
  saddr.sin_family = AF_INET;
  saddr.sin_port = 0;
  pertry.tv_sec = 10;
  pertry.tv_usec = 0;
  fs_fh.fhs_status = ETIMEDOUT;
  
  if ((clp = clntudp_create(&saddr, MOUNTPROG,
			    MOUNTVERS, pertry, &so)) == NULL)
    {
      clnt_pcreateerror("Cannot MNT RPC");
      exit(1);
    }
  
  clp->cl_auth = authunix_create_default();
  try.tv_sec = 10;
  try.tv_usec = 0;
  clnt_stat = clnt_call(clp, MOUNTPROC_MNT,
			xdr_dirpath, &fs, xdr_fhstatus, &fs_fh, try);
  if (clnt_stat != RPC_SUCCESS)
    {
      clnt_perror(clp, "Bad MNT RPC");
      exit(1);
    }

  auth_destroy(clp->cl_auth);
  clnt_destroy(clp);

  if (fs_fh.fhs_status != NFS_OK)
    {
      fprintf(stderr, "Mount RPC error %s\n", strerror(fs_fh.fhs_status));
      exit(1);
    }
  
  saddr.sin_port = 0;
  pertry.tv_sec = 1;
  pertry.tv_usec = 0;
  so = RPC_ANYSOCK;
  if ((clntp = clntudp_create(&saddr, NFS_PROGRAM, NFS_VERSION,
			      pertry, &so)) == NULL)
    {
      clnt_pcreateerror("Cannot NFS RPC");
      exit(1);
    }

  get_myaddress(&myaddr);
  myaddr.sin_family = AF_INET;
  myaddr.sin_port = xprt->xp_port;

  nfs_args.addr = &myaddr;
  nfs_args.fh = (fhandle_t *) &fs_fh.fhstatus_u.fhs_fhandle;
  nfs_args.flags = NFSMNT_HOSTNAME | NFSMNT_INT | NFSMNT_SOFT;
#ifdef NFSMNT_FSNAME
  nfs_args.flags |= NFSMNT_FSNAME;
#endif
  sprintf(fs_hostname, "%s:pid%d", progname, getpid());
  nfs_args.hostname = fs_hostname;
#ifdef NFSMNT_FSNAME
  nfs_args.fsname = nfs_args.hostname;
#endif
  strcpy(mountpoint, dir);

  /* Must fork before mounting, because the kernel wants
     our rpc service while handling the mount. */
  if ((pid = fork()) == 0)
    {
      if (MOUNT_SYSCALL(NFS_MOUNT_TYPE, mountpoint,
			M_BITS | M_NOSUID, (caddr_t)&nfs_args) == -1)
	{
	  perror("mount");
	  _exit(1);
	}
      _exit(0);
    }
}
