/*
 * afpmount -- mount an AFP server (AppleShare file server) on
 * the local host as a user-level NFS server.
 */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: afpmount.c,v 1.20 1994/09/07 21:39:23 tml Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/time.h>
#define NFSCLIENT		/* SunOS seems to need this */
#include <sys/mount.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <syslog.h>

#include <netinet/in.h>

#include <rpc/xdr.h>
#include <nfs/nfs.h>
#include <rpc/rpc.h>

#include <netat/appletalk.h>
#include <netat/afp.h>
#include <netat/afpcmd.h>
#include <netat/afpc.h>

#include "util.h"
#include "afpmount.h"

#ifdef M_NEWTYPE
#define M_BITS M_NEWTYPE
#define NFS_MOUNT_TYPE "nfs"
#else
#define M_BITS 0
#define NFS_MOUNT_TYPE MOUNT_NFS
#endif

#ifdef __hpux
#define MOUNT_SYSCALL vfsmount
#define UMOUNT_SYSCALL umount
#else
#ifdef __aux
#define MOUNT_SYSCALL fsmount
#define UMOUNT_SYSCALL unmount
#else
#define MOUNT_SYSCALL mount
#define UMOUNT_SYSCALL unmount
#endif
#endif

extern char *getenv P((const char *));

static char *progname;
static SVCXPRT *xprt;
static AddrBlock server_address;
static char mountpoint[1024];

static void usage P((void));
static void setup_xlation_tables P((void));
static void mount_server P((const char *server, const char *dir));
static void lookup P((const char *name));
static void init_client P((void));
static void close_session P((afi_user *u));
static void open_vols P((afi_user *user));
static void attn P((int srn, byte sessid, word attncode));
static void terminate P((int));
static void toggledebug P((int));

CONST char *server_name;

int debug;
static int nodaemon;

static struct timeval timeout;

static void my_svc_run P((void));

#if __STDC__
int
main(int argc,
     char **argv)
#else
int
main(argc, argv)
     int argc;
     char **argv;
#endif
{
  int c;
  extern char *optarg;
  extern int optind;
  char *revp;
  
  progname = strrchr(argv[0], '/');
  if (progname)
    progname++;
  else
    progname = argv[0];

  assert(sizeof(afi_fh) <= NFS_FHSIZE);

  mode = MODE_CAP;
  fntransl = TRANSL_LATIN1;

  revp = strchr(revision, ' ') + 1;
  *(strchr(revp, ' ')) = 0;

  guest.u_srn = SRN_NOT_OPEN;
  guest.u_name = guest.u_passwd = NULL;

  while ((c = getopt(argc, argv, "c:d:hm:tu:v")) != EOF)
    switch (c)
      {
      case 'c':
	/* Set file name translation */
	if (!strcmp(optarg, "none"))
	  fntransl = TRANSL_NONE;
	else if (!strcmp(optarg, "latin1"))
	  fntransl = TRANSL_LATIN1;
	else if (!strcmp(optarg, "cap"))
	  fntransl = TRANSL_CAP;
	else
	  {
	    fprintf(stderr, "Invalid -c option value\n");
	    usage();
	    exit(2);
	  }
	break;

      case 'd':
	/* Set CAP debugging flags */
	dbugarg(optarg);
	nodaemon = 1;
	break;

      case 'h':
	usage();
	exit(0);

      case 'm':
	/* Set Mac file format translation mode */
	if (!strcmp(optarg, "cap"))
	  mode = MODE_CAP;
	else if (!strcmp(optarg, "applesingle"))
	  mode = MODE_APPLESINGLE;
	else if (!strcmp(optarg, "appledouble"))
	  mode = MODE_APPLEDOUBLE;
	else
	  {
	    fprintf(stderr, "Invalid -m option value\n");
	    usage();
	    exit(2);
	  }
	break;

      case 't':
	debug |= DEBUG_TRACENFS;
	break;

      case 'u':
	guest.u_name = optarg;
	if ((guest.u_passwd = strchr(optarg, ':')) != NULL)
	  *guest.u_passwd = 0, guest.u_passwd++;
	break;

      case 'v':
	/* Print version. */
	fprintf(stderr, "afpmount version %s\n", revp);
	exit(0);
	break;
      }

  if (optind != argc - 2)
    usage();
  
  if (debug)
    nodaemon = 1;

  openlog(progname, LOG_PID|LOG_CONS, LOG_DAEMON);

  setup_xlation_tables();

  init_nfs_subr();

  mount_server(argv[optind], argv[optind+1]);

  signal(SIGINT, terminate);
  signal(SIGTERM, terminate);

  signal(SIGUSR1, toggledebug);

  my_svc_run();
  return 1;
}

#if __STDC__
static void
usage(void)
#else
static void
usage()
#endif  
{
  fprintf(stderr, "Usage: %s [ options ] server mountpoint\n\
Options are:\n\
    -c charset     set file name translation mode, one of: none, latin1, cap\n\
    -d flags       set CAP debugging flags\n\
    -h             print this text\n\
    -m mode        set file structure translation mode\n\
                   (cap, applesingle or appledouble)\n\
    -t             trace NFS protocol to stdout\n\
    -u user:password\n\
                   set default user name and password\n\
    -v             print program version\n\
",
	  progname);
  exit(2);
}

#if __STDC__
static void
setup_transp_xlation_tables(void)
#else
static void
setup_transp_xlation_tables()
#endif
{
  int i;

  for (i = 0; i < 0400; i++)
    local_to_afp[i] = afp_to_local[i] = i;

  local_to_afp[':'] = '/';
  local_to_afp['/'] = ':';
  afp_to_local['/'] = ':';
  afp_to_local[':'] = '/';

  local_to_afp['_'] = ' ';
  local_to_afp[' '] = '_';
  afp_to_local[' '] = '_';
  afp_to_local['_'] = ' ';
}

#if __STDC__
static void
setup_latin1_xlation_tables(void)
#else
static void
setup_latin1_xlation_tables()
#endif
{
  int i;

  /*
   * ASCII chars are mapped to shemselves, except for swapping ':' <-> '/'
   * (AFP <-> UN*X) and swapping ' ' <-> '_'.
   *
   * Latin-1 chars with no counterpart in AFP are mapped to 0xD7,
   * the "diamond" character, expect upper-case accented characters
   * not in AFP are mapped to the corresponding nonaccented character.
   *
   * AFP characters with no counterpart in Latin-1 are mapped to 
   * 0xA4, the "currency" character.
   *
   * Yes, this means that if you create a file with a name that can't
   * be translated to the AFP character set, you'll see the file
   * appearing with a different name.
   */

  for (i = 0; i < 0200; i++)
    local_to_afp[i] = afp_to_local[i] = i;

  local_to_afp[':'] = '/';
  local_to_afp['/'] = 0xD7;
  afp_to_local['/'] = ':';
  afp_to_local[':'] = 0xA4;
  local_to_afp['_'] = ' ';
  local_to_afp[' '] = '_';
  afp_to_local[' '] = '_';
  afp_to_local['_'] = ' ';

  for (i = 0200; i < 240; i++)
    local_to_afp[i] = 0xD7;

  local_to_afp[0xA0] = 0xCA;	/* nobreakspace */
  local_to_afp[0xA1] = 0xC1;	/* exclamdown */
  local_to_afp[0xA2] = 0xA2;	/* cent */
  local_to_afp[0xA3] = 0xA3;	/* sterling */
  local_to_afp[0xA4] = 0xD7;	/* currency - ?? */
  local_to_afp[0xA5] = 0xB4;	/* yen */
  local_to_afp[0xA6] = 0xD7;	/* brokenbar - ?? */
  local_to_afp[0xA7] = 0xA4;	/* section */
  local_to_afp[0xA8] = 0xAC;	/* diaeresis */
  local_to_afp[0xA9] = 0xA9;	/* copyright */
  local_to_afp[0xAA] = 0xBB;	/* ordfeminine */
  local_to_afp[0xAB] = 0xC7;	/* gillemotleft */
  local_to_afp[0xAC] = 0xC2;	/* notsign */
  local_to_afp[0xAD] = 0xD0;	/* hyphen */
  local_to_afp[0xAE] = 0xA8;	/* registered */
  local_to_afp[0xAF] = 0xD7;	/* macron - ?? */

  local_to_afp[0xB0] = 0xA1;	/* degree */
  local_to_afp[0xB1] = 0xB1;	/* plusminus */
  local_to_afp[0xB2] = 0xD7;	/* twosuperior - ?? */
  local_to_afp[0xB3] = 0xD7;	/* threesuperior - ?? */
  local_to_afp[0xB4] = 0xD5;	/* acute */
  local_to_afp[0xB5] = 0xB5;	/* mu */
  local_to_afp[0xB6] = 0xA6;	/* paragraph */
  local_to_afp[0xB7] = 0xA5;	/* periodcentered */
  local_to_afp[0xB8] = 0xD7;	/* cedilla - ?? */
  local_to_afp[0xB9] = 0xD7;	/* onesuperior - ?? */
  local_to_afp[0xBA] = 0xBC;	/* masculine */
  local_to_afp[0xBB] = 0xC8;	/* guillemotright */
  local_to_afp[0xBC] = 0xD7;	/* onequarter */
  local_to_afp[0xBD] = 0xD7;	/* onehalf */
  local_to_afp[0xBE] = 0xD7;	/* threequarters */
  local_to_afp[0xBF] = 0xC0;	/* questiondown */
  
  local_to_afp[0xC0] = 0xCB;	/* Agrave */
  local_to_afp[0xC1] = 'A';	/* Aacute - A */
  local_to_afp[0xC2] = 'A';	/* Acircumflex - A */
  local_to_afp[0xC3] = 0xCC;	/* Atilde */
  local_to_afp[0xC4] = 0x80;	/* Adiearesis */
  local_to_afp[0xC5] = 0x81;	/* Aring */
  local_to_afp[0xC6] = 0xAE;	/* AE */
  local_to_afp[0xC7] = 0x82;	/* Ccedilla */
  local_to_afp[0xC8] = 'E';	/* Egrave - E */
  local_to_afp[0xC9] = 0x83;	/* Eacute */
  local_to_afp[0xCA] = 'E';	/* Ecircumflex - E */
  local_to_afp[0xCB] = 'E';	/* Ediaeresis - E */
  local_to_afp[0xCC] = 'I';	/* Igrave - I */
  local_to_afp[0xCD] = 'I';	/* Iacute - I */
  local_to_afp[0xCE] = 'I';	/* Icircumflex - I */
  local_to_afp[0xCF] = 'I';	/* Idiaeresis - I */

  local_to_afp[0xD0] = 0xD7;	/* ETH - ?? */
  local_to_afp[0xD1] = 0x84;	/* Ntilde */
  local_to_afp[0xD2] = 'O';	/* Ograve - O */
  local_to_afp[0xD3] = 'O';	/* Oacute - O */
  local_to_afp[0xD4] = 'O';	/* Ocircumflex - O */
  local_to_afp[0xD5] = 0xCD;	/* Otilde */
  local_to_afp[0xD6] = 0x85;	/* Odiaeresis */
  local_to_afp[0xD7] = 0xD7;	/* multiply - ?? */
  local_to_afp[0xD8] = 0xAF;	/* ooblique */
  local_to_afp[0xD9] = 'U';	/* Ugrave - U */
  local_to_afp[0xDA] = 'U';	/* Uacute - U */
  local_to_afp[0xDB] = 'U';	/* Ucircumflex - U */
  local_to_afp[0xDC] = 0x86;	/* Udiaeresis */
  local_to_afp[0xDD] = 'Y';	/* Yacute - Y */
  local_to_afp[0xDE] = 0xD7;	/* THORN - ?? */
  local_to_afp[0xDF] = 0xA7;	/* ssharp */

  local_to_afp[0xE0] = 0x88;	/* agrave */
  local_to_afp[0xE1] = 0x87;	/* aacute */
  local_to_afp[0xE2] = 0x89;	/* acircumflex */
  local_to_afp[0xE3] = 0x8B;	/* atilde */
  local_to_afp[0xE4] = 0x8A;	/* adiearesis */
  local_to_afp[0xE5] = 0x8C;	/* aring */
  local_to_afp[0xE6] = 0xBE;	/* ae */
  local_to_afp[0xE7] = 0x8D;	/* ccedilla */
  local_to_afp[0xE8] = 0x8F;	/* egrave */
  local_to_afp[0xE9] = 0x8E;	/* eacute */
  local_to_afp[0xEA] = 0x90;	/* ecircumflex */
  local_to_afp[0xEB] = 0x91;	/* ediaeresis */
  local_to_afp[0xEC] = 0x93;	/* igrave */
  local_to_afp[0xED] = 0x92;	/* iacute */
  local_to_afp[0xEE] = 0x94;	/* icircumflex */
  local_to_afp[0xEF] = 0x95;	/* idiaeresis */

  local_to_afp[0xF0] = 0xD7;	/* eth - ?? */
  local_to_afp[0xF1] = 0x96;	/* ntilde */
  local_to_afp[0xF2] = 0x98;	/* ograve */
  local_to_afp[0xF3] = 0x97;	/* oacute */
  local_to_afp[0xF4] = 0x99;	/* ocircumflex */
  local_to_afp[0xF5] = 0x9B;	/* otilde */
  local_to_afp[0xF6] = 0x9A;	/* odiaeresis */
  local_to_afp[0xF7] = 0xD6;	/* division */
  local_to_afp[0xF8] = 0xBF;	/* oslash */
  local_to_afp[0xF9] = 0x9D;	/* ugrave */
  local_to_afp[0xFA] = 0x9C;	/* uacute */
  local_to_afp[0xFB] = 0x9E;	/* ucircumflex */
  local_to_afp[0xFC] = 0x9F;	/* udiaeresis */
  local_to_afp[0xFD] = 0xD7;	/* yacute - ?? */
  local_to_afp[0xFE] = 0xD7;	/* thorn - ?? */
  local_to_afp[0xFF] = 0xD8;	/* ydiaeresis */

  for (i = 0200; i < 0400; i++)
    afp_to_local[i] = 0;

  for (i = 0200; i < 0400; i++)
    if (local_to_afp[i] >= 0200)
      afp_to_local[local_to_afp[i]] = i;

  afp_to_local[0xD7] = 0xA4;

  for (i = 0200; i < 0400; i++)
    if (afp_to_local[i] == 0)
      afp_to_local[i] = 0xA4;
}

#if __STDC__
static void
setup_xlation_tables(void)
#else
static void
setup_xlation_tables()
#endif
{
  int i;

  if (fntransl == TRANSL_NONE)
    setup_transp_xlation_tables();
  else if (fntransl == TRANSL_LATIN1)
    setup_latin1_xlation_tables();

  for (i = 0; i < 0400; i++)
    afp_downcase[i] = i;
  for (i = 'A'; i <= 'Z'; i++)
    afp_downcase[i] = i + 32;
  afp_downcase[0xCB] = 0x88;
  afp_downcase[0x80] = 0x8A;
  afp_downcase[0xCC] = 0x8B;
  afp_downcase[0x81] = 0x8C;
  afp_downcase[0x82] = 0x8D;
  afp_downcase[0x83] = 0x8E;
  afp_downcase[0x84] = 0x96;
  afp_downcase[0x85] = 0x9A;
  afp_downcase[0xCD] = 0x9B;
  afp_downcase[0x86] = 0x9F;
  afp_downcase[0xAE] = 0xBE;
  afp_downcase[0xAF] = 0xBF;
  afp_downcase[0xCE] = 0xCF;

}

#if __STDC__
static void
mount_server(const char *server,
	     const char *dir)
#else
static void
mount_server(server, dir)
     char *server;
     char *dir;
#endif
{
  static struct nfs_args nfs_args;
  struct sockaddr_in sin;
  static char fs_hostname[100];
  static char namebuf[256];
  int pid;

  extern void nfs_program_2();

  abInit(TRUE);
  nbpInit();

  if (strchr(server, ':') != NULL)
    strcpy(namebuf, server);
  else
    sprintf(namebuf, "%s:AFPServer@*", server);
  server_name = namebuf;

  lookup(server_name);

  if (FPGetSrvrInfo(&server_address, &server_info) != noErr)
    {
      fprintf(stderr, "AFP Server %s not responding\n", server_name);
      exit(1);
    }
  
  init_client();

  xprt = svcudp_create(RPC_ANYSOCK);

  if (xprt == NULL)
    {
      fprintf(stderr, "Cannot create UDP service.\n");
      exit(1);
    }

  if (!svc_register(xprt, NFS_PROGRAM, NFS_VERSION, nfs_program_2, 0))
    {
      fprintf(stderr, "Unable to register RPC service.\n");
      exit(1);
    }

  /* Unless printing debugging output (CAP or our own), run as a daemon. */
  if (!nodaemon)
    {
      close(0);
      close(1);
      close(2);

      if (fork() > 0)
	exit(0);
      setsid();
    }

  get_myaddress(&sin);
  sin.sin_family = AF_INET;
  sin.sin_port = xprt->xp_port;

  nfs_args.addr = &sin;
  nfs_args.fh = (fhandle_t *) toplevel_fh;
  nfs_args.flags = 0;

  nfs_args.wsize = 1024;	/* Write in shorter chunks, necessary? */
  nfs_args.rsize = 4096;
  nfs_args.flags |= NFSMNT_WSIZE | NFSMNT_RSIZE;
  
#ifdef NFSMNT_HOSTNAME
  nfs_args.flags |= NFSMNT_HOSTNAME;
  /* We have already daemonified ourselves, so getpid() is now correct. */
  sprintf(fs_hostname, "%s:pid=%d", progname, getpid());
  nfs_args.hostname = fs_hostname;
#endif

  nfs_args.flags |= NFSMNT_INT | NFSMNT_SOFT;

#ifdef NFSMNT_FSNAME
  nfs_args.flags |= NFSMNT_FSNAME;
  nfs_args.fsname = nfs_args.hostname;
#endif

  strcpy(mountpoint, dir);

  /*
   * Must fork before mounting, because the kernel wants
   * our rpc service while handling the mount.
   */
  if ((pid = fork()) == 0)
    {
      if (MOUNT_SYSCALL(NFS_MOUNT_TYPE, mountpoint,
			M_BITS | M_NOSUID, (caddr_t)&nfs_args) == -1)
	{
	  perror("mount");
	  _exit(1);
	}
      _exit(0);
    }
}

#if __STDC__
static void
lookup(const char *name)
#else
static void
lookup(name)
     char *name;
#endif
{
  EntityName en;			/* network entity name */
  NBPTEntry nbpt[1];			/* table of entity names */
  nbpProto nbpr;			/* nbp protocol record */
  int err;

  create_entity(name, &en);

  nbpr.nbpRetransmitInfo.retransInterval = sectotick(1);
  nbpr.nbpRetransmitInfo.retransCount = 10; /* one retransmit */
  nbpr.nbpBufPtr = nbpt;		/* place to store entries */
  nbpr.nbpBufSize = sizeof(nbpt);	/* size of above */
  nbpr.nbpDataField = 1;		/* max entries */
  nbpr.nbpEntityPtr = &en;		/* look this entity up */

  /* Find all objects in specified zone */
  err = NBPLookup(&nbpr,FALSE);		/* try synchronous */
  if (err != noErr || nbpr.abResult != noErr) {
    fprintf(stderr,"NBPLookup returned err %d\n",err);
    exit(1);
  }
  if (nbpr.nbpDataField == 0) {
    printf("Couldn't find %s:%s@%s ...\n",
	   en.objStr.s,en.typeStr.s,en.zoneStr.s);
    exit(1);
  }
  NBPExtract(nbpt, nbpr.nbpDataField, 1, &en, &server_address);
}

static uam_present uams[] =
{
  { "No User Authent", 0 },
  { "Cleartxt passwrd", 0 },
  { 0, 0 },
};

#if __STDC__
static void
init_client(void)
#else
static void
init_client()
#endif
{
  char tbuf[32];
  int i, j, n;
  int some_uam_good = FALSE;

  /* What AFP version to use? */
  n = IndStrCnt(server_info.sr_avo);
  for (i = 0; i < n; i++)
    {
      GetIndStr(tbuf, server_info.sr_avo, i);
      if (strcmp("AFPVersion 1.1", tbuf) == 0 && afpversion == 0)
	afpversion = 11;
      else if(strcmp("AFPVersion 2.0", tbuf) == 0)
	afpversion = 20;
    }

  /* What authentication method? */
  n = IndStrCnt(server_info.sr_uamo);
  for (i=0; i < n; i++)
    {
      GetIndStr(tbuf, server_info.sr_uamo, i);
      for (j = 0; j < ARRAYSIZE(uams) - 1; j++)
	if (strcmp(uams[j].uam_name, tbuf) == 0)
	  {
	    uams[j].supported = TRUE;
	    some_uam_good = TRUE;
	  }
    }
  if (afpversion == 0)
    {
      fprintf(stderr, "No supported AFP Versions in server\n");
      exit(1);
    }
  if (!some_uam_good)
    {
      fprintf(stderr, "No supported User Authentication Methods in server\n");
      exit(1);
    }

  server_parms = NULL;

  /* If no guest user given, use guest login if supported.  */
  if (!guest.u_name && uams[UAM_ANON].supported)
    {
      guest.u_name = "guest";
      guest.u_passwd = NULL;
      if (!login_user(&guest, UAM_ANON))
	uams[UAM_ANON].supported = FALSE;
    }
  else
    {
      if (!guest.u_name)
	{
	  guest.u_name = getenv("AFPUSER");
	  guest.u_passwd = getenv("AFPPASSWD");
	}
      if (guest.u_name && guest.u_name[0]
	  && guest.u_passwd && guest.u_passwd[0])
	login_user(&guest, UAM_CLEAR);
    }
  
}

#if __STDC__
int
login_user(afi_user *user,
	   int uam)
#else
int
login_user(user, uam)
     afi_user *user;
     int uam;
#endif
{
  if (open_session(&user->u_srn) < 0)
    return 0;
      
  if (!login(user->u_srn, uam, user->u_name, user->u_passwd))
    {
      close_session(user);
      return 0;
    }
  open_vols(user);
  
  user->u_dtrn = DTRN_NOT_OPEN;
  
  return 1;
}

#if __STDC__
void
try_getsrvprm(afi_user *user)
#else
void
try_getsrvprm(user)
     afi_user *user;
#endif
{
  int i, comp;
  dword cr;

  comp = eFPGetSrvrParms(user->u_srn, &server_parms, &cr);
  check_errors("FPGetSrvrParms", comp, cr);

  if (comp < 0 || cr != noErr)
    {
      logout(user);
      return;
    }
  volume_parms =
    (GVPRPPtr) xmalloc(server_parms->gspr_nvols * sizeof(volume_parms[0]));

  for (i = 0; i < server_parms->gspr_nvols; i++)
    {
      comp = eFPOpenVol(user->u_srn, VP_ALL,
			server_parms->gspr_volp[i].volp_name,
			(char *)NULL, volume_parms + i, &cr);
      syslog(LOG_NOTICE, "FPOpenVol(%d, %s)", user->u_srn,
	     server_parms->gspr_volp[i].volp_name);
      check_errors("FPOpenVol", comp, cr);
      if (comp != noErr || cr != noErr)
	server_parms->gspr_volp[i].volp_name[0] = 0;
    }

  for (i = 0; i < server_parms->gspr_nvols; i++)
    if (server_parms->gspr_volp[i].volp_name[0])
      break;
  if (i == server_parms->gspr_nvols)
    {
      syslog(LOG_CRIT, "No volumes opened");
      terminate(0);
    }
}

#if __STDC__
static void
open_vols(afi_user *user)
#else
static void
open_vols(user)
     afi_user *user;
#endif
{
  int i, comp;
  dword cr;

  if (!server_parms)
    try_getsrvprm(user);
  else
    {
      for (i = 0; i < server_parms->gspr_nvols; i++)
	{
	  GetVolParmsReplyPkt vp;

	  if (!server_parms->gspr_volp[i].volp_name[0])
	    continue;

	  comp = eFPOpenVol(user->u_srn, VP_ALL,
			    server_parms->gspr_volp[i].volp_name,
			    (char *)NULL, &vp, &cr);
	  syslog(LOG_NOTICE, "FPOpenVol(%d, %s)", user->u_srn,
		 server_parms->gspr_volp[i].volp_name);
	  check_errors("FPOpenVol", comp, cr);
	}
    }
}

#if __STDC__
int
open_session(int *srn)
#else
int
open_session(srn)
     int *srn;
#endif
{
  int comp;

  SPOpenSession(&server_address, attn, srn, 2, -1, &comp);
  while (comp > 0)
    abSleep(4*9, TRUE);
  if (comp < 0) {
    syslog(LOG_ERR, "SPOpenSession failed: %d", comp);
    *srn = SRN_LOGIN_FAILED;
    return comp;
  }
  syslog(LOG_NOTICE, "opened ASP session %d", *srn);
  return 0;
}  

#if __STDC__
static void
close_session(afi_user *u)
#else
static void
close_session(u)
     afi_user *u;
#endif
{
  int comp;
  
  SPCloseSession(u->u_srn, 10, -1, &comp);
  while (comp > 0)
    abSleep(4*9, TRUE);

  if (comp < 0)
    syslog(LOG_ERR, "error closing ASP session %d: error %d", u->u_srn, comp);
  else
    syslog(LOG_NOTICE, "closed ASP session %d", u->u_srn);
  u->u_srn = SRN_NOT_OPEN;
}

#if __STDC__
void
logout(afi_user *u)
#else
void
logout(u)
     afi_user *u;
#endif
{
  int comp;
  dword cr;

  FPLogout(u->u_srn, &cr);
  syslog(LOG_NOTICE, "Logged out AFP session %d", u->u_srn);
  close_session(u);
  u->u_srn = SRN_NOT_OPEN;
}  

#if __STDC__
static void
attn(int srn,
     byte sessid,
     word attncode)
#else
static void
attn(srn, sessid, attncode)
     int srn;
     byte sessid;
     word attncode;
#endif
{
  syslog(LOG_ALERT, "Server %s, session %d, code %#x",
	 server_name, sessid, attncode);
  if ((attncode & 0xf000) == 0x8000) {
    if ((attncode & 0xfff) != 0xfff)
      syslog(LOG_ALERT,"Server %s going down in %d minutes",
	     server_name, attncode&0xfff);
    else
      syslog(LOG_ALERT, "Server %s shutdown canceled", server_name);
  }
}

#if __STDC__
int
login(int srn, int uam, const char *user, const char *passwd)
#else
int
login(srn, uam, user, passwd)
     int srn;
     int uam;
     char *user;
     char *passwd;
#endif
{
  LoginPkt lp;
  LoginReplyPkt lrp;
  int comp;
  dword cr;
  lp.log_cmd = AFPLogin;
  if (afpversion == 11)
    strcpy((char *)lp.log_ver, "AFPVersion 1.1");
  else
    strcpy((char *)lp.log_ver, "AFPVersion 2.0");
  strcpy((char *)lp.log_uam, uams[uam].uam_name);
  if (uam == UAM_ANON)
    lp.log_flag = 0;
  else
    {
      lp.log_flag = UAMP_USER|UAMP_PASS|UAMP_ZERO;
      if (user)
	strncpy((char *)lp.log_user, user, sizeof(lp.log_user));
      else
	memset(lp.log_user, 0, sizeof(lp.log_user));
      if (passwd)
	strncpy((char *)lp.log_passwd, passwd, sizeof(lp.log_passwd));
      else
	memset(lp.log_passwd, 0, sizeof(lp.log_passwd));
    }

  comp = FPLogin(srn, &lp, &lrp, &cr);

  syslog(LOG_NOTICE, "FPLogin(%d,%s,%s,%s)",
	 srn, lp.log_ver, lp.log_uam, user);
  check_errors("FPLogin", comp, cr);

  return comp == noErr && cr == noErr;
}

#if __STDC__
static void
terminate(int signo)
#else
static void
terminate(signo)
     int signo;
#endif
{
  signal(SIGINT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);

  if (fork() == 0)
    {
      if (UMOUNT_SYSCALL(mountpoint) == -1)
	syslog(LOG_ERR, "umount %s: %m", mountpoint);
      _exit(0);
    }
  exit(0);
}

#if __STDC__
static void
toggledebug(int signo)
#else
static void
toggledebug(signo)
     int signo;
#endif
{
  signal(SIGUSR1, toggledebug);

  /* If we haven't closed stderr, toggle tracing. */
  if (nodaemon)
    if (debug & DEBUG_TRACENFS)
      debug &= ~DEBUG_TRACENFS;
    else
      debug |= DEBUG_TRACENFS;
}

/*
 * Lifted from RPC 3.9 rpc/svc_run.c and modified.
 * Here is the copyright notice for that file:
 *
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user.
 * 
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 * 
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 * 
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 * 
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 * 
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */

static void
my_svc_run()
{
#ifdef FD_SETSIZE
  fd_set readfds;
#else
  int readfds;
#endif /* def FD_SETSIZE */
  extern int errno;
  time_t now;

  /*
   * It's a pity we don't know the sockets that CAP uses.  We must
   * select here with a short timeout, waiting for RPC calls from the
   * kernel, and if we got nothing call abSleep, responding to tickles
   * and maybe other stuff from CAP.
   *
   * If CAP would tell us its sockets (fdbits in abnet.c wasn't private),
   * we could have just this select, and call abSleep when there is something
   * for it to do.
   */
     
  timeout.tv_sec = 1;
  timeout.tv_usec = 0;

  for (;;) {
#ifdef FD_SETSIZE
    readfds = svc_fdset;
#else
    readfds = svc_fds;
#endif /* def FD_SETSIZE */
    switch (select(FD_SETSIZE, (int *)&readfds, (int *)0, (int *)0,
		   &timeout)) {
    case -1:
      if (errno == EINTR)
	continue;
      syslog(LOG_ERR, "svc_run: select failed: %m");
      terminate(0);

    case 0:
      abSleep(1, TRUE);
      time(&now);
      nfs_timeout(now);
      break;

    default:
      time(&now);
      svc_getreqset(&readfds);
      nfs_timeout(now);
      break;
    }
  }
}
