/* NFS RPC service procedures for afpmount */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: afpm_svc_subr.c,v 1.17 1994/09/07 21:39:15 tml Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <pwd.h>
#include <malloc.h>
#include <assert.h>
#include <time.h>

#include <syslog.h>

#include <rpc/rpc.h>
#ifndef AUTH_UNIX
#include <rpc/auth.h>
#endif
#include <sys/stat.h>

#include <netat/appletalk.h>
#include <netat/afp.h>
#include <netat/afpcmd.h>
#include <netat/afpc.h>

#include "util.h"
#include "nfs_prot.h"
#include "afpmount.h"
#include "log_nfs.h"

GetSrvrInfoReplyPkt server_info;
GSPRPPtr server_parms;
GVPRPPtr volume_parms;

int afpversion = 0;

afi_user guest;

static int toplevel_fh_allocation[NFS_FHSIZE/sizeof(int)];

afi_fh *toplevel_fh;

static afi_user *users;

static char empty_path[1];

/* The do { } while (0) is so that the macros expand to a single statement.  */

#define ENTRY(name,argtype)					\
  do {								\
    if (debug & DEBUG_TRACENFS)					\
      printf("%s\n",STRINGIFY(name)),				\
      print_auth(rqstp),					\
      CONCAT2(print_,argtype)(0,argp);				\
  } while (0)

#define LEAVE(restype)						\
  do { if (debug & DEBUG_TRACENFS)				\
	 CONCAT2(print_,restype)(1,&res);			\
	   return &res; } while (0)

#define LOG(args) (debug & DEBUG_TRACENFS) ? printf args : 0

#define print_nfs_fh print_afi_fh

#define AFP_RD 02		/* AFP bits for read */
#define AFP_WR 04		/*  write */
#define AFP_SR 01		/*  search */

#define SHIFT_WORLD 16		/* AFP world shift amount */
#define SHIFT_GROUP 8		/*  group */
#define SHIFT_OWNER 0		/*  owner */

/*
 * The fileids for the shadow .finderinfo dir, .resource dir and resource
 * fork files are the dir or file id plus an offset, to get unique fileids.
 * Additionally, all fileids are made unique by adding in the volume number.
 */
#define TYPE_FACTOR 1000000
#define VOL_FACTOR 100000000
#define UNIQ_TYPE(value, tag) (value += (tag) * TYPE_FACTOR)
#define UNIQ_VOL(value, vol) (value += ((vol)+1) * VOL_FACTOR)
#define RES_TAG		1
#define FI_TAG		2
#define FIDIR_TAG	3

#define OUR_FSID 1
#define CONTROL_FILEID	1234567890
#define TOP_FILEID 2
#define VOLUME_ROOT_FILEID 2

#define MAXTRIES 3

static afi_user *who;

static CONST char *hexdigits = "0123456789abcdef";

byte local_to_afp[256];
byte afp_to_local[256];
byte afp_downcase[256];

int mode;
int fntransl;

typedef struct afi_user_list_entry *afi_user_list;

struct afi_user_list_entry {
  afi_user *user;
  afi_user_list link;
};

#define READ 1
#define WRITE 2

typedef struct
{
  time_t timestamp;
  afi_fh fh;
  char path[MAXLFLEN];
  int mode;
  afi_user *user;
  afi_user_list piggyback;
  int refnum;
} cache_entry;

#define FORK_NOT_OPEN -1

#define CACHESIZE 3

static cache_entry cache[CACHESIZE];
static int cacheix[CACHESIZE];

static int *afp_to_nfs_uid_map, highest_mapped_afp_uid, size_of_afp_uid_map;
static int *nfs_to_afp_uid_map, highest_mapped_nfs_uid;

static void print_afi_fh P((int level, nfs_fh *value));
static int maybe_open_fork P((const afi_fh *fh, int mode, int *ixp));

#if __STDC__
void
init_nfs_subr(void)
#else
void
init_nfs_subr()
#endif
{
  int i;
  struct passwd *pwd;

  memset(toplevel_fh_allocation, 0, NFS_FHSIZE);
  toplevel_fh = (afi_fh *) toplevel_fh_allocation;
  toplevel_fh->fh_type = AFIFH_TOP;

  highest_mapped_nfs_uid = -1;

  setpwent();

  while (pwd = getpwent())
    if (pwd->pw_uid > highest_mapped_nfs_uid)
      highest_mapped_nfs_uid = pwd->pw_uid;

  nfs_to_afp_uid_map = xzalloc(sizeof(int) * (highest_mapped_nfs_uid + 1));
  
  afp_to_nfs_uid_map = xzalloc(sizeof(int));
  highest_mapped_afp_uid = 0;
  size_of_afp_uid_map = 1;
  afp_to_nfs_uid_map[0] = -1;

  setvbuf(stdout, NULL, _IOLBF, BUFSIZ);

  fh_printer = print_afi_fh;
  empty_path[0] = 0;
  users = NULL;
  for (i = 0; i < CACHESIZE; i++)
    {
      cache[i].refnum = FORK_NOT_OPEN;
      cache[i].user = NULL;
      cacheix[i] = i;
    }
}

#if __STDC__
static const char *
fh_type(const afi_fh *fh)
#else
static char *
fh_type(fh)
     afi_fh *fh;
#endif
{
  switch (fh->fh_type)
    {
    case AFIFH_TOP:
      return "top";
    case AFIFH_DIR:
      return "dir";
    case AFIFH_RESDIR:
      return "resdir";
    case AFIFH_FIDIR:
      return "fidir";
    case AFIFH_DFORK:
      return "dfork";
    case AFIFH_RFORK:
      return "rfork";
    case AFIFH_FINFO:
      return "finfo";
    case AFIFH_SINGLE:
      return "single";
    case AFIFH_DOUBLERES:
      return "doubleres";
    case AFIFH_CONTROL:
      return "control";
    default:
      return "???";
    }
}

#if __STDC__
static void
print_afi_fh(int level, nfs_fh *value)
#else
static void
print_afi_fh(level, value)
     int level;
     nfs_fh *value;
#endif
{
  CONST afi_fh *fh = (afi_fh *) value;
  printf("%s%s ", indent(level), fh_type(fh));
  switch (fh->fh_type)
    {
    case AFIFH_TOP:
    case AFIFH_CONTROL:
      break;

    default:
      printf(" volid: %d(%d), dirid: %d",
	     fh->fh_volix, volume_parms[fh->fh_volix].gvpr_volid,
	     fh->fh_dirid);
    }
  switch (fh->fh_type)
    {
    case AFIFH_DFORK:
    case AFIFH_RFORK:
    case AFIFH_FINFO:
    case AFIFH_SINGLE:
    case AFIFH_DOUBLERES:
      printf(", fileid: %d", fh->fh_fileid);
    }
  printf("\n");
}

#if __STDC__
static char *
log_fh(const afi_fh *fh)
#else
static char *
log_fh(fh)
     afi_fh *fh;
#endif
{
  static char buf[100];

  strcpy(buf, fh_type(fh));
  switch (fh->fh_type)
    {
    case AFIFH_TOP:
      break;
    default:
      sprintf(buf + strlen(buf), " volix: %d, dirid: %d",
	      fh->fh_volix, fh->fh_dirid);
    }
  switch (fh->fh_type)
    {
    case AFIFH_DFORK:
    case AFIFH_RFORK:
    case AFIFH_FINFO:
    case AFIFH_SINGLE:
    case AFIFH_DOUBLERES:
      sprintf(buf + strlen(buf), ", fileid: %d", fh->fh_fileid);
    }

  return buf;
}

#if __STDC__
static char *
log_path(const byte *path)
#else
static char *
log_path(path)
     byte *path;
#endif
{
  static char result[MAXLFLEN];
  char *p = result;
  int len = path[0];

  path++;
  while (len--)
    {
      if (!isprint(*path))
	{
	  sprintf(p, "\\%03o", *path);
	  p += strlen(p);
	}
      else if (*path == '\\')
	*p++ = '\\', *p++ = '\\';
      else
	*p++ = *path;
      path++;
    }
  *p = 0;

  return result;
}

/* local_afp_strcmp -- compare local string to AFP string */

#if __STDC__
static int
local_afp_strcmp(const char *a,
		 const char *b)
#else
static int
local_afp_strcmp(a, b)
     char *a;
     char *b;
#endif
{
  if (fntransl == TRANSL_CAP)
    {
      char n[MAXUFLEN];
      char *np = n;
      
      while (*b)
	{
	  if (isprint(*b) && *b != '/')
	    *np++ = *b;
	  else
	    *np++ = ':',
	    *np++ = hexdigits[(*b >> 4) & 0xF],
	    *np++ = hexdigits[*b & 0xF];
	  b++;
	}
      *np = 0;

      return strcasecmp(a, n);
    }

  while (*a && *b && afp_downcase[local_to_afp[*a]] == afp_downcase[*b])
      a++, b++;

  if (!*a && !*b)
    return 0;
  if (!*a)
    return -1;
  if (!*b)
    return 1;
  
  return (afp_downcase[local_to_afp[*a]] > afp_downcase[*b]) ? 1 : -1;
}

/*
 * Duplicate a string, translating Mac characters.
 */

#if __STDC__
static char *
afp_to_local_dup(const byte *src, size_t length)
#else
static char *
afp_to_local_dup(src, length)
     byte *src;
     size_t length;
#endif
{
  char tem[MAXUFLEN];
  char *p = tem;
  char *result;
  
  if (fntransl == TRANSL_CAP)
    while (length--)
      if (isprint(*src) && *src != '/')
	*p++ = *src++;
      else
	*p++ = ':',
	*p++ = hexdigits[(*src >> 4) & 0xF],
	*p++ = hexdigits[*src & 0xF],
	src++;
  else 
    while (length--)
      *p++ = afp_to_local[*src++];

  *p++ = 0;
  
  result = xmalloc(p - tem);
  strcpy((char *)result, tem);

  return result;
}

/*
 * Copy a C filename string to a counted (Pascal) string in the
 * Mac character set.
 */

#if __STDC__
static void
local_to_p_afp_cpy(char *dest,
		   const byte *src)
#else
static void
local_to_p_afp_cpy(dest, src)
     char *dest;
     byte *src;
#endif
{
  char *p;

  p = dest + 1;
  
  if (fntransl == TRANSL_CAP)
    {
      /*
       * A colon followed by two hex digits is translated
       * to the corresponding char value.
       * Any other colons are translated to a slash.
       */
      while (*src)
	{
	  if (*src != ':')
	    *p++ = *src;
	  else if (!src[1] || !src[2]
		   || !isxdigit(src[1])
		   || !isxdigit(src[2]))
	    *p++ = '/';
	  else
	    {
	      /* Normal case: "%:02x" format for non-ASCII chars.  */
	      *p++ = (HEXVAL(src[1]) << 4)
		| HEXVAL(src[2]);
	      src += 2;
	    }
	  src++;
	}
    }
  else
    while(*src)
      {
	*p++ = local_to_afp[*src];
	src++;
      }

  *dest = p - dest - 1;
}

#if __STDC__
static int
compare_uids(const void *a,
	     const void *b)
#else
static int
compare_uids(a, b)
     void *a;
     void *b;
#endif
{
  return ((afi_user *)a)->u_uxuid - ((afi_user *)b)->u_uxuid;
}

#if __STDC__
static afi_user *
session_of(int uid)
#else
static afi_user *
session_of(uid)
     int uid;
#endif
{
  afi_user *u;

  for (u = users; u && u->u_uxuid != uid; u = u->u_link)
    ;
  return u;
}  

#if __STDC__
static afi_user *
some_session(void)
#else
static afi_user *
some_session()
#endif
{
  afi_user *u;

  if (guest.u_srn >= 0)
    return  &guest;

  for (u = users; u && u->u_srn < 0; u = u->u_link)
    ;

  return u;
}  

#if __STDC__
static int
guest_user(void)
#else
static int
guest_user()
#endif
{
  if (guest.u_srn >= 0)
    {
      who = &guest;
      return NFS_OK;
    }
  return NFSERR_ACCES;
}

#if __STDC__
static int
auth_uid(const struct svc_req *rqstp)
#else
static int
auth_uid(rqstp)
     struct svc_req *rqstp;
#endif
{
  switch (rqstp->rq_cred.oa_flavor)
    {
    case AUTH_UNIX:
      return ((struct authunix_parms *) rqstp->rq_clntcred)->aup_uid;

    case AUTH_NULL:
    default:
      return 0;
    }
}  

#if __STDC__
static int
find_user(const struct svc_req *rqstp)
#else
static int
find_user(rqstp)
     struct svc_req *rqstp;
#endif
{
  int uid;
  afi_user *u;

  who = NULL;

  if ((uid = auth_uid(rqstp)) == 0)
    return guest_user();

  if (!(u = session_of(uid)))
    return guest_user();

  if (u->u_srn == SRN_LOGIN_FAILED)
    /* Her login has failed recently.  */
    return NFSERR_ACCES;
      
  who = u;
      
  if (u->u_srn >= 0)
    {
      time(&who->u_timestamp);
      return NFS_OK;
    }

  if (login_user(u, UAM_CLEAR))
    {
      who = u;
      time(&who->u_timestamp);
      return NFS_OK;
    }
  else
    return NFSERR_ACCES;
}

/*
 * Translate the owner and group ID and access rights
 * of a parent directory to the NFS mode for an entry.
 */

#if __STDC__
static u_int
afp_to_nfs_mode(sdword owner, sdword group, const FileDirParm *fdpp)
#else
static u_int
afp_to_nfs_mode(owner, group, fdpp)
     sdword owner, group;
     FileDirParm *fdpp;
#endif
{
  u_int mode;
  dword p = fdpp->fdp_parms.dp_parms.dp_accright;

  mode = 0;

  mode |= ((p >> SHIFT_OWNER) & AFP_RD) ? S_IRUSR : 0;
  mode |= ((p >> SHIFT_OWNER) & AFP_WR) ? S_IWUSR : 0;
  if (FDP_ISDIR(fdpp->fdp_flg))
    mode |= ((p >> SHIFT_OWNER) & AFP_SR) ? (S_IXUSR|S_IRUSR) : 0;

  if (group == 0)
    if (FDP_ISDIR(fdpp->fdp_flg))
      mode |= S_IRWXG;
    else
      mode |= S_IRGRP|S_IWGRP;
  else
    {
      mode |= ((p >> SHIFT_GROUP) & AFP_RD) ? S_IRGRP : 0;
      mode |= ((p >> SHIFT_GROUP) & AFP_WR) ? S_IWGRP : 0;
      if (FDP_ISDIR(fdpp->fdp_flg))
	mode |= ((p >> SHIFT_GROUP) & AFP_SR) ? (S_IXGRP|S_IRGRP) : 0;
    }
  
  if (owner == 0)
    mode |= ((mode & S_IRWXU) >> 6);
  else
    {
      mode |= ((p >> SHIFT_WORLD) & AFP_RD) ? S_IROTH : 0;
      mode |= ((p >> SHIFT_WORLD) & AFP_WR) ? S_IWOTH : 0;
      if (FDP_ISDIR(fdpp->fdp_flg))
	mode |= ((p >> SHIFT_WORLD) & AFP_SR) ? (S_IXOTH|S_IROTH) : 0;
    }
  return mode;
}

#if __STDC__
static dword
nfs_to_afp_mode(u_int mode)
#else
static dword
nfs_to_afp_mode(mode)
     u_int mode;
#endif
{
  dword accright;

  accright = 0;
  
  if (mode & (S_IXUSR|S_IRUSR))
    accright |= ((AFP_SR|AFP_RD) << SHIFT_OWNER);

  if (mode & S_IWUSR)
    accright |= (AFP_WR << SHIFT_OWNER);

  if (mode & (S_IXGRP|S_IRGRP))
    accright |= ((AFP_SR|AFP_RD) << SHIFT_GROUP);

  if (mode & S_IWGRP)
    accright |= (AFP_WR << SHIFT_GROUP);

  if (mode & (S_IXOTH|S_IROTH))
    accright |= ((AFP_SR|AFP_RD) << SHIFT_WORLD);

  if (mode & S_IWOTH)
    accright |= (AFP_WR << SHIFT_WORLD);

  return accright;
}

#if __STDC__
static int
enter_afp_to_nfs_uid(int afpuid, int uxuid)
#else
static int
enter_afp_to_nfs_uid(afpuid, uxuid)
     int afpuid;
     int uxuid;
#endif
{
  int i;

  if (afpuid >= size_of_afp_uid_map)
    {
      CONST int MORE = 100;
      afp_to_nfs_uid_map = xrealloc(afp_to_nfs_uid_map,
				    sizeof(int) * (afpuid + MORE));

      for (i = highest_mapped_afp_uid + 1;
	   i < afpuid + MORE;
	   i++)
	afp_to_nfs_uid_map[i] = -1;
      size_of_afp_uid_map = afpuid + MORE;
    }
  highest_mapped_afp_uid = afpuid;
  return afp_to_nfs_uid_map[afpuid] = uxuid;
}

#if __STDC__
static int
afp_to_nfs_uid(int uid)
#else
static int
afp_to_nfs_uid(uid)
     int uid;
#endif
{
  int nfsuid;
  afi_user *u;
  struct passwd *pwd;
  
  if (uid < 0 || uid > 100000)
    {
      /*
       * Preposterous AFP uid.  Can't handle as long as we use simple
       * table indexing for mapping.  We really should use some kind
       * of hash table, of course.
       */
      return uid - 1;
    }
      
  if (uid <= highest_mapped_afp_uid
      && (nfsuid = afp_to_nfs_uid_map[uid]) >= 0)
    return nfsuid;

  u = some_session();

  if (u)
    {
      int comp;
      dword cr;
      MapIDReplyPkt map;

      comp = eFPMapID(u->u_srn, MapID_C, uid, &map, &cr);

      check_errors("FPMapID", comp, cr);

      if (comp == noErr && cr == noErr)
	{
	  char name[MAXPSTR+1];

	  memcpy(name, map.mpir_name+1, map.mpir_name[0]);
	  name[map.mpir_name[0]] = 0;

	  LOG(("mapped afp uid %d to afp name %s\n", uid, name));

	  if ((pwd = getpwnam(name)) != NULL)
	    return (enter_afp_to_nfs_uid(uid, pwd->pw_uid));
	  else
	    return (enter_afp_to_nfs_uid(uid, 0));
	}
    }
  return uid - 1;
}

#if __STDC__
static int
afp_to_nfs_gid(int gid)
#else
static int
afp_to_nfs_gid(gid)
     int gid;
#endif
{
  return 0;
}

#if __STDC__
static int
nfs_to_afp_uid(int uid)
#else
static int
nfs_to_afp_uid(uid)
     int uid;
#endif
{
  int afpuid;
  afi_user *u;
  byte afpname[MAXPSTR+1];
  struct passwd *pwd;

  /* Do we already have a mapping? */
  if (uid > 0 && uid <= highest_mapped_nfs_uid
      && (afpuid = nfs_to_afp_uid_map[uid]) != 0)
    return afpuid;

  /* Search for (potential) session for uid.  */
  u = session_of(uid);

  if (u)
    {
      afpname[0] = strlen(u->u_name);
      memcpy(afpname+1, u->u_name, afpname[0]);
    }
  else
    {
      /* No session, try UNIX name */
      pwd = getpwuid(uid);
      if (pwd == NULL)
	return 1;
      else
	{
	  afpname[0] = strlen(pwd->pw_name);
	  memcpy(afpname+1, pwd->pw_name, afpname[0]);
	}
    }
  
  /* Ask for the mapping in any session.  */

  if (u = some_session())
    {
      int tries, comp;
      dword cr;

      tries = 0;
      while (tries++ < MAXTRIES
	     && (comp = eFPMapName(u->u_srn, MapName_C,
				   afpname, &afpuid, &cr)) == SessClosed)
	login_user(u, UAM_CLEAR);

      check_errors("FPMapName", comp, cr);

      if (comp == noErr && cr == noErr)
	{
	  LOG(("mapped nfs uid %d to afp uid %d (%.*s)\n", uid, afpuid,
	       afpname[0], afpname+1));
	  return (nfs_to_afp_uid_map[uid] = afpuid);
	}
    }
  return 1;
}

#if __STDC__
static int
nfs_to_afp_gid(int gid)
#else
static int
nfs_to_afp_gid(gid)
     int gid;
#endif
{
  return 1;
}

/*
 * Set NFS fattr fields pointed to by ATTR that are constant for all directory
 * fattrs we return. Also set the number-of-children-dependent fields
 * based on NKIDS.
 */

#if __STDC__
static void
get_dir_attr(fattr *attr)
#else
get_dir_attr(attr)
     fattr *attr;
#endif
{
  attr->type = NFDIR;
  attr->nlink = 10;		/* ??? */
  /*
   * Set size so that the cookies (= offspring index including
   * faked '.', '..', '.finderinfo' and '.resources')
   * will always be less than size.
   */
  attr->size = 10000;		/* ??? */
  attr->blocksize = 1024;	/* ??? */
  attr->blocks = 10;		/* ??? */
}

#if __STDC__
static int
map_afp_error(dword cr)
#else
static int
map_afp_error(cr)
     dword cr;
#endif
{
  switch (cr)
    {
    case aeAccessDenied:
    case aeDenyConflict:
    case aeCantRename:
      return NFSERR_ACCES;
    case aeDirNotEmpty:
      return NFSERR_NOTEMPTY;
    case aeDiskFull:
      return NFSERR_NOSPC;
    case aeObjectExists:
      return NFSERR_EXIST;
    case aeObjectNotFound:
    case aeDirNotFound:
      return NFSERR_NOENT;
    case aeVolumeLocked:
      return NFSERR_ROFS;
    case aeObjectLocked:
      return NFSERR_PERM;
    default:
      return NFSERR_IO;
    }
}

/*
 * Translate the FileDirParm in FDPP for an offspring to FH, or
 * FH itself into ATTR.
 */

#if __STDC__
static int
xlate_fdp(const afi_fh *fh,
	  fattr *attr,
	  const FileDirParm *parent_fdpp,
	  const FileDirParm *entry_fdpp,
	  int *idp)
#else
static int
xlate_fdp(fh, attr, parent_fdpp, entry_fdpp, idp)
     afi_fh *fh;
     fattr *attr;
     FileDirParm *parent_fdpp;
     FileDirParm *entry_fdpp;
     int *idp;
#endif
{
  sdword owner, group;


  attr->fsid = OUR_FSID;

  LOG((" xlate_fdp: %s\n", log_fh(fh)));

  if (entry_fdpp == NULL
      || (FDP_ISDIR(entry_fdpp->fdp_flg)
	  && (fh->fh_type == AFIFH_DIR
	      || fh->fh_dirid == entry_fdpp->fdp_parms.dp_parms.dp_dirid)))
    {
      CONST FileDirParm *fdpp = entry_fdpp ? entry_fdpp : parent_fdpp;

      /*
       * Three cases here:
       * - A normal folder
       * - '.' in .resource or .finderinfo
       * - .resource or .finderinfo
       */

      LOG((" -first case\n"));

      get_dir_attr(attr);

      owner = fdpp->fdp_parms.dp_parms.dp_ownerid;
      group = fdpp->fdp_parms.dp_parms.dp_groupid;

      attr->mode = NFSMODE_DIR | afp_to_nfs_mode(owner, group, fdpp);

      attr->uid = afp_to_nfs_uid(owner);
      attr->gid = afp_to_nfs_gid(group);

      attr->fileid = *idp = fdpp->fdp_parms.dp_parms.dp_dirid;

      if (fh->fh_type == AFIFH_RESDIR)
	UNIQ_TYPE(attr->fileid, RES_TAG);
      else if (fh->fh_type == AFIFH_FIDIR)
	UNIQ_TYPE(attr->fileid, FIDIR_TAG);

      UNIQ_VOL(attr->fileid, fh->fh_volix);

      attr->mtime.seconds = attr->atime.seconds = attr->ctime.seconds
	= fdpp->fdp_mdate;
    }
  else
    {
      /*
       * Other cases:
       * - normal files, data or resource fork
       * - entries for files and subfolders in .finderinfo
       */
      LOG((" -second case\n"));

      owner = parent_fdpp->fdp_parms.dp_parms.dp_ownerid;
      group = parent_fdpp->fdp_parms.dp_parms.dp_groupid;

      attr->type = NFREG;
      attr->uid = afp_to_nfs_uid(owner);
      attr->gid = afp_to_nfs_gid(group);
      attr->mode = NFSMODE_REG | afp_to_nfs_mode(owner, group, parent_fdpp);
      attr->nlink = 1;

      switch (fh->fh_type)
	{
	case AFIFH_DIR:
	case AFIFH_DFORK:
	  attr->size = entry_fdpp->fdp_parms.fp_parms.fp_dflen;
	  break;
	case AFIFH_RESDIR:
	case AFIFH_RFORK:
	  attr->size = entry_fdpp->fdp_parms.fp_parms.fp_rflen;
	  break;
	case AFIFH_FIDIR:
	case AFIFH_FINFO:
	  attr->size = sizeof(FileInfo);
	}
      attr->blocksize = 1024;
      attr->blocks = attr->size ? (attr->size - 1) / attr->blocksize + 1 : 0;

      attr->fileid = *idp = entry_fdpp->fdp_parms.fp_parms.fp_fileno;

      if (fh->fh_type == AFIFH_RESDIR
	  || fh->fh_type == AFIFH_RFORK)
	UNIQ_TYPE(attr->fileid, RES_TAG);
      else if (fh->fh_type == AFIFH_FIDIR
	       || fh->fh_type == AFIFH_FINFO)
	UNIQ_TYPE(attr->fileid, FI_TAG);

      UNIQ_VOL(attr->fileid, fh->fh_volix);

      attr->mtime.seconds = attr->atime.seconds = attr->ctime.seconds
	= entry_fdpp->fdp_mdate;
    }

  return NFS_OK;
}

#if __STDC__
static int
login_again(void)
#else
static int
login_again()
#endif
{
  LOG(("login_again\n"));
  logout(who);
  if (!login_user(who, UAM_CLEAR))
    return 0;
  return 1;
}  

#if __STDC__
void
check_errors(const char *function,
	     int comp,
	     dword cr)
#else
void
check_errors(function, comp, cr)
     char *function;
     int comp;
     dword cr;
#endif
{
  if (comp < 0)
    {
      syslog(LOG_ERR, "%s failed: %d", function, comp);
      LOG(("%s failed: %d\n", function, comp));
    }
  else if (cr != noErr)
    LOG(("%s error: %s\n", function, afperr(cr)));
}

#if __STDC__
static int
get_file_dir_parms(short volid,
		   int dirid,
		   const char *path,
		   const FileDirParm *fdpp,
		   dword *crp)
#else
static int
get_file_dir_parms(volid, dirid, path, fdpp, crp)
     short volid;
     int dirid;
     char *path;
     FileDirParm *fdpp;
     dword *crp;
#endif
{
  int comp;
  int tries = 0;

  while (tries++ < MAXTRIES
	 && (comp =
	     eFPGetFileDirParms(who->u_srn, volid, dirid,
				FP_ATTR|FP_MDATE|FP_FILNO|FP_DFLEN|FP_RFLEN,
				DP_ATTR|DP_MDATE|DP_DIRID|DP_CHILD
				|DP_CRTID|DP_GRPID|DP_ACCES,
				path, fdpp, crp)) == SessClosed)
      if (!login_again())
	return -1;

  check_errors("FPGetFileDirParms", comp, *crp);

  return comp;
}

/*
 * Get the NFS attributes for a directory entry.
 * FH is the parent directory or the directory itself,
 * ATTR is the fattr to fill in,
 * PATH is the relative AFP path.
 */

#if __STDC__
static int
get_attr(const afi_fh *fh,
	 fattr *attr,
	 const char *path,
	 int *idp)
#else
static int
get_attr(fh, attr, path, idp)
     afi_fh *fh;
     fattr *attr;
     char *path;
     int *idp;
#endif
{
  int comp;
  dword cr;
  FileDirParm parent, this;
  FDParmPtr parent_fdpp = &parent, this_fdpp;
  
  LOG((" get_attr %s, path: '%s'", log_fh(fh), log_path((byte *)path)));

  comp = get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
			    fh->fh_dirid,
			    empty_path, parent_fdpp, &cr);

  LOG((" parent: -> %d, %d\n", comp, cr));
  
  if (comp == noErr && cr == noErr && path[0])
    {
      this_fdpp = &this;
      comp = get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
				fh->fh_dirid,
				path, this_fdpp, &cr);
      LOG((" this: -> %d, %d\n", comp, cr));
    }
  else
    this_fdpp = NULL;
  if (comp != noErr)
    return NFSERR_IO;
  else if (cr != noErr)
    return map_afp_error(cr);

  /* If we looked up a subdir in .resource, fail */
  if (fh->fh_type == AFIFH_RESDIR
      && this_fdpp
      && FDP_ISDIR(this_fdpp->fdp_flg)
      && fh->fh_dirid != this_fdpp->fdp_parms.dp_parms.dp_dirid)
    return NFSERR_NOENT;

  return xlate_fdp(fh, attr, parent_fdpp, this_fdpp, idp);
}

#if __STDC__
static int
get_fork_parms(const afi_fh *fh,
	       const char *path,
	       const cache_entry *ce,
	       fattr *attr)
#else
static int
get_fork_parms(fh, path, ce, attr)
     afi_fh *fh;
     char *path;
     cache_entry *ce;
     fattr *attr;
#endif
{
  int comp;
  dword cr;
  FileDirParm parent, this;
  int id;

  comp = get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
			    fh->fh_dirid,
			    empty_path, &parent, &cr);

  if (comp == noErr && cr == noErr)
    {
      LOG(("FPGetForkParms(%d,%d)\n", who->u_srn, ce->refnum));
      comp = eFPGetForkParms(who->u_srn, ce->refnum,
			     FP_ATTR|FP_MDATE|FP_FILNO|
			     (fh->fh_type == AFIFH_DFORK ? FP_DFLEN : FP_RFLEN),
			     &this, &cr);
      check_errors("FPGetForkParms", comp, cr);
    }

  if (comp == SessClosed)
    return get_attr(fh, attr, path, &id);
  else if (comp != noErr)
    return NFSERR_IO;
  else if (cr != noErr)
    return map_afp_error(cr);
  
  xlate_fdp(fh, attr, &parent, &this, &id);
  return NFS_OK;
}

#if __STDC__
static int
enumerate(short volid,
	  int dirid,
	  const char *path,
	  int ix,
	  word fbitmap,
	  word dbitmap,
	  const FileDirParm *fdpp,
	  int nparms,
	  int *countp,
	  dword *crp)
#else
static int
enumerate(volid, dirid, path, ix, fbitmap, dbitmap, fdpp, nparms, countp, crp)
     short volid;
     int dirid;
     byte *path;
     int ix;
     word fbitmap;
     word dbitmap;
     FileDirParm *fdpp;
     int nparms;
     int *countp;
     dword *crp;
#endif
{
  int comp;
  int tries = 0;

  while (tries++ < MAXTRIES
	 && (comp = eFPEnumerate(who->u_srn, volid, dirid, path, ix,
				 fbitmap, dbitmap,
				 fdpp, nparms, countp, crp)) == SessClosed)
    if (!login_again())
      return -1;

  check_errors("FPEnumerate", comp, *crp);
  
  return comp;
}

/*
 * Fill in the fattr structure ATTR with the attributes for
 * the top-leel directory (containing the volumes).
 */

#if __STDC__
static int
set_top_attr(fattr *attr)
#else
static int
set_top_attr(attr)
     fattr *attr;
#endif
{
  get_dir_attr(attr);
  attr->fsid = OUR_FSID;
  attr->fileid = TOP_FILEID;
  attr->mode = NFSMODE_DIR | S_IRUSR | S_IXUSR
    | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
  attr->uid = attr->gid = 0;
}

#if __STDC__
static int
set_control_attr(fattr *attr)
#else
static int
set_control_attr(attr)
     fattr *attr;
#endif
{
  attr->type = NFREG;
  attr->mode = NFSMODE_REG | S_IWUSR | S_IWGRP | S_IWOTH;
  attr->nlink = 1;
  attr->uid = attr->gid = 0;
  attr->size = 0;
  attr->blocksize = 1024;
  attr->blocks = 0;
  attr->fsid = OUR_FSID;
  attr->fileid = CONTROL_FILEID;
}

#if __STDC__
static int
get_vol_fh_attr(int ix,
		diropokres *resp)
#else
static int
get_vol_fh_attr(ix, resp)
     int ix;
     diropokres *resp;
#endif
{
  afi_fh *result_fh = (afi_fh *) &resp->file;
  fattr *result_attr = &resp->attributes;
  int id;

  result_fh->fh_type = AFIFH_DIR;
  result_attr->fsid = OUR_FSID;
  result_fh->fh_volix = ix;
  result_attr->fileid = result_fh->fh_dirid = VOLUME_ROOT_FILEID;

/*  UNIQ_VOL(result_attr->fileid, ix); */

  return get_attr(result_fh, result_attr, empty_path, &id);
}

#if __STDC__
static int
remove_file(const afi_fh *fh, const char *path)
#else
static int
remove_file(fh, path)
     afi_fh *fh;
     char *path;
#endif
{
  int tries, comp;
  dword cr;

  tries = 0;
  while (tries++ < MAXTRIES
	 && (comp = eFPDelete(who->u_srn,
			      volume_parms[fh->fh_volix].gvpr_volid,
			      fh->fh_dirid,
			      path, &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;
  
  check_errors("FPDelete", comp, cr);
  if (comp != noErr)
    return NFSERR_IO;
  else if (cr != noErr)
    return map_afp_error(cr);

  return NFS_OK;
}

#if __STDC__
static int
search_file(const afi_fh *fh, int verbose,
	    FileDirParm *parent_fdpp, FileDirParm *entry_fdpp)
#else
static int
search_file(fh, verbose, parent_fdpp, entry_fdpp)
     afi_fh *fh;
     int verbose;
     FileDirParm *parent_fdpp;
     FileDirParm *entry_fdpp;
#endif
{
  int i, comp, ix, count;
  dword cr;
#define NFDPS 100
  FileDirParm fdps[NFDPS];
  
  LOG((" search_file(%s)", log_fh(fh)));

  ix = 1;
  
  if (parent_fdpp != NULL)
    {
      comp = get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
				fh->fh_dirid,
				empty_path, parent_fdpp, &cr);
      if (comp != noErr || cr != noErr)
	{
	  LOG((": parent parms not found\n"));
	  return 0;
	}
    }
  
  while (1)
    {
      comp = enumerate(volume_parms[fh->fh_volix].gvpr_volid, fh->fh_dirid,
		       empty_path, ix,
		       FP_FILNO | FP_LNAME, DP_DIRID | DP_LNAME,
		       fdps, NFDPS, &count, &cr);
      if (comp != noErr || cr != noErr)
	{
	  LOG((": not found\n"));
	  return 0;
	}

      for (i = 0; i < count; i++)
	if ((FDP_ISDIR(fdps[i].fdp_flg)
	     && fdps[i].fdp_parms.dp_parms.dp_dirid == fh->fh_fileid)
	    ||
	    (!(FDP_ISDIR(fdps[i].fdp_flg))
	     && fdps[i].fdp_parms.fp_parms.fp_fileno == fh->fh_fileid))
	  {
	    if (verbose)
	      {
		/* enumerate only got names and ids.  */
		comp =
		  get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
				     fh->fh_dirid,
				     fdps[i].fdp_lname, entry_fdpp, &cr);
		if (comp != noErr || cr != noErr)
		  {
		    LOG((": found, but no parms?\n"));
		    return NULL;
		  }
		memcpy(entry_fdpp->fdp_lname, fdps[i].fdp_lname,
		       sizeof(fdps[i].fdp_lname));
	      }
	    else
	      *entry_fdpp = fdps[i];
	    
	    LOG((": OK\n"));
	    return 1;
	  }
      ix += count;
  }
}
#undef NFDPS

#if __STDC__
static int
set_file_attr(const afi_fh *fh, const char *path, const sattr *attr)
#else
static int
set_file_attr(fh, path, attr)
     afi_fh *fh;
     char *path;
     sattr *attr;
#endif
{
  int tries;
  int comp;
  dword cr;
  int ix;
  
  if (attr->mtime.seconds != (u_int) -1)
    {
      FileDirParm fdp;

      tries = 0;
      fdp.fdp_mdate = attr->mtime.seconds;

      while (tries++ < MAXTRIES
	     && (comp =
		 eFPSetFileParms(who->u_srn,
				 volume_parms[fh->fh_volix].gvpr_volid,
				 fh->fh_dirid,
				 FP_MDATE, path, &fdp, &cr)) == SessClosed)
	if (!login_again())
	  return NFSERR_IO;

      check_errors("FPSetFileParms", comp, cr);

      if (comp != noErr)
	return NFSERR_IO;
      else if (cr != noErr)
	return map_afp_error(cr);
      
      return NFS_OK;
    }

  if (fh->fh_type == AFIFH_DFORK && fh->fh_fileid >= 0
      && attr->size != -1)
    {
      int res;
      FileParm fp;

      fp.fp_dflen = attr->size;
      
      if ((res = maybe_open_fork(fh, WRITE, &ix)) != NFS_OK)
	return res;
      
      tries = 0;
      while (tries++ < MAXTRIES
	     && (comp = eFPSetForkParms(who->u_srn, cache[ix].refnum,
					FP_DFLEN, &fp, &cr)) == SessClosed)
	{
	  cache[ix].refnum = FORK_NOT_OPEN;
	  if (!login_again())
	    return NFSERR_IO;
	  
	  if ((res = maybe_open_fork(fh, WRITE, &ix)) != NFS_OK)
	    return res;
	}
      
      time(&cache[ix].timestamp);

      check_errors("FPSetForkParms", comp, cr);

      if (comp != noErr)
	return NFSERR_IO;
      else if (cr != noErr)
	return map_afp_error(cr);
    }
  
  return NFS_OK;
}

#if __STDC__
static int
set_dir_attr(const afi_fh *fh, const char *path, const sattr *attr)
#else
static int
set_dir_attr(fh, path, attr)
     afi_fh *fh;
     char *path;
     sattr *attr;
#endif
{
  int tries;
  int comp;
  dword cr;
  FileDirParm fdp;
  word bitmap;

  bitmap = 0;

  if (attr->mode != -1)
    {
      fdp.fdp_parms.dp_parms.dp_accright = nfs_to_afp_mode(attr->mode);
      bitmap |= DP_ACCES;
    }
  
  if (attr->mtime.seconds != (u_int) -1)
    {
      fdp.fdp_mdate = attr->mtime.seconds;
      bitmap |= DP_MDATE;
    }

  if (attr->uid != (u_int) -1)
    {
      fdp.fdp_parms.dp_parms.dp_ownerid = nfs_to_afp_uid(attr->uid);
      bitmap |= DP_CRTID;
    }

  if (attr->gid != (u_int) -1)
    {
      fdp.fdp_parms.dp_parms.dp_groupid = nfs_to_afp_gid(attr->gid);
      bitmap |= DP_GRPID;
    }

  if (bitmap)
    {
      tries = 0;

      while (tries++ < MAXTRIES
	     && (comp =
		 eFPSetDirParms(who->u_srn,
				volume_parms[fh->fh_volix].gvpr_volid,
				fh->fh_dirid,
				bitmap, path, &fdp, &cr)) == SessClosed)
	if (!login_again())
	  return NFSERR_IO;

      check_errors("FPSetDirParms", comp, cr);

      if (comp != noErr)
	return NFSERR_IO;
      else if (cr != noErr)
	return map_afp_error(cr);
      
      return NFS_OK;
    }

  return NFS_OK;
}

#if __STDC__
static int
set_attr(const afi_fh *fh, const char *path, const sattr *attr)
#else
static int
set_attr(fh, path, attr)
     afi_fh *fh;
     char *path;
     sattr *attr;
#endif
{
  if (fh->fh_type == AFIFH_DIR)
    return set_dir_attr(fh, path, attr);
  else
    return set_file_attr(fh, path, attr);
}

#if __STDC__
static int
open_fork(const afi_fh *fh, int mode, word *refp, char *path)
#else
static int
open_fork(fh, mode, refp, path)
     afi_fh *fh;
     int mode;
     word *refp;
     char *path;
#endif
{
  FileDirParm fdp;
  int comp;
  dword cr;
  int tries = 0;

  if (!search_file(fh, 0, NULL, &fdp))
    return NFSERR_STALE;
  
  LOG(("FPOpenFork(%d, %s, %d)", fh->fh_dirid,
       log_path((byte *)fdp.fdp_lname), mode));

  while (tries++ < MAXTRIES
	 && (comp = eFPOpenFork(who->u_srn,
				volume_parms[fh->fh_volix].gvpr_volid,
				fh->fh_dirid,
				mode, fdp.fdp_lname,
				fh->fh_type == AFIFH_DFORK ? 0 : 1,
				0, &fdp, refp, &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;
  
  check_errors("FPOpenFork", comp, cr);

  if (comp != noErr || cr != noErr)
    return NFSERR_IO;

  LOG((": %d\n", *refp));

  memcpy(path, fdp.fdp_lname, fdp.fdp_lname[0]+1);

  return NFS_OK;
}

#if __STDC__
static int
read_fork(word ref, u_int offset, u_int count,
	  char **val, u_int *lenp)
#else
static int
read_fork(ref, offset, count, val, lenp)
     word ref;
     u_int offset;
     u_int count;
     char **val;
     u_int *lenp;
#endif
{
  int comp;
  dword cr;
  int tries = 0;

  *val = xmalloc(count);

  LOG(("FPRead(%d, %d, %d)", ref, offset, count));

  while (tries++ < MAXTRIES
	 && (comp = eFPRead(who->u_srn, ref, *val, count, count, offset,
			    &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;
  
  if (comp < 0)
    if (cr != aeEOFErr)
      {
	check_errors("FPRead", comp, cr);
	free(*val);
	*val = NULL;
	
	return NFSERR_IO;
      }
  *lenp = comp;
  
  LOG((": %d\n", comp));

  return NFS_OK;
}  

#if __STDC__
static int
get_fileinfo(const afi_fh *fh, FileInfo *fip, fattr *attr)
#else
static int
get_fileinfo(fh, fip, attr)
     afi_fh *fh;
     FileInfo *fip;
     fattr *attr;
#endif
{
  int comp;
  dword cr;
  FileDirParm parent, this;
  GetCommentReplyPkt comment;
  int id;

  if (!search_file(fh, 1, &parent, &this))
    return NFSERR_STALE;

  comp = eFPGetFileDirParms(who->u_srn,
			    volume_parms[fh->fh_volix].gvpr_volid,
			    fh->fh_dirid,
			    FP_FINFO, DP_FINFO,
			    this.fdp_lname, &this, &cr);

  check_errors("FPGetFileDirParms", comp, cr);
  
  if (comp != noErr || cr != noErr)
    return NFSERR_STALE;

  memcpy(fip->fi_fndr, this.fdp_finfo, FINFOLEN);
  fip->fi_attr = this.fdp_attr;
  fip->fi_magic1 = FI_MAGIC1;
  fip->fi_version = FI_VERSION;
  fip->fi_magic = FI_MAGIC;
  fip->fi_bitmap = FI_BM_MACINTOSHFILENAME;
  memcpy(fip->fi_macfilename, this.fdp_lname+1, this.fdp_lname[0]);
  fip->fi_macfilename[this.fdp_lname[0]] = 0;

  if (who->u_dtrn == DTRN_NOT_OPEN
      || who->u_dtvol != fh->fh_volix)
    {
      if (who->u_dtrn != DTRN_NOT_OPEN)
	{
	  comp = eFPCloseDT(who->u_srn, who->u_dtrn, &cr);
	  check_errors("FPCloseDT", comp, cr);
	}
      
      comp = eFPOpenDT(who->u_srn, volume_parms[fh->fh_volix].gvpr_volid, &who->u_dtrn, &cr);
      check_errors("FPOpenDT", comp, cr);
      if (comp == noErr && cr == noErr)
	who->u_dtvol = fh->fh_volix;
      else
	who->u_dtvol = DTRN_OPEN_FAILED;
    }

  fip->fi_comln = 0;
  
  if (who->u_dtvol == fh->fh_volix)
    {
      comp = eFPGetComment(who->u_srn, who->u_dtrn, fh->fh_dirid,
			   this.fdp_lname, &comment, &cr);
      check_errors("FPGetComment", comp, cr);

      if (comp == noErr && cr == noErr)
	{
	  fip->fi_comln = comment.gcmr_clen;
	  memcpy(fip->fi_comnt, comment.gcmr_ctxt, comment.gcmr_clen);
	}
    }

  xlate_fdp(fh, attr, &parent, &this, &id);

  return NFS_OK;
}

#if __STDC__
static int
write_fork(word ref, u_int offset, u_int count,
	   char *buf, u_int *lenp)
#else
static int
write_fork(ref, offset, count, buf, lenp)
     word ref;
     u_int offset;
     u_int count;
     char *buf;
     u_int *lenp;
#endif
{
  int comp;
  dword cr;
  int tries = 0;
  dword lastoffset = offset;
    
  LOG(("FPWrite(%d, %d, %d)", ref, offset, count));

  while (tries++ < MAXTRIES
	 && (comp = eFPWrite(who->u_srn, ref, buf, count, count, &count,
			     &lastoffset, &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;

  if (comp < 0 || cr != noErr)
    {
      check_errors("FPWrite", comp, cr);
      return NFSERR_IO;
    }
  *lenp = count;
  
  LOG((": %d\n", comp));

  return NFS_OK;
}  

#if __STDC__
static void
close_fork(afi_user *user, word ref)
#else
static void
close_fork(user, ref)
     afi_user *user;
     word ref;
#endif
{
  int comp;
  dword cr;

  comp = eFPCloseFork(user->u_srn, ref, &cr);
  LOG(("FPCloseFork(%d, %d): %d, %d\n", user->u_srn, ref, comp, cr));
  check_errors("FPCloseFork", comp, cr);
}

#if __STDC__
void
nfs_timeout(time_t now)
#else
void
nfs_timeout(now)
     time_t now;
#endif
{
  int i;
  afi_user *u;

  u = users;
  while (u)
    u->u_mark = 0, u = u->u_link;
  
  /* Close idle forks: no activity for TIMEOUT seconds.  */
  /* Mark sessions with non-idle forks. */
  for (i = 0; i < CACHESIZE; i++)
    if (cache[i].refnum >= 0)
      {
	if (cache[i].timestamp < now - TIMEOUT)
	  {
	    close_fork(cache[i].user, cache[i].refnum);
	    cache[i].refnum = FORK_NOT_OPEN;
	  }
	else
	  cache[i].user->u_mark = 1;
      }
	
  /*
   * Close idle sessions: sessions with no fork open, and
   * no activity for 10*TIMEOUT seconds.
   */
  u = users;
  for (u = users; u; u = u->u_link)
     if (!u->u_mark
	 && u->u_timestamp < now - 10*TIMEOUT
	 && u->u_srn >= 0)
       logout(u);
}

#if __STDC__
static int
find_cached_file(const afi_fh *fh, byte mode)
#else
static int
find_cached_file(fh, mode)
     afi_fh *fh;
     byte mode;
#endif
{
  int i, retval;
  cache_entry *ce;

  i = 0;
      
  while (i < CACHESIZE
	 && ((ce = cache + cacheix[i])->refnum == FORK_NOT_OPEN
	     || ce->fh.fh_volix != fh->fh_volix
	     || ce->fh.fh_dirid != fh->fh_dirid
	     || ce->fh.fh_fileid != fh->fh_fileid
	     || (ce->mode & mode) != mode))
    i++;
  if (i == CACHESIZE)
    return -1;

  retval = cacheix[i];

  if (i > 0)
    {
      memmove(cacheix+1, cacheix, sizeof(cacheix[0]) * i);
      cacheix[0] = retval;
    }

  return retval;
}

#if __STDC__
static void
close_cache_entry(cache_entry *ce)
#else
static void
close_cache_entry(ce)
     cache_entry *ce;
#endif     
{
  afi_user_list ul, link;

  if (ce->refnum >= 0)
    close_fork(ce->user, ce->refnum);
  ce->refnum = FORK_NOT_OPEN;
  ul = ce->piggyback;
  while (ul)
    {
      link = ul->link;
      free(ul);
      ul = link;
    }
}  

#if __STDC__
static int
enter_cache_entry(const afi_fh *fh, int mode, word ref, const char *path)
#else
static int
enter_cache_entry(fh, mode, ref, path)
     afi_fh *fh;
     int mode;
     word ref;
     char *path;
#endif
{
  int ix;
  cache_entry *ce;

  if ((ce = cache + (ix = cacheix[CACHESIZE-1]))->refnum >= 0)
    close_cache_entry(ce);
  memmove(cacheix + 1, cacheix, sizeof(cacheix[0]) * (CACHESIZE-1));
  cacheix[0] = ix;
  ce->fh = *fh;
  ce->user = who;
  ce->mode = mode;
  ce->piggyback = NULL;
  ce->refnum = ref;
  memcpy(ce->path, path, path[0]+1);

  return ix;
}

#if __STDC__
static int
find_user_of_file(afi_user_list *listp)
#else
static int
find_user_of_file(listp)
     afi_user_list *listp;
#endif
{
  afi_user_list p = *listp;

  while (p && p->user != who)
    p = p->link;

  return p != NULL;
}

#if __STDC__
static void
piggyback_user(afi_user_list *listp)
#else
static void
piggyback_user(listp)
     afi_user_list *listp;
#endif
{
  afi_user_list link = *listp;

  *listp = xmalloc(sizeof *listp);
  (*listp)->user = who;
  (*listp)->link = link;
}

#if __STDC__
static int
maybe_open_fork(const afi_fh *fh, int mode, int *ixp)
#else
static int
maybe_open_fork(fh, mode, ixp)
     afi_fh *fh;
     int mode;
     int *ixp;
#endif
{
  int ix, result;
  word ref;
  char path[MAXLFLEN];

  /* Do we have this file cached?  */
  if ((ix = find_cached_file(fh, 0)) >= 0)
    {
      /*
       * Yes; is this the original opener, or have we already
       * piggybacked this user, and is the mode suitable?
       */

      if ((cache[ix].user == who || find_user_of_file(&cache[ix].piggyback))
	  && (cache[ix].mode & mode) == mode)
	{
	  LOG(("-in cache: %d\n", ix));
	  *ixp = ix;
	  return NFS_OK;
	}

      /* Else, try to open the fork with the desired mode.  */

      LOG(("-in cache: %d, but must reopen\n", ix));
      /*
       * If we are the only user, close it first to lighten
       * the server's work. Also, open it READ+WRITE now. (???)
       */
      if (cache[ix].user == who && cache[ix].piggyback == NULL)
	{
	  close_cache_entry(cache + ix);
	  mode = READ|WRITE;
	}

      if ((result = open_fork(fh, mode, &ref, path)) != NFS_OK)
	return result;
	  
      /*
       * If the mode isn't suitable, close the fork and use the just
       * opened handle.  (Any fork is open only once in afpmount.)
       */
      if ((cache[ix].mode & mode) != mode)
	{
	  close_cache_entry(cache + ix);
	  
	  ix = enter_cache_entry(fh, mode, ref, path);
	  LOG(("-into cache: %d\n", ix));
	}
      else
	{
	  /*
	   * This user can open it; so close the unnecessary fork handle
	   * just opened and use the cached handle, piggybacking this user.
	   */
	  close_fork(who, ref);
	  piggyback_user(&cache[ix].piggyback);
	}
    }
  else
    {
      if ((result = open_fork(fh, mode, &ref, path)) != NFS_OK)
	return result;
      ix = enter_cache_entry(fh, mode, ref, path);
      LOG(("-into cache: %d\n", ix));
    }
  *ixp = ix;
  return NFS_OK;
}

#if __STDC__
static int
create_file(const afi_fh *fh, const char *path, afi_fh *result_fh)
#else
static int
create_file(fh, path, result_fh)
     afi_fh *fh;
     char *path;
     afi_fh *result_fh;
#endif
{
  int tries;
  int comp;
  dword cr;

  tries = 0;

  while (tries++ < MAXTRIES
	 && (comp = eFPCreateFile(who->u_srn,
				  volume_parms[fh->fh_volix].gvpr_volid,
				  fh->fh_dirid, 0,
				  path, &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;

  check_errors("FPCreateFile", comp, cr);

  if (comp != noErr)
    return NFSERR_IO;
  else if (cr != noErr)
    return map_afp_error(cr);

  *result_fh = *fh;
  result_fh->fh_fileid = -1;	/* Make sure it's illegal, must be filled
				   in by caller.  */
  if (fh->fh_type == AFIFH_DIR)
    if (mode == MODE_CAP)
      result_fh->fh_type = AFIFH_DFORK;
    else if (mode == MODE_APPLESINGLE)
      result_fh->fh_type = AFIFH_SINGLE;
    else if (mode == MODE_APPLEDOUBLE)
      result_fh->fh_type = AFIFH_DFORK;
    else
      {
	assert(0);
      }
  else if (fh->fh_type == AFIFH_RESDIR)
    result_fh->fh_type = AFIFH_RFORK;
  else
    {
      assert(0);		/* like abort(), but tell where */
    }

  return NFS_OK;
}
      
#if __STDC__
static int
create_dir(const afi_fh *fh, const char *path, afi_fh *result_fh)
#else
static int
create_dir(fh, path, result_fh)
     afi_fh *fh;
     char *path;
     afi_fh *result_fh;
#endif
{
  int tries;
  int comp;
  dword cr;

  tries = 0;

  *result_fh = *fh;

  while (tries++ < MAXTRIES
	 && (comp =
	     eFPCreateDir(who->u_srn,
			  volume_parms[fh->fh_volix].gvpr_volid, fh->fh_dirid, 
			  path, &result_fh->fh_dirid, &cr)) == SessClosed)
    if (!login_again())
      return NFSERR_IO;

  check_errors("FPCreateDir", comp, cr);

  if (comp != noErr)
    return NFSERR_IO;
  else if (cr != noErr)
    return map_afp_error(cr);

  return NFS_OK;
}
      
#define MAX_E_SIZE sizeof(entry) + MAXLFLEN + 16

#if __STDC__
int
dpsize(entry **e)
#else
int
dpsize(e)
     entry **e;
#endif
{
  return sizeof(entry) + strlen((*e)->name) + 16;
}

#if __STDC__
void *
ctlproc_null_1(void *argp,
	       struct svc_req *rqstp)
#else
void *
ctlproc_null_1(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  return &res;
}

#if __STDC__
void *
ctlproc_login_1(void *argp,
	       struct svc_req *rqstp)
#else
void *
ctlproc_login_1(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  return &res;
}

#if __STDC__
void *
nfsproc_null_2(void *argp,
	       struct svc_req *rqstp)
#else
void *
nfsproc_null_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  ENTRY(null,void);

  res = 0;

  LEAVE(void);
}

#if __STDC__
attrstat *
nfsproc_getattr_2(nfs_fh *argp,
		  struct svc_req *rqstp)
#else
attrstat *
nfsproc_getattr_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST afi_fh *fh = (afi_fh*) argp;

  ENTRY(getattr,nfs_fh);

  memset(&res, 0, sizeof(res));

  if (fh->fh_type == AFIFH_TOP)
    {
      set_top_attr(result_attr);
      LEAVE(attrstat);
    }

  if (fh->fh_type == AFIFH_CONTROL)
    {
      set_control_attr(result_attr);
      LEAVE(attrstat);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  if (fh->fh_type == AFIFH_DIR
      || fh->fh_type == AFIFH_RESDIR
      || fh->fh_type == AFIFH_FIDIR)
    {
      int id;

      /*
       * The .resources and .finderinfo subdirs get the same attrs
       * as the parent directory.
       */
      res.status = get_attr(fh, result_attr, empty_path, &id);
    }
#if 0
  /* Doesn't seem to work, I get AccessDenied errors from FPGetForkParms.  */
  else if ((ix = find_cached_file(fh, READ)) >= 0)
    res.status = get_fork_parms(fh, cache[ix].path, &cache[ix], result_attr);
#endif
  else
    {
      /* Must search for file in parent. */
      FileDirParm parent, this;
      int id;

      if (!search_file(fh, 1, &parent, &this))
	res.status = NFSERR_STALE;
      else
	xlate_fdp(fh, result_attr, &parent, &this, &id);
    }
  
  LEAVE(attrstat);
}

#if __STDC__
attrstat *
nfsproc_setattr_2(sattrargs *argp,
		  struct svc_req *rqstp)
#else
attrstat *
nfsproc_setattr_2(argp, rqstp)
	sattrargs *argp;
	struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST afi_fh *fh = (afi_fh *) &argp->file;
  sattr *attr = &argp->attributes;
  char *path;
  FileDirParm fdp;
  int id;

  ENTRY(setattr,sattrargs);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  switch (fh->fh_type)
    {
    case AFIFH_DIR:
      path = empty_path;
      break;
    case AFIFH_DFORK:
      if (!search_file(fh, 0, NULL, &fdp))
	{
	  res.status = NFSERR_STALE;
	  LEAVE(attrstat);
	}
      path = fdp.fdp_lname;
      break;
    default:
      res.status = NFSERR_ACCES;
      LEAVE(attrstat);
    }

  if ((res.status = set_attr(fh, path, attr)) != NFS_OK)
    LEAVE(attrstat);

  res.status = get_attr(fh, result_attr, path, &id);

  LEAVE(attrstat);
}

#if __STDC__
void *
nfsproc_root_2(void *argp,
	       struct svc_req *rqstp)
#else
void *
nfsproc_root_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;

  ENTRY(root,void);

  res = 0;

  LEAVE(void);
}

#if __STDC__
diropres *
nfsproc_lookup_2(diropargs *argp,
		 struct svc_req *rqstp)
#else
diropres *
nfsproc_lookup_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  diropokres *result = &res.diropres_u.diropres;
  fattr *result_attr = &result->attributes;

  afi_fh *result_fh = (afi_fh*) &result->file;
  CONST afi_fh *fh = (afi_fh*) &argp->dir;
  int i, t, id;
  char path[MAXLFLEN+2];

  ENTRY(lookup,diropargs);

  memset(&res, 0, sizeof(res));
  
  /* The control file? */
  if (fh->fh_type == AFIFH_TOP
      && !strcmp(argp->name, ".control"))
    {
      set_control_attr(result_attr);
      result_fh->fh_type = AFIFH_CONTROL;
      LEAVE(diropres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  switch (fh->fh_type)
    {
    case AFIFH_TOP:
      if (!strcmp(argp->name, ".") || !strcmp(argp->name, ".."))
	{
	  set_top_attr(result_attr);
	  *result_fh = *toplevel_fh;
	  LEAVE(diropres);
	}

      /* The control file was handled above.  */

      if (!server_parms)
	{
	  res.status = NFSERR_NOENT;
	  LEAVE(diropres);
	}
      
      /* Look up a volume */
      for (i = 0; i < server_parms->gspr_nvols; i++)
	if (!local_afp_strcmp
	    (argp->name, (char *) server_parms->gspr_volp[i].volp_name))
	  break;
      if (i >= server_parms->gspr_nvols)
	res.status = NFSERR_NOENT;
      else
	res.status = get_vol_fh_attr(i, result);
      break;

    case AFIFH_DIR:
      /* Look up a directory entry */
      t = AFIFH_DIR;
      if (!strcmp(argp->name, "."))
	path[0] = 0;
      else if (!strcmp(argp->name, "..")
	       && fh->fh_dirid == VOLUME_ROOT_FILEID)
	{
	  set_top_attr(result_attr);
	  *result_fh = *toplevel_fh;
	  LEAVE(diropres);
	}
      else if (!strcmp(argp->name, ".."))
	{
	  path[0] = 2;
	  path[1] = path[2] = 0;
	}
      /* .finderinfo and .resource get same attrs as dir itself... */
      else if (mode == MODE_CAP && !strcmp(argp->name, ".finderinfo"))
	{
	  t = AFIFH_FIDIR;
	  path[0] = 0;
	}
      else if (mode == MODE_CAP && !strcmp(argp->name, ".resource"))
	{
	  t = AFIFH_RESDIR;
	  path[0] = 0;
	}
      else
	local_to_p_afp_cpy(path, (byte *)argp->name);

      if ((res.status = get_attr(fh, result_attr, path, &id)) != NFS_OK)
	LEAVE(diropres);

      if (result_attr->type == NFREG)
	t = AFIFH_DFORK;
      *result_fh = *fh;
      result_fh->fh_type = t;
      if (t == AFIFH_DIR
	  || t == AFIFH_RESDIR
	  || t == AFIFH_FIDIR)
	{
	  result_fh->fh_dirid = id;
	  if (t == AFIFH_RESDIR)
	    UNIQ_TYPE(result_attr->fileid, RES_TAG);
	  else if (t == AFIFH_FIDIR)
	    UNIQ_TYPE(result_attr->fileid, FIDIR_TAG);
	}
      else
	result_fh->fh_fileid = id;
      break;

    case AFIFH_RESDIR:
    case AFIFH_FIDIR:
      if (!strcmp(argp->name, "."))
	{
	  path[0] = 0;
	  *result_fh = *fh;
	  res.status = get_attr(result_fh, result_attr, path, &id);
	}
      else if (!strcmp(argp->name, ".."))
	{
	  path[0] = 0;
	  *result_fh = *fh;
	  result_fh->fh_type = AFIFH_DIR;
	  if ((res.status = get_attr(result_fh, result_attr,
				     path, &id)) != NFS_OK)
	    LEAVE(diropres);
	}
      else
	{
	  local_to_p_afp_cpy(path, (byte *)argp->name);

	  *result_fh = *fh;
	  if (fh->fh_type == AFIFH_RESDIR)
	    result_fh->fh_type = AFIFH_RFORK;
	  else
	    result_fh->fh_type = AFIFH_FINFO;

	  if ((res.status = get_attr(fh, result_attr, path, &id)) != NFS_OK)
	    LEAVE(diropres);

	  result_fh->fh_fileid = id;

	  if (result_attr->type != NFREG)
	    {
	      res.status = NFSERR_NOENT;
	      LEAVE(diropres);
	    }
	}
      break;

    default:
      res.status = NFSERR_NOTDIR;
    };

  LEAVE(diropres);
}

#if __STDC__
readlinkres *
nfsproc_readlink_2(nfs_fh *argp,
		   struct svc_req *rqstp)
#else
readlinkres *
nfsproc_readlink_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static readlinkres res;

  ENTRY(readlink,nfs_fh);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readlinkres);

  res.status = NFSERR_IO;

  LEAVE(readlinkres);
}

#if __STDC__
readres *
nfsproc_read_2(readargs *argp,
	       struct svc_req *rqstp)
#else
readres *
nfsproc_read_2(argp, rqstp)
     readargs *argp;
     struct svc_req *rqstp;
#endif
{
  static readres res;
  fattr *result_attr = &res.readres_u.reply.attributes;
  char **result_val = &res.readres_u.reply.data.data_val;
  u_int *result_len = &res.readres_u.reply.data.data_len;
  CONST afi_fh *fh = (afi_fh *) &argp->file;
  u_int offset = argp->offset, count = argp->count;
  int ix, id;

  ENTRY(read,readargs);
  
  /* Free previous result.  */
  xdr_free(xdr_readres, &res);

  memset(&res, 0, sizeof(res));

  /* Reading from the control file always returns EOF. */
  if (fh->fh_type == AFIFH_CONTROL)
    {
      *result_val = xmalloc(1);
      *result_len = 0;
      set_control_attr(result_attr);
      LEAVE(readres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readres);
  
  if (fh->fh_type == AFIFH_DFORK
      || fh->fh_type == AFIFH_RFORK)
    {
      if ((res.status = maybe_open_fork(fh, READ, &ix)) != NFS_OK)
	LEAVE(readres);

      time(&cache[ix].timestamp);

      /* Now read the stuff she wants.  */
      if ((res.status = read_fork(cache[ix].refnum, offset, count,
				  result_val, result_len)) != NFS_OK)
	LEAVE(readres);

      res.status = get_attr(fh, result_attr, cache[ix].path, &id);
    }
  else if (fh->fh_type == AFIFH_FINFO)
    {
      if (offset >= sizeof(FileInfo))
	{
	  *result_val = xmalloc(1);
	  *result_len = 0;
	}
      else
	{
	  FileInfo fi;

	  if ((res.status = get_fileinfo(fh, &fi, result_attr)) != NFS_OK)
	    LEAVE(readres);

	  *result_len = sizeof(FileInfo) - offset;
	  *result_val = xmalloc(*result_len);
	  memcpy(*result_val, ((char *)&fi) + offset, *result_len);
	}
    }
  else
    res.status = NFSERR_IO;
  
  LEAVE(readres);
}

#if __STDC__
void *
nfsproc_writecache_2(void *argp,
		     struct svc_req *rqstp)
#else
void *
nfsproc_writecache_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  ENTRY(writecache,void);
  
  memset(&res, 0, sizeof(res));

  LEAVE(void);
}

#if __STDC__
attrstat *
nfsproc_write_2(writeargs *argp,
		struct svc_req *rqstp)
#else
attrstat *
nfsproc_write_2(argp, rqstp)
     writeargs *argp;
     struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST afi_fh *fh = (afi_fh*) &argp->file;
  u_int offset = argp->offset, count = argp->data.data_len;
  int ix, id;

  ENTRY(write,writeargs);
  
  memset(&res, 0, sizeof(res));

  /*
   * We must be able to write the control file without logging in,
   * because the purpose of writing there is to provide login authentication.
   */

  if (fh->fh_type == AFIFH_CONTROL)
    {
      char buf[MAXPSTR+1], name[MAXPSTR+1], passwd[MAXPSTR+1];
      afi_user *u;
      int uid;

      if (offset || count > MAXPSTR)
	{
	  res.status = NFSERR_IO;
	  LEAVE(attrstat);
	}

      memcpy(buf, argp->data.data_val, count);
      buf[count] = 0;

      if (sscanf(buf, "login %s %s\n", name, passwd) != 2)
	{
	  res.status = NFSERR_IO;
	  LEAVE(attrstat);
	}
      u = session_of((uid = auth_uid(rqstp)));

      if (!u)
	{
	  u = xmalloc(sizeof(*u));
	  u->u_uxuid = uid;
	  u->u_link = users;
	  users = u;
	}
      else
	{
	  if (u->u_srn >= 0)
	    logout(u);
	  free(u->u_name);
	  free(u->u_passwd);
	}
      u->u_srn = SRN_NOT_OPEN;
      u->u_name = xstrdup(name);
      u->u_passwd = xstrdup(passwd);
      time(&u->u_timestamp);
      /*
       * If nobody has logged on yet, log this user on to get
       * the server parameters (esp. volume names).
       */
      if (!server_parms)
	login_user(u, UAM_CLEAR);
      set_control_attr(result_attr);
      LEAVE(attrstat);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  if (fh->fh_type == AFIFH_DFORK
      || fh->fh_type == AFIFH_RFORK)
    {
      if ((res.status = maybe_open_fork(fh, WRITE, &ix)) != NFS_OK)
	LEAVE(attrstat);

      time(&cache[ix].timestamp);

      if (count > 0)
	{
	  /* Now write the stuff.  */
	  if ((res.status = write_fork(cache[ix].refnum, offset, count,
				       argp->data.data_val, &count)) != NFS_OK)
	    LEAVE(attrstat);
	  if (count != argp->data.data_len)
	    {
	      syslog(LOG_ERR, "Asked to write %d bytes, wrote %d\n",
		      argp->data.data_len, count);
	      res.status = NFSERR_IO;
	      LEAVE(attrstat);
	    }
	}
      
      res.status = get_attr(fh, result_attr, cache[ix].path, &id);
    }
  else if (fh->fh_type == AFIFH_FINFO)
    {
      FileInfo fi;

      if (offset >= sizeof(FileInfo))
	{
	  res.status = NFSERR_IO;
	  LEAVE(attrstat);
	}
    }
  else
    res.status = NFSERR_ACCES;
  
  LEAVE(attrstat);
}

#if __STDC__
diropres *
nfsproc_create_2(createargs *argp,
		 struct svc_req *rqstp)
#else
diropres *
nfsproc_create_2(argp, rqstp)
     createargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  CONST afi_fh *fh = (afi_fh *) &argp->where.dir;
  afi_fh *result_fh = (afi_fh *) &res.diropres_u.diropres.file;
  fattr *result_attr = &res.diropres_u.diropres.attributes;
  char path[MAXLFLEN];
	
  ENTRY(create,createargs);
  
  memset(&res, 0, sizeof(res));

  /* The control file can be "created" without loggin in. */
  if (fh->fh_type == AFIFH_TOP
      && !strcmp(argp->where.name, ".control"))
    {
      set_control_attr(result_attr);
      result_fh->fh_type = AFIFH_CONTROL;
      LEAVE(diropres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  if (fh->fh_type == AFIFH_FIDIR)
    {
      res.status = NFSERR_ACCES;
      LEAVE(diropres);
    }
  else if (fh->fh_type != AFIFH_DIR
      && fh->fh_type != AFIFH_RESDIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(diropres);
    }
  
  local_to_p_afp_cpy(path, (byte *)argp->where.name);

  if ((res.status = create_file(fh, path, result_fh)) != NFS_OK)
    if (res.status == NFSERR_EXIST)
      /*
       * RFC1094 says that CREATE must be a "create only if it is not already
       * there, but does not say what we should do *if* it is already there.
       * It seems as we should return OK in that case.
       */
      {
	res.status = NFS_OK;
	*result_fh = *fh;
	if ((res.status =
	     get_attr(fh, result_attr, path,
		      &result_fh->fh_fileid)) != NFS_OK)
	  LEAVE(diropres);
	if (result_attr->type == NFREG)
	  result_fh->fh_type =
	    (fh->fh_type == AFIFH_DIR ? AFIFH_DFORK : AFIFH_RFORK);
	else
	  res.status = NFSERR_EXIST;
	LEAVE(diropres);
      }
    else
      LEAVE(diropres);

#if 0
  /*
   * No use, we cannot set the uid and gid,
   * and the size is set to zero automatically. (?)
   */
  (void) set_attr(result_fh, path, &argp->attributes);
#endif

  res.status = get_attr(fh, result_attr, path, &result_fh->fh_fileid);

  LEAVE(diropres);
}

#if __STDC__
nfsstat *
nfsproc_remove_2(diropargs *argp,
		 struct svc_req *rqstp)
#else
nfsstat *
nfsproc_remove_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST afi_fh *fh = (afi_fh*) &argp->dir;
  char path[MAXLFLEN];
  int ix;

  ENTRY(remove,diropargs);

  res = NFS_OK;

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  local_to_p_afp_cpy(path, (byte *)argp->name);

  /*
   * If the file is open, we can't remove it.
   * We could check if we have it open ourself, but why bother.
   */

  res = remove_file(fh, path);

  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_rename_2(renameargs *argp,
		 struct svc_req *rqstp)
#else
nfsstat *
nfsproc_rename_2(argp, rqstp)
     renameargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST afi_fh *from_fh = (afi_fh *) &argp->from.dir;
  CONST afi_fh *to_fh = (afi_fh *) &argp->to.dir;
  char from_path[MAXLFLEN], to_path[MAXLFLEN];
  int tries, comp;
  dword cr;

  ENTRY(rename,renameargs);

  res = NFS_OK;

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  if (from_fh->fh_type != AFIFH_DIR || to_fh->fh_type != AFIFH_DIR)
    {
      res = NFSERR_ACCES;
      LEAVE(nfsstat);
    }
  
  if (from_fh->fh_volix != to_fh->fh_volix)
    {
      res = NFSERR_ACCES;
      LEAVE(nfsstat);
    }

  local_to_p_afp_cpy(from_path, (byte *)argp->from.name);
  local_to_p_afp_cpy(to_path, (byte *)argp->to.name);

  tries = 0;
  while (tries++ < MAXTRIES
	 && (comp = eFPMoveFile(who->u_srn,
				volume_parms[from_fh->fh_volix].gvpr_volid,
				from_fh->fh_dirid, from_path,
				to_fh->fh_dirid, empty_path,
				to_path, &cr)) == SessClosed)
    if (!login_again())
      {
	res = NFSERR_IO;
	LEAVE(nfsstat);
      }

  check_errors("FPMoveAndRename", comp, cr);

  if (comp != noErr)
    {
      res = NFSERR_IO;
      LEAVE(nfsstat);
    }
  else if (cr != noErr)
    {
      res = map_afp_error(cr);
      LEAVE(nfsstat);
    }
  
  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_link_2(linkargs *argp,
	       struct svc_req *rqstp)
#else
nfsstat *
nfsproc_link_2(argp, rqstp)
     linkargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;

  ENTRY(link,linkargs);

  memset(&res, 0, sizeof(res));

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  res = NFSERR_ACCES;
  
  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_symlink_2(symlinkargs *argp,
		  struct svc_req *rqstp)
#else
nfsstat *
nfsproc_symlink_2(argp, rqstp)
     symlinkargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;

  ENTRY(symlink,symlinkargs);

  memset(&res, 0, sizeof(res));

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  res = NFSERR_IO;

  LEAVE(nfsstat);
}

#if __STDC__
diropres *
nfsproc_mkdir_2(createargs *argp,
		struct svc_req *rqstp)
#else
diropres *
nfsproc_mkdir_2(argp, rqstp)
     createargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  CONST afi_fh *fh = (afi_fh *) &argp->where.dir;
  afi_fh *result_fh = (afi_fh *) &res.diropres_u.diropres.file;
  fattr *result_attr = &res.diropres_u.diropres.attributes;
  char path[MAXLFLEN];
  int id;
  
  ENTRY(mkdir,createargs);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  if (fh->fh_type == AFIFH_FIDIR)
    {
      res.status = NFSERR_ACCES;
      LEAVE(diropres);
    }
  else if (fh->fh_type != AFIFH_DIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(diropres);
    }
  
  local_to_p_afp_cpy(path, (byte *)argp->where.name);

  if ((res.status = create_dir(fh, path, result_fh)) != NFS_OK)
    LEAVE(diropres);

  (void) set_attr(result_fh, empty_path, &argp->attributes);

  res.status = get_attr(result_fh, result_attr, empty_path, &id);
  
  LEAVE(diropres);
}

#if __STDC__
nfsstat *
nfsproc_rmdir_2(diropargs *argp,
		struct svc_req *rqstp)
#else
nfsstat *
nfsproc_rmdir_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST afi_fh *fh = (afi_fh*) &argp->dir;
  char path[MAXLFLEN];

  ENTRY(rmdir,diropargs);

  res = NFS_OK;

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  local_to_p_afp_cpy(path, (byte *)argp->name);

  res = remove_file(fh, path);

  LEAVE(nfsstat);
}

#if __STDC__
readdirres *
nfsproc_readdir_2(readdirargs *argp,
		  struct svc_req *rqstp)
#else
readdirres *
nfsproc_readdir_2(argp, rqstp)
     readdirargs *argp;
     struct svc_req *rqstp;
#endif
{
  static readdirres res;
  CONST afi_fh *fh = (afi_fh*) &argp->dir;
  entry **e;
  FDParmPtr fdpp;
  int nalloc;
  int comp, count;
  dword cr;
  int i, j, ix, folder_ix;
  int res_size = 0;
  int double_extra = (mode == MODE_APPLEDOUBLE ? MAX_E_SIZE : 0);
  CONST int DOTDOT_COOKIE = 1;
  CONST int CONTROL_COOKIE = 2;
  CONST int TOP_EXTRA = 3;

#define NEXT_PLEASE()						\
      ix++, 							\
      memcpy((*e)->cookie, &ix, NFS_COOKIESIZE),		\
      res_size += dpsize(e),					\
      e = &((*e)->nextentry)					\

  ENTRY(readdir,readdirargs);

  /* Free previous result.  */
  xdr_free(xdr_readdirres, &res);

  memset(&res, 0, sizeof(res));
  
  if (fh->fh_type != AFIFH_TOP
      && fh->fh_type != AFIFH_DIR
      && fh->fh_type != AFIFH_FIDIR
      && fh->fh_type != AFIFH_RESDIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(readdirres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readdirres);
  
  ix = 0;
  memcpy(&ix, argp->cookie, NFS_COOKIESIZE);

  e = &(res.readdirres_u.reply.entries);
  res_size = 0;

  while (ix <= DOTDOT_COOKIE)
    {
      /* Fake '.' and '..' entries. */
      *e = (entry *) xzalloc(sizeof(entry));
      if (ix == 0)
	{
	  /* '.' entry. */
	  if (fh->fh_type == AFIFH_TOP)
	    (*e)->fileid = TOP_FILEID;
	  else
	    {
	      (*e)->fileid = fh->fh_dirid;
	      if (fh->fh_type == AFIFH_RESDIR)
		UNIQ_TYPE((*e)->fileid, RES_TAG);
	      else if (fh->fh_type == AFIFH_FIDIR)
		UNIQ_TYPE((*e)->fileid, FI_TAG);
	      UNIQ_VOL((*e)->fileid, fh->fh_volix);
	    }
	  (*e)->name = xstrdup(".");
	}
      else
	{
	  /* '..' entry. */
	  if (fh->fh_type == AFIFH_TOP
	      || fh->fh_dirid == 2)
	    /* The top and volumes have '..' entries that point to the top. */
	    (*e)->fileid = TOP_FILEID;
	  else if (fh->fh_type == AFIFH_RESDIR
		   || fh->fh_type == AFIFH_FIDIR)
	    {
	      (*e)->fileid = fh->fh_dirid;
	      UNIQ_VOL((*e)->fileid, fh->fh_volix);
	    }
	  else
	    {
	      /* Ask AFP for the parent directory. */
	      FileDirParm fdp;
	      char path[3];

	      path[0] = 2;
	      path[1] = path[2] = 0;
	      comp =
		get_file_dir_parms(volume_parms[fh->fh_volix].gvpr_volid,
				   fh->fh_dirid,
				   path, &fdp, &cr);
	      if (comp != noErr || cr != noErr)
		{
		  res.status = NFSERR_IO;
		  LEAVE(readdirres);
		}
	      assert(FDP_ISDIR(fdp.fdp_flg));
	      (*e)->fileid = fdp.fdp_parms.dp_parms.dp_dirid;
	      UNIQ_VOL((*e)->fileid, fh->fh_volix);
	    }
	  (*e)->name = xstrdup("..");
	}


      NEXT_PLEASE();

      if (res_size + MAX_E_SIZE > argp->count)
	{
	  *e = NULL;
	  res.readdirres_u.reply.eof = 0;
	  LEAVE(readdirres);
	}
    }
  
  /* The root contains the volumes and the control file. */
  if (fh->fh_type == AFIFH_TOP)
    {
      /* The .control file is used when logging in.  */
      if (ix == CONTROL_COOKIE
	  && res_size + MAX_E_SIZE < argp->count)
	{
	  *e = (entry *)xzalloc(sizeof(entry));
	  (*e)->fileid = CONTROL_FILEID;
	  (*e)->name = xstrdup(".control");
	  NEXT_PLEASE();
	}

      /* The volumes. */
      while (server_parms
	     && ix < server_parms->gspr_nvols + TOP_EXTRA
	     && res_size + MAX_E_SIZE < argp->count)
	{
	  byte *vol = server_parms->gspr_volp[ix-TOP_EXTRA].volp_name;

	  *e = (entry *)xzalloc(sizeof(entry));
	  (*e)->fileid = VOLUME_ROOT_FILEID;
	  UNIQ_VOL((*e)->fileid, ix-TOP_EXTRA);
	  (*e)->name = afp_to_local_dup(vol, strlen((char *)vol));
	  NEXT_PLEASE();
	}

      *e = NULL;
      res.readdirres_u.reply.eof =
	(ix >= server_parms->gspr_nvols + TOP_EXTRA);

      LEAVE(readdirres);
    }

  folder_ix = ix - 1;
  
  if (mode == MODE_CAP && fh->fh_type == AFIFH_DIR)
    {
      while (ix <= 3)
	{
	  /* Fake '.resource' and '.finderinfo' entries. */
	  *e = (entry *) xzalloc(sizeof(entry));
	  if (ix == 2)
	    {
	      (*e)->fileid = fh->fh_dirid;
	      UNIQ_TYPE((*e)->fileid, RES_TAG);
	      (*e)->name = xstrdup(".resource");
	    }
	  else
	    {
	      (*e)->fileid = fh->fh_dirid;
	      UNIQ_TYPE((*e)->fileid, FIDIR_TAG);
	      (*e)->name = xstrdup(".finderinfo");
	    }
	  UNIQ_VOL((*e)->fileid, fh->fh_volix);
	  NEXT_PLEASE();

	  if (res_size + MAX_E_SIZE > argp->count)
	    {
	      *e = NULL;
	      res.readdirres_u.reply.eof = 0;
	      LEAVE(readdirres);
	    }
	}
      folder_ix = ix - 3;
    }
  
  /* Maximum number of filenames we might return. */
  nalloc = (argp->count - res_size) / (sizeof(entry) + 2 + 16); 
  if (nalloc == 0)
    nalloc = 1;

  fdpp = (FDParmPtr)xmalloc(nalloc * sizeof(FileDirParm));
  
  cr = 0;
  while (((res_size + MAX_E_SIZE < argp->count
	   || e == &(res.readdirres_u.reply.entries)))
	 && cr != aeObjectNotFound)
    {
      comp = enumerate(volume_parms[fh->fh_volix].gvpr_volid, fh->fh_dirid,
		       empty_path, folder_ix, 
		       FP_LNAME|FP_FILNO, DP_LNAME|DP_DIRID,
		       fdpp, nalloc, &count, &cr);
      if (comp != noErr || (cr != 0 && cr != aeObjectNotFound))
	  {
	    if (cr != 0)
	      syslog(LOG_ERR, "FPEnumerate: %s\n", afperr(cr));
	    else
	      syslog(LOG_ERR, "eFPEnumerate: %d\n", comp);
	    free(fdpp);
	    res.status = NFSERR_IO;
	    LEAVE(readdirres);
	  }

      if (cr == noErr)
	{
	  i = 0;
	  while (res_size + MAX_E_SIZE + double_extra < argp->count
		 && i < count)
	    {
	      /* .resource only contain entries for files */
	      if (FDP_ISDIR(fdpp[i].fdp_flg) && fh->fh_type == AFIFH_RESDIR)
		{
		  ix++;
		  folder_ix++;
		  i++;
		  continue;
		}

	      *e = (entry *)xzalloc(sizeof(entry));

	      if (FDP_ISDIR(fdpp[i].fdp_flg))
		(*e)->fileid = fdpp[i].fdp_parms.dp_parms.dp_dirid;
	      else
		(*e)->fileid = fdpp[i].fdp_parms.fp_parms.fp_fileno;

	      if (fh->fh_type == AFIFH_RESDIR)
		UNIQ_TYPE((*e)->fileid, RES_TAG);
	      else if (fh->fh_type == AFIFH_FIDIR)
		UNIQ_TYPE((*e)->fileid, FI_TAG);

	      UNIQ_VOL((*e)->fileid, fh->fh_volix);

	      (*e)->name =
		afp_to_local_dup((byte *)fdpp[i].fdp_lname+1,
				 fdpp[i].fdp_lname[0]);

	      NEXT_PLEASE();
	      folder_ix++;

	      /* Resource fork in Apple Double mode. */
	      if (mode == MODE_APPLEDOUBLE && !FDP_ISDIR(fdpp[i].fdp_flg))
		{
		  byte tem[MAXLFLEN+2];
		  
		  *e = (entry *)xzalloc(sizeof(entry));

		  (*e)->fileid = fdpp[i].fdp_parms.fp_parms.fp_fileno;
		  UNIQ_TYPE((*e)->fileid, RES_TAG);
		  UNIQ_VOL((*e)->fileid, fh->fh_volix);

		  tem[0] = '%';
		  memcpy(tem+1, fdpp[i].fdp_lname+1, fdpp[i].fdp_lname[0]);

		  (*e)->name =
		    afp_to_local_dup(tem, fdpp[i].fdp_lname[0]+1);

		  NEXT_PLEASE();
		}
	      i++;
	    }
	}
    }
  *e = NULL;
  res.readdirres_u.reply.eof = (cr == aeObjectNotFound);
  free(fdpp);
  
  LEAVE(readdirres);

#undef NEXT_PLEASE
}

#if __STDC__
statfsres *
nfsproc_statfs_2(nfs_fh *argp,
		 struct svc_req *rqstp)
#else
statfsres *
nfsproc_statfs_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static statfsres res;
  statfsokres *p = &res.statfsres_u.reply;

  ENTRY(statfs,nfs_fh);

  memset(&res, 0, sizeof(res));

  p->tsize = p->bsize = 1024;
  p->blocks = p->bfree = p->bavail = 0;

  LEAVE(statfsres);
}

#ifndef SENTINEL

#if __STDC__
void *
xmalloc(size_t nbytes)
#else
void *
xmalloc(nbytes)
     size_t nbytes;
#endif
{
  void *result;

  if ((result = malloc(nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to malloc %d bytes)", nbytes);
      exit(1);
    }
  return result;
}

#if __STDC__
void *
xrealloc(void *ptr, size_t nbytes)
#else
void *
xrealloc(ptr, nbytes)
     void *ptr;
     size_t nbytes;
#endif
{
  void *result;

  if ((result = realloc(ptr, nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to realloc %d bytes)", nbytes);
      exit(1);
    }
  return result;
}  

#if __STDC__
void *
xzalloc(size_t nbytes)
#else
void *
xzalloc(nbytes)
     size_t nbytes;
#endif
{
  void *result;

  if ((result = calloc(1, nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to malloc %d bytes)", nbytes);
      exit(1);
    }
  return result;
}

#endif				/* !SENTINEL */
