/*
 * Copyright (C) 1992-94 by Gustaf Neumann, Stefan Nusser
 *
 *      Wirtschaftsuniversitaet Wien,
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6,
 *      A-1090 Vienna, Austria
 *      neumann@@wu-wien.ac.at, nusser@@wu-wien.ac.at
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 *
 * Date: Dec 24 1994
 * Author: Gustaf Neumann
 * Version: 1.0.8
 */

#define WAFESTRING_C

#include "wafe.h"

#undef WSDEBUG

#ifdef WSDEBUG
# define WSPRINT(fmtargs) printf fmtargs
# define WSREPORT(type) \
    fprintf(stderr,"%s: length=%d, max=%d %p <%s>\n", \
	    type,ws->length,ws->max,ws->buffer,ws->buffer)
#else
# define WSPRINT(fmtargs)
# define WSREPORT(type)
#endif
   
void
wafeStringInit(ws)
wafeString ws;
    {
    ws->length = 0;
    ws->buffer = ws->sBuffer;
    ws->max    = WAFESTRING_MIN;
    *ws->buffer = '\0';  /* when the buffer is used immediatedly after init */
    }

void
wafeStringClear(ws)
wafeString ws;
    {
    if (ws->buffer != ws->sBuffer) 
	{
	DBUG_PRINT("wafeString",("free size = %d",ws->max));
	WSPRINT(("freeing buffer in ws=%p\n",ws));
	XtFree(ws->buffer);
	}

    ws->length = 0;
    ws->buffer = ws->sBuffer;
    ws->max    = WAFESTRING_MIN;
    }

wafeString
wafeStringNew()
    {
    wafeString ws = (wafeString)XtMalloc(sizeof(wafeStringStruct));
    wafeStringInit(ws);
    return ws;
    }

void
wafeStringFree(ws)
wafeString ws;
    {
    wafeStringClear(ws);
    XtFree((char *)ws);
    }

/* wafeStringToString returns an XtMalloced buffer; don't call 
   wafeStringClear on ws! */
static String
wafeStringToString(ws)
wafeString ws;
    {
    return (ws->buffer == ws->sBuffer) ?
	XtNewString(ws->buffer) : ws->buffer;
    }


static String
wafeStringCheckAlloc(ws, size)
wafeString ws;
int size;
    {
    int desiredSize = size + ws->length;
    if (ws->max < desiredSize)  /* we have to grow */
	{
	if ( desiredSize > (ws->max + WAFESTRING_SLACK) )
	    ws->max = desiredSize += WAFESTRING_SLACK;
	else
	    ws->max += WAFESTRING_SLACK;

	if (ws->buffer == ws->sBuffer) 
	    {
	    DBUG_PRINT("wafeString",("malloc, new size = %d",ws->max));
	    WSPRINT(("First malloc in ws=%p, size=%d\n",ws,ws->max));
	    ws->buffer = XtMalloc(ws->max);
	    memcpy(ws->buffer,ws->sBuffer,ws->length);
	    }
	else
	    {
	    WSPRINT(("realloc in ws=%p\n",ws));
	    DBUG_PRINT("wafeString",("realloc, new size = %d",ws->max));
	    ws->buffer = XtRealloc(ws->buffer,ws->max);
	    }
	}
    return ws->buffer + ws->length;
    }

void
wafeStringAppend(ws, string)
wafeString ws;
String string;
    {
    if (string) 
	{
	int stringLength = strlen(string);
	String next      = wafeStringCheckAlloc(ws,stringLength);
	memcpy(next, string, stringLength);
	ws->length += stringLength;
	next[stringLength] = 0;
	}
    WSREPORT("stringAppend");
    }

void
wafeStringAppendN(ws, string, stringLength)
wafeString ws;
String string;
int stringLength;
    {
    String next = wafeStringCheckAlloc(ws,stringLength+1);
    strncpy(next,string,stringLength);
    ws->length += stringLength;
    next[stringLength] = 0;
    WSREPORT("stringAppendN");
    }

void
wafeStringAppendEscaped(ws, string, n)
wafeString ws;
String string;
int n;
    {
    char *p = string;
    char *next;
    int  free, f;

    if (!string)
      return;

    /* the first optimistic guess is that nothing must be escaped;
       we know that check alloc preallocates space to prevent excessive
       reallocs.
     */
    if (n>0) 
	next = wafeStringCheckAlloc(ws,n);
    else
	next = wafeStringCheckAlloc(ws,strlen(string));
    
    f = free = ws->max - ws->length;
    if (p) 
	{
	while (*p && n) 
	    {
	    if (free < 3) 
		{
		ws->length += f-free;
		next = wafeStringCheckAlloc(ws,10);
		f = free = ws->max - ws->length;
		}
	    if (*p == '"' || *p == '$' || *p == '[' || *p == '\\' ) 
		{
		*next++ = '\\';
		free--;
		}
	    *next++ = *p++;
	    free--;
	    if (n>0) n--;
	    }
	}
    *next = '\0';
    ws->length += f-free;
    WSREPORT("eString");
    }


void
wafeStringAppendLong(ws, l)
wafeString ws;
long l;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%ld", l);
    ws->length += strlen(next);
    WSREPORT("long");
    }

void
wafeStringAppendInt(ws, i)
wafeString ws;
int i;
    {
    char *next = wafeStringCheckAlloc(ws,INT_AS_STRING);
    sprintf(next, "%d", i);
    ws->length += strlen(next);
    WSREPORT("int");
    }

void
wafeStringAppendFloat(ws, f)
wafeString ws;
double f;
    {
    char *next = wafeStringCheckAlloc(ws,FLOAT_AS_STRING);
    sprintf(next, "%f", f);
    ws->length += strlen(next);
    WSREPORT("float");
    }

void
wafeStringAppendShort(ws, s)
wafeString ws;
short s;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%hd", s);
    ws->length += strlen(next);
    WSREPORT("short");
    }

void
wafeStringAppendPointer(ws, p)
wafeString ws;
XtPointer p;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%p", p);
    ws->length += strlen(next);
    WSREPORT("pointer");
    }


void
wafeStringAppendChar(ws, c)
wafeString ws;
char c;
    {
    char *next = wafeStringCheckAlloc(ws,2);
    *next++ = c;
    *next   = '\0';
    ws->length ++;
    WSREPORT("char");
    }

void
wafeStringAppendWidgetList(ws, num, widgetArray)
wafeString ws;
Cardinal   num;
WidgetList widgetArray;
    {
    if (num>0) 
	{
	int     i;
	char *output,*buffer;

	buffer = output = wafeStringCheckAlloc(ws,num * (LONG_AS_STRING+1));
	for (i=0; i<num; i++)
	    {
	    sprintf(output, "%ld ", (long)widgetArray[i]);
	    output += strlen(output);
	    }
	ws->length += output-buffer-1;
	ws->buffer[ws->length] = 0;
	}
    }

void
wafeStringAppendIntList(ws, num, intArray)
wafeString ws;
Cardinal   num;
int       *intArray;
    {
    if (num>0) 
	{
	int     i;
	char *output,*buffer;

	buffer = output = wafeStringCheckAlloc(ws,num * (INT_AS_STRING+1));
	for (i=0; i<num; i++)
	    {
	    sprintf(output, "%d ", intArray[i]);
	    output += strlen(output);
	    }
	ws->length += output-buffer-1;
	ws->buffer[ws->length] = 0;
	}
    }

String
wafeCvtIntsToList(num, intArray)
Cardinal    num;
int        *intArray;
    {
    wafeStringStruct ws_s;
    wafeString ws = &ws_s;

    wafeStringInit(ws);
    wafeStringAppendIntList(ws, num, intArray);
    return wafeStringToString(ws);
    }

String
wafeCvtWidgetListToList(num, widgetArray)
Cardinal   num;
WidgetList widgetArray;
    {
    wafeStringStruct ws_s;
    wafeString ws = &ws_s;

    wafeStringInit(ws);
    wafeStringAppendWidgetList(ws, num, widgetArray);
    return wafeStringToString(ws);
    }
