#define HANDLERS_C

#include "wafe.h"
#include "version.h"
#include <sys/wait.h>

#undef TUNNELDEBUG

#ifdef RDD
# include <rdd.h>
#endif

#define EXTRA_HANDLERS 16

/* from AIX man page */
#define CMD_BUF_SIZE		32764
#define COM_BUF_SIZE		32764
#define STDIN_BUF_SIZE		10240
#define STDERR_BUF_SIZE		10240
#define EXTRA_BUF_SIZE		10240

/* global variables for signal handling */
String wafeHandlers[HANDLERMAX];
static String tclIntHandlers[SIGMAX];

static Boolean intPending[SIGMAX];
static Boolean duringEval    = False;
static int     anyIntPending = 0;

/* global variables for io handlers */
char   wafePromptChar  = '%';
static char *comBuffer = NULL;

static char  *communicationHandlerVarname = NULL;
static char  *communicationHandlerCommand = NULL;
static Cardinal communicationHandlerLength = 0;
static Cardinal communicationCount = 0;

static Boolean stdinHandlerRegistered  = False;
static Boolean stdoutHandlerRegistered = False;
static Boolean stderrHandlerRegistered = False;
static Boolean interactiveMode;

extern int Tcl_ListCmd(
#if NeedFunctionPrototypes
	ClientData, Tcl_Interp*, int, char**
#endif
);

/* Used to buffer incomplete commands being read from stdin. */
static wafeStringStruct tclCommandStruct;
static wafeString tclCommand = &tclCommandStruct;

/*
 * Declare the additional application resource:
 * initCom (Class InitCom) can be used to start an interpreter
 */

extern void wafeExecActionProc(
#if NeedFunctionPrototypes
     Widget, XEvent *, char **, Cardinal *
#endif
);

typedef struct {
    String  init_com;
  } AppData, *AppDataPtr;

static AppData    app_data;
static XtResource resources[] = {
	{
	"initCom",
	"InitCom",
	XtRString,
	sizeof(String),
	XtOffset(AppDataPtr, init_com),
	XtRImmediate,
	"",
	}
    };

static XtActionsRec tclAction[] = {
    {"exec",  wafeExecActionProc},
    };


int
wafeEval(interp, cmd, context)
Tcl_Interp    *interp;
String         cmd;
_Xconst String context;
    {
    int rc, sig;
    static void sigcldHandler();
    
    DBUG_PRINT(context, ("command: <%s>", cmd));

    duringEval = True;
    if ((rc = Tcl_GlobalEval(interp,cmd)) == TCL_ERROR) 
	{ 
	char *line = strtok(cmd, wafe_NL);
	int   currLine = 1;

	fprintf(stderr, "Wafe(%s): %s\n", context, interp->result);
	do 
	    {
	    fprintf(stderr, "%2d: %s %s\n", 
		    currLine, currLine == interp->errorLine ? "->" : "  ",
		    line);
	    currLine ++;
	    }
	while ((line = strtok(NULL,wafe_NL)));
	}
    else
	{
	if (interactiveMode && !strcmp(context,"eval")) 
	    fprintf(stderr, "#R: {%s}\n",interp->result);
	}
    
    duringEval = False;

    if (anyIntPending) 
	{
	for (sig=0; sig < SIGMAX; sig++)
	    if (intPending[sig])
		{
		(void) wafeEval(interp, tclIntHandlers[sig], "sigPending");
		intPending[sig] = False;
		break;
		}
	anyIntPending --;
	}

    if (wafeClientPid)
	signal(SIGCLD,sigcldHandler);

    return rc;
    }


void 
wafeExit (rc)
int rc;
    {
    static char *exitCmd = "if [string compare {} [info proc onExit]] onExit";
    (void) wafeEval(wafeInterpreter, exitCmd, "exit");
    exit(rc);
    }


void
wafeFatal(part, message, arg) 
_Xconst String part;
_Xconst String message;
_Xconst String arg;
    {
    fprintf(stderr,"Wafe(%s): ",part);
    fprintf(stderr,message,arg);
    fprintf(stderr, "; aborting...\n");
    if (wafeClientPid) 
	kill(wafeClientPid, SIGTERM);
    wafeExit(-1);
    }

void
wafeWarn(part, message, arg1, arg2, arg3) 
_Xconst String part;
_Xconst String message;
_Xconst String arg1;
_Xconst String arg2;
_Xconst String arg3;
    {
    fprintf(stderr,"Wafe(%s): ",part);
    fprintf(stderr,message,arg1,arg2,arg3);
    fprintf(stderr, wafe_NL);
    }

int
wafeSetError(message, arg1, arg2, arg3) 
_Xconst String message;
_Xconst String arg1;
_Xconst String arg2;
_Xconst String arg3;
    {
    char errorBuffer[200];
    sprintf(errorBuffer, message, arg1, arg2, arg3);
    Tcl_SetResult(wafeInterpreter, errorBuffer, TCL_VOLATILE);
    return TCL_ERROR;
    }

/*
 * The following functions simplify error handling - they all are
 * invoked under certain circumstances and print an error message
 *
 */

/* component not handled yet */
int
wafeConvError (argc, argv, argNum, component, type)
int    argc;
char **argv;
int    argNum;
_Xconst String component;
_Xconst String type;
    {
    Tcl_DString errorBuffer;
    char intBuffer[INT_AS_STRING];
    char *cmd = Tcl_Merge(argc, argv);

    WAFE_UNUSED(component);

    Tcl_DStringInit(&errorBuffer);
    Tcl_DStringAppendElement(&errorBuffer, "wafeError");
    Tcl_DStringAppendElement(&errorBuffer, argv[0]);
    Tcl_DStringAppendElement(&errorBuffer, "convert");
    Tcl_DStringAppendElement(&errorBuffer, argv[argNum]);
    sprintf(intBuffer, "%d", argNum);
    Tcl_DStringAppendElement(&errorBuffer, intBuffer);
    Tcl_DStringAppendElement(&errorBuffer, type);
    Tcl_DStringAppendElement(&errorBuffer, cmd);
    XtFree(cmd);
    (void) wafeEval(wafeInterpreter, Tcl_DStringValue(&errorBuffer), "error");
    Tcl_DStringFree(&errorBuffer);
    return TCL_ERROR;
    }

void 
wafeConvWarn (proc, arg, type)
_Xconst String proc;
_Xconst String arg;
_Xconst String type;
    {
    fprintf(stderr, "Wafe(%s): could not convert '%s' to type %s\n",
	    proc, arg, type);
    }


int
wafeArgcError (argc, argv, quant, expected)
int            argc;
char         **argv;
_Xconst String quant;
int            expected;
    {
    Tcl_DString errorBuffer;
    char intBuffer[INT_AS_STRING];
    char *cmd = Tcl_Merge(argc, argv);

    Tcl_DStringInit(&errorBuffer);
    Tcl_DStringAppendElement(&errorBuffer,"wafeError");
    Tcl_DStringAppendElement(&errorBuffer,argv[0]);
    Tcl_DStringAppendElement(&errorBuffer,"argc");
    Tcl_DStringAppendElement(&errorBuffer,quant);
    sprintf(intBuffer, "%d", expected);
    Tcl_DStringAppendElement(&errorBuffer,intBuffer);
    sprintf(intBuffer, "%d", argc-1);
    Tcl_DStringAppendElement(&errorBuffer,intBuffer);
    Tcl_DStringAppendElement(&errorBuffer,cmd);
    XtFree(cmd);
    (void) wafeEval(wafeInterpreter, Tcl_DStringValue(&errorBuffer), "error");
    Tcl_DStringFree(&errorBuffer);
    return TCL_ERROR;
    }

void
wafeNoVarCompError (argc, argv, argNum, comp)
int            argc;
char         **argv;
int            argNum;
_Xconst String comp;
    {
    Tcl_DString errorBuffer;
    char *cmd = Tcl_Merge(argc, argv);

    Tcl_DStringInit(&errorBuffer);
    Tcl_DStringAppendElement(&errorBuffer, "wafeError");
    Tcl_DStringAppendElement(&errorBuffer, argv[0]);
    Tcl_DStringAppendElement(&errorBuffer, "noVarComp");
    Tcl_DStringAppendElement(&errorBuffer, comp);
    Tcl_DStringAppendElement(&errorBuffer, argv[argNum]);
    Tcl_DStringAppendElement(&errorBuffer, wafe_EMPTY);
    Tcl_DStringAppendElement(&errorBuffer, cmd);
    XtFree(cmd);
    (void) wafeEval(wafeInterpreter, Tcl_DStringValue(&errorBuffer), "error");
    Tcl_DStringFree(&errorBuffer);
    }


Boolean wafeConversionWarningMessageIssued;

static void warningMsg (name,type,class,defaultp,params,num_params)
String name,type,class,defaultp;
String* params;
Cardinal* num_params;
    {
    char buffer[4096], message[4096];
    String context;
    static String convContext = "conv";

/*
    fprintf(stderr,"name='%s', type='%s', class='%s',\n\tdefault='%s'\n",
	    name, type, class, defaultp);
*/

    XtGetErrorDatabaseText(name,type,class,defaultp, buffer, 4096);
    /*fprintf(stderr,"buffer=%s\n",buffer);*/

    if (!strncmp(name,convContext,4))
	{
	wafeConversionWarningMessageIssued = True;
	context = convContext;
	}
    else
	{
	context = name;
	}
    
    
    /*need better solution here*/
    if (params == NULL || num_params == NULL || *num_params == 0)
	{
	wafeWarn(context,buffer, NULL,NULL,NULL);
	}
    else 
	{
        Cardinal i = *num_params;
        String par[10];
        if (i > 10) i = 10;
        (void) memmove((char*)par, (char*)params, i * sizeof(String) );
        bzero ( &par[i], (10-i) * sizeof(String) );
        (void) sprintf(message, buffer, par[0], par[1], par[2], par[3],
                       par[4], par[5], par[6], par[7], par[8], par[9]);
        wafeWarn(context, message, NULL, NULL, NULL); 
        if (i != *num_params)
            XtWarning( "some arguments in previous message were lost" );
	}
    }





/*
 * SUBST_VARS can be used to activate variable substitution in 
 * lists of argument value pairs
 */
#define SUBST_VARS

Boolean
wafeMergeArguments(arguments,context,argc,argv)
char        *arguments;
char        *context;
int        * argc;
char      ***argv;
    {
    char *p = arguments;
#ifdef SUBST_VARS
    while (*p) 
	{
	if (*p == '\n') *p = ' ';
	p++;
	}

    if (Tcl_VarEval(wafeInterpreter,"list ", arguments, NULL) == TCL_ERROR)
	{
	wafeWarn(context,wafeInterpreter->result,NULL,NULL,NULL);
	return False;
	}
    
    p = wafeInterpreter->result;
#endif

    if (Tcl_SplitList(wafeInterpreter, p, argc, argv) == TCL_ERROR)
	{
	XtFree((char *)*argv);
	wafeWarn(context,wafeInterpreter->result,NULL,NULL,NULL);
	return False;
	}

#ifdef SUBST_VARS_VERBOSE
	{
	int i;
	char **arg = *argv;
	
	for (i=0; i < *argc; i++, arg++) 
	    {
	    fprintf(stderr,"arg %d: <%s>\n",i,*arg);
	    }
	}
#endif
    
    return True;
    }




/*****************************************************************************
 *    FUNCTION:  signalHandler                                             
 *               If wafe receives a signal
 *               it is forwarded to the application                
 *                                                                         
 *    Arguments: signal number                                             
 *    Returns:   nothing                                                   
 *                                                                         
 *    Used by:   main                                                      
 *****************************************************************************/

static void
signalHandler(sig)
int   sig;
    {
    DBUG_ENTER ("signalHandler");
    DBUG_PRINT("signal", ("Signal %d received", sig));

    if (tclIntHandlers[sig]) 
	{
	if (duringEval) 
	    {
	    if (!intPending[sig]) 
		{
		intPending[sig] = True;
		anyIntPending ++;
		}
	    } 
	else 
	    (void) wafeEval(wafeInterpreter, tclIntHandlers[sig], "sig");
	}
    else 
	{
	if (wafeClientPid)
	    {
	    DBUG_PRINT("signal", ("Forwarding Signal to Application"));
	    kill(wafeClientPid, sig);
	    }
	wafeExit(0); 
	}
    /* 
     * restore the signal handler 
     */
    signal(sig,signalHandler);
    DBUG_VOID_RETURN;
    }

static void
sigcldHandler(sig)
int   sig;
    {
    int status;
    WAFE_UNUSED(sig);
    if (waitpid (wafeClientPid, &status,WNOHANG) == wafeClientPid) 
	wafeExit(0);
    return;
    }

/*****************************************************************************
 *    FUNCTION:  XIOHandler                                             
 *               If wafe receives a XIOError
 *               SIGTERM is sent to the application
 *                                                                         
 *    Arguments: display, error event
 *    Returns:   nothing                                                   
 *                                                                         
 *    Used by:   main                                                      
 *****************************************************************************/

static int
IOErrorHandler(dpy)
Display *dpy;
    {
    char buffer[200];
    
    DBUG_ENTER ("IOErrorHandler");

    if (wafeHandlers[XIOERR])
	{
	sprintf(buffer,"%s %s",wafeHandlers[XIOERR],DisplayString(dpy));
	(void) wafeEval(wafeInterpreter, buffer, "XIOERR");
	}

    wafeWarn("main", "XIOError on display %s, terminating!", 
	     DisplayString(dpy), NULL,NULL);

    if (wafeClientPid) 
	kill(wafeClientPid, SIGTERM);
    else 
	wafeExit(0);
    
    DBUG_RETURN (-1);
    }





static void
outputBuffer(buffer) 
_Xconst String buffer;
    {
    if (wafeHandlers[STDOUT])
	{                    
	Tcl_SetVar(wafeInterpreter, "STDOUT", buffer, 
		   TCL_GLOBAL_ONLY);
	(void) wafeEval(wafeInterpreter, wafeHandlers[STDOUT], "STDOUT");
	}
    else
	fprintf(stdout,"%s",buffer);
    }



static void
getInput(client_data, fid, xid)
XtPointer   client_data;          /* PromptMode True oder False  */
int         *fid;
XtInputId   *xid;
    {
    char    inputBuffer[CMD_BUF_SIZE+2];
    char   *firstPtr,  *secondPtr;
    int     readBytes, leftBytes;

    /* the following two variables describe the state of the tcl command
       which is also static */
    static Boolean evalCommand = False;  /* we plan to evaluate the buffer */
    static Boolean unterminated = False; /* indicates, that the last buffer 
					    did not end with a new line; */
    DBUG_ENTER("getInput");
    WAFE_UNUSED(xid);

    firstPtr = secondPtr = inputBuffer;
  
    DBUG_PRINT("io", ("reading inputBuffer=%p, can read=%d", 
		      inputBuffer, CMD_BUF_SIZE));
 
    if ((readBytes = read(*fid, inputBuffer, CMD_BUF_SIZE)) == -1)
	perror("Wafe: Communication error: ");

    DBUG_PRINT("io", ("read inputBuffer= %p, readBytes=%d",
		      inputBuffer, readBytes));

    inputBuffer[readBytes] = '\0';
    DBUG_PRINT("io", ("BufferContents are:\n<%s>", inputBuffer));     

    if (!readBytes)
	{
	DBUG_PRINT("io",("eof on primary input channel, bye."));
	wafeExit(1);
	}    
 
    do
	{
	/* search for newline or end of string */
	while (*firstPtr && *firstPtr != '\n')
	    firstPtr++;

	if (client_data)   /* Standard Mode, checking magic chars */
	    {
	    if (unterminated || 
		(tclCommand ->length && 
		 tclCommand->buffer[tclCommand ->length-1] != '\n'
		 ))
		{  /* we are in the middle of a completion */
		DBUG_PRINT("io",("we are in a completion."));
		}
	    else
	    if (*secondPtr == wafePromptChar) 
		{
		/* DBUG_PRINT("io",("switching eval command ON")); */
		evalCommand = True;
		secondPtr++;
		}
	    else
		{
		evalCommand = False;
		/* DBUG_PRINT("io",("switching eval command OFF")); */
		}
	    }
	else 
	    evalCommand = True;

	if (*firstPtr == '\n') /* One Line is ready */
	    {
	    char saveChar;
	    String assembled, lastChar;
	    int length = 1+firstPtr-secondPtr;

	    unterminated = False;

	    saveChar = *(firstPtr+1);   /* for printing and outputBuffer */
	    *(firstPtr+1) = '\0';       /* for printing and outputBuffer */
	    
	    DBUG_PRINT("io", ("Parsed Command <%s>", secondPtr));
	    
	    /* for inner Event loops or incomplete commands
	       the command buffer has to be copied
	       */
	    if (tclCommand->length>0 || wafeInnerEventLoop) 
		{
		DBUG_PRINT("io", ("waiting for completion l = <%d>", 
				  tclCommand->length));

		wafeStringAppendN(tclCommand, secondPtr, length);
		assembled = tclCommand->buffer;
		lastChar  = &tclCommand->buffer[tclCommand->length];
		}
	    else
		{
		assembled = secondPtr;
		lastChar  = firstPtr;
		}

	    DBUG_PRINT("io", ("Assembled Command <%s>", assembled));
	    
            if (evalCommand) 
                {
		if (*assembled == '\\' || 
		    *assembled == '+'  || 
		    *assembled == '/'  || 
		    *assembled == '='
		    )
		    {
		    String namePtr = assembled + 1;
		    String valPtr  = namePtr;
		    /*
		       fprintf(stderr,
		         "first=%p, second=%p\n",firstPtr,secondPtr);
		     */
		    for(; valPtr < lastChar 
			&& *valPtr != ' ' 
			&& *valPtr != '\n'; valPtr++);

		    if (valPtr != namePtr)       /* we got a variable name */
			{
			*valPtr = '\0';          /* terminate varname */
			if (valPtr != lastChar)  /* skip over 0 */
			    valPtr++;
/*
		        fprintf(stderr,
                                "name=%p, value=%p\n",namePtr,valPtr);
 */
			switch (*assembled) 
			    {
			    case '\\':
			        if (valPtr == lastChar)
				    valPtr = wafe_NL;
				Tcl_SetVar(wafeInterpreter, namePtr, valPtr, 
					   TCL_APPEND_VALUE|TCL_GLOBAL_ONLY);
				DBUG_PRINT("assign", 
				    ("appending to variable <%s> value <%s>nl",
				    namePtr,valPtr));     
				break;

			    case '/':
				Tcl_SetVar(wafeInterpreter, namePtr, wafe_NL, 
					   TCL_APPEND_VALUE|TCL_GLOBAL_ONLY);
			        if (valPtr != lastChar)
				    {
				    *lastChar = '\0';  /* overwrite \n */
				    Tcl_SetVar(wafeInterpreter, 
					   namePtr, valPtr, 
					   TCL_APPEND_VALUE|TCL_GLOBAL_ONLY);
				    }
				DBUG_PRINT("assign", 
				    ("appending to variable <%s> value nl<%s>",
				     namePtr,valPtr));     
				break;
		    
			    case '+':
				if (valPtr != lastChar) 
				    {
				    *lastChar = '\0';  /* overwrite \n */
				    Tcl_SetVar(wafeInterpreter, 
                                           namePtr, valPtr, 
					   TCL_APPEND_VALUE|TCL_GLOBAL_ONLY);
				    DBUG_PRINT("assign", 
					       ("appending to <%s> value <%s>",
						namePtr,valPtr));     
				    }
				break;

			    case '=':
				*lastChar = '\0';  /* overwrite \n */
				Tcl_SetVar(wafeInterpreter, namePtr, valPtr, 
					   TCL_GLOBAL_ONLY);
				DBUG_PRINT("assign", 
					   ("setting variable <%s> value <%s>",
					    namePtr,valPtr));     
				}
			}
		    else
			{
			/* got no varname */
			outputBuffer(client_data ? assembled-1 : assembled);
			}
		    

		    if (tclCommand->length > 0)
			{
			DBUG_PRINT("io", ("freeing  command: %s", 
					  "assign"));
			wafeStringClear(tclCommand);
			}
		    }
		else  /* no assignment char, try to evaluate the command */
		    {
		    /* for inner Event loops or incomplete commands
		       the command buffer has to be copied
		     */
		    if (Tcl_CommandComplete(assembled)) 
			{
			(void) wafeEval(wafeInterpreter,assembled,"eval");
			if (tclCommand->length > 0)
			    {
			    DBUG_PRINT("io", 
				       ("freeing command: %s", 
					"after eval"));
			    wafeStringClear(tclCommand);
			    }
			}
		    else
			{
			if (tclCommand->length == 0)
			    { /* we have tried to avoid the copy so far */
			    wafeStringAppendN(tclCommand, assembled, length);
			    }
			}
		    }
		}
	    else   
		{ /* no evaluation wanted */
		outputBuffer(assembled);
		if (tclCommand->length > 0)
		    {
		    DBUG_PRINT("io", ("freeing command: %s", 
				      "no-eval"));
		    wafeStringClear(tclCommand);
		    }
		}
	    
	    /* we have processed something ending with \n */ 
	    /* restore saveChar in input buffer */
	    secondPtr = ++firstPtr;
	    *secondPtr = saveChar;
	    }
	else 
	    unterminated = True;
	} while (*firstPtr != '\0');

    /* if this one is the last command in the buffer and
     * it is not terminated with a \n => do buffering 
     */

    /* DBUG_PRINT("io",("appending %d chars", firstPtr-secondPtr));  */

    if ((leftBytes = firstPtr-secondPtr))
	{
	DBUG_PRINT("io", ("incomplete command: <%s>, length=%d, processed=%d", 
			  secondPtr,leftBytes, inputBuffer != secondPtr)); 
	wafeStringAppendN(tclCommand, secondPtr, leftBytes);
	}

    DBUG_VOID_RETURN;
    }


static void
stdinHandlerProc(client_data, fid, xid)
XtPointer   client_data; /* Prompt-Mode: True oder False */
int         *fid;
XtInputId   *xid;
    {
    int    readBytes;
    char   buffer[STDIN_BUF_SIZE];

    DBUG_ENTER("getInputFromStdin");

    WAFE_UNUSED(xid);
    WAFE_UNUSED(client_data);

    if ((readBytes = read(*fid, buffer, STDIN_BUF_SIZE)) == -1)
	perror("getInput");

    DBUG_PRINT("io", ("reading STDIN, got %d bytes", readBytes));
    
    if (!readBytes)
	{
	DBUG_PRINT("handler", ("EOF - condition on stdin"));
	DBUG_VOID_RETURN;
	}

/* fprintf(stderr,"HERE I AM readin STDIN, string=<%s>\n", buffer);*/

    if (wafeHandlers[STDIN])
	{
	*(buffer + readBytes) = '\0';  /* assure that the string is valid */
	Tcl_SetVar(wafeInterpreter, "STDIN", buffer, TCL_GLOBAL_ONLY);
	(void) wafeEval(wafeInterpreter, wafeHandlers[STDIN], "STDIN");
	}
    else
	write(wafeToClient, buffer, readBytes);

    DBUG_VOID_RETURN;
    }


static void
stderrHandlerProc(client_data, fid, xid)
XtPointer    client_data;
int         *fid;
XtInputId   *xid;
    {   
    int   readBytes;
    char  buffer[STDERR_BUF_SIZE];

    DBUG_ENTER("stderrHandlerProc");
    WAFE_UNUSED(xid);
    WAFE_UNUSED(client_data);

    if ((readBytes = read(*fid, buffer, STDERR_BUF_SIZE-1)) == -1)
	perror("stderrHandlerProc");

    DBUG_PRINT("io", ("reading STDERR, got %d bytes", readBytes));

    if (!readBytes)
	{
	DBUG_PRINT("handler", ("EOF - condition on stderr"));
	DBUG_VOID_RETURN;
	}

/* fprintf(stderr,"***readin STDERR, string=<%s>\n", buffer); */

    if (wafeHandlers[STDERR])
	{
	*(buffer + readBytes) = '\0';  /* assure that it is a valid string */
	Tcl_SetVar(wafeInterpreter, "STDERR", buffer, TCL_GLOBAL_ONLY);
	(void) wafeEval(wafeInterpreter, wafeHandlers[STDERR], "STDERR");
	}
    else
	write(fileno(stderr), buffer, readBytes);

    DBUG_VOID_RETURN;
    }



static void
communicationHandlerProc(client_data, fid, xid)
XtPointer    client_data;
int         *fid;
XtInputId   *xid;
    {   
    int   readBytes;
    char  buffer[COM_BUF_SIZE];

    DBUG_ENTER("communicationHandlerProc");
    WAFE_UNUSED(xid);
    WAFE_UNUSED(client_data);
    
    if ((readBytes = read(*fid, buffer, COM_BUF_SIZE-1)) == -1)
	perror("communicationHandlerProc");

#ifdef TUNNELDEBUG
    fprintf(stderr, "HandlerProc: I read %d Bytes\n", readBytes); 
#endif

    *(buffer + readBytes) = '\0';
    communicationCount += readBytes;

#ifdef TUNNELDEBUG
    fprintf(stderr, "HandlerProc: This makes now %d Bytes total\n", 
	    communicationCount); 
#endif

    if (!readBytes)
	{
	DBUG_PRINT("handler", ("EOF - condition on communication channel"));
	DBUG_VOID_RETURN;
	}

    if (!comBuffer)
	{
	comBuffer = XtNewString(buffer);
	}
    else
	{
	comBuffer = XtRealloc(comBuffer, strlen(comBuffer) + readBytes + 1);
	strcat(comBuffer, buffer);
	}
    
    if ((communicationHandlerCommand) && (communicationHandlerLength <= communicationCount))
	{
#ifdef TUNNELDEBUG
	fprintf(stderr, "HandlerProc: we got it %d <= %d, executing %s\n", 
		communicationHandlerLength, communicationCount,
		communicationHandlerCommand); 
#endif
	if (!comBuffer)
	    Tcl_SetVar(wafeInterpreter, communicationHandlerVarname, wafe_EMPTY, TCL_GLOBAL_ONLY);
	else
	    {
	    (void)Tcl_UnsetVar(wafeInterpreter, communicationHandlerVarname, 0);
	    Tcl_SetVar(wafeInterpreter, communicationHandlerVarname, comBuffer, 
		       TCL_GLOBAL_ONLY);
	    XtFree(comBuffer);
	    comBuffer = NULL;
	       }

          (void) wafeEval(wafeInterpreter, communicationHandlerCommand,
			  "setCommunicationVariable");
          
	XtFree(communicationHandlerVarname);
	XtFree(communicationHandlerCommand);
	communicationCount = 0;
	communicationHandlerLength = 0;
	communicationHandlerCommand = NULL;
	}
    else
	{
#ifdef TUNNELDEBUG
	fprintf(stderr, "HandlerProc: I do nothing, Data is not complete %d <= %d!\n",
		communicationHandlerLength,communicationCount);
#endif
	}
    DBUG_VOID_RETURN;
     }

void
wafeSetCommunicationVariableCmd(varName,length,command)
_Xconst String   varName;
int      length;
String   command;
    {

    communicationHandlerVarname = varName;
    communicationHandlerLength = length;
    communicationHandlerCommand = command;

#ifdef TUNNELDEBUG
    fprintf(stderr, "Command: setting communicationHandlerLength = %d\n",
	    communicationHandlerLength);

    fprintf(stderr, "Command: communicationHandlerLength %d <= communicationCount(global): %d\n", 
	    communicationHandlerLength, communicationCount);
#endif

    if (communicationHandlerLength <= communicationCount)
	{ 
#ifdef TUNNELDEBUG
	fprintf(stderr, "Command: data available, executing <%s>\n", 
		communicationHandlerCommand); 
#endif
	if (!comBuffer)
	    Tcl_SetVar(wafeInterpreter, communicationHandlerVarname, 
		       wafe_EMPTY, TCL_GLOBAL_ONLY);
	else
	    {
	    (void)Tcl_UnsetVar(wafeInterpreter, communicationHandlerVarname, 0);
	    if (communicationHandlerLength < communicationCount) 
		{
		char saveChar;

		saveChar = comBuffer[communicationHandlerLength+1];
#ifdef TUNNELDEBUG
		fprintf(stderr, "Command: MORE THAN WE NEED! c=<%d>\n",
			(int)saveChar);
#endif
		comBuffer[communicationHandlerLength+1] = '\0';
		Tcl_SetVar(wafeInterpreter, communicationHandlerVarname, 
			   comBuffer, TCL_GLOBAL_ONLY);
		comBuffer[communicationHandlerLength+1] = saveChar;

		memcpy(comBuffer,&comBuffer[communicationHandlerLength],
		       communicationCount-communicationHandlerLength);
		communicationCount = communicationCount - 
		    communicationHandlerLength;
		} 
	    else 
		{
		Tcl_SetVar(wafeInterpreter, communicationHandlerVarname, 
			   comBuffer, TCL_GLOBAL_ONLY);
		communicationCount = 0;
		XtFree(comBuffer);
		comBuffer = NULL;
		}
	    }  

	(void) wafeEval(wafeInterpreter, communicationHandlerCommand, 
		 "setCommunicationVariable");
          
	XtFree(communicationHandlerVarname);
	XtFree(communicationHandlerCommand);
	communicationHandlerLength = 0;
	communicationHandlerCommand = NULL;
#ifdef TUNNELDEBUG
	fprintf(stderr, "Command: setting command and length to NULL\n"); 
#endif
	}
    else
	{
#ifdef TUNNELDEBUG
	fprintf(stderr, "Command: I do nothing, because Data is not yet complete!\n");
#endif
	}
    }


/*
@handlerType = (
	    "stdin", STDIN,
	    "stdout", STDOUT,
	    "stderr", STDERR,
	    "xioerr", XIOERR,
	    @signalType
		  );
*/

void 
wafeRegisterCmd(handlerType,string)
int handlerType;
_Xconst String string;
    {
    char **table;

    switch(handlerType)
	{
    case STDIN: 
	if (!stdinHandlerRegistered)
	    {
	    wafeWarn("register","No %s-handler in this mode!",
		     "input",NULL,NULL);
	    return;
	    }
	break;
	
    case STDOUT:
	if (!stdoutHandlerRegistered)
	    {
	    wafeWarn("register","No %s-handler in this mode!",
		     "output",NULL,NULL);
	    return;
	    }
	break;

    case STDERR:
	if (!stderrHandlerRegistered)
	    {
	    wafeWarn("register","No %s-handler in this mode!",
		     "error",NULL,NULL);
	    return;
	    }
	break;

    default:
	break;
	}

    if (handlerType>HANDLERMAX)
	{
	table = tclIntHandlers;
	handlerType -= HANDLERMAX;
        if (!strcmp("IGNORE",string)) 
	    signal(handlerType,SIG_IGN);
	else
        if (!strcmp("DEFAULT",string)) 
	    signal(handlerType,SIG_DFL);
	else
	    signal(handlerType,signalHandler);
	}
    else
	table = wafeHandlers;

    if (table[handlerType]) 
	XtFree(table[handlerType]);

    table[handlerType] = XtNewString(string);
    } 


void 
wafeUnregisterCmd(handlerType)
int handlerType;
    {    
    char **table;

    if (handlerType>HANDLERMAX)
	{
	table = tclIntHandlers;
	handlerType -= HANDLERMAX;
	if (!strcmp("IGNORE",tclIntHandlers[handlerType])) 
	    signal(handlerType,signalHandler);
	else
	if (!strcmp("DEFAULT",tclIntHandlers[handlerType])) 
	    signal(handlerType,signalHandler);
	}
    else
	table = wafeHandlers;

    if (!table[handlerType]) 
	wafeWarn("unregister","Warning, no handler registered!", 
		 NULL,NULL,NULL);
    else 
        XtFree(table[handlerType]);

    table[handlerType] = NULL;
    }


/*
 * extra input handlers, which are user maintained
 */
typedef struct
    {
    String varName;
    String command;
    XtPointer mask;
    XtInputId id;
    } extraInput;

static extraInput extraInputs[EXTRA_HANDLERS];

    
static void
extraHandlerProc(clientData, fid, xid)
XtPointer    clientData;
int         *fid;
XtInputId   *xid;
    {   
    extraInput *slot = (extraInput *)clientData;

    DBUG_ENTER("extraHandlerProc");
    WAFE_UNUSED(xid);

    if (slot->mask == (XtPointer)XtInputReadMask) 
	{
	int readBytes;	
	char buffer[EXTRA_BUF_SIZE];

	if ((readBytes = read(*fid, buffer, EXTRA_BUF_SIZE-1)) == -1)
	    perror("extraHandlerProc");

	if (!readBytes)
	    {
	    DBUG_PRINT("handler", ("EOF - condition on extra"));
	    DBUG_VOID_RETURN;
	    }

	if (slot->varName) 
	    {
	    *(buffer + readBytes) = '\0'; /* make it a valid string */
	    Tcl_SetVar(wafeInterpreter,  slot->varName, buffer, 
		       TCL_GLOBAL_ONLY|TCL_APPEND_VALUE);
	    }
	}
    if (slot->command)
	(void) wafeEval(wafeInterpreter, slot->command, "extraInput");

    DBUG_VOID_RETURN;
    }


XtInputId
wafeAddInputCmd(source,condition,varName,command)
int source;
XtPointer condition;
_Xconst String varName;
_Xconst String command;
    {
    int i;
    extraInput *slot = NULL;

    if ((stdinHandlerRegistered && (fileno(stdin) == source) ) 
	|| (!fileMode && wafeFromClient == source) 
	|| (stderrHandlerRegistered && wafeErrorClient == source) 
       )
	{
	wafeWarn("addInput","cannot add input handler!", NULL,NULL,NULL);
	return 0;
	}

    for (i=0; i<EXTRA_HANDLERS; i++) 
	{
	if (!(extraInputs[i].id))
	    {
	    slot = &extraInputs[i];
	    break;
	    }
	}
    if (slot) 
	{
	slot->varName = XtNewString(varName);
	slot->command = XtNewString(command);
	slot->mask = condition;
	if (condition == (XtPointer)(1L<<3))
	    condition = (XtPointer)XtInputReadMask;
	slot->id = XtAppAddInput(wafeAppContext, source, condition,
				 extraHandlerProc, (XtPointer)slot);
	return slot->id;
	}
    else 
	{
	wafeWarn("addInput","cannot allocate slot for input handler",
		 NULL,NULL,NULL);
	return 0;
	}
    }

void
wafeRemoveInputCmd(id)
XtInputId id;
    {
    int i;
    extraInput *slot = NULL;

    for (i=0; i<EXTRA_HANDLERS; i++) 
	{
	if (extraInputs[i].id == id)
	    {
	    slot = &extraInputs[i];
	    XtRemoveInput(id);
	    slot->id = 0;
	    XtFree(slot->varName);
	    XtFree(slot->command);
	    return;
	    }
	}
    wafeWarn("removeInput","no such input %lu",id, NULL,NULL);
    }

void
wafeProcessEvents() 
    {
#ifdef RDD
    rddAppMainLoop(wafeAppContext);
#else
    XtAppMainLoop(wafeAppContext);
#endif
    }

/* no const, strings passed to tcl for execution must be writable */
static char wafeErrorProc[] =
"proc wafeError {cmd errortype a b c d} {\n\
   if [catch {set msg [wafeHelp $cmd $errortype $a $b $c $d]}] {\n\
     if ![string compare $errortype argc] {\n\
       set msg \"$cmd: $a$b arguments expected, $c arguments received\"\n\
     } elseif ![string compare $errortype convert] {\n\
       set msg \"$cmd: could not convert argument $b '$a' to type $c\"\n\
     } else {\n\
       set msg \"$cmd: no component '$a' in array '$b'\"\n\
   } }\n\
   return $msg\
}";

void
wafeInit(argc, argv, inputMode, directMode) 
int    argc;
char **argv;
Boolean inputMode;
Boolean directMode;
    {
    int i;
    char *p;
    char intBuffer[INT_AS_STRING];
    char charBuffer[200];
    Boolean canAddStdinHandler = True;
    static char *defaultFileSearchPath = wafeDefaultFileSearchPath;
    extern void wafeInitializeGeneratedCode();

    wafeInterpreter = Tcl_CreateInterp();

    /* if nobody has provided values for the channels,
     * use stdin for commands and stdout for echo
     */
    if (wafeFromClient == -1) wafeFromClient = fileno(stdin);
    if (wafeToClient   == -1) wafeToClient   = fileno(stdout);

    /* initialize handlers */
    for (i=0; i<HANDLERMAX; wafeHandlers[i]=NULL, i++); 
    for (i=0; i<SIGMAX; tclIntHandlers[i]=NULL, i++); 

    /* setup signal handlers:
     * first, set a different process group 
     */
    if (wafeClientPid) setpgrp();

    /* For the following signals we install the wafe signal handler
     * SIGCLD (death of child signal) is only handled in frontend mode
     */
    signal(SIGTERM, signalHandler);  
    signal(SIGQUIT, signalHandler);
    signal(SIGINT,  signalHandler);
    signal(SIGHUP,  signalHandler);
    signal(SIGPIPE, signalHandler);
    signal(SIGUSR1, signalHandler);
    signal(SIGUSR2, signalHandler);

    if (wafeClientPid) signal(SIGCLD, sigcldHandler);

    /* set handler for x io errors */
    XSetIOErrorHandler(IOErrorHandler);

    XtAppSetWarningMsgHandler(wafeAppContext,warningMsg);
    
    /* the default widget tree is under topLevel */
    wafeAddToEndOfWidgetList(&wafeWidgetTrees, wafeTopLevel, "topLevel");

    XtAppAddActions(wafeAppContext, tclAction, 1);
    XtVaGetApplicationResources(wafeTopLevel, &app_data, resources,
				XtNumber(resources), NULL);

    /* add various Xt inputs depending on mode */
    if ( directMode || wafeClientPid )
	{
#ifdef _IBMR2
	if (wafeClientPid) 
	    {
            /* fprintf(stderr, "setting FNDELAY for %d\n",wafeFromClient); */
	    fcntl(wafeFromClient, F_SETFL, FNDELAY);
	    }
#endif
	XtAppAddInput(wafeAppContext, wafeFromClient, 
		      (XtPointer)XtInputReadMask,
		      getInput, (XtPointer)(int)(wafePromptChar != '\0'));
	canAddStdinHandler = (wafeFromClient != fileno(stdin));
	stdoutHandlerRegistered  = (wafePromptChar != '\0');
	}
    
    if (wafeClientPid)
	{ 
#ifdef _IBMR2
	/* fprintf(stderr, "setting FNDELAY for %d\n",wafeErrorClient); */
	fcntl(wafeErrorClient, F_SETFL, FNDELAY);
#endif
	XtAppAddInput(wafeAppContext, wafeErrorClient,
		      (XtPointer)XtInputReadMask, stderrHandlerProc, 
		      (XtPointer)NULL); 
	stderrHandlerRegistered = True;
	}

    if (wafeExtraCom)
	XtAppAddInput(wafeAppContext, wafeExtraCom, 
		      (XtPointer)XtInputReadMask,
		      communicationHandlerProc, NULL); 

    if (inputMode)
	if (canAddStdinHandler)
	    {
	    XtAppAddInput(wafeAppContext, fileno(stdin), 
			  (XtPointer)XtInputReadMask,
			  stdinHandlerProc, NULL); 
	    stdinHandlerRegistered = True;
	    }
	else
	    wafeFatal("io",
		      "Wafe: cannot add input handler in this mode combination",
		      NULL);
    

    if (*(app_data.init_com))
        {
	DBUG_PRINT("start", ("Wafe: Start cmd = {%s}\n", app_data.init_com));
	write(wafeToClient, app_data.init_com, strlen(app_data.init_com));
	write(wafeToClient, wafe_NL, 1);
	}

    wafeInitializeGeneratedCode();

#ifdef ITCL
    if (Itcl_Init(wafeInterpreter) == TCL_ERROR)
	wafeWarn("init","could not initialize [incr Tcl]",NULL,NULL,NULL);
#endif
    
    /* define global variables */
#ifdef PRER5
    Tcl_SetVar(wafeInterpreter, "XVERSION", "R4", TCL_GLOBAL_ONLY);
#else
    Tcl_SetVar(wafeInterpreter, "XVERSION", "R5", TCL_GLOBAL_ONLY);
#endif
    Tcl_SetVar(wafeInterpreter, "WAFEVERSION", VERSION, TCL_GLOBAL_ONLY);

    p = Tcl_Merge(argc, argv);
    Tcl_SetVar(wafeInterpreter, "ARGV", p, TCL_GLOBAL_ONLY);
    XtFree(p);

    sprintf(intBuffer, "%d", argc);
    Tcl_SetVar(wafeInterpreter, "ARGC",  intBuffer, TCL_GLOBAL_ONLY);

    sprintf(intBuffer, "%d", argc-1);
    Tcl_SetVar(wafeInterpreter, "argc",  intBuffer, TCL_GLOBAL_ONLY);

    p = Tcl_Merge(argc-1, argv+1);
    Tcl_SetVar(wafeInterpreter, "argv", p, TCL_GLOBAL_ONLY);
    XtFree(p);

    Tcl_SetVar(wafeInterpreter, "argv0", 
	       (fileMode) ? wafeScriptName : argv[0], TCL_GLOBAL_ONLY);
    Tcl_SetVar(wafeInterpreter, "Argv0", wafeAppClass, TCL_GLOBAL_ONLY);

    Tcl_SetVar(wafeInterpreter, "tcl_interactive",
	       wafeStrings[(int) (interactiveMode = 
		           (!frontendMode && directMode && isatty(0)))], 
	       TCL_GLOBAL_ONLY);

    sprintf(intBuffer, "%d", getpid());
    Tcl_SetVar(wafeInterpreter, "PID", intBuffer, TCL_GLOBAL_ONLY);

    sprintf(intBuffer, "%d", wafeClientPid);
    Tcl_SetVar(wafeInterpreter, "CHILDPID", intBuffer, TCL_GLOBAL_ONLY);

    if ((p = (char *)getenv("XFILESEARCHPATH")))
	wafeFileSearchPath = XtNewString(p);
    else
	/* it is nesscessary to make a dynamically allocated copy for link var
	 */
	wafeFileSearchPath = XtNewString(defaultFileSearchPath);

    Tcl_LinkVar(wafeInterpreter, "FILESEARCHPATH", 
		(char *)&wafeFileSearchPath,  TCL_LINK_STRING);

    Tcl_LinkVar(wafeInterpreter, "COLORCLOSENESS", 
		(char *)&wafeColorCloseness,  TCL_LINK_INT);

    wafeStringInit(tclCommand);
    Tcl_Init(wafeInterpreter);

    if (!(p = (char *)getenv("WAFELIB")))
        p = WAFELIB;
    Tcl_SetVar(wafeInterpreter, "WAFELIB", p, TCL_GLOBAL_ONLY);
   
    sprintf(charBuffer,"%s/tcllib",p);
    Tcl_SetVar(wafeInterpreter, "auto_path", charBuffer, 
	       TCL_LIST_ELEMENT|TCL_APPEND_VALUE);
    sprintf(charBuffer,"%s/tcllib/bc",p);
    Tcl_SetVar(wafeInterpreter, "auto_path", charBuffer, 
	       TCL_LIST_ELEMENT|TCL_APPEND_VALUE);
    Tcl_SetVar(wafeInterpreter, "auto_path", "./tcllib", 
	       TCL_LIST_ELEMENT|TCL_APPEND_VALUE);
    Tcl_SetVar(wafeInterpreter, "auto_path", "../tcllib", 
	       TCL_LIST_ELEMENT|TCL_APPEND_VALUE);
    (void) wafeEval(wafeInterpreter, wafeErrorProc,"errorHandler");

    }
