/*
 * Copyright (C) 1992 by Gustaf Neumann, Stefan Nusser
 *
 *      Wirtschaftsuniversitaet Wien,
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6,
 *      A-1090 Vienna, Austria
 *      neumann@wu-wien.ac.at, nusser@wu-wien.ac.at
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 *
 * Date: Mon, Apr 13 1992
 * Author: Stefan Nusser
 * Version: 0.9
 */

/*
 *                 WAFE.C 
 *
 *                 This file contains the main function as well as some
 *                 utility procedures, which are used in several places:
 *
 *                 getTypeOfAttribute
 *                 checkArgs4String
 *                 convert
 *                 name2Widget
 *                 convert
 *                 killCallbackProc
 *                 signalHandler
 *                 main
 */


#define MAIN

#include <stdio.h>
#include <sys/types.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/AsciiText.h>

#include <unistd.h>
#include <limits.h>
#include <signal.h>

#ifndef STREAMS
#include <sys/socket.h>
#else
#include <sys/stream.h>
#include <stropts.h>
#include <fcntl.h>
#endif

#include "wafe.h"                             
#include "Version.h"


typedef struct _STRINFO
        {
        char                *addr;
        Widget               widget;
        char                *attribute;
        struct _STRINFO     *next;
        } STRINFO, *STRINFOPTR;    


STRINFOPTR strHead = NULL;


/*
 * The following functions simplify error handling - they all are
 * invoked under certain circumstances and print an error message
 *
 */


void 
convError (tclCommand, argc, tclArg, type)
char * tclCommand;
char * argc;
char * tclArg;
char * type;
    {
    fprintf(stderr,"Wafe(%s): could not convert argument %s <%s> to type %s\n",
	    tclCommand, argc, tclArg, type);
    }


void
argcError (tclCommand, quant, tclArgs, argc)
char *tclCommand;
char *quant;
int   tclArgs;
int   argc;
    {
    fprintf(stderr,"Wafe(%s): %s%d arguments expected, %d arguments received\n",
            tclCommand, quant, tclArgs, argc - 1);
    }


void
noVarCompError (tclCommand, name, comp)
char *tclCommand;
char *name;
char *comp;
    {
    fprintf(stderr,"Wafe(%s): no component %s of structure %s\n",
            tclCommand, comp, name);
     }

/***********************************************************************************
  *  The following functions maintain the global lists of ResourceLists, one for 
  *  normal widgets and one for constraint resources. They look up the
  *  resources registered for a specified widget class and - if they are queried for
  *  the first time - append the new resourceList
  ***********************************************************************************/

Cardinal
getResOfClass(wClass, rList)
WidgetClass wClass;
XtResourceList  *rList;
     {
     Cardinal  num;
     RESINFOPTR  ptr;

     if (resourceList)  /* there are allready some entries in the resource list */
          for (ptr = resourceList; ptr != NULL; ptr = ptr->next) 
	        if (ptr->wClass == wClass)
		     {
                     *rList = ptr->resources;
                     return(ptr->num);
		     }

     XtInitializeWidgetClass(wClass);  /* Just in case it's not yet done... */

     if (resourceList) /* Search for last element in list */
          {
          for (ptr = resourceList; ptr->next != NULL; ptr = ptr->next);
          ptr->next = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
          ptr = ptr->next;
	  }
     else
          ptr = resourceList = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
        
     ptr->next = NULL;
     ptr->wClass = wClass;

     XtGetResourceList(wClass, &(ptr->resources), &(ptr->num));
     *rList = ptr->resources;
     return(ptr->num);
     } 

Cardinal
getConOfClass(wClass, rList)
WidgetClass wClass;
XtResourceList  *rList;
     {
     Cardinal  num;
     RESINFOPTR  ptr;

     if (constraintList)  /* there are allready some entries in the resource list */
          for (ptr = constraintList; ptr != NULL; ptr = ptr->next) 
	        if (ptr->wClass == wClass)
		     {
                     *rList = ptr->resources;
                     return(ptr->num);
		     }

     if (constraintList) /* Search for last element in list */
          {
          for (ptr = constraintList; ptr->next != NULL; ptr = ptr->next);
          ptr->next = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
          ptr = ptr->next;
	  }
     else
          ptr = constraintList = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
        
     ptr->next = NULL;
     ptr->wClass = wClass;

     XtGetConstraintResourceList(wClass, &(ptr->resources), &(ptr->num));
     *rList = ptr->resources;
     return(ptr->num);
     } 



/*****************************************************************************
 *    FUNCTION:  getTypeOfAttribute 
 *                                  
 *    Arguments: Widgetclass, parent widget and attribute to be queried.
 *    Returns:   Type of the attribute as a string.                     
 *                                                                      
 *    Used by    getValues, convert                                     
 *****************************************************************************/


char *
getTypeOfAttribute(wClass, parent, attribute)
WidgetClass     wClass;
Widget          parent;
char           *attribute;
     {
     XtResourceList    resList, conList, tsiList, tsoList; 
     int               i, count;

     count = getResOfClass(wClass, &resList);   /* Check standard resources of this class */

     for (i = 0; i < count; i++)
          if (!strcmp(attribute, resList[i].resource_name))
               return(resList[i].resource_type);
          
     if (XtIsConstraint(parent))         /* check Constraints if there are any... */
          {
          count = getConOfClass(XtClass(parent), &resList);

          for (i = 0; i < count; i++)
               if (!strcmp(attribute, resList[i].resource_name))
                    return(resList[i].resource_type);
         }

     if (wClass == asciiTextWidgetClass)      /* check asciiText's subresources  */
          {
          count = getResOfClass((asciiSinkObjectClass), &resList);
          for (i = 0; i < count; i++)
               if (!strcmp(attribute, resList[i].resource_name))
                    return(resList[i].resource_type);

          count = getResOfClass((asciiSrcObjectClass), &resList);
          for (i = 0; i < count; i++)
               if (!strcmp(attribute, resList[i].resource_name))
                    return(resList[i].resource_type);
	  }
     return(NULL); 
     }

     
/*****************************************************************************
 *    FUNCTION:  checkArgs4String                                       
 *               Keeps track of all the string resources, which are set  
 *               in order to free the allocated memory properly!         
 *                                                                       
 *    Arguments: Widget and the Arglist, which was previously created by    
 *               convert.                                                   
 *    Returns:   Nothing                                                    
 *                                                                          
 *    Used by:   setValues, tclWidgetCommand                                
 *    Uses:      getTypeOfAttribute                                         
 *****************************************************************************/


void
checkArgs4String(w, args, numArgs)
Widget      w;
ArgList     args;
int         numArgs;
     {
     STRINFOPTR    ptr;
     int           count;
     int           found = 0;

     DBUG_ENTER("checkArgs4String");

     for (count = 0; count < numArgs; count++)
          {
          if (!strcmp("String", getTypeOfAttribute(XtClass(w), ParentWidget(w), 
                                                  args[count].name)))
               {
               for (ptr = strHead; ptr != NULL; ptr = ptr->next)
                    {
                    if ((ptr->widget == w) && (!strcmp(ptr->attribute, args[count].name)))
                         {
                         found = 1;
                         XtFree(ptr->addr);
                         ptr->addr = (char *)args[count].value;                         
                         }
		    }

               if (!found)
		    {
                    if (strHead != NULL)
                         {
                         for (ptr = strHead; ptr->next != NULL; ptr = ptr->next);
                         ptr->next = (STRINFOPTR)XtMalloc(sizeof(STRINFO));
                         ptr = ptr->next;
                         }
                    else
                         ptr = strHead = (STRINFOPTR)XtMalloc(sizeof(STRINFO));

                    ptr->next = NULL;                              
                    ptr->widget = w;
                    ptr->attribute = XtNewString(args[count].name);
                    ptr->addr = (char *)args[count].value;                              
		    }
	      }
	 }
      DBUG_VOID_RETURN;
      }


/*****************************************************************************
 *    FUNCTION:  convert                                                    
 *               Takes attribute/value pairs, both of them strings, and     
 *               constructs a proper X11 Arglist by converting the string   
 *               values to the requested data type.                         
 *                                                                          
 *    Arguments: parent widget                                              
 *               widgetClass of widget                                      
 *               argv and argc (containing the attribute/value pairs)       
 *               A Pointer to an Arglist and an int for passing the result  
 *                                                                          
 *    Returns:   1 on success, 0 on error                                   
 *                                                                          
 *    Used by:   setValues, tclWidgetCommand                                
 *    Uses:      getTypeOfAttribute                                         
 *****************************************************************************/


int
convert(parent, wClass, argv, argc, args, numArgs)
Widget        parent;    
WidgetClass   wClass;    
char        **argv;      
int           argc;      
ArgList      *args;                /* RETURN */
int          *numArgs;             /* RETURN */
     {
     int               pairNum;  
     ArgList           argPtr;   
     XrmValue          input;    
     XrmValue          output;   
     char             *type;
     int               i;
     int               count = 0; 

     DBUG_ENTER("convert");

     if (argc == 0)
          {
          *numArgs = 0;
          *args =NULL;                 
          DBUG_RETURN (1);
	  }

     if (argc % 2)
          {
          fprintf(stderr, "Wafe(convert): Wrong # of args\n");
          DBUG_RETURN (0);
	  }
    
     argPtr = *args = (ArgList)XtMalloc((argc/2)*sizeof(Arg));

     for (pairNum = 1; pairNum <= argc/2; pairNum++)
          {
          type = getTypeOfAttribute(wClass, parent, *argv);

          if(type)
               { 
               DBUG_PRINT("conv", ("N: %s, T: %s", *argv, type)); 

               argPtr->name = *argv;         /* Den Namen uebernehmen.. */  
               input.addr =   *(++argv);     /* Den Wert konvertieren.. */
               input.size =   strlen(*argv)+1;
               output.addr =  NULL;

               DBUG_PRINT("conv", ("String %s, size %d\n", input.addr,
                                    input.size));

#ifndef PRER5
               if(XtConvertAndStore(parent, "String", &input, 
                                    type, &output))
#endif

#ifdef PRER5
               XtConvert(parent, "String", &input, type, &output);
               if (output.addr)
#endif
		    {
                    DBUG_PRINT("conv", ("Konv OK, Gr: %u\n", output.size));

                    if (!strcmp(type, "String"))
                         {
		         argPtr->value = (XtArgVal)XtNewString(output.addr);
                         }
                    else 
                    if (output.size == sizeof(long))
                         argPtr->value = (XtArgVal) *(long *)output.addr;
                    else
                    if (output.size == sizeof(short))
                         argPtr->value = (XtArgVal) *(short *)output.addr;
                    else
                    if (output.size == sizeof(char))
                         argPtr->value = (XtArgVal) *(char *)output.addr;
                    else
                    if (output.size == sizeof(XtArgVal))
                         argPtr->value = *(XtArgVal *)output.addr;              
               argPtr++; 
               count++; 
               argv++;
                    }
               else 
		    {
                    argv++;
		    }
               }
          else 
	       {
               fprintf(stderr, "Wafe(convert):Attribute unknown: %s\n", *argv);
               argv+=2; 
	       }
	  }     

     *numArgs = count;
     
     DBUG_RETURN (1); 
     }

 
/*****************************************************************************
 *    FUNCTION:  name2Widget                                                
 *                                                                          
 *    Arguments: string containing an instance's name                       
 *    Returns:   Widget on success, otherwise NULL                          
 *                                                                          
 *    Used by:   nearly all tcl-commands dealing with widgets               
 *****************************************************************************/


Widget
name2Widget(name)
char  *name;
     {
     Widget result;
     char  *longname;

     if (!name)
          return(NULL);

     
     if (!(result = XtNameToWidget(topLevel, name)))
          {
          longname = XtMalloc(strlen(name) + 2);
          *longname = '*';
          strcpy (longname + 1, name);

          DBUG_PRINT("name2widget", ("Trying: %s", longname));

          if (!(result = XtNameToWidget(topLevel, longname)))
                  {           
                  DBUG_PRINT("name2widget", ("Checking topLevel"));

                  if (!strcmp("topLevel", name))
                       result = topLevel;
                  else 
	               {
                       fprintf(stderr, "Wafe(name2Widget): %s not found!\n", name);
                       return (NULL);
	               }
	           }
          XtFree(longname);
          }
     return (result);
     }

/*****************************************************************************
 *    FUNCTION:  name2Sink, name2Source                                                
 *                                                                          
 *    Arguments: string containing an instance's name                       
 *    Returns:   source/sink widget of named text widget                       
 *                                                                          
 *    Used by:   some of the generated commands in XawGen as converter proc              
 *****************************************************************************/

Widget 
name2Sink(string)
char * string;
     {
     Widget localVar1, sink;

     if (!(localVar1 = name2Widget(string)))
          {
          convError("name2Sink","1",string,"Widget");
	  return(NULL);
          }

     XtVaGetValues(localVar1, XtNtextSink, &sink, NULL);
     return(sink);
    }

Widget 
name2Source(string)
char * string;
     {
     Widget localVar1, source;

     if (!(localVar1 = name2Widget(string)))
          {
          convError("name2Source","1",string,"Widget");
	  return(NULL);
          }

     XtVaGetValues(localVar1, XtNtextSource, &source, NULL);
     return(source);
    }

   
/*****************************************************************************
 *    FUNCTION:  signalHandler                                             
 *               If wafe receives a SIGINT, SIGTERM, SIDHUP or SIGQUIT     
 *               signal, it is forwarded to the application                
 *                                                                         
 *    Arguments: signal number                                             
 *    Returns:   nothing                                                   
 *                                                                         
 *    Used by:   main                                                      
 *****************************************************************************/

void
signalHandler(sig)
int   sig;
     {
     fprintf(stderr, 
            "Wafe(main): Signal %d received, forwarding it to application!\n", sig);
 
     kill(pid, sig);

     exit(0);
     }

/*****************************************************************************
 *    FUNCTION: iSocketPair
 *            Use socketpair on BSD or ioctl on Sys 5
 *                                                                          
 *    Arguments: array of two file descriptors                       
 *    Returns:  0 if success, otherwise -1                         
 *                                                                          
 *    Used by:  main
 *****************************************************************************/

#ifdef STREAMS
#define SPX_DEVICE "/dev/spx"
int
iSocketPair(fd)
int      fd[2];
	{
	struct strfdinsert  ins;
	queue_t                  *pointer;

	if ((fd[0] = open(SPX_DEVICE, O_RDWR)) < 0)
		return(-1);

	if ((fd[1] = open(SPX_DEVICE, O_RDWR)) < 0)
		{
		close(fd[0]);
		return(-1);
		}

	ins.ctlbuf.buf = (char *) &pointer;
	ins.ctlbuf.maxlen = sizeof(queue_t *);		
	ins.ctlbuf.len = sizeof(queue_t *);		

	ins.databuf.buf = (char *) 0;
	ins.databuf.maxlen = 0;
	ins.databuf.len = -1;

	ins.fildes = fd[1];
	ins.flags = 0;
	ins.offset = 0;

	if (ioctl(fd[0], I_FDINSERT, (char *) &ins) < 0)
		{
		close(fd[0]);
		close(fd[1]);
		return(-1);
		}

	return(0);
	}

#else

int
iSocketPair(fd)
int      fd[2];
	{
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) == -1)
		return(-1);
	else
		return(0);
	}

#endif


/*****************************************************************************
 *    FUNCTION:  MAIN                                
 *               Command Line parsing, starts Xt Application, registers     
 *               addditional tcl-commands  and according                   
 *               to mode establishes the socket connection or reads in a   
 *               file or reads from stdin.                                 
 *                                                                         
 *    Arguments: argc and argv                                             
 *    Returns:   nothing                                                   
 *                                                                         
 *****************************************************************************/

void
main(argc, argv, env)
int    argc;
char **argv;
char **env;
     {
     int                    i = 0;
     int                    promptMode = YES;
     int                    execMode = NO;
     int                    args = 0; 
     int                    cargc = 0;
     char                 **cargv;    
     char                 **cargvCount;
     char                   class[50];
     char                   file[50];
     char                  *charPtr;
     char                  *execString;
     char                 application[50];

     int                    sp[2];  
     int                    ep[2];  
     int                    cp[2];

     static XtActionsRec    tclAction[] = {
                            {"exec",  execActionProc},
                            };

     DBUG_ENTER ("main");
     DBUG_PROCESS (argv[0]);

     cargvCount = cargv = (char **)XtMalloc((argc * sizeof(char *)) + 1);

     /* In standard mode, the class will be derived from the application's name,
       * in file mode from the filename and in direct  mode it is defined to be Wafe
       */

     strcpy(class, argv[0]);

     /*
      * Argument parsing loop - build new argument list without wafe's 
      * own args
      */ 

     while (args < argc)
     {
     if ((argv[args][0] == '-') && (argv[args][1] == '-') && (strlen(argv[args]) > 2))
           {
                             /***  Debug - Mode  ***/

           if (argv[args][2] == 'D')
                {                             
                DBUG_PUSH (&(argv[args][3]));  
                args++; 
	        }                             
           else
                             /***  Prompt - Mode  ***/ 
 
           if (argv[args][2] == 'n')            /* Turn prompt mode off */
	        {                              
                promptMode = NO;               
                args++;
                }
           else
                             /***  Version  ***/ 

           if (argv[args][2] == 'v')            /* Print Version */
	        {                              
		fprintf(stderr, "\nThis is Wafe, version %s\n\n", VERSION);
                fprintf(stderr, "Copyright (C) 1992 by Gustaf Neumann, Stefan Nusser\n");
                fprintf(stderr, "University of Economics and Business Administration\n"); 
                fprintf(stderr, "Vienna, Austria\n\n");
                exit(0);
                }
           else
                             /***  Input - Mode  ***/

           if (argv[args][2] == 'i')            /* Enter input mode  */
	        {                            
                inputMode  = YES;            
                args++;
                }
           else
                             /***  Execute - Mode  ***/

           if (argv[args][2] == 'e')            /* Enter execute mode  */
	        {      
		int     comLength = 0;

		args++;
                execMode  = YES;     
		fileMode = YES;
		execString = argv[args];

		if (execString == NULL)
		      {
		      fprintf(stderr, "Wafe: Option +e needs an additional argument\n");
		      exit(-1);
		    }

                strcpy(class, "Wafe");
                }
           else
                             /***  Direct - Mode  ***/

           if (argv[args][2] == 'd')            /* Enter direct mode */
	        {                           
                directMode = YES;

                toClient = fileno(stdout);
                fromClient = fileno(stdin);
                errorClient = -1;

                strcpy(class, "Wafe");
                args++; 
	        }
           else
                             /*** Change Prompt Character  ***/

           if (argv[args][2] == 'c')          
	        {                             
                promptChar = argv[args][3];
                fprintf(stderr, "PromptChar changed to %c\n", promptChar);
                args++;
	        }
           else
                             /***  File - Mode  ***/

           if (argv[args][2] == 'f')        /* Enter file mode */
                {
                if (argv[args][3] == 'i')    /* For convenience: --fi == --f --i */
		     {
                     inputMode = YES;
		     }

                fileMode = YES;
                strcpy(file, argv[args+1]);   /* Next Arg = File to execute */
                strcpy(class, file);

                args +=2;
                }
           else
                {
                fprintf(stderr, "Wafe(main): Unknown Argument, ignoring!\n");
                args++; 
                }
	   }
     else 
           {
           *cargvCount = XtNewString(argv[args]);
           cargc++;
           args++;
           cargvCount++;
           }
     }

     *cargvCount = NULL;

    /* DEBUG - Check that everybody has got his args */

     DBUG_PRINT("start", ("Arguments at startup (argc = %d):", argc));
     for (i = 0; argv[i] != NULL; i++)
          DBUG_PRINT("start",("argv[%d] = %s\n", i, argv[i]));

     DBUG_PRINT("start", ("Xt's + app's args (cargc = %d):", cargc));
     for (i = 0; cargv[i] != NULL; i++)
          DBUG_PRINT("start",("cargv[%d] = %s\n", i, cargv[i]));

     /*
       * If the program's name contains a path specification cut it off...
       */

     if (charPtr = strrchr(class, '/'))
          strcpy(class, ++charPtr);

   /*
     *  We will need this information when we execute the application process ...
     */

     strcpy(application, class); 

     /*
       * If the name contains a dot, which is not allowed in Xt's class names, substitute it
       * with a dash (keep this in mind, it will possibly affect the name of the AD file!
       */

     while(charPtr = strrchr(class, '.'))
          *charPtr = '-';
         
     /* Just because argv[0] will appear in the titlebar of the 
      * top level window - if we were in file mode the title would
      * always be wafe (which is probably not desired).
      */


     cargv[0] = XtNewString(class);

    /* 
      *  Now create class: Capitalized X, second letter capitalized, 
      *  rest remains unchanged. Under this name, the app-defaults file
      *  will be searched! 
      */

     if ((class[0] == 'x') || (class[0] == 'X'))
	  {
          class[0] = 'X';
          class[1] = toupper(class[1]);
	  }
     else
          class[0] =toupper(class[0]);

     /* Must initialize here, so that Xt can strip off its arguments */

     topLevel = XtVaAppInitialize(&appContext, class, NULL, 0, &cargc, cargv, NULL, NULL); 

     DBUG_PRINT("start", ("App's args (cargc = %d):", cargc)); 

     for(i = 0; cargv[i] != NULL; i++)
          DBUG_PRINT("start",("cargv[%d] = %s\n", i, cargv[i])); 


     if ((!directMode) && (!fileMode))     /* => standard mode            */
          {     

	 /*
          *   Main socketpair, will be connected to the client's stdin
          *   and stdout channels.
          */

          if (iSocketPair(sp) == -1)
               {
               fprintf(stderr, "Wafe(main): Couldn't create socket, aborting\n");
               exit(-1);
	       }

	 /*
          *   This pair is used to facilitate an additional communication channel.
          */

          if (iSocketPair(cp) == -1)
               {
               fprintf(stderr, "Wafe(main): Couldn't create socket, aborting\n");
               exit(-1);
	       }

          comClient = cp[0];  /* Communication endpoint for client  */
          comWafe = cp[1];    /* Communication endpoint for wafe    */


	 /*
          *  Used to catch client's error messages - wafe will never write on
          *  this channel!
          */

          if (iSocketPair(ep) == -1)
               {
               fprintf(stderr, "Wafe(main): Couldn't create socket for app's stderr\n");
               exit(-1);
	       }

          pid = fork();
    
          if (!pid)                       /* Child-process               */
               {
               dup2(sp[0], fileno(stdout));
               dup2(sp[0], fileno(stdin));
               dup2(ep[0], fileno(stderr));

               close(ep[0]);              /* Those 2 are for ..          */
               close(ep[1]);              /* getting stderr-messages.    */

               close(sp[1]);              /* For regular communication ..*/
               close(sp[0]);              /* via stdin and stdout!       */

               if ((*application != 'x') && (*application != 'X'))
		    {
                    fprintf(stderr, "Wafe(main):Wrong File was started, aborting\n");
                    exit(-1);
		  }

               cargv[0] = application+1;

               execvp(cargv[0], cargv);   /* execute the client-program  */  
   
               fprintf(stderr, "Wafe(main): Error in exec-call to <%s>, aborting\n",
                       cargv[0]);  
               exit(-1);
               }

          if (pid == -1)
               {
               fprintf(stderr, "Wafe(main): Couldn't fork!\n");
               DBUG_VOID_RETURN;
	       }

          close(sp[0]);
          close(ep[0]);

          /* Those are the four main communication-channels:
           *
           * 1) errorClient   is the client's diagnostic output,
           * 2) toClient      leads to client's stdin and
           * 3) fromClient    is client's stdout 
           * 4) wafeCom       additional communication channel
           */

          errorClient = ep[1];
          toClient = sp[1];
          fromClient = sp[1];

          /* If one of the following signals is caught, wafe exits and 
           * the same signal is sent to the application
           */

          signal(SIGTERM, signalHandler);  
          signal(SIGQUIT, signalHandler);
          signal(SIGINT, signalHandler);
          signal(SIGHUP, signalHandler);

          }

     /* Now check for invalid combination of modes */

     if ((execMode && inputMode) || (execMode && directMode))
         {
	 fprintf(stderr, "Wafe: Invalid combination of modes\n");
	 exit(-1);
	 }

     if ((directMode) && (inputMode))
          {
          fprintf(stderr, "Wafe: Invalid combination direct-mode and input-mode, aborting\n");
          exit(-1);
	  }

     if ((directMode) && (fileMode))
          {
          fprintf(stderr, "Wafe: Invalid direct-mode and file-mode, aborting\n");
          exit(-1);
	  }

    /* Now register input-handlers according to mode */

     if (!fileMode)
          {
          XtAppAddInput(appContext, fromClient, (XtPointer)XtInputReadMask,
                        getInput, (XtPointer)promptMode); 

          if (!directMode)
	       { 
               XtAppAddInput(appContext, errorClient,
                       (XtPointer)XtInputReadMask, stderrHandlerProc, 
                       (XtPointer)NULL); 

               XtAppAddInput(appContext, comWafe, (XtPointer)XtInputReadMask,
                             communicationHandlerProc, NULL); 
	       }

	  }
          
     if ((inputMode) && (!execMode))
          XtAppAddInput(appContext, fileno(stdin), (XtPointer)XtInputReadMask,
                        stdinHandlerProc, NULL); 

     XtAppAddActions(appContext, tclAction, 1);

     XtVaGetApplicationResources(topLevel, &app_data, resources,
                                 XtNumber(resources), NULL);

     interpreter = Tcl_CreateInterp();

     Tcl_SetVar(interpreter, "WAFEVERSION", VERSION, TCL_GLOBAL_ONLY);
#ifdef PRER5
     Tcl_SetVar(interpreter, "XVERSION", "R4", TCL_GLOBAL_ONLY);
#else
     Tcl_SetVar(interpreter, "XVERSION", "R5", TCL_GLOBAL_ONLY);
#endif

     /* This procedure registers all widget tcl-commands (compare 
      * to athenaW.c) and initializes all widget classes (this
      * is necessary, so that i can register my own type converters
      * afterwards without them being overwritten.
      */

     createR4WidgetCommands();

#ifndef PRER5
     createR5WidgetCommands();
#else
     createR5WarningCommands();
#endif


     /************************************************************/
     /* Now the additional tcl-commands are registered           */
     /************************************************************/

     Tcl_CreateCommand(interpreter, "mergeResources",     
                                                   com_mergeResources, NULL,NULL);
     Tcl_CreateCommand(interpreter, "setValues",   com_setValues, NULL, NULL);
     Tcl_CreateCommand(interpreter, "getValues",   com_getValues, NULL, NULL);
     Tcl_CreateCommand(interpreter, "sV",          com_setValues, NULL, NULL);
     Tcl_CreateCommand(interpreter, "gV",          com_getValues, NULL, NULL);
     Tcl_CreateCommand(interpreter, "callback",    com_callback, NULL, NULL);
     Tcl_CreateCommand(interpreter, "getResourceList",
                                                   com_getResourceList, NULL, NULL);
     Tcl_CreateCommand(interpreter, "callCallbacks",
                                                   com_callCallbacks, NULL, NULL);
     Tcl_CreateCommand(interpreter, "callActionProc",
                                                com_callActionProc, NULL, NULL);
     Tcl_CreateCommand(interpreter, "listChange",  com_listChange, NULL, NULL);
     Tcl_CreateCommand(interpreter, "listAppend",  com_listAppend, NULL, NULL);
     Tcl_CreateCommand(interpreter, "setCommunicationVariable", 
                                     com_setCommunicationVariable, NULL, NULL);
     Tcl_CreateCommand(interpreter, "asciiTextAppend",
                                                com_asciiTextAppend, NULL, NULL);
     Tcl_CreateCommand(interpreter, "getChannel",  com_getChannel, NULL, NULL);
     Tcl_CreateCommand(interpreter, "stderrHandler",
                                                   com_stderrHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "unregisterStderrHandler",
                                        com_unregisterStderrHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "stdoutHandler",
                                                 com_stdoutHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "unregisterStdoutHandler",
                                         com_unregisterStdoutHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "stdinHandler",
                                                   com_stdinHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "unregisterStdinHandler",
                                         com_unregisterStdinHandler, NULL, NULL);
     Tcl_CreateCommand(interpreter, "addTimeOut", com_addTimeOut, NULL, NULL);

     Tcl_CreateCommand(interpreter, "ownSelection",
                                                   com_ownSelection, NULL, NULL);
     /*
      * The following two include files contain the prototypings as well as the
      * Tcl_createCommand statements for the automatically gernerated functions
      */

#include "XawGen.tcc"
#include "XtGen.tcc"

#ifdef XPM
     XtAddConverter("String", "Pixmap", CvtStringToPixmapOrBitmap, NULL, 0);
     XtAddConverter("String", "Bitmap", CvtStringToPixmapOrBitmap, NULL, 0);

     Tcl_CreateCommand(interpreter, "changePixmap",
                                                   com_changePixmap, NULL, NULL);
     Tcl_CreateCommand(interpreter, "setIconPixmap",   com_setIconPixmap, NULL, NULL);
#endif

     XtAddConverter("String", "Widget", CvtStringToWidget, NULL, 0);

     if (execMode)
          {
          if(Tcl_Eval(interpreter, execString, 0, NULL) == TCL_ERROR)
	        fprintf(stderr, "Wafe(execute mode): %s\n", interpreter->result);
	  }

     if ((fileMode) && (!execMode))
          {
          char    intBuffer[20];

          Tcl_SetVar(interpreter, "ARGV", Tcl_Merge(cargc, cargv), TCL_GLOBAL_ONLY);

          sprintf(intBuffer, "%d", cargc);
          Tcl_SetVar(interpreter, "ARGC",  intBuffer, TCL_GLOBAL_ONLY);

          for (i = 0; env[i]  != NULL; i++);
          Tcl_SetVar(interpreter, "ENV", Tcl_Merge(i, env), TCL_GLOBAL_ONLY);

          if (Tcl_EvalFile(interpreter, file) == TCL_ERROR)
               {
               fprintf(stderr, "Wafe(file evaluation): %s\n", interpreter->result);
               exit(-1);
               }
          } 

     if (strcmp(app_data.init_com, "noCom"))
          {
          fprintf(stderr, "Wafe: Start mit %s\n", app_data.init_com);
          SEND(app_data.init_com);
	  }

     XtAppMainLoop(appContext);
 
     DBUG_VOID_RETURN;
     }








