/*
** RasterImport module for importing TIFF images into Xew internal
** raster format.
**
** Copyright 1992-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if SYSV_INCLUDES
#	include <memory.h>
#	include <malloc.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#else
char *malloc();
void free();
#endif
#endif
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/Xmd.h>
#include <X11/StringDefs.h>
#include <X11/Xew/RasterP.h>
#include "ImageTools.h"
#include "RasterTIFF.h"

#ifndef NO_TIFF
#	include "tiffio.h"
#	ifndef SEEK_SET
#		if ANSI_INCLUDES || SYSV_INCLUDES
#			include <unistd.h>
#		else
#			define SEEK_SET 0
#			define SEEK_CUR 1
#			define SEEK_END 2
#		endif
#	endif
#endif

static void RasterImportWarning(w, msg)
XeRasterWidget w;
char *msg;
    {
	XeWidgetWarningMsg
		((Widget)w, "rasterImportWarning", msg, (String *)NULL, 0);
    }


#ifndef NO_TIFF

/*
** AdjustLine
**	Widen the (bitspersample < 8) scanlines up to 8 bits per sample.
**	Assume this is not time critical at the moment and do it hard way.
**
**	-- Also, for now.. support only values 2 and 4.
*/
static void AdjustLine(line, bits, width)
unsigned char *line;
int bits, width;
    {
	unsigned char x, *in, *out;

	switch (bits)
	    {
	    default:
		break;
	    case 2:
		width += ((4-width) & 3); /* Make width disible by 4 */
		for (in = &line[width/4], out = &line[width]; out > line;)
		    {
			x = *--in;
			*--out = x & 0x3;
			*--out = (x>>2) & 0x3;
			*--out = (x>>4) & 0x3;
			*--out = (x>>6) & 0x3;
		    }
	    case 4:
		width += (width & 1); /* Make width even */
		for (in = &line[width/2], out = &line[width]; out > line;)
		    {
			x = *--in;
			*--out = x & 0xf;
			*--out = (x>>4) & 0xf;
		    }
		break;
	    case 16:
		for (in = out = line; --width >= 0; in +=2)
			*out++ = (*(INT16 *)in + 128) >> 8;
		break;
	    }
    }

/*
** Support routines for reading TIFF directly from a memory buffer
*/
typedef struct TiffInMemory
    {
	unsigned long length;	/* Total length of the content */
	unsigned long position;	/* Current Position in the "file" */
	char *data;		/* Address of the data */
    } TiffInMemory;

static int ReadProc(handle, buf, size)
void *handle;
char *buf;
unsigned long size;
    {
	TiffInMemory *t = (TiffInMemory *)handle;
	long remaining = t->length - t->position;
	
	if (remaining <= 0)
		return 0;
	if (size > remaining)
		size = remaining;
	memcpy(buf, t->data + t->position, (int)size);
	t->position += size;
	return size;
    }

static int WriteProc(handle, buf, size)
void *handle;
char *buf;
unsigned long size;
    {
	return -1;
    }

static long SeekProc(handle, off, whence)
void *handle;
long off;
int whence;
    {
	TiffInMemory *t = (TiffInMemory *)handle;
	switch (whence)
	    {
	    case SEEK_CUR:
		off += t->position;
		break;
	    case SEEK_END:
		off += t->length;
		break;
	    default:
		break;
	    }
	t->position = off;
	return off;
    }

static int CloseProc(handle)
void *handle;
    {
	return 0;
    }

static long SizeProc(handle)
void *handle;
    {
	return ((TiffInMemory *)handle)->length;
    }

static int MapProc(handle, pbase, psize)
void *handle;
char **pbase;
long *psize;
    {
	*pbase = ((TiffInMemory *)handle)->data;
	*psize = ((TiffInMemory *)handle)->length;
	return 1;
    }

static void UnmapProc(handle, base, size)
void *handle;
char *base;
long size;
    {
	/* Nothing to do here */
    }

static XeRawImage *LoadTIFF(w, in)
XeRasterWidget w;
TIFF *in;
    {
	unsigned long width, height;
	unsigned short *red, *green, *blue;
	unsigned short samplesperpixel, bitspersample, planarconfig;
	unsigned short photometric, orientation;
	float resX, resY;
	unsigned short resU;
	register unsigned long row;
	unsigned char *data;
	int i, linesize, planes;
	XeRawImage *raw = NULL;

	TIFFGetFieldDefaulted(in, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel);
	TIFFGetFieldDefaulted(in, TIFFTAG_BITSPERSAMPLE, &bitspersample);
	raw = _XeCreateRawImage(samplesperpixel);
	raw->samples_per_pixel = samplesperpixel;
	raw->bits_per_sample = bitspersample;
	if (bitspersample == 1)
		raw->bits_per_component = 1; /* Leave packed tight */
	else
		raw->bits_per_component = 8; /* Rest will be expanded to 8! */
	/*
	** Load Colormap if defined, regardless whether photometric
	** indicates it is needed or not.
	*/
	if (TIFFGetField(in, TIFFTAG_COLORMAP, &red, &green, &blue))
	    {
		int mapsize = (1<<bitspersample);
		unsigned char *r, *g, *b;
		int x;

		raw->alloc_map = True;
		raw->color_map = (unsigned char *)XtMalloc(3 * mapsize);
		/*
		** Convert 16-bit colormap to 8-bit colormap
		*/
		r = raw->color_map;
		g = r + mapsize;
		b = g + mapsize;
		for (x = mapsize; --x >= 0; )
		    {
			r[x] = (red[x] * 255) / 65535;
			g[x] = (green[x] * 255) / 65535;
			b[x] = (blue[x] * 255) / 65535;
		    }
	    }
	if (TIFFGetField(in, TIFFTAG_PHOTOMETRIC, &photometric) == 0)
	    {
		/*
		** Missing PHOTOMETRIC.. faulty TIFF. Make some
		** guess so that at least something will be shown.
		*/
		RasterImportWarning(w, "PHOTOMETRIC tag missing from TIFF");
		if (samplesperpixel > 1)
			photometric = PHOTOMETRIC_RGB;
		else if (raw->color_map != NULL)
			photometric = PHOTOMETRIC_PALETTE;
		else
			photometric = PHOTOMETRIC_MINISBLACK;
			
	    }
	TIFFGetFieldDefaulted(in, TIFFTAG_ORIENTATION, &orientation);
	if (orientation != ORIENTATION_TOPLEFT)
		XeWidgetWarningMsg
			((Widget)w,
			 "rasterImportWarning",
			 "Warning orientation=%d",	/* uhh.. --msa */
			 (String *)&orientation,	/* ..... --msa */
			 1);
	TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &width);
	raw->width = width;
	TIFFGetField(in, TIFFTAG_IMAGELENGTH, &height);
	raw->height = height;
	if (TIFFGetField(in, TIFFTAG_XRESOLUTION, &resX))
		raw->x_resolution = resX;
	else
		raw->x_resolution = 1.0;
	if (TIFFGetField(in, TIFFTAG_YRESOLUTION, &resY))
		raw->y_resolution = resY;
	else
		raw->y_resolution = 1.0;
	TIFFGetFieldDefaulted(in, TIFFTAG_RESOLUTIONUNIT, &resU);
	raw->resolution_unit =
		resU == RESUNIT_NONE ? XeImageUnit_NONE :
		resU == RESUNIT_INCH ? XeImageUnit_INCH :
		resU == RESUNIT_CENTIMETER ? XeImageUnit_CM : XeImageUnit_NONE;
	raw->bytes_per_line = linesize = TIFFScanlineSize(in);
	if (linesize < raw->width && bitspersample != 1)
		raw->bytes_per_line = raw->width;
	raw->alloc_data = True;
	TIFFGetField(in, TIFFTAG_PLANARCONFIG, &planarconfig);
	if (raw->num_channels > 1 && planarconfig == PLANARCONFIG_SEPARATE)
	    {
		raw->data = data = (unsigned char *)
			malloc(raw->num_channels*raw->bytes_per_line*height);
		for (i = 0; i < raw->num_channels; ++i)
		    {
			raw->channel[i].addr =
				raw->data + i * raw->bytes_per_line;
			raw->channel[i].w = raw->width;
			raw->channel[i].h = raw->height;
			raw->channel[i].inc = 1;
			raw->channel[i].line =
				raw->num_channels * raw->bytes_per_line;
		    }
		planes = raw->num_channels;
	    }
	else
	    {
		raw->data = data = (unsigned char *)
			malloc(raw->bytes_per_line * height);
		for (i = 0; i < raw->num_channels; ++i)
		    {
			raw->channel[i].addr = raw->data + i;
			raw->channel[i].w = raw->width;
			raw->channel[i].h = raw->height;
			raw->channel[i].inc = raw->num_channels;
			raw->channel[i].line = raw->bytes_per_line;
		    }
		planes = 1;
	    }
	if (data != NULL)
	    {
		for (row = 0; row < height; row++)
			for (i = 0; i < planes; ++i, data+=raw->bytes_per_line)
				if (TIFFReadScanline(in, data, row, i) < 0)
					goto bad_file;
				else if (bitspersample!=8 && bitspersample!=1)
					AdjustLine(data, bitspersample, width);
	    bad_file:
		if (photometric == PHOTOMETRIC_MINISWHITE)
		    {
			/*
			** If 0 = White, we have to invert the image
			*/
			raw->invert_image = True;
		    }
	    }
	switch (photometric)
	    {
	    case PHOTOMETRIC_MINISWHITE:
	    case PHOTOMETRIC_MINISBLACK:
		if (bitspersample == 1)
			raw->class = XeImageClass_BILEVEL;
		else
			raw->class = XeImageClass_GRAYSCALE;
		break;
	    case PHOTOMETRIC_YCBCR:
		raw->color_space = XeImageSpace_YCbCr;
		raw->class = XeImageClass_FULLCOLOR;
		break;
	    case PHOTOMETRIC_RGB:
		raw->color_space = XeImageSpace_RGB;
		raw->class = XeImageClass_FULLCOLOR;
		break;
	    case PHOTOMETRIC_PALETTE:
		raw->color_space = XeImageSpace_RGB;
		raw->class = XeImageClass_PALETTE;
		break;
	    default:
		raw->class = XeImageClass_UNKNOWN;
		break;
	    }
	if (raw->data == NULL)
	    {
		RasterImportWarning(w, "Out of memory at TIFF import");
		_XeDestroyRawImage(raw);
		raw = NULL;
	    }
	return raw;
    }
#endif

XeRawImage *XeImport_TIFF(w)
XeRasterWidget w;
    {
#ifdef NO_TIFF
	RasterImportWarning(w, "TIFF Support not compiled.");
	return NULL;
#else
	TIFF *in;
	XeRawImage *raw;
	XeDataContent content;
	TiffInMemory handle;

	_XeOpenContent((XeBasicWidget)w, &content);
	if (content.type)
		in = TIFFFdOpen(fileno(content.source.stream),
				w->basic.content_file, "r");
	else
	    {
		handle.data = content.source.string;
		handle.length = content.length;
		handle.position = 0;
		in = TIFFClientOpen
			("in-memory", "r", (void *)&handle,
			 ReadProc, WriteProc, SeekProc, CloseProc,
			 SizeProc, MapProc, UnmapProc);
	    }
	if (in)
	    {
		raw = LoadTIFF(w, in);
		TIFFClose(in);
	    }
	else
	    {
		RasterImportWarning(w, "Could not open TIFF");
		raw = NULL;
	    }
	_XeCloseContent(&content);
	return raw;
#endif
    }
