/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 */

/*
 * Shadow.c created by Andrew Lister (30 October, 1995)
 */

#include <Xm/Xm.h>
#include <Xm/XmP.h>
#if XmVersion > 1001
#include <Xm/DrawP.h>
#endif
#include <Xbae/MatrixP.h>
#include <Xbae/Shadow.h>
#include <Xbae/Draw.h>
#include <Xbae/Utils.h>

static void DrawRowShadows P(( XbaeMatrixWidget mw,
				   unsigned int redraw_reason ));
static void DrawColumnShadows P(( XbaeMatrixWidget mw,
			       unsigned int redraw_reason ));

void
xbaeDrawColumnShadow(mw, column, redraw_reason)
XbaeMatrixWidget mw;
int column;
unsigned int redraw_reason;
{
    int x, y;
    Dimension width, height;
    Dimension adjust = 2 * mw->matrix.cell_shadow_thickness;
    unsigned int clip_reason, save_clip;
    unsigned char shadow;
    Window win = XtWindow(mw);
    Display *display = XtDisplay(mw);
    Boolean set_mask = False, need_fill;
    GC hiclipgc, higc;

    if ((0 != mw->matrix.disable_redisplay) || (0 == mw->matrix.columns))
	return;

    /*
     * Save current clip mask
     */
    save_clip = mw->matrix.current_clip;

    /*
     * Determine which gc to use for highlighting.
     * If we are unhighlighting, use the drawing gc's
     * otherwise use the highlght gc
     */

    if (0xF0 & mw->matrix.highlight_location)
    {
	Pixel fg, bg;

	xbaeComputeCellColors(mw, mw->matrix.rows-1, column, &fg, &bg);

	hiclipgc = mw->matrix.draw_clip_gc;
	higc = mw->matrix.draw_gc;

	/*
	 * Set background to last cell in row
	 */
	XSetForeground(XtDisplay(mw), higc, bg);
	XSetForeground(XtDisplay(mw), hiclipgc, bg);
    }
    else
    {
	hiclipgc = mw->matrix.highlight_clip_gc;
	higc = mw->manager.highlight_GC;
    }
   
    /*
     * If our overlap adjustment isn't set, set it
     */
    if (! adjust)
	adjust = 4;
    
    /*
     * Get the shadow type. If we have a specific column shadow,
     * use that otherwise use the cell's shadow type
     */
    shadow = mw->matrix.column_shadow_types ?
	mw->matrix.column_shadow_types[column] : mw->matrix.cell_shadow_type;

    /*
     * Get the width and height for drawing the shadow around each
     * column. The height used is a bit long to prevent any overlapping
     * at the edges if there are any (trailing) fixed rows, and
     * the actual drawing position is offset the same amount for the
     * the same reason. If we need to fill the available space, set
     * height appropriately.
     */
    width = COLUMN_WIDTH(mw, column);
    height = CELL_TOTAL_HEIGHT(mw) +
	(mw->matrix.fixed_rows * adjust) +
	(mw->matrix.trailing_fixed_rows * adjust);    

    if ( ( need_fill = NEED_VERT_FILL( mw ) ) )
    {
	Dimension th = height + mw->matrix.cell_shadow_thickness;
	Dimension mv = MATRIX_VERT_VISIBLE_SPACE(mw);
	if (th > mv)	    /* picky, picky */
	    height = mv;
	else
	    height = mw->core.height - mw->manager.shadow_thickness;
    }
    
    /*
     * Get the location at which to draw our shadow
     */
    xbaeRowColToXY(mw, mw->matrix.fixed_rows, column, &x, &y);
    y -= (mw->matrix.fixed_rows * adjust);

    /*
     * Draw the shadow around the portion of the column in
     * the non-fixed rows
     */
    if ((column < mw->matrix.fixed_columns) ||
	(column >= TRAILING_HORIZ_ORIGIN(mw)))
    {
	/*
	 * In (trailing)fixed columns
	 */
	Dimension column_height, visible_height;

	/*
	 * If only redrawing the portion of the shadow in the
	 * non-fixed rows, then leave
	 */
	if (GRID_REDRAW_SCROLL_HORIZ & redraw_reason)
	    return;

	/*
	 * In this case, we've got to draw the entire shadow at once
	 * and don't get the benefit of the clip window enabling
	 * clipping to occur :(. So, we have to do a bit of extra work
	 * to get the proper visual to occur.
	 */
	if (column > mw->matrix.fixed_columns)
	    clip_reason = CLIP_TRAILING_FIXED_COLUMNS;
	else
	    clip_reason = CLIP_FIXED_COLUMNS;
	xbaeSetClipMask(mw, clip_reason | CLIP_VISIBLE_HEIGHT);
	set_mask = True;

	visible_height = ClipChild(mw)->core.height +
	    FIXED_ROW_HEIGHT(mw) + TRAILING_FIXED_ROW_HEIGHT(mw);
	
	if ((mw->matrix.fixed_rows && mw->matrix.trailing_fixed_rows) ||
	    ((! mw->matrix.fixed_rows) && mw->matrix.trailing_fixed_rows &&
	     (0 == VERT_ORIGIN(mw))) ||
	    (mw->matrix.fixed_rows && (! mw->matrix.trailing_fixed_rows) &&
	     ((mw->matrix.rows - (visible_height/ROW_HEIGHT(mw))) ==
	      VERT_ORIGIN(mw))))
	{
	    /*
	     * If we've got both fixed and trailing fixed rows,
	     * or we've got only trailing fixed rows and the top
	     * row is the first row, or we've got only fixed rows
	     * and the bottom row is the last row, then both the
	     * top and bottom edges need to be displayed
	     */
	    y = ROW_LABEL_OFFSET(mw);
	    if ( need_fill )
		column_height = mw->core.height - HORIZ_SB_SPACE(mw) -
		    mw->manager.shadow_thickness - y;
	    else
		column_height = visible_height;
	}
	else
	{
	    /*
	     * If we have fixed rows, we need to always display the top edge.
	     */
	    if (mw->matrix.fixed_rows)
		y = ROW_LABEL_OFFSET(mw);
	    else
		y -= FIXED_ROW_HEIGHT(mw) -
		    (mw->matrix.fixed_rows * adjust);

	    if ( need_fill )
		column_height = mw->core.height - HORIZ_SB_SPACE(mw) -
		    mw->manager.shadow_thickness - y;
	    else
		column_height = height + FIXED_ROW_HEIGHT(mw) +
		    TRAILING_FIXED_ROW_HEIGHT(mw) -
		    (mw->matrix.fixed_rows * adjust);
	}

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness, x, y,
			width, column_height, shadow);

	if (mw->matrix.highlighted_cells &&
	    ((HighlightColumn & mw->matrix.highlighted_cells[0][column]) ||
	     (0xF0 & mw->matrix.highlight_location)))
	{
	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc,
			      x + mw->matrix.cell_shadow_thickness,
			      y + mw->matrix.cell_shadow_thickness,
			      width - 2*mw->matrix.cell_shadow_thickness,
			      column_height - 2 *
			        mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}
    }
    else
    {
	/*
	 * Non fixed columns
	 */
	
	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, XtWindow(ClipChild(mw)),
			mw->manager.top_shadow_GC,
			mw->manager.bottom_shadow_GC,
			mw->matrix.cell_shadow_thickness, x, y,
			width, height, shadow);

	if (mw->matrix.highlighted_cells && (
	    (HighlightColumn & mw->matrix.highlighted_cells[0][column]) ||
	    (0xF0 & mw->matrix.highlight_location)))
	{
	    if (mw->matrix.fixed_rows)
		y -= mw->matrix.cell_shadow_thickness;
	    else
		y += mw->matrix.cell_shadow_thickness;

	    if (mw->matrix.trailing_fixed_rows)
		height += mw->matrix.cell_shadow_thickness;
	    else if (0 == mw->matrix.fixed_rows)
		height -= 2 * mw->matrix.cell_shadow_thickness;

	    _XmDrawHighlight( XtDisplay(mw), XtWindow(ClipChild(mw)), higc,
			      x + mw->matrix.cell_shadow_thickness, y,
			      width - 2*mw->matrix.cell_shadow_thickness,
			      height,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}

	/*
	 * In this case, 'x' is relative to the clip window origin
	 * because this is a nonfixed row. So we need to make it
	 * relative to the matrix window origin for drawing the
	 * shadows in any (trailing) fixed columns
	 */
	x += FIXED_COLUMN_LABEL_OFFSET(mw);
    }

    /*
     * Draw the shadow around the portion of the column in
     * the fixed rows
     */
    if (mw->matrix.fixed_rows &&
	(column >= mw->matrix.fixed_columns) &&
	(column < TRAILING_HORIZ_ORIGIN(mw)) )
    {
	/*
	 * Adjust 'y' such that the leading edge of the
	 * shadow is always drawn. Nicer visually.
	 */
	y = ROW_LABEL_OFFSET(mw);

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness, x, y,
			width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    ((HighlightColumn & mw->matrix.highlighted_cells[0][column]) ||
	    (0xF0 & mw->matrix.highlight_location)))

	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc,
			      x + mw->matrix.cell_shadow_thickness,
			      y + mw->matrix.cell_shadow_thickness,
			      width - 2*mw->matrix.cell_shadow_thickness,
			      height - 2*mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
    }

    /*
     * Draw the shadow around the portion of the column in
     * the trailing fixed rows
     */
    if (mw->matrix.trailing_fixed_rows ||
	( need_fill &&
	  (column >= mw->matrix.fixed_columns ) &&
	  (column < TRAILING_HORIZ_ORIGIN(mw)) ) )
    {
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);
	set_mask = True;

	/*
	 * Adjust 'y' such that the trailing edge of the
	 * shadow is always drawn. Nicer visually.
	 */
	y = TRAILING_FIXED_ROW_LABEL_OFFSET(mw) - adjust;

	/*
	 * If we are to extend the shadow to the limit to "fill" the
	 * matrix's available space, then set the height to fill the
	 * available space, otherwise use the appropriate amount.
	 */
	if ( need_fill )
	    height = mw->core.height - HORIZ_SB_SPACE(mw) -
		mw->manager.shadow_thickness - y;
	else
	    height = TRAILING_FIXED_ROW_HEIGHT(mw) + adjust;

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness,
			x, y, width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    ((HighlightColumn & mw->matrix.highlighted_cells[0][column]) ||
	    (0xF0 & mw->matrix.highlight_location)))

	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc,
			      x + mw->matrix.cell_shadow_thickness, y,
			      width - 2*mw->matrix.cell_shadow_thickness,
			      height - mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
    }

    /*
     * Reset clipping mask if necessary
     */
    if (set_mask || (save_clip != mw->matrix.current_clip))
	xbaeSetClipMask(mw, save_clip);
}

void
xbaeDrawRowShadow(mw, row, redraw_reason)
XbaeMatrixWidget mw;
int row;
unsigned int redraw_reason;
{
    int x, y;
    Dimension width, height;
    Dimension adjust = 2 * mw->matrix.cell_shadow_thickness;
    unsigned int clip_reason, save_clip;
    unsigned char shadow;
    Window win = XtWindow(mw);
    Display *display = XtDisplay(mw);
    Boolean set_mask = False, need_fill;
    GC hiclipgc, higc;

    if ((0 != mw->matrix.disable_redisplay) || (0 == mw->matrix.rows))
	return;

    /*
     * Save current clip mask
     */
    save_clip = mw->matrix.current_clip;
    
    /*
     * Determine which gc to use for highlighting.
     * If we are unhighlighting, use the drawing gc's
     * otherwise use the highlght gc
     */

    if (0xF0 & mw->matrix.highlight_location)
    {
	Pixel fg, bg;
	
	xbaeComputeCellColors(mw, row, mw->matrix.columns-1, &fg, &bg);

	hiclipgc = mw->matrix.draw_clip_gc;
	higc = mw->matrix.draw_gc;

	/*
	 * Set background to last cell in row
	 */
	XSetForeground(XtDisplay(mw), higc, bg);
	XSetForeground(XtDisplay(mw), hiclipgc, bg);
    }
    else
    {
	hiclipgc = mw->matrix.highlight_clip_gc;
	higc = mw->manager.highlight_GC;
    }

    /*
     * If our overlap adjustment isn't set, set it
     */
    if (! adjust)
	adjust = 4;
    
    /*
     * Get the shadow type. If we have a specific row shadow,
     * use that otherwise use the cell's shadow type
     */
    shadow = mw->matrix.row_shadow_types ?
	mw->matrix.row_shadow_types[row] : mw->matrix.cell_shadow_type;

    /*
     * Get the width and height for drawing the shadow around each
     * row. The width used is a bit long to prevent any overlapping
     * at the edges if there are any (trailing) fixed columns, and
     * the actual drawing position is offset the same amount for the
     * the same reason. If we need to fill the available space, adjust
     * width appropriately.
     */
    height = ROW_HEIGHT(mw);
    width = CELL_TOTAL_WIDTH(mw) +
	(mw->matrix.fixed_columns * adjust) +
	(mw->matrix.trailing_fixed_columns * adjust);

    if ( ( need_fill = NEED_HORIZ_FILL( mw ) ) )
    {
	if ((width + mw->matrix.cell_shadow_thickness) >
	    MATRIX_HORIZ_VISIBLE_SPACE(mw))
	    width += FILL_HORIZ_WIDTH(mw);  /* damn picky, but it works */
	else
	    width = mw->core.width - mw->manager.shadow_thickness;
    }

    /*
     * Get the location at which to draw our shadow
     */
    xbaeRowColToXY(mw, row, mw->matrix.fixed_columns, &x, &y);
    x -= (mw->matrix.fixed_columns * adjust);

    /*
     * Draw the shadow around the portion of the row in
     * the non-fixed columns
     */
    if ((row < mw->matrix.fixed_rows) || (row >= TRAILING_VERT_ORIGIN(mw)))
    {
	if (row > mw->matrix.fixed_rows)
	{
	    clip_reason = CLIP_TRAILING_FIXED_ROWS;
	    xbaeSetClipMask(mw, clip_reason);
	    set_mask = True;
	}
	else
	    clip_reason = CLIP_NONE;

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness, x, y,
			width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    HighlightRow & mw->matrix.highlighted_cells[row][0])
	{
	    if (mw->matrix.fixed_columns)
		x -= mw->matrix.cell_shadow_thickness;
	    else
		x += mw->matrix.cell_shadow_thickness;

	    if (mw->matrix.trailing_fixed_columns)
		width += mw->matrix.cell_shadow_thickness;
	    else if (0 == mw->matrix.fixed_columns)
		width -= 2 * mw->matrix.cell_shadow_thickness;

	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc, x,
			      y + mw->matrix.cell_shadow_thickness,
			      width, height - 2*mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}
    }
    else
    {
	clip_reason = CLIP_BETWEEN_FIXED_ROWS;

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, XtWindow(ClipChild(mw)),
			mw->manager.top_shadow_GC,
			mw->manager.bottom_shadow_GC,
			mw->matrix.cell_shadow_thickness, x, y,
			width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    HighlightRow & mw->matrix.highlighted_cells[row][0])
	{
	    if (mw->matrix.fixed_columns)
		x -= mw->matrix.cell_shadow_thickness;
	    else
		x += mw->matrix.cell_shadow_thickness;

	    if (mw->matrix.trailing_fixed_columns)
		width += mw->matrix.cell_shadow_thickness;
	    else if (0 == mw->matrix.fixed_columns)
		width -= 2 * mw->matrix.cell_shadow_thickness;

	    _XmDrawHighlight( XtDisplay(mw), XtWindow(ClipChild(mw)), higc, x,
			      y + mw->matrix.cell_shadow_thickness,
			      width, height - 2*mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}

	/*
	 * In this case, 'y' is relative to the clip window origin
	 * because this is a nonfixed row. So we need to make it
	 * relative to the matrix window origin for drawing the
	 * shadows in any (trailing) fixed columns
	 */
	y += FIXED_ROW_LABEL_OFFSET(mw);
    }

    /*
     * If only redrawing the portion of the shadow in the
     * non-fixed columns, then leave
     */
    if (GRID_REDRAW_SCROLL_HORIZ & redraw_reason)
    {
	/*
	 * Reset clipping mask if necessary
	 */
	if (set_mask)
	    xbaeSetClipMask(mw, CLIP_NONE);

	return;
    }

    /*
     * Draw the shadow around the portion of the row in
     * the fixed columns
     */
    if (mw->matrix.fixed_columns)
    {
	xbaeSetClipMask(mw, CLIP_FIXED_COLUMNS | clip_reason);
	set_mask = True;

	/*
	 * Adjust 'x' such that the leading edge of the
	 * shadow is always drawn. Nicer visually.
	 */
	x = COLUMN_LABEL_OFFSET(mw);
	width = FIXED_COLUMN_WIDTH(mw) + adjust;

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness, x, y,
			width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    HighlightRow & mw->matrix.highlighted_cells[row][0])
	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc,
			      x + mw->matrix.cell_shadow_thickness,
			      y + mw->matrix.cell_shadow_thickness,
			      width, height - 2*mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
    }
    
    /*
     * Draw the shadow around the portion of the row in
     * the trailing fixed columns, or the portion of the
     * row that needs to be extended for filling
     */
    
    if (mw->matrix.trailing_fixed_columns || need_fill)
    {
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_COLUMNS | clip_reason);
	set_mask = True;

	/*
	 * Adjust 'x' such that the trailing edge of the
	 * shadow is always drawn. Nicer visually.
	 */
	x = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) - adjust;

	/*
	 * If we are to extend the shadow to the limit to "fill" the
	 * matrix's available space, then bump up the width by the
	 * necessary amount.
	 */
	if ( need_fill )
	    width = mw->core.width - VERT_SB_SPACE(mw) -
		mw->manager.shadow_thickness - x;
	else
	    width = TRAILING_FIXED_COLUMN_WIDTH(mw) + adjust;

	if (mw->matrix.cell_shadow_thickness &&
	    !(GRID_REDRAW_HIGHLIGHT & redraw_reason))
	    DRAW_SHADOW(display, win,
			mw->matrix.cell_top_shadow_clip_gc,
			mw->matrix.cell_bottom_shadow_clip_gc,
			mw->matrix.cell_shadow_thickness,
			x, y, width, height, shadow);

	if (mw->matrix.highlighted_cells &&
	    ((HighlightRow & mw->matrix.highlighted_cells[row][0]) ||
	    (0xF0 & mw->matrix.highlight_location)))
	{
	    if (! need_fill)
	    {
		if (mw->matrix.trailing_fixed_columns)
		    x -= mw->matrix.cell_shadow_thickness;
		else
		{
		    x += mw->matrix.cell_shadow_thickness;
		    width -= mw->matrix.cell_shadow_thickness;
		}
	    }
	    else
		width -= mw->matrix.cell_shadow_thickness;

	    _XmDrawHighlight( XtDisplay(mw), win, hiclipgc,
			      x, y + mw->matrix.cell_shadow_thickness,
			      width, height - 2*mw->matrix.cell_shadow_thickness,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}
    }

    /*
     * Reset clipping mask if necessary
     */
    if (set_mask || (save_clip != mw->matrix.current_clip))
	xbaeSetClipMask(mw, save_clip);
}


static void
DrawRowShadows(mw, redraw_reason)
XbaeMatrixWidget mw;
unsigned int redraw_reason;
{
    int m, end;

    /*
     * Draw the shadow around the non-fixed rows.
     */
    xbaeGetVisibleRows(mw, &m, &end);
    for (; m <= end; m++)
	xbaeDrawRowShadow(mw, m, redraw_reason);

    /*
     * If we're scrolling vertically, don't need to
     * redraw any fixed rows, so leave.
     */
    if (GRID_REDRAW_SCROLL_VERT & redraw_reason)
	return;

    /*
     * Draw the shadow around the fixed rows.
     */
    if (mw->matrix.fixed_rows)
	for (m = 0; m < mw->matrix.fixed_rows; m++)
	    xbaeDrawRowShadow(mw, m, redraw_reason);

    /*
     * Draw the shadow around the trailing fixed rows.
     */
    if (mw->matrix.trailing_fixed_rows)
	for (m = TRAILING_VERT_ORIGIN(mw); m < mw->matrix.rows; m++)
	    xbaeDrawRowShadow(mw, m, redraw_reason);
}



static void
DrawColumnShadows(mw, redraw_reason)
XbaeMatrixWidget mw;
unsigned int redraw_reason;
{
    int m, end;

    /*
     * Draw the shadow around the non-fixed columns.
     */
    xbaeGetVisibleColumns(mw, &m, &end);
    for (; m <= end; m++)
	xbaeDrawColumnShadow(mw, m, redraw_reason);

    /*
     * If we're scrolling horizontally, don't need to
     * redraw any fixed columns, so leave.
     */
    if (GRID_REDRAW_SCROLL_HORIZ & redraw_reason)
	return;

    /*
     * Draw the shadow around the fixed columns.
     */
    if (mw->matrix.fixed_columns)
	for (m = 0; m < mw->matrix.fixed_columns; m++)
	    xbaeDrawColumnShadow(mw, m, redraw_reason);

    /*
     * Draw the shadow around the trailing fixed columns.
     */
    if (mw->matrix.trailing_fixed_columns)
	for (m = TRAILING_HORIZ_ORIGIN(mw); m < mw->matrix.columns; m++)
	    xbaeDrawColumnShadow(mw, m, redraw_reason);
}


void
xbaeDrawGridShadows(mw, redraw_reason)
XbaeMatrixWidget mw;
unsigned int redraw_reason;
{
    if (0 == mw->matrix.disable_redisplay)
    {
	if (XmGRID_ROW_SHADOW == mw->matrix.grid_type)
	    DrawRowShadows(mw, redraw_reason);
	else if (XmGRID_COLUMN_SHADOW == mw->matrix.grid_type)
	    DrawColumnShadows(mw, redraw_reason);
    }
}
