#!/bin/sh
# \
exec wish "$0" "$@"

# main.tcl --
#
#	GUI driver for xmlpowertool
#
# Copyright (c) 2001-2002 Zveno pty Ltd
# http://www.zveno.com/
#
# $Id: main.tcl,v 1.2 2002/05/23 10:00:44 balls Exp $

# Look for the lib directory
# On Darwin, this script may not be in a bin subdirectory
# so lib will be in the same directory

if {[file isdir [file join [file dirname [info script]] lib]]} {
    # Darwin style
    set libdir [file join [file dirname [info script]] lib]
} elseif {[file isdir [file join [file dirname [file dirname [info script]]] lib]]} {
    # TclKit style
    set libdir [file join [file dirname [file dirname [info script]]] lib]
}

lappend auto_path $libdir
lappend auto_path [file join $libdir resources]

# For TclKit
catch {package require Tk}

package require BWidget 1.4

# Setup message catalog stuff
package require msgcat
namespace import ::msgcat::*
::msgcat::mcload [file join [file dirname [info script]] msgs]

set version 2.0
set patchlevel Beta3

set configParamNS http://www.zveno.com/xmltool/parameters

set filetypes [list				\
	[list [mc {XML Documents}] {.xml .XML}]	\
	[list [mc {XSL Documents}] {.xsl .XSL}]	\
	[list [mc {All Files}] *]		\
	]

# We need TclDOM for reading config file
package require dom

# If we have TclXSLT then we can run XSLT stylesheets in the config file
set haveBuiltinXSLT 0
if {![catch {package require xslt}]} {
    set haveBuiltinXSLT 1

    # Now we need access to resources as an XSLT extension

    package require resources

    ::xslt::extension add http://www.zveno.com/resources ::resources

}

# We need tools for running XSL processors and XML parsers
package require tools

# Display a simple GUI

wm title . [mc "XML Power Tool"]

menu .menu -tearoff 0
. configure -menu .menu
.menu add cascade -label [mc File] -menu .menu.file
.menu add cascade -label [mc Edit] -menu .menu.edit
.menu add cascade -label [mc Help] -menu .menu.help

set m [menu .menu.file -tearoff 0]
#$m add command -label [mc "Load Configuration"] -command InitFromFile
$m add command -label [mc "View Configuration"] -command ShowConfig
$m add separator
$m add command -label [mc Quit] -command Quit

set m [menu .menu.edit -tearoff 0]
$m add command -label [mc Cut] -state disabled
$m add command -label [mc Copy] -state disabled
$m add command -label [mc Paste] -state disabled
set debugging 0
if {$debugging} {
    $m add separator
    $m add command -label [mc Debug] -command {
	console show
    }
}

set m [menu .menu.help -tearoff 0]
$m add command -label [mc About...] -command tkAboutDialog

set w {}
set win .

font create main -family Arial -size 14
font create mainBold -family Arial -size 14 -weight bold

set nb [NoteBook $w.nb -homogeneous 1 -width 650 -height 500]

set winsys {}
catch {set winsys $::tcl_platform(windowingsystem)}
if {[string equal $winsys "aqua"]} {
    $nb configure -activeforeground black \
	-activebackground white
}

grid $nb -row 0 -column 0 -sticky news
grid rowconfigure $win 0 -weight 1
grid columnconfigure $win 0 -weight 1

# UIInit --
#
#	Initialise the UI
#
# Arguments:
#	nb	Notebook widget used to display processing options
#
# Results:
#	Notebook is populated with tabs and content

proc UIInit {nb} {
    global initialised page tab stage_phases stage_intro

    if {!$initialised} {
	return
    }

    set cfgdom [::tools::getconfig]
    set defaultPage {}

    foreach tabNode [dom::libxml2::selectNode $cfgdom /configuration/display/tab] {
	set page [dom::element getAttribute $tabNode name]
	set tab [dom::element getAttribute $tabNode label]
	if {![string length $tab]} {
	    set tab $page
	}

	set stage_intro($page) [dom::element getAttribute $tabNode description]
	if {![string length $stage_intro($page)]} {
	    set stage_intro($page) "Use the wizard-style interface to process an XML document."
	}

	if {![string compare [dom::element getAttribute $tabNode role] "default"]} {
	    set defaultPage $page
	}

	set stagelist {{} intro}
	foreach stepNode [dom::libxml2::node selectNode $tabNode *] {
	    lappend stagelist [dom::element getAttribute $stepNode label] \
		[dom::node cget $stepNode -nodeName]
	}

	set stage_phases($page) $stagelist

	set tabwin [$nb insert end $page -text [mc $tab]]
	$nb itemconfigure $page -raisecmd [list Raised $page $tabwin] \
		-leavecmd [list Leave $page $tabwin]

	grid columnconfigure $tabwin 1 -weight 1
	grid rowconfigure $tabwin 0 -weight 1

	labelframe $tabwin.stages -text [mc Wizard] -font main -foreground black
	set st [text $tabwin.stages.wizard -width 24 -font main]
	labelframe $tabwin.f -text [mc Introduction] -font main -foreground black

	set c [frame $tabwin.controls]
	button $c.next -text [mc Next] -command Next -font {Arial 18 bold}
	button $c.stop -text [mc Stop] -command New -font {Arial 18 bold} -state disabled

	grid $st -row 0 -column 0 -sticky news
	grid rowconfigure $tabwin.stages 0 -weight 1
	grid columnconfigure $tabwin.stages 0 -weight 1

	grid $tabwin.stages -row 0 -column 0 -sticky news
	grid $tabwin.f -row 0 -column 1 -sticky news
	grid $c - -row 1 -column 0 -sticky ew

	grid $c.next $c.stop

	foreach {descr stage} $stagelist {
	    if {![string length $descr]} {
		continue
	    }
	    $st insert end \n {} \
		[mc $descr] $stage \
		\n\n {}
	    $st tag configure $stage -font main
	    $st tag bind $stage <ButtonRelease-1> [list SkipTo $stage]
	}

	$st tag configure intro -font mainBold
	$st tag bind intro <ButtonRelease-1> New
	$st tag configure current -font main -foreground red

	$st configure -state disabled -cursor [. cget -cursor]
    }

    if {[string length $defaultPage]} {
	set page $defaultPage
    }

    return {}
}

# Display splash screen

proc tkAboutDialog {{dismissable 1}} {
    global libdir

    if {[winfo exists .about]} {
	raise .about
	return {}
    }

    set t [toplevel .about]
    wm title $t "About XMLPowerTool"

    if {!$dismissable} {
	wm overrideredirect $t on
	$t configure -border 5 -relief raised
    }

    image create photo zvenoLogo -data {R0lGODlhlgCNAPf/AP///0dIR1lYV+Xd5istG///9P8RIv9ocDo6OvP2890m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}

    wm geometry $t +[expr ([winfo screenwidth $t] - 500) / 2]+[expr ([winfo screenheight $t] - 300) / 2]
    grid [canvas $t.can -width 500 -height 300 -background white] -sticky news
    $t.can create image 100 0 -image zvenoLogo -anchor n
    $t.can create text 100 200 -text [mc "XML and Scripting Specialists"] -font {arial 10 italic}
    $t.can create text 100 250 -text "http://www.zveno.com/" -font {arial 14}

    $t.can create line 210 0 210 350 -width 5 -fill #d9d9d9

    $t.can create text 350 25 -text "XMLPowerTool" -font {arial 24}

    $t.can create text 350 60 -text "Version $::version$::patchlevel" -font {arial 14}
    $t.can create text 350 100 -text [mc "Copyright (c) 1999-2002 Zveno Pty Ltd"] -font {arial 10}
    $t.can create text 250 150 -text [mc "Credits:"] -font {arial 10} -anchor nw
    $t.can create text 260 175 -text [mc "Programming: Steve Ball"] -font {arial 10} -anchor nw

    return $t
}

proc Quit {} {
    destroy .
}

proc InitFromFile {} {
    global initialcfgdir cfgfile

    set initialcfgdir [pwd]
    catch {set initialcfgdir [file dirname $cfgfile]}

    set newcfgfile [tk_getOpenFile -parent . -title [mc {Open Configuration}] -defaultextension .xml -filetypes $filetypes -initialdir $initialcfgdir]
    if {![string length $newcfgfile]} {
	return {}
    }

    set cfgfile $newcfgfile

    Init $cfgfile

    return {}
}

proc Init cfgfile {
    global initialised nb

    if {[catch {::tools::init $cfgfile} msg]} {
	tk_messageBox -message "Unable to configure application due to \"$msg\"" \
		-icon error -type ok
	Quit
    }

    set initialised 1

    UIInit $nb

    return {}
}

proc ShowConfig {} {
    global initialised

    if {$initialised} {
	set cfgdom [::tools::getconfig]
	catch {destroy .config}
	set c [toplevel .config]
	text $c.t -yscrollcommand [list $c.y set] -xscrollcommand [list $c.x set]
	scrollbar $c.y -command [list $c.t yview] -orient vertical
	scrollbar $c.x -command [list $c.t xview] -orient horizontal
	grid $c.t -row 0 -column 0 -sticky news
	grid $c.y -row 0 -column 1 -sticky ns
	grid $c.x -row 1 -column 0 -sticky ew
	grid rowconfigure $c 0 -weight 1
	grid columnconfigure $c 0 -weight 1

	$c.t insert end [::dom::libxml2::serialize $cfgdom]

    } else {
	tk_messageBox -parent . -message [mc "Application has not been initialised"] -type ok -icon error
    }
}

proc Raised {raisedpage tabwin} {
    global nb page phase

    set page $raisedpage

    New

    return {}
}

proc Leave {leftpage tabwin} {
    global nb page

    # Stop processes

    set page {}

    return {}
}

proc ClearFrame {} {
    global nb page

    set tabwin [$nb getframe $page]
    foreach child [winfo children $tabwin.f] {
	destroy $child
    }
    $tabwin.f configure -text [mc Introduction]
    set st $tabwin.stages.wizard
    catch {eval $st tag remove current [$st tag ranges current]}
}

proc New {} {
    global initialised page phase nb stage_phases stage_intro

    ClearFrame

    set tabwin [$nb getframe $page]
    $tabwin.controls.stop configure -state disabled -text [mc "Stop"]

    message $tabwin.f.message -aspect 500
    grid $tabwin.f.message -row 0 -column 0 -sticky news
    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0
    grid columnconfigure $tabwin.f 0 -weight 1

    if {!$initialised} {
	$tabwin.f.message configure -text [mc "The application has not been initialised."]
	$tabwin.controls.next configure -state disabled
	return {}
    }

    $tabwin.controls.next configure -state normal -text [mc Next] -command Next

    set stagelist $stage_phases($page)
    set phase [lindex $stagelist 3]

    $tabwin.f.message configure -text $stage_intro($page)

    return {}
}

proc Next {} {
    global nb page phase stage_phases

    set tabwin [$nb getframe $page]
    set st $tabwin.stages.wizard
    $tabwin.controls.stop configure -state normal -text [mc "Stop"]
    $tabwin.controls.next configure -state normal
    ClearFrame

    eval $st tag add current [$st tag ranges $phase]

    uplevel #0 Phase_$phase

    return {}
}

proc SkipTo to {
    global nb page phase stage_phases

    switch $to {
	chooseFiles {
	    New
	    Next
	    return {}
	}
	chooseSSheet {
	    # If we have already chosen files then this is OK
	    # otherwise beep
	    switch $phase {
		intro -
		chooseFiles {
		    bell
		    return {}
		}
		default {
		    set phase chooseSSheet
		    Next
		    return {}
		}
	    }
	}
	chooseResult {
	    # If we have already chosen files and a stylesheet then OK
	    # otherwise beep
	    switch $phase {
		intro -
		chooseFiles -
		chooseSSheet {
		    bell
		    return {}
		}
		default {
		    set phase chooseResult
		    Next
		    return {}
		}
	    }
	}
	checkWF {
	    switch $phase {
		intro -
		chooseFiles {
		    bell
		    return {}
		}
	    }
	}
	transform {
	    # If we have already chosen files, a stylesheet and result filename then OK
	    # otherwise beep
	    switch $phase {
		intro -
		chooseFiles -
		chooseSSheet -
		chooseResult {
		    bell
		    return {}
		}
		default {
		    set phase transform
		    Next
		    return {}
		}
	    }
	}
	validate {
	    switch $phase {
		intro -
		chooseFiles {
		    bell
		    return {}
		}
		default {
		    set phase validate
		    Next
		    return {}
		}
	    }
	}
	validate-result {
	    switch $phase {
		intro -
		chooseFiles -
		chooseSSheet {
		    bell
		    return {}
		}
		default {
		    set phase validate-result
		    Next
		    return {}
		}
	    }
	}
    }

    return {}
}

proc GetNextPhase {} {
    global page phase stage_phases

    set stagelist $stage_phases($page)
    set phaselist {}
    foreach {tab ph} $stagelist {
	lappend phaselist $ph
    }
    set idx [lsearch $phaselist $phase]
    incr idx
    set nextphase [lindex $phaselist $idx]
    if {![string length $nextphase]} {
	set nextphase intro
    }

    return $nextphase
}

proc Phase_chooseFiles {} {
    global nb phase page currdir dirlist

    if {![info exists currdir]} {
	set currdir [pwd]
    }

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Choose Source Documents:"]
    set l [listbox $tabwin.f.view -yscrollcommand [list $tabwin.f.yscroll set] -listvar dirlist -selectmode extended]
    scrollbar $tabwin.f.yscroll -command [list $l yview] -orient vertical

    grid $l -row 0 -column 0 -sticky news
    grid $tabwin.f.yscroll -row 0 -column 1 -sticky ns

    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0

    bind $l <ButtonRelease-1> [list SelectFile $l currdir selectedfiles]
    bind $l <Double-1> [list OpenFile $l currdir selectedfiles %x %y]
    bindtags $l [list Listbox $l . all]

    PopulateListbox $l currdir

    set phase [GetNextPhase]
    $tabwin.controls.next configure -state disabled

    return {}
}

proc Phase_checkWF {} {
    global nb page phase currdir selectedfiles

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Checking Well-Formedness:"]
    set pt [text $tabwin.f.selfiles -yscrollcommand [list $tabwin.f.yscroll set] -xscrollcommand [list $tabwin.f.xscroll set] -wrap none]
    scrollbar $tabwin.f.yscroll -command [list $pt yview] -orient vertical
    scrollbar $tabwin.f.xscroll -command [list $pt xview] -orient horizontal

    foreach file $selectedfiles {
	$pt insert end $file [list result$file hyperlink] \n {}
	$pt tag bind result$file <1> [list View [file join $currdir $file]]
	$pt tag bind result$file <Enter> [list Hyperlink_Enter $pt result$file]
	$pt tag bind result$file <Leave> [list Hyperlink_Leave $pt result$file]
    }

    $pt tag configure current -foreground red
    $pt tag configure hyperlink -foreground blue
    $pt tag configure msg -lmargin1 25 -lmargin2 25
    $pt configure -state disabled

    grid $pt -row 0 -column 0 -sticky news
    grid $tabwin.f.yscroll -row 0 -column 1 -sticky ns
    grid $tabwin.f.xscroll -row 1 -column 0 -sticky ew

    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0
    grid columnconfigure $tabwin.f 0 -weight 1

    $tabwin.controls.next configure -state disabled

    doCheckWF $pt $currdir $selectedfiles

    return {}
}

proc Phase_chooseSSheet {} {
    global nb page phase currssheetdir dirlist

    if {![info exists currssheetdir]} {
	set currssheetdir [pwd]
    }

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Choose Stylesheet:"]
    set l [listbox $tabwin.f.view -yscrollcommand [list $tabwin.f.yscroll set] -listvar dirlist]
    scrollbar $tabwin.f.yscroll -command [list $l yview] -orient vertical

    grid $l -row 0 -column 0 -sticky news
    grid $tabwin.f.yscroll -row 0 -column 1 -sticky ns

    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0

    bind $l <ButtonRelease-1> [list SelectFile $l currssheetdir ssheet]
    bind $l <Double-1> [list OpenFile $l currssheetdir ssheet %x %y]
    bindtags $l [list Listbox $l . all]

    PopulateListbox $l currssheetdir

    set phase [GetNextPhase]
    $tabwin.controls.next configure -state disabled

    return {}
}

proc Phase_chooseResult {} {
    global nb page phase currssheetdir ssheet dirlist result_filenames selectedfiles result

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Set Result Filename:"]
    set e [entry $tabwin.f.result -width 30 -textvariable result_template]
    grid $e -row 0 -column 0 -sticky new

    # Have we been here before?
    # If not then set a reasonable default

    switch -glob [llength $selectedfiles] {
	1 {
	    if {[info exists result_filenames([lindex $selectedfiles 0],$ssheet)]} {
		set default_result_filename $result_filenames([lindex $selectedfiles 0],$ssheet)
	    } else {
		set default_result_filename [GetResultName [lindex $selectedfiles 0]]
	    }
	    button $tabwin.f.browse -text [mc "Browse"] -command {
		set browse_filename [tk_getSaveFile -title [mc "Choose Result File"] -message [mc "Choose where to save result"]]
		if {[string length $browse_filename]} {
		    set result_template $browse_filename
		}
	    }
	    grid $tabwin.f.browse -row 1 -column 0 -sticky nw
	}
	default {
	    set default_result_filename [GetResultName %F]

	    label $tabwin.f.multi -text [mc "Source filename is substituted for %F"]
	    grid $tabwin.f.multi -row 1 -column 0 -sticky nw
	}
    }

    grid rowconfigure $tabwin.f 0 -weight 0
    grid rowconfigure $tabwin.f 1 -weight 1
    grid rowconfigure $tabwin.f 2 -weight 0

    set ::result_template $default_result_filename

    set phase [GetNextPhase]

    return {}
}

proc Phase_setParameters {} {
    global nb page phase currdir currssheetdir ssheet dirlist parameters

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Set Stylesheet Parameters:"]

    frame $tabwin.f.params
    grid $tabwin.f.params -row 0 -column 0 -sticky news
    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0

    grid columnconfigure $tabwin.f.params 1 -weight 1

    # Display default set of parameters

    catch {unset parameters}
    array set parameters {}

    set cfgdom [::tools::getconfig]
    set row -1
    foreach paramNode [dom::libxml2::selectNode $cfgdom /configuration/transform/parameters/*] {
	set name [dom::libxml2::node cget $paramNode -nodeName]
	set value [dom::libxml2::node children $paramNode]

	if {[catch {label $tabwin.f.params.name$name -text $name}]} {
	    continue
	}
	entry $tabwin.f.params.value$name -width 30 -textvariable parameters($name)
	button $tabwin.f.params.delete$name -text [mc Delete] -command [list phase_setParameters:Delete $tabwin.f.params $name] -fg red
	grid $tabwin.f.params.name$name -row [incr row] -column 0 -sticky ne
	grid $tabwin.f.params.value$name -row $row -column 1 -sticky new
	grid $tabwin.f.params.delete$name -row $row -column 2 -sticky nw
    }

    button $tabwin.f.params.add -text [mc "Add Parameter"] -command [list phase_setParameters:Add $tabwin.f.params]
    grid $tabwin.f.params.add -row [incr row] -column 0 -sticky nw
    grid rowconfigure $tabwin.f.params $row -weight 1

    phase_setParameters:Update [set phase [GetNextPhase]] $cfgdom

    return {}
}

proc phase_setParameters:Add {w} {
    global parameters

    set t [toplevel .parameterAdd]
    wm title $t "Add Parameter"
    grid [label $t.label -text [mc "Parameter Name:"]] -row 0 -column 0 -sticky w
    grid [entry $t.name -width 20] -row 0 -column 0 -sticky ew
    grid [button $t.ok -text [mc OK] -command {set ::setParameters_Add OK}] -row 1 -column 0 -sticky w
    grid [button $t.cancel -text [mc Cancel] -command {set ::setParameters_Add Cancel}] -row 1 -column 1 -sticky e
    grid columnconfigure $t 1 -weight 1

    tkwait variable ::setParameters_Add

    if {$::setParameters_Add == "Cancel"} {
	return {}
    }

    set name [$t.name get]
    destroy $t

    if {[info exists parameters($name)] || \
	    ![regexp ^$::xml::QName\$ $name]} {
	bell
	return {}
    }

    foreach {col row} [grid size $w] break

    grid forget $w.add
    grid rowconfigure $w [incr row -1] -weight 0

    label $w.name$name -text $name
    entry $w.value$name -width 30 -textvariable parameters($name)
    button $w.delete$name -text [mc Delete] -command [list phase_setParameters:Delete $w $name] -fg red
    grid $w.name$name -row $row -column 0 -sticky ne
    grid $w.value$name -row $row -column 1 -sticky new
    grid $w.delete$name -row $row -column 2 -sticky nw

    grid rowconfigure $w $row -weight 0

    grid $w.add -row [incr row] -column 0 -sticky nw
    grid rowconfigure $w $row -weight 1

    return {}
}

proc phase_setParameters:Delete {w name} {
    global parameters

    unset parameters($name)
    destroy $w.name$name
    destroy $w.value$name
    destroy $w.delete$name

    return {}
}
proc phase_setParameters:Update {nextphase cfgdom} {
    global parameters phase taskid currdir selectedfiles

    if {[string compare $phase $nextphase]} {
	return {}
    }

    foreach name [array names parameters] {

	set paramsNodeList [lindex [dom::libxml2::selectNode $cfgdom /configuration/transform/parameters/$name] 0]

	if {[llength $paramsNodeList] == 0} {
	    continue
	}

	set paramsNode [lindex $paramsNodeList 0]
	set parameters($name) {}
	foreach child [dom::libxml2::node children $paramsNode] {
	    switch [dom::libxml2::node cget $child -nodeType] {
		element {
		    switch [dom::libxml2::node cget $child -nodeName] {
			filename {
			    if {[llength $selectedfiles] == 1} {
				append parameters($name) [lindex $selectedfiles 0]
			    } else {
				append parameters($name) %F
			    }
			}
			directory {
			    append parameters($name) $currdir
			}
			clock {
			    set format [dom::libxml2::element getAttribute $child format]
			    append parameters($name) [clock format [clock seconds] -format $format]
			}
			default {
			    # Ignore it
			}
		    }
		}
		textNode {
		    append parameters($name) [dom::libxml2::node cget $child -nodeValue]
		}
		default {
		    # Ignore it
		}
	    }
	}
    }

    set taskid [after 1000 [list phase_setParameters:Update $nextphase $cfgdom]]

    return {}
}

proc Phase_transform {} {
    global nb page phase currdir selectedfiles currssheetdir ssheet result_template result_filenames

    set tabwin [$nb getframe $page]

    frame $tabwin.f.ssheet
    label $tabwin.f.ssheet.label -text [mc "Stylesheet:"]
    label $tabwin.f.ssheet.fname -text $ssheet
    grid $tabwin.f.ssheet.label $tabwin.f.ssheet.fname -sticky w
    grid $tabwin.f.ssheet - -row 0 -column 0 -sticky w

    $tabwin.f configure -text [mc "Transforming files:"]

    set pt [text $tabwin.f.selfiles -yscrollcommand [list $tabwin.f.yscroll set] -xscrollcommand [list $tabwin.f.xscroll set] -wrap none]
    scrollbar $tabwin.f.yscroll -command [list $pt yview] -orient vertical
    scrollbar $tabwin.f.xscroll -command [list $pt xview] -orient horizontal
    foreach file $selectedfiles {
	if {[llength $selectedfiles] == 1} {
	    set resultfname $result_template
	    set result_filenames([lindex $selectedfiles 0],$ssheet) $result_template
	} else {
	    regsub %F $result_template $file resultfname
	}
	$pt insert end $file [list tag$file hyperlink] { -> } {} $resultfname [list result$resultfname hyperlink] \n {}
	$pt tag bind tag$file <1> [list View [file join $currdir $file]]
	$pt tag bind tag$file <Enter> [list Hyperlink_Enter $pt tag$file]
	$pt tag bind tag$file <Leave> [list Hyperlink_Leave $pt tag$file]
	$pt tag bind result$resultfname <1> [list View [file join $currdir $resultfname]]
	$pt tag bind result$resultfname <Enter> [list Hyperlink_Enter $pt result$resultfname]
	$pt tag bind result$resultfname <Leave> [list Hyperlink_Leave $pt result$resultfname]
    }
    $pt tag configure current -foreground red
    $pt tag configure hyperlink -foreground blue
    $pt tag configure msg -lmargin1 25 -lmargin2 25
    $pt configure -state disabled

    grid $pt -row 1 -column 0 -sticky news
    grid $tabwin.f.yscroll -row 1 -column 1 -sticky ns
    grid $tabwin.f.xscroll -row 2 -column 0 -sticky ew

    grid rowconfigure $tabwin.f 0 -weight 0
    grid rowconfigure $tabwin.f 1 -weight 1
    grid rowconfigure $tabwin.f 2 -weight 0
    grid rowconfigure $tabwin.f 3 -weight 0
    grid columnconfigure $tabwin.f 0 -weight 1

    $tabwin.controls.next configure -state disabled

    doXSLT $pt $currdir $selectedfiles

    return {}
}

proc Phase_validate {{isresult 0}} {
    global nb page phase currdir selectedfiles

    set tabwin [$nb getframe $page]

    $tabwin.f configure -text [mc "Validating:"]

    set pt [text $tabwin.f.selfiles -yscrollcommand [list $tabwin.f.yscroll set] -xscrollcommand [list $tabwin.f.xscroll set] -wrap none]
    scrollbar $tabwin.f.yscroll -command [list $pt yview] -orient vertical
    scrollbar $tabwin.f.xscroll -command [list $pt xview] -orient horizontal
    set validatefiles {}
    foreach file $selectedfiles {
	if {$isresult} {
	    set file [GetResultName $file]
	}
	lappend validatefiles $file
	$pt insert end $file [list result$file hyperlink] \n {}
	$pt tag bind result$file <1> [list View [file join $currdir $file]]
	$pt tag bind result$file <Enter> [list Hyperlink_Enter $pt result$file]
	$pt tag bind result$file <Leave> [list Hyperlink_Leave $pt result$file]
    }
    $pt tag configure current -foreground red
    $pt tag configure hyperlink -foreground blue
    $pt tag configure msg -lmargin1 25 -lmargin2 25
    $pt configure -state disabled

    grid $pt -row 0 -column 0 -sticky news
    grid $tabwin.f.yscroll -row 0 -column 1 -sticky ns
    grid $tabwin.f.xscroll -row 1 -column 0 -sticky ew

    grid rowconfigure $tabwin.f 0 -weight 1
    grid rowconfigure $tabwin.f 1 -weight 0
    grid rowconfigure $tabwin.f 2 -weight 0
    grid columnconfigure $tabwin.f 0 -weight 1

    $tabwin.controls.next configure -state disabled

    doValidate $pt $currdir $validatefiles

    return {}
}

proc Phase_validate-result {} {
    Phase_validate 1
}

proc Hyperlink_Enter {w tag} {
    $w tag configure $tag -borderwidth 3 -background [$w cget -background] -relief raised
}

proc Hyperlink_Leave {w tag} {
    $w tag configure $tag -background [$w cget -background] -relief {}
}

proc doCheckWF {pt dir files} {
    global nb page phase

    if {[string compare $phase "checkWF"]} {
	return {}
    }

    catch {eval $pt tag remove current [$pt tag ranges current]}

    set fname [lindex $files 0]
    set files [lrange $files 1 end]

    eval $pt tag add current [$pt tag ranges result$fname]
    set cursor [. cget -cursor]
    . configure -cursor watch
    $pt configure -cursor watch
    update

    if {[string compare $phase "checkWF"]} {
	. configure -cursor $cursor
	$pt configure -cursor $cursor
	return {}
    }

    catch {::tools::wellformed [file join $dir $fname]} msg

    . configure -cursor $cursor
    $pt configure -cursor $cursor

    $pt configure -state normal
    $pt insert "[lindex [$pt tag ranges result$fname] 0] lineend" \n$msg\n msg
    $pt configure -state disabled

    if {[llength $files]} {
	after idle [list doCheckWF $pt $dir $files]
    } elseif {[string compare $phase "checkWF"]} {
	return {}
    } else {
	catch {eval $pt tag remove current [$pt tag ranges current]}
	set tabwin [$nb getframe $page]
	set phase [GetNextPhase]
	if {[string compare $phase "intro"]} {
	    $tabwin.controls.next configure -state normal
	} else {
	    $tabwin.controls.next configure -state disabled
	    $tabwin.controls.stop configure -text [mc "New"]
	}
    }

    return {}
}

proc doXSLT {pt dir files} {
    global nb page currdir currssheetdir ssheet phase haveBuiltinXSLT selectedfiles parameters

    if {[string compare $phase "transform"]} {
	return {}
    }

    catch {eval $pt tag remove current [$pt tag ranges current]}

    set fname [lindex $files 0]
    set files [lrange $files 1 end]

    eval $pt tag add current [$pt tag ranges tag$fname]
    set cursor [. cget -cursor]
    . configure -cursor watch
    $pt configure -cursor watch
    update

    if {[string compare $phase "transform"]} {
	. configure -cursor $cursor
	return {}
    }

    if {[llength $selectedfiles] == 1} {
	set result $::result_template
    } else {
	regsub %F $::result_template $fname result
    }

    set cfgdom [::tools::getconfig]
    set customParameters {}
    if {$haveBuiltinXSLT} {

	# Add custom parameters

	if {0} {
	foreach parametersNode [dom::libxml2::selectNode [::tools::getconfig] /configuration/parameters/*] {
	    lappend customParameters [dom::libxml2::node cget $parametersNode -nodeName]
	    set paramSSheet [dom::libxml2::serialize [dom::node selectNode $parametersNode *]]
	    set value [::xslt::transform $paramSSheet "<parameters>
  <source-directory>$dir</source-directory>
  <source-filename>$fname</source-filename>
  <result-directory>[file dirname [file join $dir $result]]</result-directory>
  <result-filename>[file tail $result]</result-filename>
  <current-date>[clock format [clock seconds] -format %Y%m%d]</current-date>
  <current-time>[clock format [clock seconds] -format %H%M%S]</current-time>
</parameters>
"]
	    lappend customParameters $value
	}
	}
    }

    set ssheetParameters {}
    foreach name [array names parameters] {
	set paramsNodeList [lindex [dom::libxml2::selectNode $cfgdom /configuration/transform/parameters/$name] 0]
	if {[llength $paramsNodeList] == 0} {
	    lappend ssheetParameters $name '$parameters($name)'
	} else {
	    set paramsNode [lindex $paramsNodeList 0]
	    set value {}
	    foreach child [dom::libxml2::node children $paramsNode] {
		switch [dom::libxml2::node cget $child -nodeType] {
		    element {
			switch [dom::libxml2::node cget $child -nodeName] {
			    filename {
				append value $result
			    }
			    directory {
				append value $currdir
			    }
			    clock {
				set format [dom::libxml2::element getAttribute $child format]
				append value [clock format [clock seconds] -format $format]
			    }
			    default {
				# Ignore it
			    }
			}
		    }
		    textNode {
			append value [dom::libxml2::node cget $child -nodeValue]
		    }
		    default {
			# Ignore it
		    }
		}
	    }

	    lappend ssheetParameters $name '$value'

	}
    }

    catch {eval ::tools::xslt [list [file join $dir $fname] [file join $currssheetdir $ssheet] [file join $dir $result]] $ssheetParameters} msg

    . configure -cursor $cursor
    $pt configure -cursor $cursor

    $pt configure -state normal
    $pt insert "[lindex [$pt tag ranges tag$fname] 0] lineend" \n$msg\n msg
    $pt configure -state disabled

    if {[llength $files]} {
	after idle [list doXSLT $pt $dir $files]
    } elseif {[string compare $phase "transform"]} {
	return {}
    } else {
	catch {eval $pt tag remove current [$pt tag ranges current]}
	set tabwin [$nb getframe $page]
	set phase [GetNextPhase]
	if {[string compare $phase "intro"]} {
	    $tabwin.controls.next configure -state normal
	} else {
	    $tabwin.controls.next configure -state disabled
	    $tabwin.controls.stop configure -text [mc "New"]
	}
    }
}

proc GetResultName fname {
    set cfgdom [::tools::getconfig]

    set result [file root $fname]_result[file extension $fname]

    set destNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/destination] 0]

    if {[llength $destNode] == 1} {

	set dirNode [lindex [dom::libxml2::selectNode $destNode dir] 0]
	set dir {}
	catch {set dir [dom::libxml2::node stringValue $dirNode]}

	set suffixNode [lindex [dom::libxml2::selectNode $destNode suffix] 0]
	set suffix {}
	catch {set suffix [dom::libxml2::node stringValue $suffixNode]}

	set result [eval file join $dir [list [file root $fname]$suffix[file extension $fname]]]
    }

    return $result
}

proc GetLinkbaseName fname {
    set filename [file join [pwd] [file dirname [file dirname $fname]] linkbase [file root [file tail $fname]]_linkbase.xml]
    if {![file exists $filename]} {
	set filename {}
    } else {
	switch $::tcl_platform(platform) {
	    windows {
		return file:/$filename
	    }
	    default {
		return file://$filename
	    }
	}
    }
}

proc doValidate {pt dir files} {
    global nb page phase

    if {[string compare $phase "validate"] && [string compare $phase "validate-result"]} {
	return {}
    }

    catch {eval $pt tag remove current [$pt tag ranges current]}

    set file [lindex $files 0]
    set files [lrange $files 1 end]

    eval $pt tag add current [$pt tag ranges result$file]
    set cursor [. cget -cursor]
    . configure -cursor watch
    $pt configure -cursor watch
    update

    if {[string compare $phase "validate"] && [string compare $phase "validate-result"]} {
	. configure -cursor $cursor
	$pt configure -cursor $cursor
	return {}
    }

    catch {::tools::validate [file join $dir $file]} msg

    . configure -cursor $cursor
    $pt configure -cursor $cursor

    $pt configure -state normal
    $pt insert "[lindex [$pt tag ranges result$file] 0] lineend" \n$msg\n msg
    $pt configure -state disabled

    if {[llength $files]} {
	after idle [list doValidate $pt $dir $files]
    } elseif {[string compare $phase "validate"] && [string compare $phase "validate-result"]} {
	return {}
    } else {
	catch {eval $pt tag remove current [$pt tag ranges current]}
	set tabwin [$nb getframe $page]
	set phase [GetNextPhase]
	if {[string compare $phase "intro"]} {
	    $tabwin.controls.next configure -state normal
	} else {
	    $tabwin.controls.next configure -state disabled
	    $tabwin.controls.stop configure -text [mc "New"]
	}
    }

    return {}
}

proc SelectFile {l dirvar selvar} {
    global nb page
    upvar #0 $dirvar currdir
    upvar #0 $selvar selectedfiles

    set files {}
    set dirselected 0

    foreach i [$l curselection] {
	set entry [$l get $i]
	if {[string equal $entry {.. Up one level}]} {
	    set entry ..
	}
	if {[file isdir [file join $currdir $entry]]} {
	    set dirselected 1
	} else {
	    lappend files $entry
	}
    }

    set tabwin [$nb getframe $page]

    if {$dirselected} {
	$tabwin.controls.next configure -state disabled
    } else {
	set selectedfiles $files
	$tabwin.controls.next configure -state normal
    }

    return {}
}

proc OpenFile {l dirvar selvar x y} {
    upvar #0 $dirvar currdir
    upvar #0 $selvar selectedfiles

    $l selection clear 0 end

    set fname [$l get [$l index @$x,$y]]
    if {[string equal $fname {.. Up one level}]} {
	set fname ..
    }

    if {[file isdir [file join $currdir $fname]]} {
	set currdir [file join $currdir $fname]
	PopulateListbox $l $dirvar
    } else {
	Next
    }

    return {}
}

proc PopulateListbox {l dirvar} {
    global dirlist
    upvar #0 $dirvar currdir

    set pwd [pwd]
    cd $currdir
    set dirlist [glob -nocomplain *]
    cd $pwd

    set dirlist [linsert [lsort -dictionary $dirlist] 0 {.. Up one level}]

    switch $::tcl_platform(platform) {
	windows -
	macintosh {
	    eval lappend dirlist [file volume]
	}
    }

    return {}
}

set ViewCounter 0

proc View fname {
    global ViewCounter

    if {![file exists $fname]} {
	tk_messageBox -message [mc "File \"$fname\" does not exist"] -type ok -icon error
	return {}
    }

    set top [toplevel .view[incr ViewCounter]]
    wm title $top "[file tail $fname] : Document View"

    upvar #0 view.line$top line
    upvar #0 view.column$top column

    set t [text $top.t -width 100 -height 50 -yscrollcommand [list $top.y set] -xscrollcommand [list $top.x set]]
    scrollbar $top.x -orient horizontal -command [list $top.t xview]
    scrollbar $top.y -orient vertical -command [list $top.t yview]

    $top.t tag configure highlight -foreground red

    frame $top.f
    label $top.f.linelabel -text Line:
    entry $top.f.l -textvariable view.line$top
    label $top.f.collabel -text Column:
    entry $top.f.c -textvariable view.column$top

    grid $top.t -row 0 -column 0 -sticky news
    grid $top.x -row 1 -column 0 -sticky ew
    grid $top.y -row 0 -column 1 -sticky ns
    grid $top.f - -row 2 -column 0 -sticky ew
    grid $top.f.linelabel -row 0 -column 0 -sticky ew
    grid $top.f.l -row 0 -column 1 -sticky ew
    grid $top.f.collabel -row 0 -column 2 -sticky ew
    grid $top.f.c -row 0 -column 3 -sticky ew

    grid rowconfigure $top 0 -weight 1
    grid columnconfigure $top 0 -weight 1
    grid rowconfigure $top.f 0 -weight 1
    grid columnconfigure $top.f 0 -weight 1
    grid columnconfigure $top.f 1 -weight 1
    grid columnconfigure $top.f 2 -weight 1
    grid columnconfigure $top.f 3 -weight 1

    bind update <Key> [list keyupdate %W]
    bindtags $top.f.l "[bindtags $top.f.l] update"
    bindtags $top.f.c "[bindtags $top.f.c] update"

    set line 1
    set column 0
    set ch [open $fname]
    $top.t insert end [read $ch]
    close $ch
    $top.t configure -state disabled
    #keyupdate $top.t

}

proc keyupdate win {
    set top [winfo toplevel $win]

    upvar #0 view.line$top line
    upvar #0 view.column$top column
    upvar #0 view.flashid$top flashid

    catch {after cancel $flashid}

    if {![string length $line] || ![string length $column]} {
	return
    }

    $top.t see ${line}.$column
    $top.t mark set insert ${line}.$column

    flash $top.t ${line}.$column 10 10

    return {}
}

# TODO: set a destroy handler on the window to
# properly cleanup the after handler

proc flash {w index offset count} {
    if {![winfo exists $w]} {
	return {}
    }

    set top [winfo toplevel $w]
    upvar #0 view.flashid$top flashid

    $w tag remove highlight 1.0 end

    if {[expr $count % 2] == 1} {
	$w tag add highlight "$index - $offset chars" "$index + $offset chars"
    }
    if {$count > 0} {
	set flashid [after 250 [list flash $w $index $offset [expr $count - 1]]]
    } else {
	$w tag add highlight "$index - $offset chars" "$index + $offset chars"
    }
}

### Main program

set initialised 0
set page {}
set tab {}
set stage start

array set cmdlineargs {}
catch {array set cmdlineargs $argv}

if {![info exists env(XMLPOWERTOOLDEBUG)]} {
    catch {console hide}
}
if {[info exists cmdlineargs(-debug)] && $cmdlineargs(-debug)} {
    catch {console show}
}

if {[info exists cmdlineargs(-config)] > 0} {
    Init $cmdlineargs(-config)
} elseif {[info exists env(XMLPOWERTOOLCONFIGRC)]} {
    Init $env(XMLPOWERTOOLCONFIGRC)
} elseif {[file exists [file join $env(HOME) .xmlpowertoolrc]]} {
    Init [file join $env(HOME) .xmlpowertoolrc]
} elseif {[file exists [file join [pwd] config.xml]]} {
    Init [file join [pwd] config.xml]
} elseif {[file exists [file join [file dirname [file dirname [info script]]] etc config.xml]]} {
    Init [file join [file dirname [file dirname [info script]]] etc config.xml]
} elseif {[file exists [file join [file dirname [info script]] etc config.xml]]} {
    Init [file join [file dirname [info script]] etc config.xml]
}

if {!$initialised} {
    tk_messageBox -title "No Configuration" -message "Can't find configuration"
    exit 1
}

# Default initial page determined by config doc
$nb raise $page

# Start from user's home directory
# Should this be configurable?
catch {set currdir $env(HOME)}

New
