/*
 * saxHandlers.c --
 *
 *	This file implements callbacks from the Xerces
 *      XML parser.
 *
 * Copyright (c) 2000 Ajuba Solutions.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: $Id: saxHandlers.cpp,v 1.38 2000/07/27 16:22:33 awb Exp $
 */

#include "XercesSax.h"
#ifdef _WINDOWS
#include <memory>
#else
#include <memory.h>
#endif


/*
 *--------------------------------------------------------------
 *
 * SetStartLocation
 *
 *      Record the starting location (line, column) for a node.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void SetStartLocation (XercesParserData *parserPtr, Tdp_Node node, 
    int width=-1) 
{
    if (parserPtr->currentLine == parserPtr->previousLine) {
	Tdp_SetStartLocation(node, parserPtr->previousLine,
		parserPtr->previousColumn-1, 
		(width == -1) ? 
		parserPtr->currentColumn-parserPtr->previousColumn:width,
		parserPtr->currentLine, parserPtr->currentColumn-1);
    } else {
	Tdp_SetStartLocation(node, parserPtr->previousLine,
		parserPtr->previousColumn-1, width,
		parserPtr->currentLine, parserPtr->currentColumn-1);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SetEndLocation
 *
 *      Record the end location (line, column) for a node.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void SetEndLocation (XercesParserData *parserPtr, Tdp_Node node) 
{
    if (parserPtr->currentLine == parserPtr->previousLine) {
	Tdp_SetEndLocation(node, parserPtr->previousLine,
		parserPtr->previousColumn-1, 
		parserPtr->currentColumn-parserPtr->previousColumn,
		parserPtr->currentLine, parserPtr->currentColumn-1);
    } else {
	Tdp_SetEndLocation(node, parserPtr->previousLine,
		parserPtr->previousColumn-1, (unsigned int) -1,
		parserPtr->currentLine, parserPtr->currentColumn-1);
    }
    
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::SaxTclHandler --
 *
 *      Constructor for the SAXTclHandler class. This class
 *      handles callback for the SAX API.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclHandler::SAXTclHandler(
    const bool doEscapes,
    Tcl_Interp *setInterp,
    XercesParserData *setParserPtr)
    : fDoEscapes(doEscapes)
{
    interp = setInterp;
    parserPtr = setParserPtr;
    handlerNamespace = NULL;
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::SaxTclHandler --
 *
 *      Destructor for the SAXTclHandler class.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclHandler::~SAXTclHandler()
{
    if (handlerNamespace) {
	ckfree(handlerNamespace);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::genericError --
 *
 *      Perform processing common to all error handlers.
 *      Invoke the SAX callback script if the saxNamespace
 *      option is defined. Append the error information to a
 *      list variable.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void 
SAXTclHandler::genericError(const SAXParseException& e, char *saxType)
{
    Tcl_DString errorString;
    Tcl_DString cmdString;
    Tcl_DString appendedError;
    int result;
    char work[64];

    /*
     * Save the error in the parser object
     */

    Tcl_DStringInit(&errorString);
    Tcl_UniCharToUtfDString(e.getMessage(), 
	    XMLString::stringLen(e.getMessage()), &errorString);

    Tcl_DStringInit(&appendedError);

    sprintf(work, "line: %d", e.getLineNumber());
    Tcl_DStringAppend(&appendedError, work, -1);
    sprintf(work, " column: %d ", e.getColumnNumber());
    Tcl_DStringAppend(&appendedError, work, -1);
    Tcl_DStringAppend(&appendedError, Tcl_DStringValue(&errorString), -1);

    /*
     * We save only errors in "parseErrors"; 
     * we save everything in "parseErrorsAndWarnings"
     */

    if (strcmp(saxType, "warning") != 0) {
	/*
	 * Not a warning
	 */

#if defined(SOLARIS) || defined(HP_UX)
	/*
	 * The "throw" mechanism doesn't currently work with shared 
	 * libraries on some platforms. This means we may have already
	 * had an error in a Tcl callback that we need to check for.
	 */
	if (Tcl_DStringLength(&parserPtr->parseErrors) == 0) {
	    Tcl_DStringAppend(&parserPtr->parseErrors, 
		    Tcl_DStringValue(&appendedError), -1);
	}
#else
	Tcl_DStringAppend(&parserPtr->parseErrors, 
		Tcl_DStringValue(&appendedError), -1);
#endif
    }
    Tcl_DStringAppendElement(&parserPtr->parseErrorsAndWarnings, 
	    Tcl_DStringValue(&appendedError));

    Tcl_DStringFree(&appendedError);

    if (handlerNamespace) {
        Tcl_DStringInit(&cmdString);
        Tcl_DStringAppend(&cmdString, handlerNamespace, -1);
        Tcl_DStringAppend(&cmdString, "::", -1);
	Tcl_DStringAppend(&cmdString, saxType, -1);
        sprintf(work, " %d", e.getLineNumber());
        Tcl_DStringAppend(&cmdString, work, -1);
        sprintf(work, " %d ", e.getColumnNumber());
        Tcl_DStringAppend(&cmdString, work, -1);
        Tcl_DStringAppendElement(&cmdString, 
		Tcl_DStringValue(&errorString));
        result = Tcl_EvalEx(interp, Tcl_DStringValue(&cmdString), 
		Tcl_DStringLength(&cmdString), TCL_EVAL_GLOBAL);
        if (result != TCL_OK) {
            printf("%s: %s\n", saxType, Tcl_GetVar(interp, "errorInfo", 
		    TCL_GLOBAL_ONLY));
        }
        Tcl_DStringFree(&cmdString);
    }   
    Tcl_DStringFree(&errorString);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::error --
 *
 *      Handler for the SAX error event.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::error(
    const SAXParseException& e)
{
#if !defined(SOLARIS) && !defined(HP_UX)
    throw e;
#endif
    genericError(e, "error");
    parserPtr->parseStatus = TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::fatalError --
 *
 *      Handler for the SAX fatalError event.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::fatalError(
    const SAXParseException& e)
{
#if !defined(SOLARIS) && !defined(HP_UX)
    throw e;
#endif
    genericError(e, "fatalError");
    parserPtr->parseStatus = TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::warning --
 *
 *      Handler for the SAX warning event.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::warning(const SAXParseException& e)
{
    genericError(e, "warning");
    if (parserPtr->treatWarningsAsErrors) {
	parserPtr->parseStatus = TCL_ERROR;
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::unparsedEntityDecl --
 *
 *      Handler for the SAX unparsedEntityDecl event of the
 *      SAX DTDHandler interface.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::unparsedEntityDecl(
    const XMLCh* const name,
    const XMLCh* const publicId,
    const XMLCh* const systemId,
    const XMLCh* const notationName)
{
#if defined(_WINDOWS) && defined(_DEBUG)
    _asm int 3
#endif
	// Not used at this time
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::unparsedEntityDecl --
 *
 *      Handler for the SAX notationDecl event of the
 *      SAX DTDHandler interface.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::notationDecl(
    const XMLCh* const name,
    const XMLCh* const publicId,
    const XMLCh* const systemId)
{
#if defined(_WINDOWS) && defined(_DEBUG)
    _asm int 3
#endif
	// Not used at this time
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::startDocument --
 *
 *      Handler for the SAX startDocument event of the
 *      SAX DocumentHandler interface.  Invoke the SAX callback 
 *      script if the saxNamespace option is defined.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::startDocument()
{
    Tcl_DString cmdString;
    int result;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    if (handlerNamespace) {
	Tcl_DStringInit(&cmdString);
	Tcl_DStringAppend(&cmdString, handlerNamespace, -1);
	Tcl_DStringAppend(&cmdString, "::startDocument", -1);
	result = Tcl_EvalEx(interp, Tcl_DStringValue(&cmdString),
		Tcl_DStringLength(&cmdString), TCL_EVAL_GLOBAL);
	if (result != TCL_OK) {
	    printf("startDocumentError: %s\n", Tcl_GetVar(interp,
		    "errorInfo", TCL_GLOBAL_ONLY));
	}
	Tcl_DStringFree(&cmdString);
    } 

    if (parserPtr->useDom) {
	parserPtr->currentDocument = Tdp_CreateDocument(interp);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::endDocument --
 *
 *      Handler for the SAX endDocument event of the
 *      SAX DocumentHandler interface.  Invoke the SAX callback 
 *      script if the saxNamespace option is defined.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */


void
SAXTclHandler::endDocument()
{
    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::startElement --
 *
 *      Handler for the SAX startElement event of the
 *      SAX DocumentHandler interface.  Invoke a Tcl callback.
 *      If DOM tree generation is enabled, then add the node
 *      to the current DOM tree.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::startElement(
    const XMLCh* const name,       
    AttributeList&  attributes)
{
    int length, i, result;
    Tcl_DString nameString;
    Tcl_DString attributeNameString;
    Tcl_DString attributeValueString;
    Tcl_Obj *cmdObjPtr;
    Tcl_Obj *attributeObj;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    Tcl_DStringInit(&nameString);
    Tcl_UniCharToUtfDString(name, XMLString::stringLen(name), &nameString);

    length = attributes.getLength();

    if (parserPtr->elementStartCommand || parserPtr->handlerNamespace
	    || parserPtr->defaultCommand) {
	if (parserPtr->elementStartCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->elementStartCommand);
	} else if (parserPtr->handlerNamespace) {
	    Tcl_DString cmdString;
	    Tcl_DStringInit(&cmdString);
	    Tcl_DStringAppend(&cmdString,
		    Tcl_GetString(parserPtr->handlerNamespace), -1);
	    Tcl_DStringAppend(&cmdString, "::startElement", -1);
	    cmdObjPtr = Tcl_NewStringObj(Tcl_DStringValue(&cmdString), -1);
	    Tcl_IncrRefCount(cmdObjPtr);

	    Tcl_DStringFree(&cmdString);
	} else if (parserPtr->defaultCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->defaultCommand);
	}

	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);
	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&nameString), -1)); 

	attributeObj = Tcl_NewListObj(0, NULL);

	length = attributes.getLength();
	for (i = 0; i < length; i++) {
	    Tcl_DStringInit(&attributeNameString);
	    Tcl_DStringInit(&attributeValueString);
	    Tcl_UniCharToUtfDString(attributes.getName(i),
		    XMLString::stringLen(attributes.getName(i)), 
		    &attributeNameString);
	    Tcl_UniCharToUtfDString(attributes.getValue(i),
		    XMLString::stringLen(attributes.getValue(i)), 
		    &attributeValueString);
	    Tcl_ListObjAppendElement(interp, attributeObj,
		    Tcl_NewStringObj(Tcl_DStringValue(&attributeNameString),
			    -1));
	    Tcl_ListObjAppendElement(interp, attributeObj,
		    Tcl_NewStringObj(Tcl_DStringValue(&attributeValueString),
			    -1));
            
	    Tcl_DStringFree(&attributeNameString);
	    Tcl_DStringFree(&attributeValueString);
	}
	Tcl_ListObjAppendElement(interp, cmdObjPtr, attributeObj);
        
	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
    } 

    if (parserPtr->useDom) {
	TdpDomError domError;
	Tdp_Element element;
	Tdp_Node parentNode;
	domError = Tdp_CreateElement(interp, parserPtr->currentDocument, 
		Tcl_DStringValue(&nameString), &element);

	for (i = 0; i < length; i++) {
	    Tcl_DStringInit(&attributeNameString);
	    Tcl_DStringInit(&attributeValueString);
	    Tcl_UniCharToUtfDString(attributes.getName(i),
		    XMLString::stringLen(attributes.getName(i)), 
		    &attributeNameString);
	    Tcl_UniCharToUtfDString(attributes.getValue(i),
		    XMLString::stringLen(attributes.getValue(i)), 
		    &attributeValueString);
	    Tdp_SetAttribute(interp, element,
		    Tcl_DStringValue(&attributeNameString),
		    Tcl_DStringValue(&attributeValueString));
	    Tcl_DStringFree(&attributeNameString);
	    Tcl_DStringFree(&attributeValueString);
	}

	if (parserPtr->depth == 0) {
	    parentNode = (Tdp_Node) Tdp_GetDocumentElement(
		parserPtr->currentDocument);
	} else {
	    parentNode = parserPtr->currentNode;
	}
	domError = Tdp_AppendChild(interp, parentNode, (Tdp_Node) element);
	SetStartLocation(parserPtr, (Tdp_Node) element);
	parserPtr->currentNode = (Tdp_Node) element;
	parserPtr->depth++;
    }

    Tcl_DStringFree(&nameString);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::endElement --
 *
 *      Handler for the SAX endElement event of the
 *      SAX DocumentHandler interface.  Invoke a Tcl callback.
 *      If DOM tree generation is enabled, then finish processing
 *      the open element node.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */


void
SAXTclHandler::endElement(
    const XMLCh* const name)
{
    int result;
    Tcl_DString nameString;
    Tcl_Obj *cmdObjPtr;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    if (parserPtr->elementEndCommand || parserPtr->handlerNamespace
	    || parserPtr->defaultCommand) {
	if (parserPtr->elementEndCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->elementEndCommand);
	} else if (parserPtr->handlerNamespace) {
	    Tcl_DString cmdString;
	    Tcl_DStringInit(&cmdString);
	    Tcl_DStringAppend(&cmdString,
		    Tcl_GetString(parserPtr->handlerNamespace), -1);
	    Tcl_DStringAppend(&cmdString, "::endElement", -1);
	    cmdObjPtr = Tcl_NewStringObj(Tcl_DStringValue(&cmdString), -1);
	    Tcl_DStringFree(&cmdString);
	} else if (parserPtr->defaultCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->defaultCommand);
	}
	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);

	Tcl_DStringInit(&nameString);
	Tcl_UniCharToUtfDString(name, XMLString::stringLen(name), &nameString);
	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&nameString), -1));
	Tcl_DStringFree(&nameString);

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
    } 
    if (parserPtr->useDom) {
	parserPtr->depth--;

	SetEndLocation(parserPtr, parserPtr->currentNode);
	if (parserPtr->depth != 0) {
	    parserPtr->currentNode = Tdp_GetParentNode(parserPtr->currentNode);
	}
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::characters --
 *
 *      Handler for the SAX characters event of the
 *      SAX DocumentHandler interface.  Invoke a Tcl callback.
 *      If DOM tree generation is enabled, then append a Text
 *      node at the current tree position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::characters(
    const XMLCh* const chars,
    const unsigned int length)
{
    int result;
    Tcl_DString charsString;
    Tcl_Obj *cmdObjPtr;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    Tcl_DStringInit(&charsString);
    Tcl_UniCharToUtfDString(chars, length, &charsString);

    if (parserPtr->characterDataCommand || parserPtr->handlerNamespace
	    || parserPtr->defaultCommand) {
	if (parserPtr->characterDataCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->characterDataCommand);
	} else if (parserPtr->handlerNamespace) {
	    Tcl_DString cmdString;
	    Tcl_DStringInit(&cmdString);
	    Tcl_DStringAppend(&cmdString,
		    Tcl_GetString(parserPtr->handlerNamespace), -1);
	    Tcl_DStringAppend(&cmdString, "::characters", -1);
	    cmdObjPtr = Tcl_NewStringObj(Tcl_DStringValue(&cmdString), -1);
	    Tcl_DStringFree(&cmdString);
	} else if (parserPtr->defaultCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->defaultCommand);
	}
	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);
	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&charsString), length));

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
    }   
    
    if (parserPtr->useDom) {
	TdpNodeType nodeType;
	Tdp_TextNode textNode;
	Tdp_Node parentNode, lastChild;
	TdpDomError domError;
	parentNode = parserPtr->currentNode;
	lastChild = Tdp_GetLastChild(parentNode);

	if (lastChild && ((nodeType = Tdp_GetNodeType(lastChild))
		== TDP_TEXT_NODE)) {
	    unsigned int nodeLine, nodeColumn, nodeWidth;
	    char *s;
	    Tcl_DString newValueString;
	    /*
	     * Normalize text -- i.e., combine sibling text nodes
	     */
	    s = Tdp_GetNodeValue(lastChild);
	    Tdp_GetStartLocation((Tdp_Node) lastChild, &nodeLine, &nodeColumn,
		    &nodeWidth);
	    Tcl_DStringInit(&newValueString);
	    Tcl_DStringAppend(&newValueString, s, -1);
	    Tcl_DStringAppend(&newValueString, Tcl_DStringValue(&charsString),
		    -1);
	    Tdp_SetNodeValue(lastChild, Tcl_DStringValue(&newValueString));
	    Tdp_SetStartLocation(lastChild, nodeLine, nodeColumn,
		    Tcl_DStringLength(&newValueString), parserPtr->currentLine, 
		    parserPtr->currentColumn-1);
	    SetEndLocation(parserPtr, (Tdp_Node) lastChild);
	    Tcl_DStringFree(&newValueString);
	} else {
	    if (Tdp_GetNodeType(parentNode) == TDP_ELEMENT_NODE) {
		textNode = Tdp_CreateTextNode(interp,
			parserPtr->currentDocument,
			Tcl_DStringValue(&charsString));
		domError = Tdp_AppendChild(interp, parentNode,
			(Tdp_Node) textNode);
		SetStartLocation(parserPtr, (Tdp_Node) textNode, length);
		SetEndLocation(parserPtr, (Tdp_Node) textNode);
	    } else {
#if defined(_WINDOWS) && defined(_DEBUG)
		_asm int 3
#endif
	    }
	}
    }
    Tcl_DStringFree(&charsString);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::ignorableWhitespace --
 *
 *      Handler for the SAX ignorableWhitepspace event of the
 *      SAX DocumentHandler interface.  Invoke the "characters"
 *      routine depending on the state of the "ignoreWhiteSpace"
 *      flag.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::ignorableWhitespace(
    const   XMLCh* const chars,
    const  unsigned int length)
{
    if (!parserPtr->ignoreWhiteSpace) {
	characters(chars, length);
    } else {
	XMLScanner& scanner
	    = (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

	parserPtr->previousLine = parserPtr->currentLine;
	parserPtr->previousColumn = parserPtr->currentColumn;

	scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
		parserPtr->currentPubId, ID_SIZE, 
		parserPtr->currentLine,  
		parserPtr->currentColumn);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::processingInstruction --
 *
 *      Handler for the SAX processingInstruction event of the
 *      SAX DocumentHandler interface.  Invoke a Tcl callback.
 *      If DOM tree generation is enabled, then append a PI
 *      node at the current tree position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::processingInstruction(
    const  XMLCh* const target,
    const XMLCh* const data)
{
    int result;
    Tcl_DString targetString;
    Tcl_DString dataString;
    Tcl_Obj **objv;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    Tcl_DStringInit(&targetString);
    Tcl_UniCharToUtfDString(target, XMLString::stringLen(target),
	&targetString);

    Tcl_DStringInit(&dataString);
	Tcl_UniCharToUtfDString(data, XMLString::stringLen(data), &dataString);

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    if (parserPtr->processingInstructionCommand
	    || parserPtr->handlerNamespace || parserPtr->defaultCommand) {
	objv = (Tcl_Obj **) ckalloc(sizeof(Tcl_Obj *) * 3);
	if (parserPtr->processingInstructionCommand) {
	    objv[0] = parserPtr->processingInstructionCommand;
	} else if (parserPtr->handlerNamespace) {
	    Tcl_DString cmdString;
	    Tcl_DStringInit(&cmdString);
	    Tcl_DStringAppend(&cmdString,
		    Tcl_GetString(parserPtr->handlerNamespace), -1);
	    Tcl_DStringAppend(&cmdString, "::processingInstruction", -1);
	    objv[0] = Tcl_NewStringObj(Tcl_DStringValue(&cmdString), -1);
	    Tcl_DStringFree(&cmdString);
	} else if (parserPtr->defaultCommand) {
	    objv[0] = parserPtr->defaultCommand;
	}

	objv[1] = Tcl_NewStringObj(Tcl_DStringValue(&targetString), -1);
	Tcl_IncrRefCount(objv[1]);
        
	objv[2] = Tcl_NewStringObj(Tcl_DStringValue(&dataString), -1);
	Tcl_IncrRefCount(objv[2]);
       
	result = Tcl_EvalObjv(interp, 3, objv, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(objv[1]);
	Tcl_DecrRefCount(objv[2]);
    } 
    if (parserPtr->useDom) {
	TdpDomError domError;
	Tdp_Node parentNode;
	Tdp_PINode piNode;
	if (parserPtr->depth == 0) {
	    parentNode = (Tdp_Node) Tdp_GetDocumentElement(
		parserPtr->currentDocument);
	} else {
	    parentNode = parserPtr->currentNode;
	}
	piNode = Tdp_CreateProcessingInstructionNode(interp,
		parserPtr->currentDocument, Tcl_DStringValue(&targetString),
		Tcl_DStringValue(&dataString));
	domError = Tdp_AppendChild(interp, parentNode,
		(Tdp_Node) piNode);
	SetStartLocation(parserPtr, (Tdp_Node) piNode);
	SetEndLocation(parserPtr, (Tdp_Node) piNode);
    }

    Tcl_DStringFree(&targetString);
    Tcl_DStringFree(&dataString);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::setDocumentLocator --
 *
 *      Handler for the SAX setDocumentLocator event of the
 *      SAX DocumentHandler interface.  This callback appears
 *      not to be implemented in Xerces 1.1. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::setDocumentLocator(
    Locator* const locator)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::setTclNamespace --
 *
 *      Set a namespace in which to invoke the SAX API events
 *      as Tcl scripts.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclHandler::setTclNamespace(
    Tcl_Interp *myInterp,
    char *tclNamespace)
{
    interp = myInterp;
    if (handlerNamespace) {
	ckfree(handlerNamespace);
    }
    handlerNamespace = ckalloc(strlen(tclNamespace));
    strcpy(handlerNamespace, tclNamespace);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::SaxTclHandler --
 *
 *      Constructor for the SAXTclResolver class. Methods of this
 *      class are invoked by Xerces to give applications a
 *      chance to resolve entities. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclResolver::SAXTclResolver(
    Tcl_Interp *setInterp,
    XercesParserData *setParserPtr)
{
    interp = setInterp;
    parserPtr = setParserPtr;
}

/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::SaxTclHandler --
 *
 *      Destructor for the SAXTclResolver class.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclResolver::~SAXTclResolver()
{
}

/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::resolveEntity --
 *
 *      Method to resolve entities. Invoke a Tcl script that
 *      may optionally resolve the entity. If the script
 *      chooses to resolve the entity, if should return the
 *      entity value in its return statement. If it can't
 *      resolve the entity, it should return TCL_CONTINUE.
 *      Any errors in the script are treated as parsing errors
 *      and are reported as such.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

InputSource *
SAXTclResolver::resolveEntity(
    const XMLCh* const publicId,
    const XMLCh* const systemId)
{
    int result;
    Tcl_DString publicIdString;
    Tcl_DString systemIdString;
    Tcl_Obj *cmdObjPtr;
    MemBufInputSource *resolvedValuePtr;

    if (parserPtr->externalEntityCommand || parserPtr->handlerNamespace
	    || parserPtr->defaultCommand) {
	if (parserPtr->externalEntityCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->externalEntityCommand);
	} else if (parserPtr->handlerNamespace) {
	    Tcl_DString cmdString;
	    Tcl_DStringInit(&cmdString);
	    Tcl_DStringAppend(&cmdString,
		    Tcl_GetString(parserPtr->handlerNamespace), -1);
	    Tcl_DStringAppend(&cmdString, "::resolveEntity", -1);
	    cmdObjPtr = Tcl_NewStringObj(Tcl_DStringValue(&cmdString), -1);
	    Tcl_DStringFree(&cmdString);
	} else if (parserPtr->defaultCommand) {
	    cmdObjPtr = Tcl_DuplicateObj(parserPtr->defaultCommand);
	}

	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);

	/*
	 * Append "names" argument
	 */

	Tcl_ListObjAppendElement(interp, cmdObjPtr, Tcl_NewStringObj("", 0));

	/*
	 * Append "base" argument
	 */

	Tcl_ListObjAppendElement(interp, cmdObjPtr, Tcl_NewStringObj("", 0));

	Tcl_DStringInit(&publicIdString);
	Tcl_UniCharToUtfDString(publicId, XMLString::stringLen(publicId),
		&publicIdString);
	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&publicIdString), -1));
	Tcl_DStringFree(&publicIdString);
        
	Tcl_DStringInit(&systemIdString);
	Tcl_UniCharToUtfDString(systemId, XMLString::stringLen(systemId),
		&systemIdString);
	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&systemIdString), -1));
	Tcl_DStringFree(&systemIdString);

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
	if (result == TCL_OK) {
	    char *value = Tcl_GetStringResult(interp);
	    resolvedValuePtr = new MemBufInputSource((const XMLByte *) value,
		    strlen(value), 
		    "TestCase", 0);
	    return resolvedValuePtr;
	} else if (result == TCL_CONTINUE) {
	    return NULL;
	} else {
	    Tcl_DString errorString;
	    Tcl_DStringInit(&errorString);
	    Tcl_UtfToUniCharDString(Tcl_GetStringResult(interp), -1, 
		&errorString);
#if defined(SOLARIS) || defined(HP_UX)
	    /*
	     * The exception mechanism doesn't currently work with shared
	     * libraries on some platforms, so we can't throw an exception.
	     * Just save the error info, and return NULL, which will cause
	     * some other error further along within Xerces itself. We'll 
	     * clean things up when we get the second error
	     */
	    char work[64];

	    sprintf(work, "line: %d", parserPtr->currentLine);
	    Tcl_DStringAppend(&parserPtr->parseErrors, work, -1);
	    sprintf(work, " column: %d ", parserPtr->currentColumn);
	    Tcl_DStringAppend(&parserPtr->parseErrors, work, -1);
	    Tcl_DStringAppend(&parserPtr->parseErrors, 
		    Tcl_DStringValue(&errorString), -1);

	    Tcl_DStringAppend(&parserPtr->parseErrors,
		    Tcl_GetStringResult(interp), -1);
	    Tcl_DStringAppendElement(&parserPtr->parseErrorsAndWarnings,
		    Tcl_GetStringResult(interp));
	    return NULL;
#else
	    /*
	     * We're on a well-behaved platform, so do the right thing
	     */
	    SAXParseException toThrow = SAXParseException(
		(const XMLCh *) Tcl_DStringValue(&errorString),
		(const XMLCh *) NULL,
		(const XMLCh *) NULL,
		parserPtr->currentLine,
		parserPtr->currentColumn
	    );
	    Tcl_DStringFree(&errorString);
	    throw toThrow;
#endif
	}
    } 
#ifdef UNDEF
    MemBufInputSource *newSourcePtr
	= new MemBufInputSource((const XMLByte *) "xxx", 3, "TestCase", 0);
#endif
    return NULL;
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::SaxAdvTclHandler --
 *
 *      Constructor for the SAXAdvTclHandler class. Methods of this
 *      class are invoked by Xerces to give applications a
 *      chance to handle events that aren't covered by the SAX
 *      API. We use this to handle XML comments. Note that 
 *      arguably this API could have been used instead of the 
 *      SAX API; I'm not sure there's a big difference.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXAdvTclHandler::SAXAdvTclHandler(
    Tcl_Interp *interpValue,
    XercesParserData *parserValue)
{
    interp = interpValue;
    parserPtr = parserValue;
}

/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::~SaxAdvTclHandler --
 *
 *      Destructor for the SAXAdvTclHandler class. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXAdvTclHandler::~SAXAdvTclHandler()
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::docCharacters --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::docCharacters(
    const XMLCh * const chars,
    const unsigned int length,
    const bool cdataSection)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXAdvTclHandler::docComment --
 *
 *      Handler for the docComment callback from the Xerces
 *      parser.  Invoke a Tcl callback.
 *      If DOM tree generation is enabled, then append a Comment
 *      node at the current tree position.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::docComment(
    const XMLCh * const chars)
{
    int result;
    Tcl_DString commentString;
    Tcl_Obj *cmdObjPtr;

    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;


    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

    Tcl_DStringInit(&commentString);
    Tcl_UniCharToUtfDString(chars, XMLString::stringLen(chars),
	&commentString);

    if (parserPtr->commentCommand) {
	cmdObjPtr = Tcl_DuplicateObj(parserPtr->commentCommand);
        
	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);

	Tcl_ListObjAppendElement(interp, cmdObjPtr,
		Tcl_NewStringObj(Tcl_DStringValue(&commentString), -1));

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
    } 

    if (parserPtr->useDom) {
	TdpDomError domError;
	Tdp_CommentNode commentNode;
	Tdp_Node parentNode;
	commentNode = Tdp_CreateCommentNode(interp,
		parserPtr->currentDocument, Tcl_DStringValue(&commentString));
	if (parserPtr->depth == 0) {
	    parentNode = (Tdp_Node) Tdp_GetDocumentElement(
		parserPtr->currentDocument);
	} else {
	    parentNode = parserPtr->currentNode;
	}
	domError = Tdp_AppendChild(interp, parentNode,
		(Tdp_Node) commentNode);
	SetStartLocation(parserPtr, (Tdp_Node) commentNode);
	SetEndLocation(parserPtr, (Tdp_Node) commentNode);
    }

    Tcl_DStringFree(&commentString);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::docPI --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::docPI(
    const XMLCh * const chars,
    const XMLCh * const data)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::endDocument --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::endDocument()
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::ignorableWhitespace --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::ignorableWhitespace(
    const XMLCh * const chars,
    const unsigned int length,
    const bool cdataSection)
{
    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::resetDocument --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::resetDocument()
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::startDocument --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::startDocument()
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::startElement --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::startElement(
    const XMLElementDecl& elemDecl,
    const unsigned int uriId,
    const XMLCh* const prefixName,
    const RefVectorOf<XMLAttr>& attrList,
    const unsigned int attrCount,
    const bool isEmpty,
    const bool isRoot)
{
    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::endElement --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::endElement(
    const XMLElementDecl& elemDecl,
    const unsigned int uriId,
    const bool isRoot)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::startEntityReference --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::startEntityReference(
    const XMLEntityDecl& entityDecl)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::endEntityReference --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::endEntityReference(
    const XMLEntityDecl& entDecl)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclHandler::XMLDecl --
 *
 *      Not used -- do nothing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXAdvTclHandler::XMLDecl(
    const XMLCh * const versionStr,
    const XMLCh * const encodingStr,
    const XMLCh * const standaloneStr,
    const XMLCh * const autoEncodingStr)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::SaxTclDocTypeHandler --
 *
 *      Constructor for the SAXTclDocTypeHandler class. 
 *      Methods of this
 *      class are invoked by Xerces during DTD processing.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclDocTypeHandler::SAXTclDocTypeHandler(
    Tcl_Interp *interpValue,
    XercesParserData *parserValue)
{
    interp = interpValue;
    parserPtr = parserValue;
    attrListPtr = NULL;
}

/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::~SaxTclDocTypeHandler --
 *
 *      Destructor for the SAXTclDocTypeHandler class. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

SAXTclDocTypeHandler::~SAXTclDocTypeHandler()
{
    if (attrListPtr) {
	Tcl_DecrRefCount(attrListPtr);
	attrListPtr = NULL;
    }
}




/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::docTypeComment --
 *
 *      Process a comment in a DOCTYPE declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::doctypeComment(
    const XMLCh* const comment)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::docTypeDecl --
 *
 *      Process the DOCTYPE declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::doctypeDecl(
    const DTDElementDecl& elemDecl,
    const XMLCh* const publicId,
    const XMLCh* const systemId,
    const bool hasIntSubset)
{
    Tcl_DString nameString;
    Tcl_DString publicIdString;
    Tcl_DString systemIdString;

    Tcl_DStringInit(&nameString);
    Tcl_DStringInit(&publicIdString);
    Tcl_DStringInit(&systemIdString);

    Tcl_UniCharToUtfDString(elemDecl.getBaseName(),
	    XMLString::stringLen(elemDecl.getBaseName()), &nameString);
    if (publicId) {
	Tcl_UniCharToUtfDString(publicId, XMLString::stringLen(publicId),
		&publicIdString);
    }
    if (systemId) {
	Tcl_UniCharToUtfDString(systemId, XMLString::stringLen(systemId),
		&systemIdString);
    }

    if (parserPtr->useDom) {
	Tdp_DocumentType documentType;
	
	documentType = Tdp_CreateDocumentType(interp,
		parserPtr->currentDocument, 
		Tcl_DStringValue(&nameString), 
		publicId ? Tcl_DStringValue(&publicIdString) : NULL, 
		systemId ? Tcl_DStringValue(&systemIdString) : NULL);

	Tdp_SetDocumentType(interp, parserPtr->currentDocument, documentType);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::docTypePI --
 *
 *      Process a PI in a DOCTYPE declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */
void
SAXTclDocTypeHandler::doctypePI(
    const XMLCh* const target,
    const XMLCh* const data)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::doctypeWhitespace --
 *
 *      Process whitespace within a DOCTYPE declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */
void
SAXTclDocTypeHandler::doctypeWhitespace(
    const XMLCh* const chars,
    const unsigned int length)
{
    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);
}

void SAXTclDocTypeHandler::FormatNodes(Tcl_Obj *specObjPtr, 
    const ContentSpecNode *curNode, 
    ContentSpecNode::NodeTypes  parentType) 
{
    const ContentSpecNode* first = curNode->getFirst();
    const ContentSpecNode* second = curNode->getSecond();
    const ContentSpecNode::NodeTypes curType = curNode->getType();
    int id;
    XMLElementDecl *elem;
    Tcl_DString workString;
    Tcl_Obj *objv[1];
    int objc;
    Tcl_Obj *leftObj, *rightObj;
    int startPos, endPos;
    DTDValidator& validator = 
	    (class DTDValidator &) parserPtr->saxParserPtr->getValidator();

    Tcl_DStringInit(&workString); 

    // Get the type of the first node
    const ContentSpecNode::NodeTypes firstType = first ?
                                                 first->getType() :
                                                 ContentSpecNode::Leaf;

    // Calculate the parens flag for the rep nodes
    bool doRepParens = false;
    if (((firstType != ContentSpecNode::Leaf) && (parentType != -1))
	    ||  ((firstType == ContentSpecNode::Leaf) && (parentType == -1))) {
        doRepParens = true;
    }

    // Now handle our type
    unsigned int tmpVal;
    switch(curType) {
        case ContentSpecNode::Leaf :
            tmpVal = curNode->getElemId();
	    if (curNode->isPCData()) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("#PCDATA", -1));
	    } else {
		id = curNode->getElemId();
		elem = validator.getElemDecl(id);
		Tcl_UniCharToUtfDString(elem->getBaseName(),
		    XMLString::stringLen(elem->getBaseName()), &workString);
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(Tcl_DStringValue(&workString), -1));
	    }
            break;

        case ContentSpecNode::ZeroOrOne :
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("(", 1));
	    }
	    Tcl_ListObjLength(interp, specObjPtr, &startPos);

            FormatNodes(specObjPtr, first, curType);
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(")", 1));
	    }


	    /* 
	     * trim parentheses 
	     */
	    Tcl_ListObjLength(interp, specObjPtr, &endPos);
	    if (startPos && endPos) {
		Tcl_ListObjIndex(interp, specObjPtr, startPos, &leftObj);
		Tcl_ListObjIndex(interp, specObjPtr, endPos-2, &rightObj);
		if ((strcmp("(", Tcl_GetString(leftObj)) == 0) &&
		    (strcmp(")", Tcl_GetString(rightObj)) == 0)) {
		    Tcl_ListObjReplace(interp, specObjPtr, endPos-2, 1, 0, 
			    NULL);
		    Tcl_ListObjReplace(interp, specObjPtr, startPos, 1, 0, 
			    NULL);
		}
	    }
	    /* 
	     * Replace the last close paren with ")?"
	     */
	    objv[0] = Tcl_NewStringObj(")?", 2);
	    Tcl_ListObjLength(interp, specObjPtr, &objc);
	    Tcl_ListObjReplace(interp, specObjPtr, objc-1, 1, 1, objv);
            break;

        case ContentSpecNode::ZeroOrMore :
	    // printf("ZeroOrMore self: %d parent: %d\n", curType, parentType);
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("(", 1));
	    }
	    Tcl_ListObjLength(interp, specObjPtr, &startPos);

	    
            FormatNodes(specObjPtr, first, curType);
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(")", 1));
	    } 

	    /* 
	     * trim parentheses 
	     */
	    Tcl_ListObjLength(interp, specObjPtr, &endPos);
	    if (startPos && endPos) {
		Tcl_ListObjIndex(interp, specObjPtr, startPos, &leftObj);
		Tcl_ListObjIndex(interp, specObjPtr, endPos-2, &rightObj);
		if ((strcmp("(", Tcl_GetString(leftObj)) == 0) &&
		    (strcmp(")", Tcl_GetString(rightObj)) == 0)) {
		    Tcl_ListObjReplace(interp, specObjPtr, endPos-2, 1, 0, 
			    NULL);
		    Tcl_ListObjReplace(interp, specObjPtr, startPos, 1, 0, 
			    NULL);
		}
	    }

	    /* 
	     * Replace the last close paren with ")*"
	     */
	    objv[0] = Tcl_NewStringObj(")*", 2);
	    Tcl_ListObjLength(interp, specObjPtr, &objc);
	    Tcl_ListObjReplace(interp, specObjPtr, objc-1, 1, 1, objv);

            break;

        case ContentSpecNode::OneOrMore :
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("(", 1));
	    }
	    Tcl_ListObjLength(interp, specObjPtr, &startPos);

            FormatNodes(specObjPtr, first, curType);
            if (doRepParens) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(")", 1));
	    } 


	    /* 
	     * trim parentheses 
	     */
	    Tcl_ListObjLength(interp, specObjPtr, &endPos);
	    if (startPos && endPos) {
		Tcl_ListObjIndex(interp, specObjPtr, startPos, &leftObj);
		Tcl_ListObjIndex(interp, specObjPtr, endPos-2, &rightObj);
		if ((strcmp("(", Tcl_GetString(leftObj)) == 0) &&
		    (strcmp(")", Tcl_GetString(rightObj)) == 0)) {
		    Tcl_ListObjReplace(interp, specObjPtr, endPos-2, 1, 0, 
			    NULL);
		    Tcl_ListObjReplace(interp, specObjPtr, startPos, 1, 0, 
			    NULL);
		}
	    }

	    /* 
	     * Replace the last close paren with ")+"
	     */
	    objv[0] = Tcl_NewStringObj(")+", 2);
	    Tcl_ListObjLength(interp, specObjPtr, &objc);
	    Tcl_ListObjReplace(interp, specObjPtr, objc-1, 1, 1, objv);
            break;

        case ContentSpecNode::Choice :
            if (parentType != curType) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("(", 1));
	    }
            FormatNodes(specObjPtr, first, curType);
	    Tcl_ListObjAppendElement(interp, specObjPtr, 
		    Tcl_NewStringObj("|", 1));
            FormatNodes(specObjPtr, second, curType);
            if (parentType != curType) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(")", 1));
	    }
            break;

        case ContentSpecNode::Sequence :
            if (parentType != curType) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj("(", 1));
	    }
            FormatNodes(specObjPtr, first, curType);
	    Tcl_ListObjAppendElement(interp, specObjPtr, 
		    Tcl_NewStringObj(",", 1));
            FormatNodes(specObjPtr, second, curType);
            if (parentType != curType) {
		Tcl_ListObjAppendElement(interp, specObjPtr, 
			Tcl_NewStringObj(")", 1));
	    }
            break;
    }
    return;
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::elementDecl --
 *
 *      Process an ELEMENT declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */
void
SAXTclDocTypeHandler::elementDecl(
    const DTDElementDecl& decl, 
    const bool isIgnored)
{
    int result;
    Tcl_DString nameString;
    Tcl_Obj *cmdObjPtr;
    Tcl_Obj *childrenObjPtr;
    const ContentSpecNode *contentSpec;
    const ContentSpecNode *firstSpec;

    ContentSpecNode::NodeTypes nodeType;

    if (parserPtr->elementDeclCommand) {
	cmdObjPtr = Tcl_DuplicateObj(parserPtr->elementDeclCommand);
	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);

	Tcl_DStringInit(&nameString);
	Tcl_UniCharToUtfDString(decl.getBaseName(),
	    XMLString::stringLen(decl.getBaseName()), &nameString);

	Tcl_ListObjAppendElement(interp, cmdObjPtr, 
          Tcl_NewStringObj(Tcl_DStringValue(&nameString), -1));

	switch (decl.getModelType()) {
	    case DTDElementDecl::Empty:
		Tcl_ListObjAppendElement(interp, cmdObjPtr, 
			Tcl_NewStringObj("EMPTY", -1));
		break;
	    case DTDElementDecl::Any:
		Tcl_ListObjAppendElement(interp, cmdObjPtr, 
			Tcl_NewStringObj("ANY", -1));
		break;
	    case DTDElementDecl::Mixed:
		contentSpec = decl.getContentSpec();
		childrenObjPtr = Tcl_NewListObj(0, NULL);
		Tcl_ListObjAppendElement(interp, childrenObjPtr, 
		    Tcl_NewStringObj("(", 1));
		while (contentSpec) {
		    nodeType = contentSpec->getType();
		    if (nodeType == ContentSpecNode::ZeroOrMore) {
			contentSpec = contentSpec->getFirst();
		    } else if (nodeType == ContentSpecNode::Choice) {
			firstSpec = contentSpec->getFirst();
			FormatNodes(childrenObjPtr, firstSpec, 
				(ContentSpecNode::NodeTypes) -1);
			contentSpec = contentSpec->getSecond();
		    } else {		    
			FormatNodes(childrenObjPtr, contentSpec, 
				(ContentSpecNode::NodeTypes) -1);
		    }
		    if (nodeType == ContentSpecNode::Leaf) break;
		}
		Tcl_ListObjAppendElement(interp, childrenObjPtr, 
		    Tcl_NewStringObj(")*", 2));
		Tcl_ListObjAppendElement(interp, cmdObjPtr, childrenObjPtr);
		break;
	    case DTDElementDecl::Children:
		contentSpec = decl.getContentSpec();
		childrenObjPtr = Tcl_NewListObj(0, NULL);
		FormatNodes(childrenObjPtr, contentSpec, 
			(ContentSpecNode::NodeTypes) -1);
		Tcl_ListObjAppendElement(interp, cmdObjPtr, childrenObjPtr);
		break;
	    case DTDElementDecl::ModelTypes_Count:
		break;
	    default:
		break;
	}

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
	if (result == TCL_OK) {
	} else if (result == TCL_CONTINUE) {
	} else {
	    Tcl_DString errorString;
	    Tcl_DStringInit(&errorString);
	    Tcl_UtfToUniCharDString(Tcl_GetStringResult(interp), -1, 
		&errorString);
	   
#if !defined(SOLARIS) && !defined(HP_UX)
	    SAXParseException toThrow = SAXParseException(
		(const XMLCh *) Tcl_DStringValue(&errorString),
		(const XMLCh *) NULL,
		(const XMLCh *) NULL,
		parserPtr->currentLine,
		parserPtr->currentColumn
	    );
	    Tcl_DStringFree(&errorString);
	    throw toThrow;
#else
	    Tcl_DStringAppend(&parserPtr->parseErrors,
		    Tcl_GetStringResult(interp), -1);
	    Tcl_DStringAppendElement(&parserPtr->parseErrorsAndWarnings,
		    Tcl_GetStringResult(interp));
#endif
	}
    } 
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::endIntSubset --
 *
 *      Process the end of the internal subset.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::endIntSubset()
{
     XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::endExtSubset --
 *
 *      Process the end of the external subset.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::endExtSubset()
{
    XMLScanner& scanner
	= (class XMLScanner &) parserPtr->saxParserPtr->getScanner();

    
    parserPtr->previousLine = parserPtr->currentLine;
    parserPtr->previousColumn = parserPtr->currentColumn;

    scanner.getLastExtLocation(parserPtr->currentSysId, ID_SIZE, 
	    parserPtr->currentPubId, ID_SIZE, 
	    parserPtr->currentLine,  
	    parserPtr->currentColumn);

}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::entityDecl --
 *
 *      Process an ENTITY declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::entityDecl(
    const DTDEntityDecl& entityDecl,
    const bool isPEDecl,
    const bool isIgnored)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::resetDocType --
 *
 *      Process an <TBD>
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::resetDocType()
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::notationDecl --
 *
 *      Process a NOTATION declaration.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::notationDecl(
    const XMLNotationDecl& notDecl,
    const bool isIgnored)
{
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::startAttList --
 *
 *      Process the start of an Attribute list
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::startAttList(
    const DTDElementDecl& elemDecl)
{
    if (parserPtr->attlistDeclCommand) {
	if (attrListPtr) {
	    Tcl_DecrRefCount(attrListPtr);
	    attrListPtr = NULL;
	}
	attrListPtr = Tcl_NewListObj(0, NULL);
	Tcl_IncrRefCount(attrListPtr);
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::attDef --
 *
 *      Process an attribute definition.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::attDef(
    const DTDElementDecl& elemDecl,
    const DTDAttDef& attDef,
    const bool ignoring)
{
    if (parserPtr->attlistDeclCommand) {
	Tcl_DString attrNameString;
	Tcl_DStringInit(&attrNameString);
	Tcl_UniCharToUtfDString(attDef.getFullName(), 
		XMLString::stringLen(attDef.getFullName()), 
		&attrNameString);
	Tcl_ListObjAppendElement(interp, attrListPtr, 
		Tcl_NewStringObj(Tcl_DStringValue(&attrNameString), -1));
	Tcl_DStringFree(&attrNameString);	
    }
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::endAttList --
 *
 *      Process the end of an attribute list.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::endAttList(
    const DTDElementDecl& elemDecl)
{
    int result;
    Tcl_DString nameString;
    Tcl_Obj *cmdObjPtr;

    if (parserPtr->attlistDeclCommand) {
	cmdObjPtr = Tcl_DuplicateObj(parserPtr->attlistDeclCommand);
	Tcl_IncrRefCount(cmdObjPtr);
	Tcl_Preserve(interp);

	Tcl_DStringInit(&nameString);
	Tcl_UniCharToUtfDString(elemDecl.getBaseName(),
	    XMLString::stringLen(elemDecl.getBaseName()), &nameString);

	Tcl_ListObjAppendElement(interp, cmdObjPtr, 
	    Tcl_NewStringObj(Tcl_DStringValue(&nameString), -1));
	Tcl_ListObjAppendElement(interp, cmdObjPtr, attrListPtr);

	result = Tcl_EvalObjEx(interp, cmdObjPtr, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(cmdObjPtr);
	Tcl_Release((ClientData) interp);
	if (result == TCL_OK) {
	} else if (result == TCL_CONTINUE) {
	} else {
	    Tcl_DString errorString;
	    Tcl_DStringInit(&errorString);
	    Tcl_UtfToUniCharDString(Tcl_GetStringResult(interp), -1, 
		&errorString);
	   
#if !defined(SOLARIS) && !defined(HP_UX)
	    SAXParseException toThrow = SAXParseException(
		(const XMLCh *) Tcl_DStringValue(&errorString),
		(const XMLCh *) NULL,
		(const XMLCh *) NULL,
		parserPtr->currentLine,
		parserPtr->currentColumn
	    );
	    Tcl_DStringFree(&errorString);
	    throw toThrow;
#else
	    Tcl_DStringAppend(&parserPtr->parseErrors,
		    Tcl_GetStringResult(interp), -1);
	    Tcl_DStringAppendElement(&parserPtr->parseErrorsAndWarnings,
		    Tcl_GetStringResult(interp));
#endif
	}
    } 
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::startIntSubset --
 *
 *      Process the start of the internal subset.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::startIntSubset()
{
    // printf("startIntSubset\n");
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::startExtSubset --
 *
 *      Process the start of the external subset.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::startExtSubset()
{
    // printf("startExtSubset\n");
}


/*
 *--------------------------------------------------------------
 *
 * SAXTclDocTypeHandler::textDecl --
 *
 *      Process a Text declaration in a DOCTYPE.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
SAXTclDocTypeHandler::TextDecl(
    const XMLCh* const versionStr,
    const XMLCh* const encodingStr)
{
}


TclMemoryBuffer::TclMemoryBuffer(   const   XMLByte* const  srcDocBytes
                                        , const unsigned int    byteCount
                                        , const XMLCh* const    bufId
                                        , const bool            adoptBuffer) :
    InputSource(bufId)
   
{
    fSrcBytes = srcDocBytes;
    fCopyBufToStream = true;
    fByteCount = byteCount;
    fAdopted = adoptBuffer;
}

TclMemoryBuffer::TclMemoryBuffer(   const   XMLByte* const  srcDocBytes
                                        , const unsigned int    byteCount
                                        , const char* const     bufId
                                        , const bool            adoptBuffer) :
    InputSource(bufId)
{
    fSrcBytes = srcDocBytes;
    fCopyBufToStream = true;
    fByteCount = byteCount;
    fAdopted = adoptBuffer;
    fStream = new BinMemInputStream
    (
        fSrcBytes
        , fByteCount
        , fCopyBufToStream ? BinMemInputStream::BufOpt_Copy
                           : BinMemInputStream::BufOpt_Reference
    );
}

BinInputStream* TclMemoryBuffer::makeStream() const
{
    //
    //  Create a memory input stream over our buffer. According to our
    //  fCopyBufToStream flag, we either tell it to copy the buffer or to
    //  just reference it.
    //
    return fStream;
}

unsigned int TclMemoryBuffer::curPos()
{
    return fStream->curPos();
}
